/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views;

import java.util.Iterator;
import java.util.Set;

import de.uni_paderborn.fujaba.asg.ASGElement;
import de.uni_paderborn.fujaba.uml.*;
import de.upb.tools.fca.FEmptyIterator;
import de.upb.tools.fca.FHashSet;


/**
 * UMLDiagram that is a View of another Diagram. <p>
 *
 * It consists of several ViewDefinitions and an include-list defining the actual view. <h2>
 * Associations</h2> <pre>
 *              0..n    diagram    0..1
 * ViewDiagram ------------------------- UMLDiagram
 *              views           diagram
 *
 *              0..n       additionalItems        0..n
 * ViewDiagram ---------------------------------------- UMLDiagramItem
 *              revAdditionalItems     additionalItems
 *
 *              0..1    viewDefinitions    0..n
 * ViewDiagram --------------------------------- ViewDefinition
 *              diagram         viewDefinitions
 * </pre>
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.27 $
 * @see       de.uni_paderborn.fujaba.views.ViewDefinition
 */
public class ViewDiagram extends UMLDiagram
{
   /**
    * Constructor for class ViewDiagram
    */
   public ViewDiagram()
   {
      super();
   }


   /**
    * Constructor for class ViewDiagram
    *
    * @param name     No description provided
    * @param project  No description provided
    */
   public ViewDiagram (String name, UMLProject project)
   {
      super (name, project);
   }


   /**
    * <pre>
    *              0..n    diagram    0..1
    * ViewDiagram ------------------------- UMLDiagram
    *              views           diagram
    * </pre>
    */
   private UMLDiagram diagram;


   /**
    * Sets the diagram attribute of the ViewDiagram object
    *
    * @param value  The new diagram value
    * @return       No description provided
    */
   public boolean setDiagram (UMLDiagram value)
   {
      if (this.diagram != value)
      {
         if (this.diagram != null)
         {
            UMLDiagram oldValue = this.diagram;
            this.diagram = null;
            removeAllFromViewDefinitions();
            oldValue.removeFromViews (this);
         }
         this.diagram = value;

         removeAllFromElements();
         removeAllFromAdditionalItems();

         if (value != null)
         {
            this.diagram.addToViews (this);
            update();
         }
         return true;
      }

      return false;
   }


   /**
    * Get the diagram attribute of the ViewDiagram object
    *
    * @return   The diagram value
    */
   public UMLDiagram getDiagram()
   {
      return this.diagram;
   }


   /**
    * Get the originalDiagram attribute of the ViewDiagram object
    *
    * @return   The originalDiagram value
    */
   public UMLDiagram getOriginalDiagram()
   {
      UMLDiagram result = this;

      while (result != null && result instanceof ViewDiagram)
      {
         result =  ((ViewDiagram) result).getDiagram();
      }
      return result;
   }


   /**
    * <pre>
    *              0..n       additionalItems        0..n
    * ViewDiagram ---------------------------------------- UMLDiagramItem
    *              revAdditionalItems     additionalItems
    * </pre>
    */
   private FHashSet additionalItems;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToAdditionalItems (UMLDiagramItem value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.additionalItems == null)
         {
            this.additionalItems = new FHashSet();
         }
         changed = this.additionalItems.add (value);
         if (changed)
         {
            value.addToRevAdditionalItems (this);
            this.addToElements (value);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInAdditionalItems (UMLDiagramItem value)
   {
      return  ( (this.additionalItems != null) &&
          (value != null) &&
         this.additionalItems.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfAdditionalItems()
   {
      return  ( (this.additionalItems == null)
         ? FEmptyIterator.get()
         : this.additionalItems.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfAdditionalItems()
   {
      return  ( (this.additionalItems == null)
         ? 0
         : this.additionalItems.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromAdditionalItems (UMLDiagramItem value)
   {
      boolean changed = false;

      if ( (this.additionalItems != null) &&  (value != null))
      {
         changed = this.additionalItems.remove (value);
         if (changed)
         {
            value.removeFromRevAdditionalItems (this);
            checkAndRemoveFromItems (value, null);
            if (value instanceof UMLClass)
            {
               Iterator iter = isolateUmlClass ((UMLClass) value);
               while (iter.hasNext())
               {
                  removeFromAdditionalItems ((UMLDiagramItem) iter.next());
               }
            }
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromAdditionalItems()
   {
      UMLDiagramItem tmpValue;
      Iterator iter = this.iteratorOfAdditionalItems();

      while (iter.hasNext())
      {
         tmpValue = (UMLDiagramItem) iter.next();
         this.removeFromAdditionalItems (tmpValue);
      }
   }



   /**
    * <pre>
    *              0..1    viewDefinitions    0..n
    * ViewDiagram --------------------------------- ViewDefinition
    *              diagram         viewDefinitions
    * </pre>
    */
   private FHashSet viewDefinitions;


   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    * @return       No description provided
    */
   public boolean addToViewDefinitions (ViewDefinition value)
   {
      boolean changed = false;

      if (value != null)
      {
         if (this.viewDefinitions == null)
         {
            this.viewDefinitions = new FHashSet();
         }
         changed = this.viewDefinitions.add (value);
         if (changed)
         {
            // update of view contents is done in ViewDefinition
            value.setDiagram (this);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean hasInViewDefinitions (ViewDefinition value)
   {
      return  ( (this.viewDefinitions != null) &&
          (value != null) &&
         this.viewDefinitions.contains (value));
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfViewDefinitions()
   {
      return  ( (this.viewDefinitions == null)
         ? FEmptyIterator.get()
         : this.viewDefinitions.iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public int sizeOfViewDefinitions()
   {
      return  ( (this.viewDefinitions == null)
         ? 0
         : this.viewDefinitions.size());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param value  No description provided
    * @return       No description provided
    */
   public boolean removeFromViewDefinitions (ViewDefinition value)
   {
      boolean changed = false;

      if ( (this.viewDefinitions != null) &&  (value != null))
      {
         changed = this.viewDefinitions.remove (value);
         if (changed)
         {
            //Note: If somebody should change this back to
            //setDiagram(null) please take care of the startItems.
            //They do not have a link to their ViewDefinitions, so
            //item.removeYou() will not fully isolate it, if the
            //association of this ViewDiagram to its diagram or to
            //the ViewDefinition is lost!

            // update of view contents is done in ViewDefinition
            value.removeYou(); //setDiagram (null);

         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromViewDefinitions()
   {
      ViewDefinition tmpValue;
      Iterator iter = this.iteratorOfViewDefinitions();

      while (iter.hasNext())
      {
         tmpValue = (ViewDefinition) iter.next();
         this.removeFromViewDefinitions (tmpValue);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void flushAdditionalItems()
   {
      removeAllFromAdditionalItems();
      update();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void update()
   {
      update (false);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param force  No description provided
    */
   public void update (boolean force)
   {
      Iterator viewDefIter = iteratorOfViewDefinitions();
      Set contextItems = new ItemSet();
      Set tmpSet = null;

      if (additionalItems != null)
      {
         contextItems.addAll (additionalItems);
      }

      while (viewDefIter.hasNext())
      {
         ViewDefinition viewDef = (ViewDefinition) viewDefIter.next();
         if (force || viewDef.isUpdate())
         {
            tmpSet = viewDef.evaluate();
         }
         else
         {
            tmpSet = viewDef.getItems();
         }

         if (tmpSet != null)
         {
            contextItems.addAll (tmpSet);
         }
      }

      Iterator itemIter = iteratorOfElements();
      while (itemIter.hasNext())
      {
         ASGElement asgElement = (ASGElement) itemIter.next();
         if (contextItems.contains (asgElement))
         {
            contextItems.remove (asgElement);
         }
         else if (asgElement instanceof UMLDiagramItem)
         {
            removeFromItemsWithoutIncrementAge ((UMLDiagramItem) asgElement);
         } // end of else
      }
   }


   /**
    * Access method for an one to n association.
    *
    * @param item  The object added.
    * @return      No description provided
    */
   public boolean addToElements (UMLDiagramItem item)
   {
      boolean changed = false;
      if (item != null && !hasInElements (item))
      {
         super.addToElements (item);

         if ( (getDiagram() == null || !getDiagram().hasInElements (item)) && !UMLProject.isLoading())
         {
            addToAdditionalItems (item);
         }
         item.addToDiagrams (getDiagram());

         if (getOriginalDiagram() != null)
         {
            item.addToDiagrams (this.getOriginalDiagram());
         } // if
      }
      return changed;
   }


   /**
    * This is needed for parsing XML-files
    *
    * @param item  The object added.
    * @param diag  The object added.
    * @return      No description provided
    */
   public boolean addToItems (UMLDiagramItem item, UMLDiagram diag)
   {
      boolean changed = false;
      if (item != null && !hasInElements (item))
      {
         super.addToElements (item);

         if ( (getDiagram() == null || !diag.hasInElements (item)) && !UMLProject.isLoading())
         {
            addToAdditionalItems (item);
         }

         if (diag != null)
         {
            item.addToDiagrams (diag);
         } // if
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item  No description provided
    * @return      No description provided
    */
   public boolean removeFromElements (UMLDiagramItem item)
   {
      return removeFromItemsWithoutIncrementAge (item);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item  No description provided
    * @return      No description provided
    */
   public boolean removeFromItemsWithoutIncrementAge (UMLDiagramItem item)
   {
      boolean changed = removeFromItemsNoParentSideEffects (item);
      if (changed)
      {
         item.removeFromDiagrams (getDiagram());
         Iterator viewIter = iteratorOfViewDefinitions();
         while (viewIter.hasNext())
         {
            ViewDefinition viewDef = (ViewDefinition) viewIter.next();
            viewDef.removeFromStartItems (item);
            viewDef.removeFromItems (item);
         }
         update();
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item  No description provided
    * @return      No description provided
    */
   public boolean removeFromItemsNoParentSideEffects (UMLDiagramItem item)
   {
      boolean changed = false;
      if (item != null && hasInElements (item))
      {
         super.removeFromElementsWithSideEffects (item);
         removeFromAdditionalItems (item);

         if (item instanceof UMLClass)
         {
            Iterator iter = isolateUmlClass ((UMLClass) item);
            while (iter.hasNext())
            {
               removeFromItemsNoParentSideEffects ((UMLDiagramItem) iter.next());
            }
         }
         changed = true;
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromElements()
   {
      UMLDiagramItem item;
      Iterator iter = iteratorOfElements();
      while (iter.hasNext())
      {
         item = (UMLDiagramItem) iter.next();
         removeFromItemsNoParentSideEffects (item);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item       No description provided
    * @param ignoreDef  No description provided
    * @return           No description provided
    */
   protected boolean checkAndRemoveFromItems (UMLDiagramItem item, ViewDefinition ignoreDef)
   {
      boolean changed = false;
      if (item != null && hasInElements (item))
      {
         boolean found = false;
         Iterator defIter = iteratorOfViewDefinitions();
         while (defIter.hasNext() && !found)
         {
            ViewDefinition tmpDef = (ViewDefinition) defIter.next();
            if (tmpDef != ignoreDef && tmpDef.hasInItems (item))
            {
               found = true;
            }
         }
         if (!found)
         {
            changed = removeFromItemsNoParentSideEffects (item);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param items      No description provided
    * @param ignoreDef  No description provided
    * @return           No description provided
    */
   protected boolean checkAndRemoveFromItems (Set items, ViewDefinition ignoreDef)
   {
      boolean changed = false;
      if (items != null && items.size() > 0 && sizeOfElements() > 0)
      {
         Set tmpItems = null;
         Iterator viewIter = iteratorOfViewDefinitions();
         while (viewIter.hasNext() && items.size() > 0)
         {
            ViewDefinition tmpDef = (ViewDefinition) viewIter.next();
            if (tmpDef != ignoreDef)
            {
               tmpItems = tmpDef.getItems();
               if (tmpItems != null)
               {
                  items.removeAll (tmpItems);
               }
            }
         }
         Iterator itemIter = items.iterator();
         while (itemIter.hasNext())
         {
            UMLDiagramItem delItem = (UMLDiagramItem) itemIter.next();
            changed = changed | removeFromItemsNoParentSideEffects (delItem);
         }
      }
      return changed;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeYou()
   {
      setDiagram (null);
      removeAllFromViewDefinitions();
      super.removeYou();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public String toString()
   {
      return getName();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item  No description provided
    * @return      No description provided
    */
   private Iterator isolateUmlClass (UMLClass item)
   {
      Set removeItems = new ItemSet();
      Iterator iter;

      //----- remove all assocs
      UMLAssoc assoc;
      UMLRole role;
      iter = item.iteratorOfRoles();
      while (iter.hasNext())
      {
         role = (UMLRole) iter.next();
         assoc = role.getAssoc();
         removeItems.add (assoc);
      }

      //----- remove all inheritance
      UMLGeneralization gen;
      iter = item.iteratorOfRevSubclass();
      while (iter.hasNext())
      {
         gen = (UMLGeneralization) iter.next();
         removeItems.add (gen);
      }
      iter = item.iteratorOfRevSuperclass();
      while (iter.hasNext())
      {
         gen = (UMLGeneralization) iter.next();
         removeItems.add (gen);
      }

      return removeItems.iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void delete()
   {
      this.removeYou();
   }
}

/*
 * $Log: ViewDiagram.java,v $
 * Revision 1.27  2004/10/20 17:50:31  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
