/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) 1997-2004 Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact adress:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.uni_paderborn.fujaba.views.gui;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;

import javax.swing.*;
import javax.swing.border.BevelBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileFilter;

import de.uni_paderborn.fujaba.gui.comp.FujabaDialogPanel;
import de.uni_paderborn.fujaba.preferences.GeneralPreferences;
import de.uni_paderborn.fujaba.views.*;


/**
 * Part of the FilterEditorDialog. This is the Panel that holds the list of avaliable filters
 * and a info field
 *
 * @author    $Author: schneider $
 * @version   $Revision: 1.10 $
 */
public class FilterInfoPanel extends FujabaDialogPanel implements ActionListener,
   ListSelectionListener
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JList filterList = null;
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private JEditorPane descriptionTextPane = null;


   /**
    * Constructor for class FilterInfoPanel
    *
    * @param hostingDialog  No description provided
    */
   public FilterInfoPanel (FilterEditorDialog hostingDialog)
   {
      super();

      setHostingDialog (hostingDialog);
      guiBuild();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void guiBuild()
   {
      setLayout (new GridBagLayout());

      GridBagConstraints constraints = new GridBagConstraints();
      constraints.gridx = 0;
      constraints.gridy = 0;
      constraints.gridwidth = 2;
      constraints.gridheight = 2;
      constraints.weightx = 0.5;
      constraints.weighty = 1.0;
      constraints.anchor = GridBagConstraints.CENTER;
      constraints.fill = GridBagConstraints.BOTH;

      JPanel filterPanel = new JPanel();
      filterPanel.setBorder (new TitledBorder ("Available Filters"));
      filterPanel.setLayout (new GridLayout (1, 1, 0, 0));

      JScrollPane filterScrollPane = new JScrollPane();
      filterScrollPane.setBorder (new BevelBorder (BevelBorder.LOWERED));
      filterList = new JList();
      filterList.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
      filterList.addListSelectionListener (this);
      filterScrollPane.getViewport().setLayout (new GridLayout (1, 1, 0, 0));
      filterScrollPane.getViewport().setView (filterList);

      filterPanel.add (filterScrollPane);

      add (filterPanel, constraints);

      JPanel descriptionPanel = new JPanel();
      descriptionPanel.setBorder (new TitledBorder ("Description"));
      descriptionPanel.setLayout (new GridLayout (1, 1, 0, 0));

      JScrollPane descriptionScrollPane = new JScrollPane();
      descriptionScrollPane.setBorder (new BevelBorder (BevelBorder.LOWERED));
      descriptionTextPane = new JEditorPane();
      descriptionTextPane.setContentType ("text/html");
      descriptionTextPane.setText ("<i>Select a Filter to see information about it</i>");
      descriptionTextPane.setEditable (false);
      descriptionScrollPane.getViewport().setLayout (new GridLayout (1, 1, 0, 0));
      descriptionScrollPane.getViewport().setView (descriptionTextPane);

      descriptionPanel.add (descriptionScrollPane);

      constraints.gridx = 3;
      add (descriptionPanel, constraints);

      JPanel filterButtonPanel = new JPanel();
      filterButtonPanel.setLayout (new GridLayout (2, 1, 0, 0));

      JButton importButton = new JButton();
      importButton.setText ("Import");
      importButton.setActionCommand ("import");
      importButton.addActionListener (this);
      filterButtonPanel.add (importButton);

      JButton removeButton = new JButton();
      removeButton.setText ("Remove");
      removeButton.setActionCommand ("remove");
      removeButton.addActionListener (this);
      filterButtonPanel.add (removeButton);

      constraints.gridx = 2;
      constraints.gridy = 1;
      constraints.gridwidth = 1;
      constraints.gridheight = 1;
      constraints.weightx = 0.0;
      constraints.weighty = 0.0;
      constraints.fill = GridBagConstraints.NONE;
      add (filterButtonPanel, constraints);

      setName ("Available Filters");
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FilterEditorDialog hostingDialog = null;


   /**
    * The FilterEditorDialog this panel is nested in
    *
    * @param hostingDialog  The new hostingDialog value
    */
   protected void setHostingDialog (FilterEditorDialog hostingDialog)
   {
      this.hostingDialog = hostingDialog;
   }


   /**
    * The FilterEditorDialog this panel is nested in
    *
    * @return   The hostingDialog value
    */
   public FilterEditorDialog getHostingDialog()
   {
      return hostingDialog;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void refreshDialog()
   {
      if (getHostingDialog() != null)
      {
         getHostingDialog().setCurrentFilter ((Filter) filterList.getSelectedValue());
      }
      unparse();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void unparse()
   {
      Filter filter = null;
      if (getHostingDialog() != null)
      {
         filter = getHostingDialog().getCurrentFilter();
      }
      if (filter != null)
      {
         filterList.setSelectedValue (filter, true);
      }
      else
      {
         filterList.clearSelection();
      }

      if (filter != null)
      {
         String desc = filter.getDescription();
         if (desc == null || desc.length() == 0)
         {
            desc = "<i>No description available</i>";
         }

         String className = filter.getClass().getName();
         int lastDot = className.lastIndexOf (".");
         if (lastDot > -1)
         {
            className = className.substring (lastDot + 1);
         }
         className = className.replace ('$', '.');

         String title = "<h3>" + filter.getName() + "<br>" +
            "<font size=-2>(" + className + ")</font></h3>";

         descriptionTextPane.setText (title + desc);
      }
      else
      {
         descriptionTextPane.setText ("<i>Select a Filter to see<br>information about it</i>");
      }
   }


   /**
    * Sets the model attribute of the FilterInfoPanel object
    *
    * @param model  The new model value
    */
   public void setModel (ComboBoxModel model)
   {
      filterList.setModel (model);
   }


   /**
    * Get the model attribute of the FilterInfoPanel object
    *
    * @return   The model value
    */
   public ComboBoxModel getModel()
   {
      return (ComboBoxModel) filterList.getModel();
   }


   /**
    * Sets the renderer attribute of the FilterInfoPanel object
    *
    * @param renderer  The new renderer value
    */
   public void setRenderer (ListCellRenderer renderer)
   {
      filterList.setCellRenderer (renderer);
   }


   /**
    * Get the renderer attribute of the FilterInfoPanel object
    *
    * @return   The renderer value
    */
   public ListCellRenderer getRenderer()
   {
      return filterList.getCellRenderer();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void actionPerformed (ActionEvent e)
   {
      String command = e.getActionCommand();
      if (command.equals ("import"))
      {
         importAction();
      }
      else if (command.equals ("remove"))
      {
         removeAction();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private boolean removeAction()
   {
      Filter filter = (Filter) filterList.getSelectedValue();
      if (filter == null)
      {
         return false;
      }

      int result = JOptionPane.showConfirmDialog (this,
         "Removing a Filter results in removing all\n" +
         "the associated files!\n\nContinue?", "Remove",
         JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);
      if (result == JOptionPane.NO_OPTION)
      {
         return false;
      }

      FilterManager.get().removeFromFilters (filter);
      if (filter instanceof ConfigurableFilter)
      {
         File settingsFile = FilterManager.get().
            getFromSettingsFiles ((ConfigurableFilter) filter);

         if (settingsFile != null && settingsFile.exists())
         {
            settingsFile.delete();
         }

         FilterManager.get().removeKeyFromSettingsFiles ((ConfigurableFilter) filter);
      }
      else
      {
         String name = filter.getClass().getName();
         String pack = null;
         int lastDot = name.lastIndexOf (".");
         if (lastDot > -1)
         {
            pack = name.substring (0, lastDot);
            name = name.substring (lastDot + 1);
         }
         else
         {
            pack = "";
         }

         name = name.replace ('.', File.pathSeparatorChar);
         File file = new File (GeneralPreferences.get().getViewFilterFolder(), name + ".class");
         if (file.exists() && !file.isDirectory())
         {
            file.delete();
         }
         file = new File (file, pack + File.pathSeparatorChar + name + ".class");
         if (file.exists() && file.isDirectory())
         {
            file.delete();
         }
      }
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private boolean importAction()
   {
      File target = new File (GeneralPreferences.get().getViewFilterFolder());
      if (!checkDir (target))
      {
         return false;
      }

      JFileChooser fileChooser = new JFileChooser (GeneralPreferences.get().getWorkspaceFolder());
      fileChooser.setFileFilter (
         new FileFilter()
         {
            public boolean accept (File file)
            {
               String name = file.getName().toLowerCase();
               return  (file.isDirectory() ||
                  name.endsWith ("." + FilterManager.SETTINGS_FILE_EXT) ||
                  name.endsWith (".jar") ||
                  name.endsWith (".zip") ||
                  name.endsWith (".class"));
            }


            public String getDescription()
            {
               return "Filter Files (*." + FilterManager.SETTINGS_FILE_EXT + ", *.zip, *.jar, *.class)";
            }
         }
         );

      fileChooser.setMultiSelectionEnabled (true);
      int result = fileChooser.showOpenDialog (this);

      if (result == JFileChooser.CANCEL_OPTION)
      {
         return false;
      }

      File selection = fileChooser.getSelectedFile();
      if (selection.isDirectory() || !selection.exists())
      {
         return false;
      }

      File targetFile = new File (target, selection.getName());
      if (targetFile.exists())
      {
         result = JOptionPane.showConfirmDialog (this,
            "File " + targetFile.getName() + " already exists in target directory!\n\nOverwrite it?", "Import",
            JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE);

         if (result == JOptionPane.NO_OPTION)
         {
            return false;
         }
      }
      if (!copyFile (selection, targetFile))
      {
         JOptionPane.showMessageDialog (this,
            "Unable to copy File " + targetFile.getName(), "Import",
            JOptionPane.ERROR_MESSAGE);
      }

      FilterManager.get().rescanDir();
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param source  No description provided
    * @param target  No description provided
    * @return        No description provided
    */
   private boolean copyFile (File source, File target)
   {
      if (!source.canRead())
      {
         return false;
      }

      BufferedInputStream in = null;
      BufferedOutputStream out = null;

      Cursor cursor = this.getCursor();
      this.setCursor (Cursor.getPredefinedCursor (Cursor.WAIT_CURSOR));

      try
      {
         if (!target.exists())
         {
            target.createNewFile();
         }

         if (!target.canWrite())
         {
            target.delete();
            return false;
         }

         in = new BufferedInputStream (new FileInputStream (source));
         out = new BufferedOutputStream (new FileOutputStream (target));
         int lastRead = 0;
         byte[] buffer = new byte[131072];
         while (lastRead > -1)
         {
            lastRead = in.read (buffer, 0, 131072);
            if (lastRead > -1)
            {
               out.write (buffer, 0, lastRead);
            }
         }
      }
      catch (IOException e)
      {
         target.delete();
         return false;
      }
      finally
      {
         try
         {
            if (in != null)
            {
               in.close();
            }

            if (out != null)
            {
               out.flush();
               out.close();
            }
         }
         catch (IOException e)
         {
         }
         this.setCursor (cursor);
      }
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param target  No description provided
    * @return        No description provided
    */
   private boolean checkDir (File target)
   {
      if (!target.exists())
      {
         JOptionPane.showMessageDialog (this,
            "Target Directory \n   " + target.getAbsolutePath() + "\ndoes not exist!\n\nPlease create it!", "Import",
            JOptionPane.ERROR_MESSAGE);
         return false;
      }
      if (!target.isDirectory())
      {
         JOptionPane.showMessageDialog (this,
            "Target Directory \n   " + target.getAbsolutePath() + "\nis no Directory!\n\nPlease change it!", "Import",
            JOptionPane.ERROR_MESSAGE);
         return false;
      }
      return true;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param e  No description provided
    */
   public void valueChanged (ListSelectionEvent e)
   {
      if (! (e.getSource() == filterList))
      {
         return;
      }
      refreshDialog();
   }
}

/*
 * $Log: FilterInfoPanel.java,v $
 * Revision 1.10  2004/10/20 17:50:33  schneider
 * Introduction of interfaces for class diagram classes
 *
 */
