/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.plugins;

import java.io.*;
import java.net.*;
import java.util.*;

import javax.swing.*;

import org.apache.log4j.Logger;
import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.uni_paderborn.lib.java.io.*;
import de.upb.lib.userinterface.UserInterfaceManager;
import de.upb.tools.fca.*;

import sun.misc.*;



/**
 * Manages all plugins of an application. This manager class is realized as singleton design
 * pattern.
 *
 * @author    student research group Reddmom
 * @version   $Revision: 1.68 $
 */
public class PluginManager
{
   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final String STABLE_MENU = "stable";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final String PLUGIN_FILE = "plugin";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private final String NEW_PLUGIN_FILE = "fujabaPlugin";

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int DEBUG_OFF = 0;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public final static int DEBUG_ON = 1;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private int debugLevel = DEBUG_ON;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private KernelInterface kernelInterface;

   /**
    * The role attribute for the properties association between PluginManager and PluginProperty.
    * <pre>
    *               +------+ 0..1   properties   0..1
    * PluginManager | key  |-------------------------- PluginProperty
    *               +------+ manager       properties
    * </pre>
    *
    * @see   #hasInProperties
    * @see   #hasKeyInProperties
    * @see   #iteratorOfProperties
    * @see   #keysOfProperties
    * @see   #entriesOfProperties
    * @see   #sizeOfProperties
    * @see   #getFromProperties
    * @see   #addToProperties
    * @see   #removeFromProperties
    * @see   #removeKeyFromProperties
    * @see   #removeAllFromProperties
    */
   private FHashMap properties;

   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private static FHashMap pluginManagerInstances = new FHashMap();


   /**
    * Constructor for class PluginManager
    */
   private PluginManager()
   {
      kernelInterface = null;
   }


   /**
    * Constructor for class PluginManager
    *
    * @param kernelInterface  No description provided
    */
   private PluginManager (KernelInterface kernelInterface)
   {
      this.kernelInterface = kernelInterface;
   }


   /**
    * Get the instance attribute of the PluginManager class
    *
    * @param theManagerID  No description provided
    * @return              The instance value
    */
   public static synchronized PluginManager getInstance (KernelInterface theManagerID)
   {
      PluginManager manager = null;
      if (pluginManagerInstances.containsKey (theManagerID))
      {
         manager = (PluginManager) pluginManagerInstances.get (theManagerID);
      }
      else
      {
         manager = new PluginManager (theManagerID);
         pluginManagerInstances.put (theManagerID, manager);
      }
      return manager;
   }


   /**
    * access method to get the information if the given value is element in the set of properties.
    *
    * @param value  the property which is checked.
    * @return       true if the given value is member of the set.
    * @see          #properties
    */
   public boolean hasInProperties (PluginProperty value)
   {
      return  ( (this.properties != null) &&
          (value != null) &&  (value.getPluginID() != null) &&
          (this.properties.get (value.getPluginID()) == value));
   }


   /**
    * access method to get the information if the given key is already in the set of properties.
    *
    * @param key  the key which is checked
    * @return     true if the key is in the set of properties
    * @see        #properties
    */
   public boolean hasKeyInProperties (String key)
   {
      return  ( (this.properties != null) &&
          (key != null) &&
         this.properties.containsKey (key));
   }


   /**
    * access method to get an Iterator over all elements in the set of properties.
    *
    * @return   always an Iterator, if the set of properties is empty the Iterator is an FEmptyIterator
    *      instance
    * @see      #properties
    */
   public Iterator iteratorOfProperties()
   {
      return  ( (this.properties == null)
         ? FEmptyIterator.get()
         : this.properties.values().iterator());
   }


   /**
    * access method to get an Iterator over all keys in the set of properties.
    *
    * @return   always an Iterator
    * @see      #properties
    */
   public Iterator keysOfProperties()
   {
      return  ( (this.properties == null)
         ? FEmptyIterator.get()
         : this.properties.keySet().iterator());
   }


   /**
    * access method to get an Iterator over all elements in the set of properties.
    *
    * @return   always an Iterator.
    * @see      #properties
    */
   public Iterator entriesOfProperties()
   {
      return  ( (this.properties == null)
         ? FEmptyIterator.get()
         : this.properties.entrySet().iterator());
   }


   /**
    * access method to get the count of elements in the set of properties.
    *
    * @return   the count of elements
    * @see      #properties
    */
   public int sizeOfProperties()
   {
      return  ( (this.properties == null)
         ? 0
         : this.properties.size());
   }


   /**
    * access method to get the property specified by the key.
    *
    * @param key  the specified key
    * @return     if exists the property specified by the key, null if the property doesn't
    *      exist.
    * @see        #properties
    */
   public PluginProperty getFromProperties (String key)
   {
      return  ( ( (this.properties == null) ||  (key == null))
         ? null
         : (PluginProperty) this.properties.get (key));
   }


   /**
    * access method to add a new property to the set of properties.
    *
    * @param value  the property
    * @return       true if the property was added.
    * @see          #properties
    */
   public boolean addToProperties (PluginProperty value)
   {
      boolean changed = false;
      if ( (value != null) &&  (value.getPluginID() != null))
      {
         if (this.properties == null)
         {
            this.properties = new FHashMap(); // or FTreeMap ()
         }
         PluginProperty oldValue = (PluginProperty) this.properties.put (value.getPluginID(), value);
         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setManager (null);
            }
            value.setManager (this);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * access method to remove the specified property from the list of properties.
    *
    * @param value  the property
    * @return       true if the property was deleted from the list
    * @see          #properties
    */
   public boolean removeFromProperties (PluginProperty value)
   {
      boolean changed = false;
      if ( (this.properties != null) &&  (value != null) &&  (value.getPluginID() != null))
      {
         PluginProperty oldValue = (PluginProperty) this.properties.get (value.getPluginID());
         if (oldValue == value)
         {
            this.properties.remove (value.getPluginID());
            value.setManager (null);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * access method to remove the property with the specified key from the list of properties.
    *
    * @param key  the specified key
    * @return     true if the property with the specified key was found in the list removed.
    * @see        #properties
    */
   public boolean removeKeyFromProperties (String key)
   {
      boolean changed = false;
      if ( (this.properties != null) &&  (key != null))
      {
         PluginProperty tmpValue = (PluginProperty) this.properties.get (key);
         if (tmpValue != null)
         {
            this.properties.remove (key);
            tmpValue.setManager (null);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * remove all properties from the list of properties.
    *
    * @see   #properties
    */
   public void removeAllFromProperties()
   {
      PluginProperty tmpValue;
      Iterator iter = this.iteratorOfProperties();
      while (iter.hasNext())
      {
         tmpValue = (PluginProperty) iter.next();
         this.removeFromProperties (tmpValue);
      }
   }


   /**
    * role attribute of the jarFiles association between PluginManager and PluginJarFile. <pre>
    *               +----------+ 0..1   jarFiles   0..1
    * PluginManager | fileName |------------------------ PluginJarFile
    *               +----------+ manager       jarFiles
    * </pre>
    *
    * @see   #hasInJarFiles
    * @see   #hasKeyInJarFiles
    * @see   #iteratorOfJarFiles
    * @see   #keysOfJarFiles
    * @see   #entriesOfJarFiles
    * @see   #sizeOfJarFiles
    * @see   #getFromJarFiles
    * @see   #addToJarFiles
    * @see   #removeFromJarFiles
    * @see   #removeKeyFromJarFiles
    * @see   #removeAllFromJarFiles
    */
   private FHashMap jarFiles;


   /**
    * access method to get the information if the specified jar file is in the list of jar
    * files.
    *
    * @param value  the specified jar file
    * @return       true if the specified jar file exists
    * @see          #jarFiles
    */
   public boolean hasInJarFiles (PluginJarFile value)
   {
      return  ( (this.jarFiles != null) &&
          (value != null) &&  (value.getFileName() != null) &&
          (this.jarFiles.get (value.getFileName()) == value));
   }


   /**
    * access method to get the information if a jar file with the specified key exists in the
    * list of jar files.
    *
    * @param key  the specified key
    * @return     true if a jar file exists
    * @see        #jarFiles
    */
   public boolean hasKeyInJarFiles (String key)
   {
      return  ( (this.jarFiles != null) &&
          (key != null) &&
         this.jarFiles.containsKey (key));
   }


   /**
    * access method to get an iterator over all elements in the list of jar files.
    *
    * @return   an iterator which may be empty (not null).
    * @see      #jarFiles
    */
   public Iterator iteratorOfJarFiles()
   {
      return  ( (this.jarFiles == null)
         ? FEmptyIterator.get()
         : this.jarFiles.values().iterator());
   }


   /**
    * access method to get an iterator over all key in the list of jar files.
    *
    * @return   an iterator which may be empty (not null).
    * @see      #jarFiles
    */
   public Iterator keysOfJarFiles()
   {
      return  ( (this.jarFiles == null)
         ? FEmptyIterator.get()
         : this.jarFiles.keySet().iterator());
   }


   /**
    * access method to get an iterator over all elements in the list of jar files.
    *
    * @return   an iterator which may be empty (not null).
    * @see      #jarFiles
    */
   public Iterator entriesOfJarFiles()
   {
      return  ( (this.jarFiles == null)
         ? FEmptyIterator.get()
         : this.jarFiles.entrySet().iterator());
   }


   /**
    * access method to get the count of elements in the list of jar files.
    *
    * @return   the count of jar files.
    * @see      #jarFiles
    */
   public int sizeOfJarFiles()
   {
      return  ( (this.jarFiles == null)
         ? 0
         : this.jarFiles.size());
   }


   /**
    * access method to get the element with the specified key from the list of jar files.
    *
    * @param key  the specified key.
    * @return     the jar file if found.
    * @see        #jarFiles
    */
   public PluginJarFile getFromJarFiles (String key)
   {
      return  ( ( (this.jarFiles == null) ||  (key == null))
         ? null
         : (PluginJarFile) this.jarFiles.get (key));
   }


   /**
    * access method to add a new jar file to the list of jar files.
    *
    * @param value  the jar file
    * @return       true if the jar file was added
    * @see          #jarFiles
    */
   public boolean addToJarFiles (PluginJarFile value)
   {
      boolean changed = false;
      if ( (value != null) &&  (value.getFileName() != null))
      {
         if (this.jarFiles == null)
         {
            this.jarFiles = new FHashMap(); // or FTreeMap ()
         }
         PluginJarFile oldValue = (PluginJarFile) this.jarFiles.put (value.getFileName(), value);
         if (oldValue != value)
         {
            if (oldValue != null)
            {
               oldValue.setManager (null);
            }
            value.setManager (this);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * access method to remove a jar file from the list of jar files.
    *
    * @param value  the jar file
    * @return       true if the jar file could be removed
    * @see          #jarFiles
    */
   public boolean removeFromJarFiles (PluginJarFile value)
   {
      boolean changed = false;
      if ( (this.jarFiles != null) &&
          (value != null) &&
          (value.getFileName() != null))
      {
         PluginJarFile oldValue = (PluginJarFile)
            this.jarFiles.get (value.getFileName());

         if (oldValue == value)
         {
            this.jarFiles.remove (value.getFileName());
            value.setManager (null);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * access method to remove a jar file with the specified key from the list of jar files.
    *
    * @param key  No description provided
    * @return     true if the jar file could be removed
    * @see        #jarFiles
    */
   public boolean removeKeyFromJarFiles (String key)
   {
      boolean changed = false;
      if ( (this.jarFiles != null) &&  (key != null))
      {

         PluginJarFile tmpValue = (PluginJarFile)
            this.jarFiles.get (key);

         if (tmpValue != null)
         {
            this.jarFiles.remove (key);
            tmpValue.setManager (null);
            changed = true;
         }
      }

      return changed;
   }


   /**
    * access method to remove all jar files from the list of jar files.
    *
    * @see   #jarFiles
    */
   public void removeAllFromJarFiles()
   {
      PluginJarFile tmpValue;
      Iterator iter = this.iteratorOfJarFiles();
      while (iter.hasNext())
      {
         tmpValue = (PluginJarFile) iter.next();
         this.removeFromJarFiles (tmpValue);
      }
   }


   /**
    * If defaultKey is unequal null, all initiated plugins get defaultkey as PluginID. This
    * is needed e.g. by Dobs, cause all classes there have to be loaded within one classloader.
    *
    * @see   #getDefaultKey
    * @see   #setDefaultKey
    */
   private String defaultKey = null;


   /**
    * returns the current set default key.
    *
    * @return   the current set error level.
    */
   public String getDefaultKey()
   {
      return defaultKey;
   }


   /**
    * set the default key to the new value.
    *
    * @param value  the new error level.
    */
   public void setDefaultKey (String value)
   {
      if (defaultKey != value)
      {
         defaultKey = value;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void terminate()
   {
      Iterator iter = iteratorOfProperties();
      while (iter.hasNext())
      {
         PluginProperty property = (PluginProperty) iter.next();
         PluginInterface plugin = property.getPlugin();
         if (plugin != null)
         {
            plugin.terminate();
         }
      }
   }


   /**
    * logging
    */
   private final static Logger LOGGER = Logger.getLogger (PluginManager.class);


   /**
    * Get a plugin property instance specified by the file. The file must be an xml file named
    * plugin.xml which defines the keys in the main section: <pre>
    * Name       : A user readable name of the plugin
    * PluginID   : The pluginID the manager identifies a plugin
    * Class-Name : The full qualified class which implements the PluginInterface
    * Class-Path : The class path extension which is needed the plugin works correct
    * Depends    : Dependencies to other plugins
    * Description: A description of the plugin for what the plugin is good for
    * </pre> Name, Key and Class-Name must be defined to initialize the plugin. Class-Path,
    * Depends and Description are optional settings.
    *
    * @param file  No description provided
    * @return      the property for the file if the file is a jar file with a plugin specific
    */
   private PluginProperty getPluginProperty (URL file)
   {
      PluginProperty property = null;

      try
      {
         if (getDebugLevel() == DEBUG_ON)
         {
            LOGGER.info ("Loading plug-in " + file + ":");
         }

         XMLToPluginProperty xmlToProperty = new XMLToPluginProperty (this);
         property = xmlToProperty.parsePluginProperty (file);

         if (property != null)
         {
            property.setAbsolutePath (file.getFile()); //FIX ME: absolute path may not be in local filesystem
            if (getDefaultKey() != null)
            {
               property.setClassLoaderKey (getDefaultKey());
            }
            else
            {
               property.setClassLoaderKey (property.getPluginID());
            }

            if (getDebugLevel() == DEBUG_ON)
            {
               LOGGER.info ("loading plugin successfull");
            }
         }
         else
         {
            LOGGER.info ("File " + file + " is not a Fujaba plugin file and will be ignored.");
         }
      }
      catch (Exception exc)
      {
         exc.printStackTrace();

         if (getDebugLevel() == DEBUG_ON)
         {
            LOGGER.error ("Plugin " + file + ":");
            LOGGER.error ("  could not get the xml description file of the plugin");
            LOGGER.error ("  Load of plugin skipped.\n");
         }
      }

      return property;
   }


   /**
    * internal used method to parse a file which is a candidate for a plugin.
    *
    * @param pluginFile   the current plugin's property file
    * @param menuVersion  No description provided
    * @param pluginDir    No description provided
    * @see                #getPluginProperty(URL)
    */
   private void parsePlugin (String menuVersion, URL pluginFile, File pluginDir)
   {
      PluginProperty property = null;
      PluginJarFile jarFile = null;
      try
      {
         property = getPluginProperty (pluginFile);

         if (property != null && !hasInProperties (property))
         {
            addToProperties (property);

            // add the current jar file to the list of
            // jar files the PluginManager is responsible for.
            String jarFileName = new File (pluginDir, property.getPluginJarFile()).getAbsolutePath();
            jarFile = getFromJarFiles (jarFileName);
            if (jarFile == null)
            {
               jarFile = new PluginJarFile();
               jarFile.setFileName (jarFileName);
               File tmpJarFile = new File (jarFileName);
               jarFile.setFile (tmpJarFile);
               addToJarFiles (jarFile);
            }
            property.addToJarFiles (jarFile);
            jarFile.setProperty (property);

            URL menuURL = findMenuXMLFile (menuVersion, pluginFile, property);
            if (menuURL != null)
            {
               property.setXMLFile (menuURL.toString());
            }

            // at this point the property file could be loaded.
            property.setDirty (false);

            Iterator iter = property.iteratorOfClassPaths();
            if (iter != null)
            {
               File parentDir = pluginDir;
               String classPath = null;
               while (iter.hasNext() && !property.isDirty())
               {
                  classPath = (String) iter.next();
                  File tmpFile = new File (parentDir, classPath);
                  if (tmpFile.isDirectory())
                  {
                     UPBClassLoader.get (property.getClassLoaderKey()).addClassPath (tmpFile.getAbsolutePath());
                  }
                  else
                  {

                     // add class pathes relative to the parent dircetory
                     // of the current jar file.
                     int pos = classPath.lastIndexOf ("\\") + classPath.lastIndexOf ("/") + 1;
                     String pathAdd = "";
                     if (pos > 0)
                     {
                        pathAdd = classPath.substring (0, pos);
                        classPath = classPath.substring (pos + 1, classPath.length());
                     }
                     JarFileFilter packageFilter = new JarFileFilter();
                     packageFilter.setFilter (classPath);

                     String pdir = parentDir.toString() + File.separator + pathAdd + File.separator;
                     File[] pkgFiles = new File (pdir).listFiles (packageFilter);

                     if (pkgFiles != null)
                     {
                        for (int i = 0; i < pkgFiles.length; i++)
                        {
                           String path = pkgFiles[i].getAbsolutePath();
                           jarFile = getFromJarFiles (path);

                           if (jarFile == null)
                           {
                              jarFile = new PluginJarFile();
                              jarFile.setFileName (path);
                              jarFile.setFile (pkgFiles[i]);
                              addToJarFiles (jarFile);
                           }
                           property.addToJarFiles (jarFile);
                           jarFile.setProperty (property);
                        }
                     }
                     else
                     {
                        if (property.getAbsolutePath().indexOf ("!/") < 0)
                        {
                           System.err.println ("Library '" + classPath + "' can not be found for plugin '"
                              + property.getName() + "'!");
                           System.err.println ("The plugin can not be loaded!");

                           if (getDebugLevel() == DEBUG_ON)
                           {
                              LOGGER.warn ("Plugin " + property.getAbsolutePath() + ":");
                              LOGGER.warn ("  Jar File \"" + classPath + "\" not found in directory of the plugin.");
                              LOGGER.warn ("  Load of plugin skipped.");
                           }

                           property.setDirty (true);
                        }
                        else
                        {
                           if (getDebugLevel() == DEBUG_ON)
                           {
                              LOGGER.warn ("Plugin " + property.getAbsolutePath() + ":");
                              LOGGER.warn ("  Jar File \"" + classPath + "\" not found in directory of the plugin.");
                           }
                        }
                     }
                  }
               }
            }
         }
      }
      catch (Exception ex)
      {
         LOGGER.error ("An error occured while parsing plug-ins.");
         LOGGER.error (ex.getMessage());
         ex.printStackTrace();
         if (property != null)
         {
            property.setDirty (true);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param menuVersion  No description provided
    * @param pluginFile   No description provided
    * @param property     No description provided
    * @return             No description provided
    */
   private URL findMenuXMLFile (String menuVersion, URL pluginFile, PluginProperty property)
   {
      URL url = null;
      try
      {
         url = new URL (pluginFile, menuVersion + ".xml");
      }
      catch (MalformedURLException e)
      {
      }

      //try to read
      try
      {
         url.openStream().close();
      }
      catch (IOException e)
      {
         url = null;
      }

      if (url == null)
      {
         url = property.getResource (menuVersion + ".xml");

         if (url == null)
         {
            url = property.getResource (menuVersion + ".xml");
         }
      }

      return url;
   }


   /**
    * internal method to remove dirty plugins. A plugin is dirty if not all dependencies of
    * the plugin could be solved.
    */
   private void removeDirtyPlugins()
   {
      PluginProperty property = null;

      Iterator propIter = iteratorOfProperties();
      while (propIter.hasNext())
      {
         property = (PluginProperty) propIter.next();
         if (property.isDirty())
         {
            if (getDebugLevel() == DEBUG_ON)
            {
               LOGGER.warn ("Removing Plug-In: " + property.getPluginID());
            }

            property.removeYou();
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param menubar  No description provided
    * @return         No description provided
    */
   public JMenuBar init (JMenuBar menubar)
   {
      return menubar;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @author    $Author: creckord $
    * @version   $Revision: 1.68 $
    */
   class FileCompare implements Compare
   {
      /**
       * No comment provided by developer, please add a comment to improve documentation.
       *
       * @param o   No description provided
       * @param o1  No description provided
       * @return    No description provided
       */
      public int doCompare (Object o, Object o1)
      {
         return  ((File) o).getName().compareTo ( ((File) o1).getName());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginDir  No description provided
    */
   public void scanPlugins (File pluginDir)
   {
      scanPlugins (pluginDir, NEW_PLUGIN_FILE, STABLE_MENU);
      scanPlugins (pluginDir, PLUGIN_FILE, STABLE_MENU);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginDirs  No description provided
    */
   public void scanPlugins (Vector pluginDirs)
   {
      scanPlugins (pluginDirs, NEW_PLUGIN_FILE, STABLE_MENU);
      scanPlugins (pluginDirs, PLUGIN_FILE, STABLE_MENU);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginDirs      No description provided
    * @param pluginFileName  No description provided
    * @param menuFileName    No description provided
    */
   public void scanPlugins (Vector pluginDirs, String pluginFileName, String menuFileName)
   {
      try
      {
         String pluginDir = null;
         File dir;
         for (int i = 0; i < pluginDirs.size(); i++)
         {
            pluginDir = (String) pluginDirs.elementAt (i);
            dir = new File (pluginDir);
            if (dir.isDirectory())
            {
               if (getDebugLevel() == DEBUG_ON)
               {
                  LOGGER.info ("Scanning directory \"" + dir.getCanonicalPath() + "\" for plug-ins");
               }
               scanPlugins (dir, pluginFileName, menuFileName);
            }
            else if (dir.getName().endsWith (".jar"))
            {
               if (getDebugLevel() == DEBUG_ON)
               {
                  LOGGER.info ("Scanning jar file \"" + dir.getCanonicalPath() + "\" for plug-in");
               }
               scanPlugins (dir, pluginFileName, menuFileName);
            }
         }
      }
      catch (Exception ex)
      {
         LOGGER.error ("An Exception is occured while scanning for plugins");
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param pluginDir       No description provided
    * @param pluginFileName  No description provided
    * @param menuFileName    No description provided
    */
   public void scanPlugins (File pluginDir, String pluginFileName, String menuFileName)
   {
      if (pluginDir.isDirectory())
      {
         File[] pluginEntries = pluginDir.listFiles();
         Sort.quicksort (pluginEntries, new FileCompare());
         for (int i = 0; i < pluginEntries.length; i++)
         {
            if (pluginEntries[i].isDirectory())
            {
               scanPlugins (pluginEntries[i], pluginFileName, menuFileName);
            }
         }

         File pluginFile = new File (pluginDir, pluginFileName + ".xml");

         if (pluginFile.isFile())
         {
            try
            {
               parsePlugin (menuFileName, pluginFile.toURL(), pluginDir);
            }
            catch (MalformedURLException e)
            {
               //should not happen
               e.printStackTrace();
            }
         }
      }
      else if (pluginDir.getName().endsWith (".jar"))
      {
         try
         {
            URL pluginFileURL = new URL ("jar:" + pluginDir.toURL().toExternalForm() + "!/" + pluginFileName + ".xml");
            pluginFileURL.openStream().close();
            parsePlugin (menuFileName, pluginFileURL, pluginDir.getParentFile());
         }
         catch (MalformedURLException e)
         {
            if (getDebugLevel() == DEBUG_ON)
            {
               LOGGER.info ("Failed to access jar \"" + pluginDir.getAbsolutePath() + "\" for plug-in loading");
               e.printStackTrace();
            }
         }
         catch (FileNotFoundException e)
         {
            // ok, no plugin
         }
         catch (IOException e)
         {
            //entry not accessible
            e.printStackTrace();
            JOptionPane.showMessageDialog (null, "Error reading jar: " + e.toString(), "Load plugins", JOptionPane.ERROR_MESSAGE);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void loadPlugins()
   {
      Iterator iter = iteratorOfProperties();
      if (iter != null)
      {
         PluginProperty property = null;
         while (iter.hasNext())
         {
            property = (PluginProperty) iter.next();

            //test whether plug-in requires the right kernel version
            if (property.getNeededKernelMajor() == kernelInterface.getMajorVersion() &&
               property.getNeededKernelMinor() <= kernelInterface.getMinorVersion())
            {
               ;
            }
            else
            {
               property.setDirty (true);
               LOGGER.warn ("  Plug-In: " + property.getPluginID() + " can not be loaded.");
               LOGGER.warn ("  Kernel version: " + kernelInterface.getMajorVersion() + "." + kernelInterface.getMinorVersion());
               LOGGER.warn ("  Plug-In requires: " + property.getNeededKernelMajor() + "." + property.getNeededKernelMinor());
               continue;
            }

            // set of plugins needed by property
            Iterator dependIter = property.iteratorOfDependsOn();
            if (dependIter != null)
            {
               String tmpKey = null;
               String key = null;
               int major;
               int minor;
               while (dependIter.hasNext())
               {
                  major = 0;
                  minor = 0;
                  // tmpKey has the form pluginClass#majornr%minornr
                  tmpKey = (String)  (dependIter.next());

                  int keyPos = tmpKey.indexOf ("#");
                  int versionPos = tmpKey.indexOf ("%");

                  if (keyPos > 0)
                  {
                     key = tmpKey.substring (0, keyPos);
                     if (versionPos > 0)
                     {
                        major = new Integer (tmpKey.substring (keyPos + 1, versionPos)).intValue();
                        minor = new Integer (tmpKey.substring (versionPos + 1, tmpKey.length())).intValue();
                     }
                  }
                  else
                  {
                     key = tmpKey;
                  }
                  PluginProperty neededProp = getFromProperties (key);

                  if ( (hasKeyInProperties (key) && neededProp != null)
                     &&  ( (neededProp.getMajor() == major || neededProp.getMajor() < 0)
                     &&  (neededProp.getMinor() >= minor || neededProp.getMinor() < 0)))
                  {
                     UPBClassLoader.get (property.getClassLoaderKey()).addToClassLoaderChain (UPBClassLoader.get (neededProp.getClassLoaderKey()));
                  }
                  else
                  {
                     LOGGER.error ("Plugin " + property.getPluginID() + " cannot be loaded: "
                        + "needs " +  (neededProp == null ? key :
                         (neededProp.getPluginID() + " "
                        + major + "." + minor + ", found "
                        + neededProp.getMajor() + "." + neededProp.getMinor())));
                     property.setDirty (true);
                     Iterator neededIter = property.iteratorOfNeededBy();
                     if (neededIter != null)
                     {
                        String neededKey = null;
                        int npos = -1;
                        while (neededIter.hasNext())
                        {
                           neededKey = (String) neededIter.next();
                           npos = neededKey.indexOf ("#");
                           if (npos > 0)
                           {
                              neededKey = neededKey.substring (0, npos);
                              neededProp = getFromProperties (neededKey);
                           }
                           if (neededProp != null && !neededProp.getPluginID().equals (property.getPluginID()))
                           {
                              neededProp.setDirty (true);

                              if (getDebugLevel() == DEBUG_ON)
                              {
                                 LOGGER.warn ("Plugin: " + property.getPluginID() + " can not be loaded.");
                                 LOGGER.warn ("Plugin: " + property.getPluginID() + " needs " + neededProp.getPluginID());
                                 LOGGER.warn ("load of plugin skipped.");
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
      }

      removeDirtyPlugins();
      initJarFiles();
      createPlugins();
      removeDirtyPlugins();
      prepareXMLMenus();
   }


   /**
    * append the given jar files to the jar files of the internal used classloader.
    */
   private void initJarFiles()
   {
      PluginJarFile jarFile = null;

      PluginProperty pluginProperty = null;

      UPBClassLoader loader;
      Iterator jarIter = iteratorOfJarFiles();
      try
      {
         while (jarIter.hasNext())
         {
            jarFile = (PluginJarFile) jarIter.next();

            pluginProperty = jarFile.getProperty();

            // here we want instantiate for each plug-in a class loader
            loader = UPBClassLoader.get (pluginProperty.getClassLoaderKey());

            if (jarFile.getFile() != null)
            {
               loader.addClassPath (jarFile.getFile().getAbsolutePath());
               File parentDir = jarFile.getFile().getParentFile();
               if (parentDir != null)
               {
                  loader.addResourcePath (parentDir.getAbsolutePath());
               }
            }
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void createPlugins()
   {
      PluginProperty property = null;
      PluginInterface plugin = null;

      UPBClassLoader loader; // = UPBClassLoader.get();
      String className = null;

      Iterator propIter = iteratorOfProperties();

      while (propIter.hasNext())
      {
         property = (PluginProperty) propIter.next();

         if (property.getRootClass() != null)
         {
            className = property.getRootClass();

            // here we want instantiate for each plug-in a class loader
            loader = UPBClassLoader.get (property.getClassLoaderKey());
            try
            {
               plugin = (PluginInterface)
                  Class.forName (className, true, loader).newInstance();
               plugin.setKey (property.getPluginID());
               String tmpString = property.getAbsolutePath();

               int position = tmpString.lastIndexOf ("/");
               if (position == -1)
               {
                  position = tmpString.lastIndexOf (System.getProperty ("file.separator"));
               }

               if (position >= 0)
               {
                  String directory = tmpString.substring (0, position);
                  plugin.setInstallationPath (directory);
               }

               if (plugin.initialize())
               {
                  property.setPlugin (plugin);
               }
               else
               {
                  property.setDirty (true);

                  if (getDebugLevel() == DEBUG_ON)
                  {
                     LOGGER.warn ("Plugin " + property.getPluginID() + ":");
                     LOGGER.warn ("  plugin interface \"" + className + "\" could not be initialized.");
                     LOGGER.warn ("  Load of plugin skipped.");
                  }
               }
            }
            catch (Throwable exception)
            {
               plugin = null;
               exception.printStackTrace();
               property.setDirty (true);

               if (getDebugLevel() == DEBUG_ON)
               {
                  LOGGER.warn ("Plugin " + property.getPluginID() + ":");
                  LOGGER.warn ("  An instance of the plugin interface \"" + className + "\" could not be created.");
                  LOGGER.warn ("  Load of plugin skipped.");
               }
            }
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void prepareXMLMenus()
   {
      UserInterfaceManager uiManager = UserInterfaceManager.get();
      Iterator iter = iteratorOfProperties();
      PluginProperty currentProperty = null;

      // list of (unique) xml documents to be loaded by the ui manager
      FLinkedUniqueElementList list = new FLinkedUniqueElementList();
      while (iter.hasNext())
      {

         // the current plugin property object
         currentProperty = (PluginProperty) iter.next();
         Iterator dependIter = currentProperty.iteratorOfDependsOn();

         // plugin needed by the current plugin
         PluginProperty depProp = null;
         String key = null;
         int index = -1;

         list.add (currentProperty);

         while (dependIter.hasNext())
         {
            // extract the plugin id from string pluginid#Major%Minor
            key = (String) dependIter.next();
            index = key.indexOf ("#");
            index =  (index < 0 ? 0 : index);
            key = key.substring (0, index);

            depProp = getFromProperties (key);

            // the needed plugin is not in the list, so put it before current plugin
            if (!list.contains (depProp))
            {
               //if(i>-1&&i<list.size())
               list.add (list.indexOf (currentProperty), depProp); // addFirst(depProp);

            }
            else
            { // the needed plugin is also in the list,
               // test weather the needed plugin has a smaller index as current plugin
               if (list.isBefore (depProp, currentProperty))
               {
                  continue;
               } // ok
               else
               { // add the needed plugin before the current plugin
                  Iterator iterator = depProp.iteratorOfDependsOn(); //currentProperty.iteratorOfDependsOn();
                  int tmpIndex = -1;
                  int listIndex = -1;
                  String tmpKey = null;
                  PluginProperty tmpProp = null;
                  boolean canMove = true;
                  while (iterator.hasNext() && canMove)
                  {
                     tmpKey = (String) iterator.next();
                     tmpIndex = tmpKey.indexOf ("#");
                     tmpIndex =  (tmpIndex < 0 ? 0 : tmpIndex);
                     tmpKey = tmpKey.substring (0, tmpIndex);
                     tmpProp = getFromProperties (tmpKey);
                     if (list.contains (tmpProp))
                     {
                        listIndex = list.indexOf (tmpProp);
                        if (! (listIndex < 0) && listIndex < list.indexOf (currentProperty))
                        {
                           continue;
                        }
                        else
                        {
                           canMove = false;
                           LOGGER.warn ("Detected a cyle in plug-in dependencies.");
                        } // may be a cycle
                     }
                  }
                  if (canMove)
                  { // dependency contains no cycle
                     list.remove (depProp);
                     list.add (list.indexOf (currentProperty), depProp);
                  }
               }
            }

         }

      } // end of while ()

      // add the plugin xml documents with corresponded plugin ids to ui manager
      iter = list.iterator();
      while (iter.hasNext())
      {
         currentProperty = (PluginProperty) iter.next();
         uiManager.addToDocuments (currentProperty.getClassLoaderKey(), currentProperty.getXMLFile());
      }
   }


   /**
    * access method to delete all references from the PluginManager instance so that the java
    * garbage collector can free it.
    */
   public void removeYou()
   {
      terminate();
      removeAllFromProperties();
      removeAllFromJarFiles();
   }


   /**
    * Get the kernelInterface attribute of the PluginManager object
    *
    * @return   The kernelInterface value
    */
   public KernelInterface getKernelInterface()
   {
      return kernelInterface;
   }


   /**
    * If returned value is 0 the debug is off else the returned value is 1 the debug is on
    *
    * @return   the debug level
    */
   public int getDebugLevel()
   {
      return debugLevel;
   }


   /**
    * Sets the debugLevel attribute of the PluginManager object
    *
    * @param level  The new debugLevel value
    */
   public void setDebugLevel (int level)
   {
      debugLevel = level;
   }

}

/*
 * $Log: PluginManager.java,v $
 * Revision 1.68  2006/01/03 16:12:07  creckord
 * Fixed NPE from change in RuntimeTools
 *
 */
