/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.plugins;

import java.io.*;
import java.net.*;
import java.util.*;

import javax.xml.parsers.*;

import org.apache.log4j.Logger;
import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import de.uni_paderborn.lib.classloader.UPBClassLoader;


/**
 * No comment provided by developer, please add a comment to improve
 * documentation.
 *
 * @author    $Author: lowende $
 * @version   $Revision: 1.33 $
 */
public class XMLToPluginProperty extends DefaultHandler
{
   /**
    * URL used in Plugin definitions (XML-files) to identify the DTD. DTD given
    * as "SYSTEM" (non-public) definition.
    */
   private final static String PLUGIN_SYSTEM_ID = "http://www.upb.de/cs/fujaba/DTDs/Plugin.dtd";

   /**
    * Alternative URL used in Plugin definitions (XML-files) to identify the
    * DTD. DTD given as "SYSTEM" (non-public) definition.
    */
   private final static String PLUGIN_SYSTEM_ID2 = "http://www.fujaba.de/DTDs/Plugin.dtd";

   /**
    * Name used in Plugin definitions (XML-files) to identify the DTD. DTD given
    * as "PUBLIC" definition.
    */
   private final static String PLUGIN_PUBLIC_ID = "-//Fujaba//Fujaba Plugin Definition//EN//1.0";

   /**
    */
   private final static String PLUGIN_DTD_RESOURCE = "DTDs/Plugin.dtd";

   /**
    */
   private static URL pluginDTD;

   /**
    */
   private final static String PLUGIN_LIST = "PluginList";

   /**
    */
   private final static String PLUGIN = "Plugin";

   /**
    */
   private final static String NAME = "Name";

   /**
    */
   private final static String VERSION = "Version";

   /**
    */
   private final static String SOURCE = "Source";

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private final static String KERNEL = "Kernel";

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private final static String PLUGIN_LIB = "PluginLib";

   /**
    */
   private final static String CLASSPATH = "Classpath";

   /**
    */
   private final static String PATH = "Path";

   /**
    */
   private final static String REQUIRED_PLUGINS = "RequiredPlugins";

   /**
    */
   private final static String PLUGIN_ID = "PluginId";

   /**
    */
   private final static String DESCRIPTION = "Description";

   /**
    */
   private final static String SHORT_DESCRIPTION = "ShortDescription";

   /**
    */
   private final static String DETAILED_DESCRIPTION = "DetailedDescription";

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private final static String HELP = "Help";

   /**
    */
   private final static String VENDOR = "Vendor";

   /**
    */
   private final static String CONTACT = "Contact";

   /**
    * State
    */
   private final static int S_NONE = 0;

   /**
    * State
    */
   private final static int S_PLUGIN = 1;

   /**
    * State
    */
   private final static int S_NAME = 2;

   /**
    * State
    */
   private final static int S_VERSION = 3;

   /**
    * State
    */
   private final static int S_SOURCE = 4;

   /**
    * State
    */
   private final static int S_CLASSPATH = 5;

   /**
    * State
    */
   private final static int S_PATH = 6;

   /**
    * State
    */
   private final static int S_REQUIRED_PLUGINS = 7;

   /**
    * State
    */
   private final static int S_PLUGIN_ID = 8;

   /**
    * State
    */
   private final static int S_DESCRIPTION = 9;

   /**
    * State
    */
   private final static int S_SHORT_DESCRIPTION = 10;

   /**
    * State
    */
   private final static int S_DETAILED_DESCRIPTION = 11;

   /**
    * State
    */
   private final static int S_VENDOR = 12;

   /**
    * State
    */
   private final static int S_CONTACT = 13;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private final static int S_PLUGIN_LIB = 14;

   /**
    */
   private int state = S_NONE;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   public final static char SEPARATOR = ';';

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private PluginProperty curProperty;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private Hashtable pluginProperties = null;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private PluginManager manager;

   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    */
   private String currentXMLFile;


   /**
    * Constructor for class XMLToPluginProperty
    */
   private XMLToPluginProperty() { }


   /**
    * Constructor for class XMLToPluginProperty
    *
    * @param manager  No description provided
    */
   public XMLToPluginProperty (PluginManager manager)
   {
      this.manager = manager;
      this.pluginProperties = new Hashtable();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param xmlFile  No description provided
    * @return         A vector containing a PluginList object and
    * one or more serialized "PluginProperties" objects
    */
   public Vector parsePluginList (String xmlFile)
   {
      Vector result;
      PluginList pluginList = new PluginList();
      pluginList.setUrl (xmlFile);

      try
      {
         parse (new URL (xmlFile));
      }
      catch (Exception e)
      {
         pluginList.setParseException (e);
      }

      pluginList.setName (getPluginListName());

      result = new Vector (this.pluginProperties.size() + 1);
      result.add (pluginList);
      result.addAll (this.pluginProperties.values());

      return result;
   }


   /**
    * Parses a single plugin description.
    *
    *
    * @param xmlFile  No description provided
    * @return         No description provided
    */
   public PluginProperty parsePluginProperty (URL xmlFile)
   {
      try
      {
         parse (xmlFile);
      }
      catch (Exception e)
      {
         return new PluginProperty (manager.getKernelInterface().getMajorVersion(), manager.getKernelInterface().getMinorVersion());
      }

      return this.curProperty;
   }


   /**
    * Parses a plugin list or a single plugin xml file.
    *
    * @param xmlFile     the list or plugin
    * @throws Exception  If an error occured during parsing.
    */
   private void parse (URL xmlFile) throws Exception
   {
      this.currentXMLFile = xmlFile.getFile();

      try
      {
         Thread.currentThread().setContextClassLoader (UPBClassLoader.get());
         SAXParserFactory factory = SAXParserFactory.newInstance();
         factory.setValidating (true);
         factory.setNamespaceAware (true);

         XMLReader xmlReader = factory.newSAXParser().getXMLReader();
         xmlReader.setContentHandler (this);
         xmlReader.setErrorHandler (this);
         xmlReader.setEntityResolver (this);
         final EntityResolver entityResolver = xmlReader.getEntityResolver();
         xmlReader.setEntityResolver (
            new EntityResolver()
            {
               /**
                * @param publicId                   The public identifier of the external entity being
                *           referenced, or null if none was supplied.
                * @param systemId                   The system identifier of the external entity being
                *           referenced.
                * @return                           An InputSource object describing the new input source, or
                *         null to request that the parser open a regular URI
                *         connection to the system identifier.
                * @throws org.xml.sax.SAXException  Any SAX exception, possibly wrapping another
                *            exception.
                * @throws java.io.IOException       A Java-specific IO exception, possibly the result of
                *            creating a new InputStream or Reader for the
                *            InputSource.
                * @see                              org.xml.sax.InputSource
                * @throws SAXException              Exception description not provided
                * @throws IOException               Exception description not provided
                */
               public InputSource resolveEntity (String publicId, String systemId) throws SAXException,
                  IOException
               {
                  String upbURL = "http://www.upb.de/cs/fujaba/DTDs";
                  if (systemId != null && systemId.startsWith (upbURL))
                  {
                     File localFile = new File ("./DTDs" + systemId.substring (upbURL.length()));
                     if (localFile.exists() && !localFile.isDirectory())
                     {
                        return new InputSource (new FileInputStream (localFile));
                     }
                  }
                  // System.out.println ("resolving " + publicId + ", " +
                  // systemId);
                  return entityResolver.resolveEntity (publicId, systemId);
               }
            });

         xmlReader.parse (new InputSource (xmlFile.openStream()));
      }
      catch (SAXException e)
      {
         e.printStackTrace();
         throw new Exception (e.getClass().getName() + ": " + e.getMessage());
      }
      catch (ParserConfigurationException e)
      {
         e.printStackTrace();
         throw new Exception (e.getClass().getName() + ": " + e.getMessage());
      }
      catch (IOException e)
      {
         e.printStackTrace();

         String newMessage;

         // examine the exception type, so we are able to produce a
         // "pretty" error message, that can be displayed later

         // the "message" of a FileNotFoundException is the name of the file
         // because of that we don't use the message
         if (e instanceof FileNotFoundException)
         {
            newMessage = xmlFile + ": file not found";
         }
         else if (e instanceof SocketException)
         {
            newMessage = xmlFile + ": socket error: " + e.getMessage();
         }
         //else if (e instanceof )
         //	newMessage = "" + e.getMessage();
         else
         {
            newMessage = xmlFile + ": " + e.getClass().getName() + ": " + e.getMessage();
         }

         throw new Exception (newMessage);
      }

      this.currentXMLFile = null;
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param exception  No description provided
    */
   public void error (SAXParseException exception)
   {
      Logger.getLogger (XMLToPluginProperty.class).debug ("XML Parse Error in document " + currentXMLFile + ", line "
         + exception.getLineNumber() + ":");
      Logger.getLogger (XMLToPluginProperty.class).debug ("  " + exception.getMessage());
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param exception  No description provided
    */
   public void fatalError (SAXParseException exception)
   {
      Logger.getLogger (XMLToPluginProperty.class).error ("Fatal XML Parse Error in document " + currentXMLFile + ", line "
         + exception.getLineNumber() + ":");
      Logger.getLogger (XMLToPluginProperty.class).error ("  " + exception.getMessage());
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param exception  No description provided
    */
   public void warning (SAXParseException exception)
   {
      Logger.getLogger (XMLToPluginProperty.class).warn ("XML Parse warning in document " + currentXMLFile + ", line "
         + exception.getLineNumber() + ":");
      Logger.getLogger (XMLToPluginProperty.class).warn ("  " + exception.getMessage());
   }


   /**
    * Method implements interface EntityResolver.
    *
    * @param publicId       No description provided
    * @param systemId       No description provided
    * @return               No description provided
    * @throws SAXException  Exception description not provided
    */
   public InputSource resolveEntity (String publicId, String systemId) throws SAXException
   {
      InputSource inputSource = null;

      if (PLUGIN_PUBLIC_ID.equals (publicId) || PLUGIN_SYSTEM_ID.equals (systemId)
         || PLUGIN_SYSTEM_ID2.equals (systemId))
      {
         if (pluginDTD == null)
         {
            pluginDTD = UPBClassLoader.get().getResource (PLUGIN_DTD_RESOURCE);
         }
         if (pluginDTD != null)
         {
            try
            {
               inputSource = new InputSource (pluginDTD.openStream());
            }
            catch (Exception e)
            {
               System.out.println ("Could not resolve SYSTEM or PUBLIC reference for DTD.");
               throw new SAXException (e);
            }
         }
      }

      return inputSource;
   } // resolveEntity


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param uri    No description provided
    * @param name   No description provided
    * @param qName  No description provided
    * @param atts   No description provided
    */
   public void startElement (String uri, String name, String qName, Attributes atts)
   {
      String value;
      try
      {
         if (qName.equals (PLUGIN_LIST))
         {
            setPluginListName (atts.getValue ("name"));
         }
         else if (qName.equals (PLUGIN))
         {
            state = S_PLUGIN;
            value = atts.getValue ("pluginClass");

            curProperty = new PluginProperty (manager.getKernelInterface().getMajorVersion(), manager.getKernelInterface().getMinorVersion());

            curProperty.setRootClass (value);
            curProperty.setPluginID (value);
         }
         else if (qName.equals (NAME))
         {
            state = S_NAME;
         }
         else if (qName.equals (VERSION))
         {
            state = S_VERSION;
            curProperty.setMajorVersion (atts.getValue ("major"));
            curProperty.setMinorVersion (atts.getValue ("minor"));
            try
            {
               String buildNo = atts.getValue ("build");
               int buildNumber = Integer.parseInt (buildNo);
               curProperty.setBuildNumber (buildNumber);
            }
            catch (Exception ex)
            {
            }

         }
         else if (qName.equals (SOURCE))
         {
            state = S_SOURCE;
         }
         else if (qName.equals (KERNEL))
         {
            try
            {
               curProperty.setNeededKernelMajor (new Integer (atts.getValue ("major")).intValue());
               curProperty.setNeededKernelMinor (new Integer (atts.getValue ("minor")).intValue());
            }
            catch (Exception ex)
            {
               System.out.println ("Warrning: Invalid Kernel major and/or minor number.\nUsing the default.");
               curProperty.setNeededKernelMajor (manager.getKernelInterface().getMajorVersion());
               curProperty.setNeededKernelMinor (manager.getKernelInterface().getMinorVersion());
            }
         }
         else if (qName.equals (CLASSPATH))
         {
            state = S_CLASSPATH;
         }
         else if (qName.equals (PATH))
         {
            state = S_PATH;
         }
         else if (qName.equals (REQUIRED_PLUGINS))
         {
            state = S_REQUIRED_PLUGINS;
         }
         else if (qName.equals (PLUGIN_ID))
         {
            state = S_PLUGIN_ID;
            String plugin = atts.getValue ("pluginClass") + "#" + atts.getValue ("major") + "%"
               + atts.getValue ("minor");
            curProperty.addToDependsOn (plugin);
         }
         else if (qName.equals (DESCRIPTION))
         {
            state = S_DESCRIPTION;
         }
         else if (qName.equals (SHORT_DESCRIPTION))
         {
            state = S_SHORT_DESCRIPTION;
         }
         else if (qName.equals (DETAILED_DESCRIPTION))
         {
            state = S_DETAILED_DESCRIPTION;
         }
         else if (qName.equals (HELP))
         {
            value = atts.getValue ("helpURL");
            curProperty.setPluginHelp (value);
         }
         else if (qName.equals (VENDOR))
         {
            state = S_VENDOR;
         }
         else if (qName.equals (CONTACT))
         {
            state = S_CONTACT;
         }
         else if (qName.equals (PLUGIN_LIB))
         {
            state = S_PLUGIN_LIB;
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param ch      No description provided
    * @param start   No description provided
    * @param length  No description provided
    */
   public void characters (char[] ch, int start, int length)
   {
      try
      {
         String data = new String (ch, start, length);

         switch (state)
         {
            case S_NAME:
               curProperty.setName (data);
               break;
            case S_SOURCE:
               curProperty.setSource (data);
               break;
            case S_PATH:
               curProperty.addToClassPaths (data);
               break;
            case S_REQUIRED_PLUGINS:
               break;
            case S_PLUGIN_ID:
               break;
            case S_PLUGIN_LIB:
               curProperty.setPluginJarFile (data);
               break;
            case S_SHORT_DESCRIPTION:
               curProperty.setShortDescription (data);
               break;
            case S_DETAILED_DESCRIPTION:
               curProperty.setDescription (data);
               break;
            case S_VENDOR:
               curProperty.setVendor (data);
               break;
            case S_CONTACT:
               curProperty.setContact (data);
               break;
            default:
               break;
         }
      }
      catch (Exception ex)
      {
         ex.printStackTrace();
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve
    * documentation.
    *
    * @param uri    No description provided
    * @param name   No description provided
    * @param qName  No description provided
    */
   public void endElement (String uri, String name, String qName)
   {
      if (qName.equals (PLUGIN))
      {
         state = S_NONE;
         this.pluginProperties.put (curProperty.getPluginID() + SEPARATOR + curProperty.getMajorVersion()
            + SEPARATOR + curProperty.getMinorVersion() + SEPARATOR
            + curProperty.getBuildNumber(), curProperty);
      }
      switch (state)
      {
         case S_NAME:
            state = S_NONE;
            break;
         default:
            state = S_NONE;
            break;
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private String pluginListName = "";


   /**
    * Sets the pluginListName attribute of the XMLToPluginProperty object
    *
    * @param value  The new pluginListName value
    */
   private void setPluginListName (String value)
   {
      if (value != null && value.length() > 0)
      {
         pluginListName = value;
      }
   }


   /**
    * Get the pluginListName attribute of the XMLToPluginProperty object
    *
    * @return   The pluginListName value
    */
   private String getPluginListName()
   {
      return pluginListName;
   }

}

/*
 * $Log: XMLToPluginProperty.java,v $
 * Revision 1.33  2006/02/07 17:04:13  lowende
 * Applied patch send by Felix Klar: Connection error handling in plugin download dialog enhanced.
 *
 */
