/*
 * The FUJABA ToolSuite project:
 *
 *   FUJABA is the acronym for 'From Uml to Java And Back Again'
 *   and originally aims to provide an environment for round-trip
 *   engineering using UML as visual programming language. During
 *   the last years, the environment has become a base for several
 *   research activities, e.g. distributed software, database
 *   systems, modelling mechanical and electrical systems and
 *   their simulation. Thus, the environment has become a project,
 *   where this source code is part of. Further details are avail-
 *   able via http://www.fujaba.de
 *
 *      Copyright (C) Fujaba Development Group
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free
 *   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *   MA 02111-1307, USA or download the license under
 *   http://www.gnu.org/copyleft/lesser.html
 *
 * WARRANTY:
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *   GNU Lesser General Public License for more details.
 *
 * Contact address:
 *
 *   Fujaba Management Board
 *   Software Engineering Group
 *   University of Paderborn
 *   Warburgerstr. 100
 *   D-33098 Paderborn
 *   Germany
 *
 *   URL  : http://www.fujaba.de
 *   email: info@fujaba.de
 *
 */
package de.upb.lib.userinterface;

import java.awt.Point;
import java.io.*;
import java.net.URL;
import java.util.*;

import javax.swing.*;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;

import org.apache.log4j.Logger;
import org.xml.sax.*;

import de.uni_paderborn.lib.classloader.UPBClassLoader;
import de.upb.lib.userinterface.action.AbstractMultiplexAction;
import de.upb.lib.userinterface.action.MultiplexAction;
import de.upb.tools.fca.*;



/**
 * <h2>Associations</h2>
 *
 * <pre>
 *                          0..n                  0..1
 * UserInterfaceSaxHandler ---------------------------- UserInterfaceManager
 *                          handler            manager
 * </pre>
 *
 * @author    $Author: creckord $
 * @version   $Revision: 1.34 $
 */
public class UserInterfaceManager
{
   /**
    * Singleton instance
    */
   private static UserInterfaceManager theUserInterfaceManager;


   /**
    * Private constructor for singleton class UserInterfaceManager
    */
   private UserInterfaceManager()
   {
      setMultiplexActionClass (MultiplexAction.class);
   } // constructor


   /**
    * @return   The singleton instance
    */
   public static UserInterfaceManager get()
   {
      if (theUserInterfaceManager == null)
      {
         theUserInterfaceManager = new UserInterfaceManager();
      }

      return theUserInterfaceManager;
   } // get


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Class multiplexActionClass;


   /**
    * The MultiplexAction class is used as an adapter to all actions. Before calling the actionPerformed()
    * method of the adapted class, the {@link java.awt.event.ActionListener#actionPerformed}
    * method of this class is called. The class must have a parameter less constructor.
    *
    * @param newMultiplexActionClass    The new MultiplexAction class implementing {@link AbstractMultiplexAction}
    * @throws IllegalArgumentException  Exception is thrown if newMultiplexActionClass is null,
    *      is interface or doesn't implement {@link AbstractMultiplexAction}
    */
   public void setMultiplexActionClass (Class newMultiplexActionClass)
       throws IllegalArgumentException
   {
      // check, if argument is not null, is not interface and implements AbstractMultiplexAction
      if ( (newMultiplexActionClass != null) &&
          (!newMultiplexActionClass.isInterface()) &&
          (AbstractMultiplexAction.class.isAssignableFrom (newMultiplexActionClass)))
      {
         // check, if there is a parameter less constructor
         try
         {
            newMultiplexActionClass.getConstructor (new Class[]{});
         }
         catch (NoSuchMethodException e)
         {
            throw new IllegalArgumentException ("MultiplexAction does not have a parameter less constructor!");
         }

         this.multiplexActionClass = newMultiplexActionClass;
      }
      else
      {
         throw new IllegalArgumentException ("Argument is null, is interface, or does not implement AbstractMultiplexAction!");
      }
   }


   /**
    * The MultiplexAction class is used as an adapter to all actions. Before calling the actionPerformed()
    * method of the adapted class, the {@link java.awt.event.ActionListener#actionPerformed}
    * method of this class is called.
    *
    * @return   class that implements the {@link AbstractMultiplexAction} interface
    */
   public Class getMultiplexActionClass()
   {
      return this.multiplexActionClass;
   }


   /**
    * <pre>
    *                          0..n                  0..1
    * UserInterfaceSaxHandler ---------------------------- UserInterfaceManager
    *                          handler            manager
    * </pre>
    */
   private Hashtable handler = new Hashtable();


   /**
    * Sets the handler attribute of the UserInterfaceManager object
    *
    * @param value  The new handler value
    * @return       No description provided
    */
   boolean setHandler (UserInterfaceSaxHandler value)
   {
      if (this.handler != null)
      {
         Iterator iter = handler.values().iterator();
         UserInterfaceSaxHandler tmp;
         while (iter.hasNext())
         {
            tmp = (UserInterfaceSaxHandler)  (iter.next());
            if (tmp != value)
            {
               if (tmp != null)
               {
                  UserInterfaceSaxHandler oldValue = tmp;
                  tmp = null;
                  oldValue.setManager (null);
               }
               if (value != null)
               {
                  value.setManager (this);
               }

               return true;
            }

            return false;
         }
      }

      if (value != null)
      {
         value.setManager (this);
      }
      return true;
   }


   /**
    * Get the handler attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The handler value
    */
   /*
    *  package
    */
   UserInterfaceSaxHandler getHandler (String key)
   {
      UserInterfaceSaxHandler tmp = (UserInterfaceSaxHandler) this.handler.get (key);

      if (tmp == null)
      {
         tmp = new UserInterfaceSaxHandler();
         tmp.setKey (key);
         handler.put (key, tmp);
      }
      return tmp;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param key  No description provided
    * @return     No description provided
    */
   private XMLReader createXMLReader (String key)
   {
      XMLReader xmlReader = null;
      UserInterfaceSaxHandler handler = getHandler (key);
      try
      {
         SAXParserFactory factory = SAXParserFactory.newInstance();
         factory.setValidating (true);
         factory.setNamespaceAware (true);

         xmlReader = factory.newSAXParser().getXMLReader();
         xmlReader.setContentHandler (handler);
         xmlReader.setErrorHandler (handler);
         xmlReader.setEntityResolver (handler);
      }
      catch (SAXException sax)
      {
         sax.printStackTrace();
         throw new RuntimeException ("Unable to instantiate XML Parser: " + sax.getMessage());
      }
      catch (ParserConfigurationException pce)
      {
         pce.printStackTrace();
         throw new RuntimeException ("Unable to instantiate XML Parser: " + pce.getMessage());
      }

      return xmlReader;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private Point lastPointerPosition;


   /**
    * Returns the mouse pointer position for popup menu.
    *
    * @return   The lastPointerPosition value
    */
   public Point getLastPointerPosition()
   {
      return lastPointerPosition;
   }


   /**
    * Used to save the mouse pointer position for popup menu.
    *
    * @param newPos  The new lastPointerPosition value
    */
   public void setLastPointerPosition (Point newPos)
   {
      this.lastPointerPosition = newPos;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private EventSource eventSource;


   /**
    * Returns the interface for getting sources for performing events.
    *
    * @return   The eventSource value
    */
   public EventSource getEventSource()
   {
      return eventSource;
   }


   /**
    * Sets the interface for getting sources for performing events.
    *
    * @param eventSource  The new eventSource value
    */
   public void setEventSource (EventSource eventSource)
   {
      this.eventSource = eventSource;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap actions = new HashMap();


   /*
    *  package
    */
   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    */
   void addToActions (String key, AbstractAction value)
   {
      actions.put (key, value);
   }


   /**
    * Get the fromActions attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromActions value
    */
   public AbstractAction getFromActions (String key)
   {
      return (AbstractAction) actions.get (key);
   }


   /*
    *  package
    */
   /**
    * Get the keyForAction attribute of the UserInterfaceManager object
    *
    * @param value  No description provided
    * @return       The keyForAction value
    */
   public String getKeyForAction (Action value)
   {
      String key = null;

      Set actionsSet = actions.entrySet();
      Iterator iter = actionsSet.iterator();
      while (iter.hasNext() &&  (key == null))
      {
         Map.Entry entry = (Map.Entry) iter.next();
         if (entry.getValue().equals (value))
         {
            key = (String) entry.getKey();
         }
      }

      return key;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap initializers = new HashMap();


   /*
    *  package
    */
   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    */
   void addToInitializers (String key, ElementInitializer value)
   {
      initializers.put (key, value);
   }


   /**
    * Get the fromInitializers attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromInitializers value
    */
   public ElementInitializer getFromInitializers (String key)
   {
      ElementInitializer result = null;

      result = (ElementInitializer) initializers.get (key);

      if (result == null)
      {
         // try actions if nothing was found
         AbstractAction action = getFromActions (key);

         if (action != null && action instanceof ElementInitializer)
         {
            result = (ElementInitializer) action;

            //remember result
            addToInitializers (key, result);
         }
      }

      return result;
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap menus = new HashMap();


   /*
    *  package
    */
   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    */
   void addToMenus (String key, JMenu value)
   {
      menus.put (key, value);
   } // addToMenus


   /**
    * Get the fromMenus attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromMenus value
    */
   public JMenu getFromMenus (String key)
   {
      return (JMenu) menus.get (key);
   } // getFromMenus


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap menuBars = new HashMap();


   /*
    *  package
    */
   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    */
   void addToMenuBars (String key, JMenuBar value)
   {
      menuBars.put (key, value);
   }


   /**
    * Get the fromMenuBars attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromMenuBars value
    */
   public JMenuBar getFromMenuBars (String key)
   {
      return (JMenuBar) menuBars.get (key);
   } // getMenuBar


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap popupMenus = new HashMap();


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    */
   private void addToPopupMenus (String key, JPopupMenu value)
   {
      popupMenus.put (key, value);
   } // addToPopupMenus


   /**
    * Get the fromPopupMenus attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromPopupMenus value
    */
   public JPopupMenu getFromPopupMenus (String key)
   {
      return (JPopupMenu) popupMenus.get (key);
   } // getFromPopupMenus


   /**
    * Get the fromPopupMenus attribute of the UserInterfaceManager object
    *
    * @param key          No description provided
    * @param popupSource  No description provided
    * @return             The fromPopupMenus value
    */
   public JPopupMenu getFromPopupMenus (String key, Object popupSource)
   {
      JPopupMenu menu = getFromPopupMenus (key);
      if (menu != null)
      {
         initMenu (menu, popupSource);
      }
      return menu;
   } // getFromPopupMenus


   /**
    * Calls initialize (ElementInitializers) on elements of JMenuBar
    *
    * @param menu    JMenuBar with elements to initialize
    * @param source  Object that wishes to initialize the elements
    */
   public void initMenu (JMenuBar menu, Object source)
   {
      if (menu == null)
      {
         System.err.println ("UserInterfaceManger.initMenu: ERROR: menu == null. That should not happen");
         return;
      }
      MenuElement[] elements = menu.getSubElements();

      for (int i = 0; i < elements.length; i++)
      {
         JMenuItem item = (JMenuItem) elements[i];

         if (item instanceof JMenu)
         {
            initMenu ((JMenu) item, source);
         }
         else
         {
            initMenuItem (item, source);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param menu    No description provided
    * @param source  No description provided
    */
   public void initMenu (JPopupMenu menu, Object source)
   {
      MenuElement[] elements = menu.getSubElements();

      for (int i = 0; i < elements.length; i++)
      {
         JMenuItem item = (JMenuItem) elements[i];

         if (item instanceof JMenu)
         {
            initMenu ((JMenu) item, source);
         }
         else
         {
            initMenuItem (item, source);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param menu    No description provided
    * @param source  No description provided
    */
   public void initMenu (JMenu menu, Object source)
   {
      MenuElement[] elements = menu.getSubElements();

      for (int i = 0; i < elements.length; i++)
      {
         if (elements[i] instanceof JMenu)
         {
            initMenu ((JMenu) elements[i], source);
         }
         else if (elements[i] instanceof JPopupMenu)
         {
            initMenu ((JPopupMenu) elements[i], source);
         }
         else if (elements[i] instanceof JMenuItem)
         {
            initMenuItem ((JMenuItem) elements[i], source);
         }
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param item    No description provided
    * @param source  No description provided
    */
   public void initMenuItem (JMenuItem item, Object source)
   {
      Action action = item.getAction();
      if (action instanceof ElementInitializer)
      {
          ((ElementInitializer) action).initialize (item, source);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap toolBars = new HashMap();


   /**
    * Access method for an one to n association.
    *
    * @param key    The object added.
    * @param value  The object added.
    */
   private void addToToolBars (String key, JToolBar value)
   {
      toolBars.put (key, value);
   } // addToToolBars


   /**
    * Get the fromToolBars attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromToolBars value
    */
   public JToolBar getFromToolBars (String key)
   {
      return (JToolBar) toolBars.get (key);
   } // getFromToolBars


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfToolBars()
   {
      return  ( (this.toolBars == null)
         ? FEmptyIterator.get()
         : this.toolBars.values().iterator());
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private HashMap containers = new HashMap();


   /*
    *  package
    */
   /**
    * Access method for an one to n association.
    *
    * @param value  The object added.
    */
   void addToContainers (SectionContainer value)
   {
      containers.put (value.getId(), value);
   }


   /*
    *  package
    */
   /**
    * Get the fromContainers attribute of the UserInterfaceManager object
    *
    * @param key  No description provided
    * @return     The fromContainers value
    */
   SectionContainer getFromContainers (String key)
   {
      return (SectionContainer) containers.get (key);
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   private Iterator iteratorOfContainers()
   {
      return containers.values().iterator();
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private FDuplicatedLinkedHashMap documents = new FDuplicatedLinkedHashMap();


   /**
    * Access method for an one to n association.
    *
    * @param document  The object added.
    * @param key       The object added.
    */
   public void addToDocuments (String key, String document)
   {
      if (key != null && document != null)
      {
         this.documents.add (key, document);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   public void removeAllFromDocuments()
   {
      documents.clear();
   }


   /**
    * <pre>
    *          				   executionListeners  0..n
    * UserInterfaceManager ---------------------------------> ActionExcutionListener
    *                                    executionListeners
    * </pre>
    */
   private FHashSet executionListeners;


   /**
    * Access method for an one to n association.
    *
    * @param l  The object added.
    */
   public void addExecutionListener (ActionExecutionListener l)
   {
      if (l != null &&  (this.executionListeners == null || !this.executionListeners.contains (l)))
      {
         if (this.executionListeners == null)
         {
            this.executionListeners = new FHashSet();
         }

         this.executionListeners.add (l);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param l  No description provided
    */
   public void removeExecutionListener (ActionExecutionListener l)
   {
      if (l != null && this.executionListeners != null && this.executionListeners.contains (l))
      {
         this.executionListeners.remove (l);
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @return   No description provided
    */
   public Iterator iteratorOfExecutionListeners()
   {
      return this.executionListeners == null ? FEmptyIterator.get() : this.executionListeners.iterator();
   }


   /**
    * Initializes the ActionManager by loading the given XML documents and building actions,
    * menus and toolbars.
    */
   public void loadDocuments()
   {
      String currentKey = null;
      String currentFileName = null;

      Iterator iter = documents.iterator();
      while (iter.hasNext())
      {
         Map.Entry entry = (Map.Entry) iter.next();
         currentKey = (String) entry.getKey();
         Collection collect = (Collection) entry.getValue();
         Iterator collectIter = collect.iterator();
         while (collectIter.hasNext())
         {
            currentFileName = (String) collectIter.next();
            loadXMLDocument (currentKey, currentFileName);
         }
      }

      generateStructures();
   } // initialize


   /**
    * logging
    */
   private final static Logger L = Logger.getLogger (UserInterfaceManager.class);


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    *
    * @param fileName  No description provided
    * @param key       No description provided
    */
   public void loadXMLDocument (String key, String fileName)
   {
      XMLReader xmlReader = createXMLReader (key);
      L.info ("Loading user interface " + fileName);

      try
      {
         InputSource inputSource = null;

         InputStream stream = UPBClassLoader.get (key).getResourceAsStream (fileName);

         if (stream == null)
         {
            // Try to open the Stream from fileName as URL.
            URL url = new URL (fileName);
            stream = url.openStream();
         }

         if (stream != null)
         {
            inputSource = new InputSource (stream);
         }
         else
         {
            // Try to open the input source using the file path.
            inputSource = new InputSource (new FileReader (fileName));
         }

         if (inputSource != null)
         {
            xmlReader.parse (inputSource);
            stream.close();
         }
         else
         {
            throw new IOException ("No input source found!");
         }
      }
      catch (IOException e)
      {
         System.err.println ("Error loading XML document " + fileName + ":\n" + e.getMessage());
      }
      catch (SAXParseException e)
      {
         File currentDir = new File (".");
         System.err.println ("loadXMLDocument: we are in Directory " + currentDir.getAbsolutePath());

         System.err.println ("An error occurred parsing XML document " + fileName + ", line " + e.getLineNumber() + ":\n" + e.getMessage());
      }
      catch (SAXException e)
      {
         System.err.println ("An error occurred parsing XML document " + fileName + ":\n" + e.getMessage());
      }
   }


   /**
    * No comment provided by developer, please add a comment to improve documentation.
    */
   private void generateStructures()
   {
      Iterator iter = iteratorOfContainers();
      while (iter.hasNext())
      {
         SectionContainer container = (SectionContainer) iter.next();
         JComponent component = container.generateComponent();

         if (component instanceof JMenuBar)
         {
            addToMenuBars (container.getId(), (JMenuBar) component);
         }
         else if (component instanceof JPopupMenu)
         {
            addToPopupMenus (container.getId(), (JPopupMenu) component);
         }
         else if (component instanceof JToolBar)
         {
            addToToolBars (container.getId(), (JToolBar) component);
         }
      }
   } // generateStructures

}

/*
 * $Log: UserInterfaceManager.java,v $
 * Revision 1.34  2006/01/03 16:12:08  creckord
 * Fixed NPE from change in RuntimeTools
 *
 */
