/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#ifndef _GLE_FILE_INFO_H
#define _GLE_FILE_INFO_H

#include <QtGui>

//! Class used to describe the open file
class GLEFileInfo
{
private:
	//! The GLE file name
	QString gleFileName;
	//! The EPS file name
	QString epsFileName;
	//! The Image without any rendered objects
	QImage epsImage;
	//! The GLE code used to draw the basic image
	QStringList code;
	//! A flag noting this object as a temporary file
	bool tempFile;
	//! The image with rendered objects
	QImage epsImageWithExtras;
	//! The time stamp
	QDateTime lastModified;

public:
	//! Constructor: initialise variables
	GLEFileInfo();
	
	//! Return the GLE file name
	inline const QString& gleFile() { return gleFileName; }
	//! Set the GLE file name
	inline void setGleFile(const QString& newGLEFile) { gleFileName = newGLEFile; setHasFileName(true);}
	//! Return the EPS file name
	inline const QString& epsFile() { return epsFileName; }
	//! Set the EPS file name
	inline void setEpsFile(const QString& newEPSFile) { epsFileName = newEPSFile; }
	//! Get the rendered base image
	inline const QImage& image() { return epsImage; }
	//! Set the rendered base image
	void setImage(const QImage& newImage, double dpi);
	//! Return the rendered combined image
	inline const QImage& imageWithExtras() { return epsImageWithExtras; }
	//! Set the rendered combined image
	inline void setImageWithExtras(const QImage& newImage) { epsImageWithExtras = newImage; }
	//! Get the GLE base code
	inline const QStringList& gleCode() { return code; };
	//! Set the GLE base code
	void setGLECode(const QStringList& newCode);
	//! Is this a temporary file?
	inline const bool& isTemp() { return tempFile; }
	//! Set whether this is a temporary file
	inline void setTemp(const bool& newTemp) { tempFile = newTemp; }
	//! Set whether we have a file name
	inline void setHasFileName(const bool& newFlag) { hasFileNameFlag = newFlag; }
	//! Do we have a file name?
	inline const bool& hasFileName() { return hasFileNameFlag; }

	QSizeF GLEPageSize();
	void setGLEPageSize(QSizeF newSize);

	//! Flag noting whether we have a file name
	bool hasFileNameFlag;
	//! Read in the GLE code from gleFile()
	void readGLECode();
	//! Update the time stamp
	void updateTimeStamp();
	//! Has the file changed externally?
	bool hasChanged();
	//! Is the primary file a GLE file?
	bool isGLE();
	//! Is the primary file an EPS file?
	bool isEPS();
	//! What's the file name of the primary file?
	QString primaryFile();
	//! What's the GLE page size
	QSizeF glePageSize;
	//! Clear the object
	void clear();
};




#endif
