/*
 * (c) 2004 Jan Struyf
 *
 */

#include "all.h"
#include "tokens/stokenizer.h"
#include "SourceLine.h"
#include "tokens/Tokenizer.h"
#include "core.h"
#include "mygraph.h"
#include "file_io.h"
#include "texinterface.h"
#include "cutils.h"
#include "gprint.h"
#include "cmdline.h"
#include "config.h"
#include "mem_limits.h"
#include "token.h"
#include "var.h"
#include "name.h"
#include "run.h"

#define BEGINDEF extern
#include "begin.h"

void decode_utf8_basic(string& sc);

extern ConfigCollection g_Config;

class FourDoubleList {
protected:
	double m_List[4];
	int m_Pos;
public:
	FourDoubleList();
	double get(int idx);
	void add(double value);
};

FourDoubleList::FourDoubleList() {
	m_Pos = 0;
}

double FourDoubleList::get(int idx) {
	return m_List[(m_Pos + idx) % 4];
}

void FourDoubleList::add(double value) {
	m_List[m_Pos] = value;
	m_Pos = (m_Pos + 1) % 4;
}

TeXInterface TeXInterface::m_Instance;

TeXInterface::TeXInterface() {
}

TeXInterface::~TeXInterface() {
	cleanUpObjects();
	cleanUpHash();
}

TeXObject* TeXInterface::draw(const char* str) {
	TeXObjectInfo info;
	return draw(str, info, 1);
}

TeXObject* TeXInterface::drawUTF8(const char* str) {
	TeXObjectInfo info;
	string utf8 = str;
	decode_utf8_basic(utf8);
	return draw(utf8.c_str(), info, 1);
}

TeXObject* TeXInterface::draw(const char* str, int nblines) {
	TeXObjectInfo info;
	return draw(str, info, nblines);
}

TeXObject* TeXInterface::draw(const char* str, TeXObjectInfo& info, int nblines) {
	/* Load hash */
	tryLoadHash();
	/* Get object from hash */
	string obj_str = str;
	scaleObject(obj_str);
	TeXHashObject* hobj = getHashObject(obj_str);
	hobj->setNbLines(nblines);
	hobj->setUsed(true);
	/* Construct object and add to list */
	return drawObj(hobj, info);
}

void TeXInterface::scaleObject(string& obj_str) {
	int scaleMode = getScaleMode();
	if (scaleMode != TEX_SCALE_MODE_NONE) {
		double hei;
		TeXPreambleInfo* preamble = getCurrentPreamble();
		if (!preamble->hasFontSizes()) checkTeXFontSizes();
		g_get_hei(&hei);
		if (scaleMode == TEX_SCALE_MODE_FIXED) {
			int best_size = preamble->getBestSizeFixed(hei);
			if (best_size != -1) {
				string prefix = "{\\" + getFontSize(best_size)->getName() + " ";
				obj_str = prefix + obj_str + "}";
			}
		} else {
			int best_size = preamble->getBestSizeScaled(hei);
			if (best_size != -1) {
				double scale = hei / preamble->getFontSize(best_size);
				stringstream sstr;
				sstr << "\\scalebox{" << scale << "}{{\\";
				sstr << getFontSize(best_size)->getName();
				sstr << " " << obj_str << "}}";
				obj_str = sstr.str();
			}
		}
	}
}

TeXObject* TeXInterface::drawObj(TeXHashObject* hobj, TeXObjectInfo& info) {
	/* Update unknown fields of info structure */
	info.initializeAll();
	/* Check if dimensions available */
	double width = 1.0, height = 0.5, baseline = 0.1;
	if (hobj->hasDimensions()) {
		width = hobj->getWidth();
		height = hobj->getHeight();
		baseline = hobj->getBaseline();
		// cout << "OBJ: " << width << "x" << height << " + " << baseline << endl;
	}
	/* Handle justify */
	double xp = info.getXp();
	double yp = info.getYp();
	int just = info.getJustify();
	g_dotjust(&xp, &yp, 0.0, width, height, 0.0, just);
	if ((just & 0x100) != 0) {
		yp -= baseline;
	}
	/* Update bounding box */
	g_set_bounds(xp, yp+height);
	g_set_bounds(xp+width, yp);
	/* Create object */
	if ((info.getFlags() & TEX_OBJ_INF_DONT_PRINT) == 0 && !g_is_dummy_device()) {
		TeXObject* obj = new TeXObject();
		obj->setObject(hobj);
		obj->setXY(xp, yp);
		m_TeXObjects.push_back(obj);
		obj->setColor(info.getColor());
		/* Apply input transformation */
		double devx, devy;
		g_dev(xp, yp, &devx, &devy);
		obj->setDeviceXY(devx/72.0*CM_PER_INCH, devy/72.0*CM_PER_INCH);
		/* Find out if text should be rotated */
		double angle = g_get_angle_deg();
		if (fabs(angle) > 1e-6) {
			obj->setAngle(angle);
		}
		/* Return object */
		return obj;
	} else {
		return NULL;
	}
}

void TeXInterface::checkObjectDimensions() {
	PSGLEDevice* psdev = (PSGLEDevice*)g_get_device_ptr();
	double x0 = (double)psdev->getBBXOrigin()/72.0*CM_PER_INCH;
	double y0 = (double)psdev->getBBYOrigin()/72.0*CM_PER_INCH;
	double x1 = (double)(psdev->getBBXOrigin()+psdev->getBBWidth())/72.0*CM_PER_INCH;
	double y1 = (double)(psdev->getBBYOrigin()+psdev->getBBHeight())/72.0*CM_PER_INCH;
	for (int i = 0; i < m_TeXObjects.size(); i++) {
		TeXObject* obj = m_TeXObjects[i];
		TeXHashObject* hobj = obj->getObject();
		if (hobj != NULL && hobj->hasDimensions()) {
			double cos_alpha = cos(obj->getAngle()*GLE_PI/180.0);
			double sin_alpha = sin(obj->getAngle()*GLE_PI/180.0);
			double ox0 = obj->getDXp();
			double oy0 = obj->getDYp();
			double ox1 = ox0 + hobj->getWidth()*cos_alpha;
			double oy1 = oy0 + hobj->getWidth()*sin_alpha;
			double ox2 = ox1 - hobj->getHeight()*sin_alpha;
			double oy2 = oy1 + hobj->getHeight()*cos_alpha;
			double ox3 = ox0 - hobj->getHeight()*sin_alpha;
			double oy3 = oy0 + hobj->getHeight()*cos_alpha;
			if (ox0 < x0 || ox0 > x1 || oy0 < y0 || oy0 > y1 ||
			    ox1 < x0 || ox1 > x1 || oy1 < y0 || oy1 > y1 ||
			    ox2 < x0 || ox2 > x1 || oy2 < y0 || oy2 > y1 ||
			    ox3 < x0 || ox3 > x1 || oy3 < y0 || oy3 > y1) {
			    	string error = "TeX object '";
				hobj->addFirstLine(&error);
				error += "' outside bounding box";
				g_message(error.c_str());
				inc_nb_errors();
			}
		}
	}
}

int TeXInterface::createObj(const char* str) {
	/* Load hash */
	tryLoadHash();
	/* Create object given string */
	string obj_str = str;
	scaleObject(obj_str);
	int result = getHashObjectIndex(obj_str);
	m_TeXHash[result]->setUsed(true);
	return result;
}

void TeXInterface::addHashObject(TeXHashObject* obj) {
	m_TeXHash.push_back(obj);
}

int TeXInterface::getHashObjectIndex(const string& line) {
	for (int i = 0; i < m_TeXHash.size(); i++) {
		if (m_TeXHash[i]->getLine() == line) {
			return i;
		}
	}
	TeXHashObject* hobj = new TeXHashObject(line);
	addHashObject(hobj);
	m_HashModified = 1;
	return m_TeXHash.size()-1;
}

TeXHashObject* TeXInterface::getHashObject(const string& line) {
	return m_TeXHash[getHashObjectIndex(line)];
}

TeXHashObject* TeXInterface::getHashObject(int idx) {
	return m_TeXHash.getHashObject(idx);
}

void TeXInterface::initialize(const string& name, const string& oname) {
	/* cleanUp */
	cleanUpObjects();
	cleanUpHash();
	/* Initialize flags */
	m_HashLoaded = TEX_INTERFACE_HASH_LOADED_NONE;
	m_HashModified = 0;
	/* File name without extension */
	string dir, file;
	GetMainName(name, m_MainName);
	GetMainName(oname, m_MainOutputName);
	/* Hash file name without extension */
	SplitFileName(m_MainName, m_HashName, file);
	m_HashName += ".gle";
	m_HashName += DIR_SEP;
	m_HashName += file;
	m_HashName += DIR_SEP;
	m_HashName += file;
	m_HashName += "_tex";
	/* Get font sizes */
	initTeXFontScales();
}

void TeXInterface::reset() {
	resetPreamble();
	cleanUpObjects();
	setScaleMode(TEX_SCALE_MODE_FIXED);
	m_HashModified = 0;
	/* Remove unused objects */
	for (int i = m_TeXHash.size()-1; i >= 0; i--) {
		TeXHashObject* hobj = m_TeXHash[i];
		if (!hobj->isUsed()) {
			delete hobj;
			m_TeXHash.erase(m_TeXHash.begin() + i);
		}
	}
}

void TeXInterface::resetPreamble() {
	m_Preambles.selectDefault();
}

void TeXInterface::cleanUpObjects() {
	for (int i = 0; i < m_TeXObjects.size(); i++) {
		delete m_TeXObjects[i];
	}
	m_TeXObjects.clear();
}

void TeXInterface::cleanUpHash() {
	m_TeXHash.cleanUp();
}

void TeXInterface::tryLoadHash() {
	if (m_HashLoaded != TEX_INTERFACE_HASH_LOADED_FULL) {
		if (m_HashLoaded != TEX_INTERFACE_HASH_LOADED_PARTIAL) {
			loadTeXLines();
		}
		m_TeXHash.loadTeXPS(m_HashName);
		m_HashModified = 0;
		m_HashLoaded = TEX_INTERFACE_HASH_LOADED_FULL;
	}
}

void TeXInterface::loadTeXLines() {
	string strfile = m_HashName;
	strfile += ".texlines";
	ifstream str_file(strfile.c_str());
	if (str_file.is_open()) {
		string line;
		while (!str_file.eof()) {
			int len = ReadFileLine(str_file, line);
			if (len != 0) {
				if (strncmp("tex", line.c_str(), 3)  == 0) {
					line.erase(0, 4);
					TeXHashObject* hobj = new TeXHashObject(line);
					addHashObject(hobj);
				} else {
					line.erase(0, 9);
					string object_string;
					int nblines = atoi(line.c_str());
					for (int i = 0; i < nblines; i++) {
						ReadFileLine(str_file, line);
						if (object_string.length() == 0) {
							object_string = line;
						} else {
							object_string += "\7";
							object_string += line;
						}
					}
					TeXHashObject* hobj = new TeXHashObject(object_string);
					addHashObject(hobj);
				}
			}
		}
		str_file.close();
	}
}

void TeXInterface::writeInc(ostream& out, const char* prefix) {
	/* First output include for GLE .eps/.pdf file */
	out << "\\setlength{\\unitlength}{1cm}%" << endl;
	// out << "\\begingroup\\makeatletter\\ifx\\SetFigFont\\undefined%" << endl;
	// out << "\\gdef\\SetFigFont#1#2#3#4#5{%" << endl;
	// out << "   \\reset@font\\fontsize{#1}{#2pt}%" << endl;
	// out << "   \\fontfamily{#3}\\fontseries{#4}\\fontshape{#5}%" << endl;
	// out << "   \\selectfont}%" << endl;
	// out << "\\fi\\endgroup%" << endl;
	/* Get page size */
	double width, height, pic_a, pic_b;
	if (g_is_fullpage()) {
		g_get_pagesize(&width, &height);
		pic_a = width;
		pic_b = height;
	} else {
		g_get_usersize(&width, &height);
		pic_a = width + 0.075;
		pic_b = height + 0.075;
	}
	double pic_c = 0;
	double pic_d = 0;
	/* Output picture commands */
	out << "\\noindent{}\\begin{picture}(" << pic_a << "," << pic_b << ")";
	out <<                 "(" << pic_c << "," << pic_d << ")%" << endl;
	out << "\\put(0,0)";
	// double offs = -CM_PER_INCH/72;
	// out << "\\put(" << offs << "," << offs << ")";
	string incname;
	SplitFileNameNoDir(m_MainOutputName, incname);
	out << "{\\includegraphics{" << prefix << incname << "_inc}}" << endl;
	for (int i = 0; i < m_TeXObjects.size(); i++) {
		TeXObject* obj = m_TeXObjects[i];
		obj->output(out);
	}
	out << "\\end{picture}" << endl;
}

void TeXInterface::createInc(const string& prefix) {
	if (m_TeXObjects.size() != 0) {
		string inc_name = m_MainOutputName;
		inc_name += ".inc";
		ofstream inc_file(inc_name.c_str());
		writeInc(inc_file, prefix.c_str());
		inc_file.close();
	}
}

void TeXInterface::createPreamble(ostream& tex_file) {
	ConfigSection* tex = g_Config.getSection(GLE_CONFIG_TEX);
	CmdLineArgSet* texsys =	(CmdLineArgSet*)tex->getOptionValue(GLE_TEX_SYSTEM);
	tex_file << getDocumentClass() << endl;
	if (texsys->hasValue(GLE_TEX_SYSTEM_VTEX)) {
		tex_file << "\\usepackage{graphics}" << endl;
	} else {
		tex_file << "\\usepackage[dvips]{graphics}" << endl;
	}
	for (int i = 0; i < getNbPreamble(); i++) {
		tex_file << getPreamble(i) << endl;
	}
}

void TeXInterface::createTeX(bool usegeom) {
	if (m_TeXObjects.size() != 0) {
		int m_type;
		double m_width, m_height, pic_a, pic_b;
		if (g_is_fullpage()) {
			g_get_pagesize(&m_width, &m_height, &m_type);
			pic_a = m_width;
			pic_b = m_height;
		} else {
			g_get_usersize(&m_width, &m_height);
			pic_a = m_width + 0.075;
			pic_b = m_height + 0.075;
			m_type = GLE_PAPER_UNKNOWN;
		}
		string tex_name = m_MainOutputName;
		tex_name += ".tex";
		ofstream tex_file(tex_name.c_str());
		createPreamble(tex_file);
		tex_file << "\\usepackage{color}" << endl;
		if (usegeom) {
			tex_file << "\\usepackage{geometry}" << endl;
			tex_file << "\\geometry{%" << endl;
			tex_file << "  paperwidth=" << pic_a << "cm," << endl;
			tex_file << "  paperheight=" << pic_b << "cm," << endl;
			tex_file << "  left=0in," << endl;
			tex_file << "  right=0in," << endl;
			tex_file << "  top=0in," << endl;
			tex_file << "  bottom=0in" << endl;
			tex_file << "}" << endl;
		}
		tex_file << "\\pagestyle{empty}" << endl;
		tex_file << "\\begin{document}" << endl;
		writeInc(tex_file, "");
		tex_file << "\\end{document}" << endl;
		tex_file.close();
	}
}

int TeXInterface::tryCreateHash() {
	if (m_HashModified != 0 && m_TeXObjects.size() != 0) {
		createHiddenDir();
		saveTeXLines();
		m_TeXHash.saveTeXPS(m_HashName, this);
		createTeXPS();
		m_HashLoaded = TEX_INTERFACE_HASH_LOADED_PARTIAL;
		return 1;
	} else {
		return 0;
	}
}

void TeXInterface::createHiddenDir() {
	string dir;
	GetDirName(m_HashName, dir);
	EnsureMkDir(dir);
}

void TeXInterface::saveTeXLines() {
	string strfile = m_HashName;
	strfile += ".texlines";
	ofstream str_file(strfile.c_str());
	for (int i = 0; i < m_TeXHash.size(); i++) {
		TeXHashObject* obj = m_TeXHash[i];
		if (obj->isUsed()) obj->outputLog(str_file);
	}
	str_file.close();
}

void TeXInterface::createTeXPS() {
	createTeXPS(m_HashName);
}

void TeXInterface::createTeXPS(const string& filestem) {
	string dir, file;
	SplitFileName(filestem, dir, file);
	if (!run_latex(dir, file)) return;
	run_dvips(filestem, "", false);
}

void TeXInterface::initTeXFontScales() {
	addSize(new TeXSize("tiny"));
	addSize(new TeXSize("scriptsize"));
	addSize(new TeXSize("footnotesize"));
	addSize(new TeXSize("small"));
	addSize(new TeXSize("normalsize"));
	addSize(new TeXSize("large"));
	addSize(new TeXSize("Large"));
	addSize(new TeXSize("LARGE"));
	addSize(new TeXSize("huge"));
	addSize(new TeXSize("Huge"));
}

void TeXInterface::checkTeXFontSizes() {
	TeXPreambleInfo* preamble = getCurrentPreamble();
	if (preamble->hasFontSizes()) return;
	/* Create .gle subdirectory */
	string dir, file;
	SplitFileName(m_MainName, dir, file);
	dir += ".gle";
	EnsureMkDir(dir);
	dir += DIR_SEP;
	dir += "texpreamble";
	/* Try to load from file */
	m_Preambles.load(dir, this);
	if (preamble->hasFontSizes()) return;
	/* Font sizes not found, recompute */
	TeXHash tex_hash;
	for (int i = 0; i < getNbFontSizes(); i++) {
		string obj_name;
		TeXSize* size = getFontSize(i);
		size->createObject(&obj_name);
		TeXHashObject* obj = new TeXHashObject(obj_name);
		tex_hash.push_back(obj);
		obj->setUsed(true);
	}
	/* Get sizes of these fonts */
	tex_hash.saveTeXPS(dir, this);
	createTeXPS(dir);
	tex_hash.loadTeXPS(dir);
	/* Save this info in the current preamble */
	retrieveTeXFontSizes(tex_hash, preamble);
	/* Save font sizes */
	m_Preambles.save(dir);
}

void TeXInterface::retrieveTeXFontSizes(TeXHash& tex_hash, TeXPreambleInfo* preamble) {
	for (int i = 0; i < getNbFontSizes(); i++) {
		string obj_name;
		TeXSize* size = getFontSize(i);
		size->createObject(&obj_name);
		TeXHashObject* obj = tex_hash.getHashObjectOrNULL(obj_name);
		if (obj == NULL || !obj->hasDimensions()) {
			cout << ">>> error: did not get size for TeX font!" << endl;
		} else {
			// Round values by converting them to a string
			// -> Results in same rounding as if values are read from .pinfo file!
			stringstream str_value;
			double size_value = obj->getHeight() * FONT_HEI_FACTOR;
			str_value << size_value;
			str_value >> size_value;
			preamble->setFontSize(i, size_value);
			// cout << "font: " << size->getName() << " size: " << size_value << endl;
		}
	}
	preamble->setHasFontSizes(true);
}

bool TeXPreambleKey::equals(const TeXPreambleKey* key) const {
	if (getDocumentClass() != key->getDocumentClass()) return false;
	int nb = getNbPreamble();
	if (nb != key->getNbPreamble()) return false;
	for (int i = 0; i < nb; i++) {
		if (getPreamble(i) != key->getPreamble(i)) return false;
	}
	return true;
}

void TeXPreambleKey::copyFrom(const TeXPreambleKey* other) {
	setDocumentClass(other->getDocumentClass());
	int nb = other->getNbPreamble();
	for (int i = 0; i < nb; i++) {
		addPreamble(other->getPreamble(i));
	}
}

TeXPreambleInfo::TeXPreambleInfo() {
	m_HasFontSizes = false;
}

void TeXPreambleInfo::setFontSize(int font, double size) {
	while (m_FontSizes.size() <= font) {
		m_FontSizes.push_back(0.0);
	}
	m_FontSizes[font] = size;
}

double TeXPreambleInfo::getFontSize(int font) {
	return font >= m_FontSizes.size() ? 1.0 : m_FontSizes[font];
}

void TeXPreambleInfo::save(ostream& os) {
	int nb = getNbPreamble();
	os << "preamble: " << nb << endl;
	os << getDocumentClass() << endl;
	for (int i = 0; i < nb; i++) {
		os << getPreamble(i) << endl;
	}
	for (int i = 0; i < getNbFonts(); i++) {
		if (i != 0) os << " ";
		os << getFontSize(i);
	}
	os << endl;
}

void TeXPreambleInfo::load(istream& is, TeXInterface* iface) {
	for (int i = 0; i < iface->getNbFontSizes(); i++) {
		double size;
		is >> size;
		setFontSize(i, size);
	}
	setHasFontSizes(true);
}

int TeXPreambleInfo::getBestSizeFixed(double hei) {
	int best_size = -1;
	double best_dist = 1e30;
	for (int i = 0; i < getNbFonts(); i++) {
		double dist = fabs(hei - getFontSize(i));
		if (dist < best_dist) {
			best_dist = dist;
			best_size = i;
		}
	}
	return best_size;
}

int TeXPreambleInfo::getBestSizeScaled(double hei) {
	for (int i = 0; i < getNbFonts(); i++) {
		double size_i = getFontSize(i);
		if (size_i >= hei) {
			return i;
		}
	}
	return getNbFonts()-1;
}

TeXPreambleInfoList::TeXPreambleInfoList() {
	m_Current = new TeXPreambleInfo();
	m_Current->setDocumentClass("\\documentclass{article}");
	addPreamble(m_Current);
}

TeXPreambleInfoList::~TeXPreambleInfoList() {
	for (int i = 0; i < getNbPreambles(); i++) {
		delete getPreamble(i);
	}
}

TeXPreambleInfo* TeXPreambleInfoList::findOrAddPreamble(const TeXPreambleKey* pre_key) {
	for (int i = 0; i < getNbPreambles(); i++) {
		TeXPreambleInfo* info = getPreamble(i);
		if (pre_key->equals(info)) return info;
	}
	TeXPreambleInfo* info = new TeXPreambleInfo();
	info->copyFrom(pre_key);
	addPreamble(info);
	return info;
}

void TeXPreambleInfoList::save(const string& filestem) {
	string file = filestem + ".pinfo";
	ofstream pream_file(file.c_str());
	for (int i = 0; i < getNbPreambles(); i++) {
		if (getPreamble(i)->hasFontSizes()) {
			getPreamble(i)->save(pream_file);
		}
	}
	pream_file.close();
}

void TeXPreambleInfoList::load(const string& filestem, TeXInterface* iface) {
	string file = filestem + ".pinfo";
	ifstream pream_file(file.c_str());
	if (pream_file.is_open()) {
		string line;
		TeXPreambleKey pre_key;
		while (!pream_file.eof()) {
			int len = ReadFileLine(pream_file, line);
			if (len != 0) {
				if (strncmp("preamble:", line.c_str(), 9)  == 0) {
					line.erase(0, 10);
					int nblines = atoi(line.c_str());
					ReadFileLine(pream_file, line);
					pre_key.clear();
					pre_key.setDocumentClass(line);
					for (int i = 0; i < nblines; i++) {
						ReadFileLine(pream_file, line);
						pre_key.addPreamble(line);
					}
					TeXPreambleInfo* info = findOrAddPreamble(&pre_key);
					info->load(pream_file, iface);
				} else {
					/* error in file format? */
					return;
				}
			}
		}
	}
	pream_file.close();
}

TeXHash::TeXHash() {
}

TeXHash::~TeXHash() {
	cleanUp();
}

void TeXHash::cleanUp() {
	for (int i = 0; i < size(); i++) {
		delete get(i);
	}
	clear();
}

TeXHashObject* TeXHash::getHashObject(int idx) {
	if (idx >= size()) return NULL;
	return get(idx);
}

TeXHashObject* TeXHash::getHashObjectOrNULL(const string& line) {
	for (int i = 0; i < size(); i++) {
		if (get(i)->getLine() == line) {
			return get(i);
		}
	}
	return NULL;
}

void TeXHash::saveTeXPS(const string& filestem, TeXInterface* iface) {
	string tex = filestem;
	tex += ".tex";
	ofstream hash_file(tex.c_str());
	/* Output preamble */
	iface->createPreamble(hash_file);
	hash_file << "\\pagestyle{empty}" << endl;
	hash_file << "\\begin{document}" << endl;
	/* First object is for calibrating */
	hash_file << "\\newpage" << endl;
	hash_file << "\\noindent{}\\rule{1cm}{0.025cm}\\framebox{\\rule{1cm}{1cm}}" << endl << endl;
	/* Output other objects */
	for (int i = 0; i < size(); i++) {
		TeXHashObject* obj = get(i);
		if (obj->isUsed()) obj->outputMeasure(hash_file);
	}
	hash_file << "\\end{document}" << endl;
	hash_file.close();
}

void TeXHash::loadTeXPS(const string& filestem) {
	int objindex = -1;
	double adjWidth = 0.0, adjHeight = 0.0, adjBaseline = 0.0;
	string ps_name = filestem;
	ps_name += ".ps";
	StreamTokenizerMax tokens(ps_name,' ', 50);
	while (tokens.hasMoreTokens()) {
		const char* token = tokens.nextToken();
		if (str_i_equals(token, "%%PAGE:")) {
			int found = 0;
			FourDoubleList list;
			double unitsPerCm = 0.0, width = 0.0, height = 0.0, baseline = 0.0, adjustY = 0.0;
			while (found < 3 && tokens.hasMoreTokens()) {
				token = tokens.nextToken();
				if (str_i_equals(token, "v")) {
					double d1 = list.get(0);
					double d2 = list.get(1);
					double d3 = list.get(2);
					double d4 = list.get(3);
					switch (found) {
						case 0:	/* Units/cm and baseline */
							unitsPerCm = d3;
							adjustY = d2;
							break;
						case 1:	/* Width */
							width = d3;
							break;
						case 2:	/* Height and baseline */
							height = d4;
							baseline = d2-adjustY;
							break;

					}
					// cout << "v = " << d1 << " " << d2 << " " << d3 << " " << d4 << endl;
					found++;
				} else {
					char* pos;
					double value = strtod(token, &pos);
					list.add(value);
				}
			}
			if (found == 3 && unitsPerCm != 0.0) {
				width /= unitsPerCm; height /= unitsPerCm; baseline /= unitsPerCm;
				if (objindex == -1) {
					/* Reference object is 1cm x 1cm square, no baseline */
					adjWidth = width-1.0;
					adjHeight = height-1.0;
					adjBaseline = baseline;
				} else {
					width -= adjWidth; height -= adjHeight;	baseline -= adjBaseline;
					TeXHashObject* hobj = getHashObject(objindex);
					if (hobj != NULL) {
						// cout << width << "x" << height << " + " << baseline << " " << hobj->getLine() << endl;
						hobj->setDimension(width, height, baseline);
					}
				}
				// cout << "width = " << width << " height = " << height << endl;
			}
			objindex++;
		}
	}

	tokens.close();
}

TeXSize::TeXSize(const char* name) {
	m_Name = name;
}

void TeXSize::createObject(string* name) {
	*name =  "{\\";
	*name += getName();
	*name += " H}";
}

TeXObject::TeXObject() {
	m_Xp = 0.0;
	m_Yp = 0.0;
	m_DXp = 0.0;
	m_DYp = 0.0;
	m_Angle = 0.0;
	m_Object = NULL;
}

void TeXObject::output(ostream& os) {
	if (!hasObject()) {
		return;
	}
	int closeb = 1;
	double angle = m_Angle;
	double pic_x = m_DXp;
	double pic_y = m_DYp;
	os << "\\put(" << pic_x << "," << pic_y << "){";
	if (angle != 0.0) {
		os << "\\rotatebox{" << angle << "}{";
		closeb++;
	}
	os << "\\makebox(0,0)[lb]{";
	if (!isBlack()) {
		rgb01 c1;
		g_colortyp_to_rgb01(getColor(), &c1);
		os << "\\color[rgb]{" << c1.red << "," << c1.green << "," << c1.blue << "}";
	}
	getObject()->outputLines(os);
	for (int i = 0; i < closeb; i++) {
		os << "}";
	}
	os << "}" << endl;
}

void TeXObject::getDimensions(double* x1, double *y1, double *x2, double *y2) {
	*x1 = m_Xp;
	*y1 = m_Yp;
	*x2 = m_Xp + getWidth();
	*y2 = m_Yp + getHeight();
}

int TeXObject::isBlack() {
	return g_is_black(getColor());
}

TeXHashObject::TeXHashObject(const string& line) : m_Line(line) {
	m_Width = 10.0;
	m_Height = 10.0;
	m_Baseline = 0.0;
	m_HasDimensions = 0;
	m_Used = 0;
	m_NbLines = 0;
}

void TeXHashObject::outputLines(ostream& os) {
	if (getNbLines() <= 1) {
		os << getLine();
	} else {
		char_separator sep("\7");
		tokenizer<char_separator> tok(getLine(), sep);
		os << "%" << endl;
		while (tok.has_more()) {
			os << tok.next_token() << endl;
		}
	}
}

void TeXHashObject::addFirstLine(string* str) {
	if (getNbLines() <= 1) {
		(*str) += getLine();
	} else {
		char_separator sep("\7");
		tokenizer<char_separator> tok(getLine(), sep);
		if (tok.has_more()) (*str) += tok.next_token();
	}
}

void TeXHashObject::outputMeasure(ostream& os) {
	os << "\\newpage" << endl;
	os << "\\noindent{}\\rule{1cm}{0.025cm}\\framebox{";
	outputLines(os);
	os << "}" << endl << endl;
}

void TeXHashObject::outputLog(ostream& os) {
	if (getNbLines() <= 1) {
		os << "tex " << getLine() << endl;
	} else {
		char_separator sep("\7");
		tokenizer<char_separator> tok(getLine(), sep);
		os << "multitex " << getNbLines() << endl;
		while (tok.has_more()) {
			os << tok.next_token() << endl;
		}
	}
}

void TeXHashObject::setDimension(double width, double height, double baseline) {
	m_Width = width;
	m_Height = height;
	m_Baseline = baseline;
	m_HasDimensions = 1;
}

TeXObjectInfo::TeXObjectInfo() {
	m_Status = 0;
}

void TeXObjectInfo::setJustify(int just) {
	m_Just = just;
	m_Status |= TEX_OBJ_INF_HAS_JUSTIFY;
}

void TeXObjectInfo::setColor(colortyp* color) {
	m_Color = *color;
	m_Status |= TEX_OBJ_INF_HAS_COLOR;
}

void TeXObjectInfo::setPosition(double xp, double yp) {
	m_Xp = xp; m_Yp = yp;
	m_Status |= TEX_OBJ_INF_HAS_POSITION;
}

void TeXObjectInfo::initializeAll() {
	if ((m_Status & TEX_OBJ_INF_HAS_POSITION) == 0) {
		g_get_xy(&m_Xp, &m_Yp);
	}
	if ((m_Status & TEX_OBJ_INF_HAS_JUSTIFY) == 0) {
		g_get_just(&m_Just);
	}
	if ((m_Status & TEX_OBJ_INF_HAS_COLOR) == 0) {
		g_get_colortyp(&m_Color);
	}
}

bool create_tex_eps_file(const string& fname) {
	string main_name, file, dir;
	ConfigSection* tex = g_Config.getSection(GLE_CONFIG_TEX);
	CmdLineArgSet* texsys =	(CmdLineArgSet*)tex->getOptionValue(GLE_TEX_SYSTEM);
	GetMainName(fname, main_name);
	SplitFileName(main_name, dir, file);
	if (!run_latex(dir, file)) return false;
	if (!run_dvips(main_name, "", true)) return false;
        DeleteFileWithExt(main_name, ".aux");
	if (texsys->hasValue(GLE_TEX_SYSTEM_VTEX)) {
	        DeleteFileWithExt(main_name, ".ps");
	} else {
		DeleteFileWithExt(main_name, ".dvi");
	}
	DeleteFileWithExt(main_name, ".log");
	return true;
}

bool create_ps_file_latex_dvips(const string& fname) {
	string main_name, file, dir;
	ConfigSection* tex = g_Config.getSection(GLE_CONFIG_TEX);
	CmdLineArgSet* texsys =	(CmdLineArgSet*)tex->getOptionValue(GLE_TEX_SYSTEM);
	GetMainName(fname, main_name);
	SplitFileName(main_name, dir, file);
	if (!run_latex(dir, file)) return false;
	if (!run_dvips(main_name, "", false)) return false;
        DeleteFileWithExt(main_name, ".aux");
	if (!texsys->hasValue(GLE_TEX_SYSTEM_VTEX)) {
		DeleteFileWithExt(main_name, ".dvi");
	}
	DeleteFileWithExt(main_name, ".log");
	return true;
}

bool adjust_bounding_box(const string& name, int* width, int* height) {
	int b1 = 0, b2 = 0, b3 = 0, b4 = 0;
	string in_fname = name + ".eps";
	string out_fname = name + "_temp.eps";
	/* Find bounding box */
	StreamTokenizerMax tokens(in_fname,' ', 50);
	while (tokens.hasMoreTokens()) {
		const char* token = tokens.nextToken();
		if (str_i_str(token, "BoundingBox") != NULL) {
			b1 = tokens.hasMoreTokens() ? atoi(tokens.nextToken()) : 0;
			b2 = tokens.hasMoreTokens() ? atoi(tokens.nextToken()) : 0;
			b3 = tokens.hasMoreTokens() ? atoi(tokens.nextToken()) : 0;
			b4 = tokens.hasMoreTokens() ? atoi(tokens.nextToken()) : 0;
			break;
		}
	}
	if (b1 != 0 || b2 != 0 || b3 != 0 || b4 != 0) {
		ofstream out(out_fname.c_str());
		out << "%!PS-Adobe-2.0 EPSF-2.0" << endl;
		out << "%%BoundingBox: 0 0 " << (b3-b1) << " " << (b4-b2) << endl;
		if (b1 != 0 || b2 != 0) {
			out << "gsave " << (-b1) << " " << (-b2) << " translate" << endl;
			*width = b3-b1+1;
			*height = b4-b2+1;
		}
		char ch;
		bool has_output = false;
		ifstream& strm = tokens.getFile();
		while (!strm.eof()) {
			strm.read(&ch, 1);
			if (ch == '%') {
				/* Remove other comments from file */
				while (!strm.eof()) {
					strm.read(&ch, 1);
					if (ch == '\n') break;
				}
				if (has_output) out << endl;
			} else {
				if (ch != '\r') {
					out << ch;
					has_output = ch != '\n';
				}
			}
		}
		if (b1 != 0 || b2 != 0) {
			out << endl << "grestore" << endl;
		}
		out.close();
	}
	tokens.close();
	return true;
}

bool create_bitmap_file(const string& fname, int device, int dpi, int width, int height, bool bw, bool tex) {
	string main_name;
	char temp_str[80];
	GetMainName(fname, main_name);
	string gsargs = "-q -DNOPLATFONTS -dTextAlphaBits=4 -dGraphicsAlphaBits=4 -dBATCH -dNOPAUSE -r";
	sprintf(temp_str, "%d", dpi);
	gsargs += temp_str;
	gsargs += " -g";
	if (tex) {
		adjust_bounding_box(main_name, &width, &height);
	}
	int img_wd = (int)floor((double)dpi/72.0*width+1);
	int img_hi = (int)floor((double)dpi/72.0*height+1);
	sprintf(temp_str, "%dx%d", img_wd, img_hi);
	gsargs += temp_str;
	gsargs += " -sDEVICE=";
	switch (device) {
		case GLE_DEVICE_PNG: gsargs += bw ? "pnggray" : "png16m"; break;
		case GLE_DEVICE_JPEG: gsargs += bw ? "jpeggray" : "jpeg"; break;
	}
	gsargs += " -sOutputFile=\"";
	gsargs += main_name;
	switch (device) {
		case GLE_DEVICE_PNG: gsargs += ".png"; break;
		case GLE_DEVICE_JPEG: gsargs += ".jpg"; break;
	}
	gsargs += "\" ";
	gsargs += main_name;
	if (tex) {
		gsargs += "_temp";
	}
	gsargs += ".eps";
	return run_ghostscript(gsargs);
}

bool create_pdf_file(const string& fname, int dpi, int width, int height, bool tex) {
	string main_name;
	char temp_str[80];
	GetMainName(fname, main_name);
	stringstream gsargs;
	gsargs << "-q";
	int compr_mode = g_get_pdf_image_format();
	switch (compr_mode) {
		case PDF_IMG_COMPR_AUTO:
			gsargs << " -dAutoFilterColorImages=true";
			gsargs << " -dAutoFilterGrayImages=true";
			gsargs << " -dEncodeColorImages=true";
			gsargs << " -dEncodeGrayImages=true";
			gsargs << " -dEncodeMonoImages=false"; break;
		case PDF_IMG_COMPR_ZIP:
			gsargs << " -dAutoFilterColorImages=false";
			gsargs << " -dAutoFilterGrayImages=false";
			gsargs << " -dEncodeColorImages=true";
			gsargs << " -dEncodeGrayImages=true";
			gsargs << " -dEncodeMonoImages=true";
			gsargs << " -dColorImageFilter=/FlateEncode";
			gsargs << " -dGrayImageFilter=/FlateEncode";
			gsargs << " -dMonoImageFilter=/FlateEncode"; break;
		case PDF_IMG_COMPR_JPEG:
			gsargs << " -dAutoFilterColorImages=false";
			gsargs << " -dAutoFilterGrayImages=false";
			gsargs << " -dEncodeColorImages=true";
			gsargs << " -dEncodeGrayImages=true";
			gsargs << " -dEncodeMonoImages=true";
			gsargs << " -dColorImageFilter=/DCTEncode";
			gsargs << " -dGrayImageFilter=/DCTEncode";
			gsargs << " -dMonoImageFilter=/FlateEncode"; break;
		case PDF_IMG_COMPR_PS:
			gsargs << " -dAutoFilterColorImages=false";
			gsargs << " -dAutoFilterGrayImages=false";
			gsargs << " -dEncodeColorImages=false";
			gsargs << " -dEncodeGrayImages=false";
			gsargs << " -dEncodeMonoImages=false"; break;
	}
	gsargs << " -dBATCH -dNOPAUSE -r" << dpi;
	int img_wd = (int)floor((double)dpi/72.0*width+0.5);
	int img_hi = (int)floor((double)dpi/72.0*height+0.5);
	gsargs << " -g" << img_wd << "x" << img_hi;
	gsargs << " -sDEVICE=pdfwrite -sOutputFile=\"" << main_name << ".pdf\" ";
	gsargs << main_name;
	if (tex) {
		int dummy_w, dummy_h;
		adjust_bounding_box(main_name, &dummy_w, &dummy_h);
		gsargs << "_temp";
	}
	gsargs << ".eps";
	return run_ghostscript(gsargs.str());
}

bool show_process_error(int result, const char* procname, const string& cmd) {
	if (result == GLE_SYSTEM_OK) {
		return true;
	} else {
		gprint("Error running %s: {%s}", procname, cmd.c_str());
		return false;
	}
}

bool create_pdf_file_pdflatex(const string& fname) {
	string main_name, file, dir;
	GetMainName(fname, main_name);
	SplitFileName(main_name, dir, file);
	ConfigSection* tools = g_Config.getSection(GLE_CONFIG_TOOLS);
	string pdftex_cmd = ((CmdLineArgString*)tools->getOptionValue(GLE_TOOL_PDFTEX_CMD))->getValue();
	str_try_add_quote(pdftex_cmd);
	string cmdline = pdftex_cmd + string(" ") + file + ".tex";
	cout << "[Running: " << cmdline << "]" << endl;
	int result = GLESystem(cmdline);
	bool res = show_process_error(result, "PdfLaTeX", cmdline);
        DeleteFileWithExt(main_name, ".aux");
	DeleteFileWithExt(main_name, ".log");
	return res;
}

bool run_ghostscript(const string& args) {
	ConfigSection* tools = g_Config.getSection(GLE_CONFIG_TOOLS);
	string gs_cmd = ((CmdLineArgString*)tools->getOptionValue(GLE_TOOL_GHOSTSCRIPT_CMD))->getValue();
	str_try_add_quote(gs_cmd);
	string cmdline = gs_cmd + string(" ") + args;
	cout << "[Running: " << cmdline << "]" << endl;
	int result = GLESystem(cmdline);
	return show_process_error(result, "GhostScript", cmdline);
}

bool run_latex(const string& dir, const string& file) {
	string cmdline, crdir;
	if (dir != "") {
		GLEGetCrDir(&crdir);
		if (!GLEChDir(dir)) {
			gprint("Can't find directory: {%s}", dir.c_str());
			return false;
		}
	}
	ConfigSection* tools = g_Config.getSection(GLE_CONFIG_TOOLS);
	string latex_cmd = ((CmdLineArgString*)tools->getOptionValue(GLE_TOOL_LATEX_CMD))->getValue();
	str_try_add_quote(latex_cmd);
	cmdline = latex_cmd + string(" ") + file + ".tex";
	cout << "[Running: " << cmdline << "]" << endl;
	int result = GLESystem(cmdline);
	if (crdir.length() != 0) GLEChDir(crdir);
	return show_process_error(result, "LaTeX", cmdline);
}

bool run_dvips(const string& file, const char* opts, bool eps) {
        string cmdline;
	ConfigSection* tex = g_Config.getSection(GLE_CONFIG_TEX);
	CmdLineArgSet* texsys =	(CmdLineArgSet*)tex->getOptionValue(GLE_TEX_SYSTEM);
	if (texsys->hasValue(GLE_TEX_SYSTEM_VTEX)) {
		// VTeX creates directly PS or PDF, no DVI!
		// use ghostscript to convert PS to EPS
		if (eps) {
			string gsargs;
            		gsargs += "-dNOPAUSE -sDEVICE=epswrite -sOutputFile=";
	            	gsargs += file;
			// for one-page PS files use:
			gsargs += ".eps -q -sBATCH ";
			// for multi-page PS files use (%03d -> 001,002,...):
			// gsargs += ".%03d -q -sBATCH ";
			gsargs += file;
			gsargs += ".ps";
			return run_ghostscript(gsargs);
		} else {
			return true;
		}
	} else {
		ConfigSection* tools = g_Config.getSection(GLE_CONFIG_TOOLS);
		string dvips_cmd = ((CmdLineArgString*)tools->getOptionValue(GLE_TOOL_DVIPS_CMD))->getValue();
		str_try_add_quote(dvips_cmd);
		cmdline += dvips_cmd + string(" ") + opts;
		if (eps) cmdline += " -E";
		cmdline += " -o ";
		cmdline += file;
		cmdline += eps ? ".eps " : ".ps ";
		cmdline += file;
		cmdline += ".dvi";
		cout << "[Running: " << cmdline << "]" << endl;
		int result = GLESystem(cmdline);
		return show_process_error(result, "DVIPS", cmdline);
	}
}

void begin_tex_preamble(int *pln, int *pcode, int *cp) {
	TeXInterface* iface = TeXInterface::getInstance();
	iface->resetPreamble();
	// Start with pcode from the next line
	(*pln)++;
	begin_init();
	TeXPreambleKey pre_key;
	pre_key.setDocumentClass(iface->getDocumentClass());
	while (true) {
		int st = begin_token(&pcode,cp,pln,srclin,tk,&ntk,outbuff);
		if (!st) {
			/* exit loop */
			break;
		}
		string mline = srclin;
		str_trim_both(mline);
		if (str_i_str(mline.c_str(), "\\documentclass") != NULL) {
			pre_key.setDocumentClass(mline);
		} else {
			pre_key.addPreamble(mline);
		}
	}
	TeXPreambleInfo* info = iface->getPreambles()->findOrAddPreamble(&pre_key);
	iface->getPreambles()->select(info);
}

void begin_tex(int *pln, int *pcode, int *cp) {
	TeXInterface* iface = TeXInterface::getInstance();
	// Get optional params
	char ostr[200];
	string name;
	double add = 0.0, x;
	int ptr = *(pcode + (*cp)); /* add */
	if (ptr) {
		int zzcp = 0, otyp;
		eval(pcode + (*cp) + ptr, &zzcp, &add, ostr, &otyp);
	}
	(*cp) = (*cp) + 1;
	ptr = *(pcode + (*cp)); /* name */
	if (ptr) {
		int zzcp = 0, otyp;
		eval(pcode + (*cp) + ptr, &zzcp, &x, ostr, &otyp);
		name = ostr;
	}
	// Start with pcode from the next line
	(*pln)++;
	begin_init();
	string text_block;
	int nblines = 0;
	while (true) {
		int st = begin_token(&pcode,cp,pln,srclin,tk,&ntk,outbuff);
		if (!st) {
			/* exit loop */
			break;
		}
		string mline = srclin;
		str_trim_left(mline);
		if (text_block.length() == 0) {
			text_block = mline;
		} else {
			text_block += "\7";
			text_block += mline;
		}
		nblines++;
	}
	decode_utf8_basic(text_block);
	TeXObject* tex_obj = TeXInterface::getInstance()->draw(text_block.c_str(), nblines);
	// Name object
	if (tex_obj != NULL && name.length() != 0) {
		/* FIXME: better not return null objects (?) */
		/*        (this happens if drawing on dummy device.) */
		double x1, x2, y1, y2;
		tex_obj->getDimensions(&x1,&y1,&x2,&y2);
		x1 -= add; x2 += add; y1 -= add; y2 += add;
		name_set((char*)name.c_str(),x1,y1,x2,y2);
	}
}
