/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include "circle.h"
#include "qgle_statics.h"

// The constructor for the circle object
GLECircle::GLECircle(double resolution, int imageHeight, QObject *parent) :
	GLEDrawingObject(resolution, imageHeight, parent)
{
	// Make sure the circle is updated if a point changes or 
	// the image size changes
	connect(this, SIGNAL(pointChanged()),
			this, SLOT(updateCircle()));
	connect(this, SIGNAL(imageChanged()),
			this, SLOT(updateCircle()));
}

// Update the painter path
void GLECircle::updateCircle()
{
	QPointF p;
	QPair<QPointF,int> snap;

	// Only do this if both the centre and radius have been set
	if (isSet(CentrePoint) && isSet(Radius))
	{
		// This is guaranteed to have been created in the constructor
		// of GLEDrawingObject
		delete(paintPath);
		// Create a new path starting at circleCentre
		paintPath = new QPainterPath(getQtPoint(CentrePoint));
		// Add the circle based on the qrect
		paintPath->addEllipse(circleRect());

		// Update the GLE Code
		gleCode->clear();
		gleCode->append(QString("! %1")
				.arg(tr("Circle added using QGLE", "Comment added to source file")));
		gleCode->append(QString("amove %1")
				.arg(QGLE::GLEToStr(getGLEPoint(CentrePoint))));
		gleCode->append(QString("circle %1")
				.arg(QGLE::GLEToStr(getGLEDouble(Radius))));
		gleCode->append(QString(""));

		// Now we add the osnap handles
		osnapHandles.clear();
		for(int i=0;i<360;i+=90)
		{
			p.setX(getQtDouble(Radius)*cos(QGLE::degreesToRadians(i)));
			p.setY(getQtDouble(Radius)*sin(QGLE::degreesToRadians(i)));
			p += getQtPoint(CentrePoint);
			snap.first = p;
			snap.second = QGLE::QuadrantSnap;
			osnapHandles.append(snap);
		}
		snap.first = getQtPoint(CentrePoint);
		snap.second = QGLE::CentreSnap;
		osnapHandles.append(snap);
	}
}

void GLECircle::addRelativeOSnaps(QPointF p)
{
	if (isSet(CentrePoint) && isSet(Radius))
	{
		qDebug() << "Adding osnaps relative to " << QGLE::absQtToGLE(p,dpi,pixmapHeight);
		QGLE::flushIO();

		relativeOSnaps.clear();

		// The first perpendicular osnap is defined as the nearest point
		QPointF np;
		distanceToPoint(p,&np);
		qDebug() << "Nearest point: " << QGLE::absQtToGLE(np,dpi,pixmapHeight);
		relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::PerpendicularSnap));

		// The second perpendicular osnap is diametrically opposite the first one
		double angleToCentre = QGLE::angleBetweenTwoPoints(np,getQtPoint(CentrePoint));
		double radius = getQtDouble(Radius);
		double diameter = 2*radius;

		np = np + QPointF(diameter*cos(angleToCentre),diameter*sin(angleToCentre));
		relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::PerpendicularSnap));

		if (!isInside(p))
		{
			// Now we need the tangential ones
			double distanceToCentre = QGLE::distance(p, getQtPoint(CentrePoint));
			double angleOffset = asin(radius/distanceToCentre);
			double distanceToTangent = sqrt(pow(distanceToCentre,2)-pow(radius,2));
			angleToCentre = QGLE::angleBetweenTwoPoints(p,getQtPoint(CentrePoint));

			double angle = angleToCentre + angleOffset;
			np = p + QPointF(distanceToTangent*cos(angle),distanceToTangent*sin(angle));
			relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::TangentSnap));

			angle = angleToCentre - angleOffset;
			np = p + QPointF(distanceToTangent*cos(angle),distanceToTangent*sin(angle));
			relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::TangentSnap));

		}

	}
	QGLE::flushIO();
}

void GLECircle::draw(QPainter *p)
{
	if (selected)
	{
		QPen selPen;
		selPen = pen();
		selPen.setColor(Qt::red);
//		selPen.setStyle(Qt::DotLine);
		p->setPen(selPen);
	}
	else
	{
		p->setPen(pen());
	}
	p->drawEllipse(circleRect());
}

double GLECircle::distanceToPoint(QPointF p, QPointF *nearestPoint)
{
	QPointF c = getQtPoint(CentrePoint);
	if (nearestPoint)
	{
		double r = getQtDouble(Radius);
		double theta = QGLE::angleBetweenTwoPoints(c,p);
		nearestPoint->setX(r*cos(theta)+c.x());
		nearestPoint->setY(r*sin(theta)+c.y());
	}
	// Calculations in QT coordinates
	return(fabs(QGLE::distance(c,p)-getQtDouble(Radius)));
}

// Set a point (start or end in the case of a line)
void GLECircle::setPoint(int pointChoice, QPointF p)
{
	double radius;
	switch(pointChoice)
	{
		// Note that circumference point also runs
		// the centre point code: this is intentional
		case CircumferencePoint:
			if (!isSet(CentrePoint))
				return;
			radius = QGLE::distance(QGLE::absQtToGLE(p,dpi,pixmapHeight),getGLEPoint(CentrePoint));
			if (radius > 0)
				pointHash[Radius] = QPointF(radius, 0);
			else
				pointHash.remove(Radius);
		case CentrePoint:
			pointHash[pointChoice] = QGLE::absQtToGLE(p, dpi, pixmapHeight);
			break;
	}

	updateCircle();
}

QRectF GLECircle::circleRect()
{
	// Work in Qt coordinates
	QPointF c = getQtPoint(CentrePoint);
	double r = getQtDouble(Radius);

	return(QRectF(c.x() - r,
				c.y() - r,
				2*r, 2*r));
}

QList<QPointF> GLECircle::intersections(double qtm, double qtc, bool vertical)
{
	QPointF one, two;
	// For circles, we'll deal with points individually:
	if (vertical)
	{
		one.setX(qtm);
		two.setX(qtm);
		one.setY(0.0);
		two.setY(1.0);
	}
	else
	{
		one.setX(0.0);
		one.setY(qtc);
		two.setX(1.0);
		two.setY(qtm + qtc);
	}

	return(intersections(one,two));
}


QList<QPointF> GLECircle::intersections(QPointF qtp1, QPointF qtp2)
{
	QList<QPointF> pointArray;

	QPointF cp = getQtPoint(CentrePoint);
	double r = getQtDouble(Radius);
	double a,b,c;
	double bac;
	double u;
	QPointF p;

	a = pow((qtp2.x() - qtp1.x()),2) + pow((qtp2.y() - qtp1.y()),2);
	b = 2 * ( (qtp2.x() - qtp1.x())*(qtp1.x()-cp.x()) + (qtp2.y()-qtp1.y())*(qtp1.y()-cp.y()));
	c = pow(cp.x(),2)+pow(cp.y(),2)+pow(qtp1.x(),2)+pow(qtp1.y(),2)
		- 2*(cp.x()*qtp1.x()+cp.y()*qtp1.y()) - pow(r,2);

	bac = pow(b,2)-4*a*c;
	if (bac == 0.0)
	{
		u = - b / (2*a);
		p.setX(qtp1.x()+u*(qtp2.x()-qtp1.x()));
		p.setY(qtp1.y()+u*(qtp2.y()-qtp1.y()));
		pointArray.append(p);
	}
	else if (bac > 0.0)
	{
		u = (-b + sqrt(bac))/(2*a);
		p.setX(qtp1.x() + u*(qtp2.x()-qtp1.x()));
		p.setY(qtp1.y() + u*(qtp2.y()-qtp1.y()));
		pointArray.append(p);
		u = (-b - sqrt(bac))/(2*a);
		p.setX(qtp1.x() + u*(qtp2.x()-qtp1.x()));
		p.setY(qtp1.y() + u*(qtp2.y()-qtp1.y()));
		pointArray.append(p);
	}

	return(pointArray);
}

QList<QPointF> GLECircle::intersections(QPointF qtp1, double angle)
{
	// This intersection code must determine the intersections in
	// a particular direction from a start point
	
	// First get a list based on an infinite line:
	QPointF qtp2 = qtp1 + QPointF(1.0*cos(angle),1.0*sin(angle));
	QList<QPointF> allIntersections = intersections(qtp1,qtp2);

	// Now go through the list and determine which are in the right
	// direction
	QList<QPointF> correctIntersections;
	QPointF pt;
	double ptAngle;

	foreach(pt, allIntersections)
	{
		ptAngle = QGLE::angleBetweenTwoPoints(qtp1, pt);
		if (QGLE::quadrant(ptAngle) == QGLE::quadrant(angle))
			correctIntersections.append(pt);
	}

	return(correctIntersections);

}

bool GLECircle::isInside(QPointF p)
{
	QPointF cp = getQtPoint(CentrePoint);
	if (QGLE::distance(cp,p) < getQtDouble(Radius))
		return(true);
	else
		return(false);
}
