/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

// Implementation of GLEDrawingObject, the base class
// for all drawing objects in QGLE
#include "drawingobject.h"

// Constructor: store the resolution and image height; set up defaults
GLEDrawingObject::GLEDrawingObject(double resolution, int imageHeight, QObject *parent) : QObject(parent)
{
	// Defaults:
	dpi = resolution;
	lineWidth = 0.02;
	pixmapHeight = imageHeight;
	selected = false;
	gleCode = new QStringList();
	// We don't actually use this at present, but it's kept
	// for possible future use
	paintPath = new QPainterPath();
	// Default pen
	paintPen = QPen(Qt::black, 
			QGLE::relGLEToQt(lineWidth, 0, dpi).x(), 
			Qt::SolidLine, 
			Qt::SquareCap,
			Qt::BevelJoin);
}

// At the moment this will always return an empty QStringList
QStringList GLEDrawingObject::GLECode()
{
	return(*gleCode);
}

// The selected property tells the object whether it has been
// selected by the pointer tool
void GLEDrawingObject::setSelected(bool sel)
{
	selected = sel;
}

bool GLEDrawingObject::isSelected()
{
	return(selected);
}

// Determine whether a parameter has been set
bool GLEDrawingObject::isSet(int pointChoice)
{
	return(pointHash.contains(pointChoice));
}

// Get the value of a parameter (as a point in GLE coordinates)
QPointF GLEDrawingObject::getGLEPoint(int pointChoice)
{
	return(pointHash.value(pointChoice, QPointF(0,0)));
}

// Get the value of a parameter (as a point in QT coordinates)
QPointF GLEDrawingObject::getQtPoint(int pointChoice)
{
	QPointF gle = getGLEPoint(pointChoice);
	return(QGLE::absGLEToQt(gle, dpi, pixmapHeight));
}

// Get the value of a parameter (as a double in GLE coordinates)
double GLEDrawingObject::getGLEDouble(int pointChoice)
{
	return(pointHash.value(pointChoice, QPointF(0,0)).x());
}

// Get the value of a parameter (as a double in QT coordinates)
double GLEDrawingObject::getQtDouble(int pointChoice)
{
	double gleX = getGLEDouble(pointChoice);
	return(QGLE::relGLEToQt(gleX, dpi));
}

// This returns a QPainterPath object that can be used to draw
// the requested object (although we should just use draw())
QPainterPath GLEDrawingObject::path()
{
	return(*paintPath);
}

// A slot that is called whenever the resolution changes
void GLEDrawingObject::setDPI(double newDPI)
{
	if (dpi != newDPI)
	{
		dpi = newDPI;
		paintPen.setWidthF(QGLE::relGLEToQt(lineWidth, 0, dpi).x());
		emit imageChanged();
	}
}

// A slot that is called whenever the image size changes
void GLEDrawingObject::setPixmapSize(QSize newPixmapSize)
{
	if (pixmapHeight != newPixmapSize.height())
	{
		pixmapHeight = newPixmapSize.height();
		emit imageChanged();
	}
}

void GLEDrawingObject::drawOSnaps(QPainter *p)
{
	QPair<QPointF,int> osnap;
	QList< QPair<QPointF,int> > allOSnaps = osnapHandles + relativeOSnaps;
	foreach(osnap, allOSnaps)
	{
		//QGLE::drawBox(p,osnap.first,OSNAP_BOX_SIZE, pen);
		QGLE::drawOSnap(p,osnap);
	}
}

double GLEDrawingObject::nearestOSnap(QPointF p, QPointF *osnap)
{
	QPair<QPointF,int> o;
	double dist;
	double shortest_distance = 1e6;
	QList< QPair<QPointF,int> > allOSnaps = osnapHandles + relativeOSnaps;
	*osnap = QPointF(0.0,0.0);
	foreach(o, allOSnaps)
	{
		dist = QGLE::distance(o.first,p);
		if (dist < shortest_distance)
		{
			shortest_distance = dist;
			*osnap = o.first;
		}
	}
	return(shortest_distance);
}

void GLEDrawingObject::clearRelativeOSnaps()
{
	qDebug() << "Clearing relative OSnaps";
	relativeOSnaps.clear();
}
