/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include "fileinfo.h"
#include "qgle_statics.h"

/**********************************************************************
 * A class that holds various bits of information about the open file *
 **********************************************************************/

// Clear the file info object on creation
GLEFileInfo::GLEFileInfo()
{
	clear();
}

void GLEFileInfo::setImage(const QImage& newImage, double dpi)
{
	epsImage = newImage;
	QSize sz = epsImage.size();
	qDebug() << "Image size: " << sz;
	QSizeF gsz = QGLE::qtSizeToGLE(QSizeF(sz),dpi);

	// THIS IS NOT GUARANTEED ACCURATE!
	qDebug() << "Setting GLE page size: " << gsz.width() << ", " << gsz.height();
	setGLEPageSize(gsz);
}

// Clear the fileinfo object by setting everything to null
void GLEFileInfo::clear()
{
	gleFileName = "";
	epsFileName = "";
	epsImage = QImage();
	code.clear();
	tempFile = false;
}

// Read the GLE code from the given file name
void GLEFileInfo::readGLECode()
{
	QFile source(gleFileName);
	code.clear();

	// Open the file
	if(!source.open(QIODevice::ReadOnly | QIODevice::Text))
		return;

	// Read the file in line by line
	QString line;
	qDebug() << "Reading file";
	while(!source.atEnd())
	{
		QByteArray data = source.readLine();
		QString line = QString::fromUtf8(data.constData());
		code.append(line);
	}

	// Close the file
	source.close();
}

void GLEFileInfo::setGLECode(const QStringList& newCode)
{
	QString line;
	QRegExp rx("^[ \\t]*size[ \\t]*([0-9.]*)[ \\t]*([0-9.]*)");
	code = newCode;
	foreach(line, code)
	{
		if (rx.exactMatch(line))
			setGLEPageSize(QSizeF(rx.capturedTexts()[1].toDouble(),rx.capturedTexts()[2].toDouble()));
	}
}

void GLEFileInfo::setGLEPageSize(QSizeF newSize)
{
	glePageSize = newSize;
}

QSizeF GLEFileInfo::GLEPageSize()
{
	return(glePageSize);
}

// Read the time stamp from the file and save it.
void GLEFileInfo::updateTimeStamp()
{
	QFileInfo fi;
	QString mainFile = primaryFile();
	if (mainFile.isEmpty())
	{
		lastModified = QDateTime();
		return;
	}
	
	fi.setFile(mainFile);

	lastModified = fi.lastModified();
}

// Return the filename of the main file (i.e. the GLE
// file if we have one and the EPS file otherwise)
QString GLEFileInfo::primaryFile()
{
	if (isGLE())
		return(gleFileName);
	else if (isEPS())
		return(epsFileName);
	else
		return QString();
}

// Is it a GLE file?
bool GLEFileInfo::isGLE()
{
	return(!gleFileName.isEmpty());
}

// Is it an EPS file?
bool GLEFileInfo::isEPS()
{
	if(isGLE())
		return false;
	
	return(!epsFileName.isEmpty());
}

// Check whether the file has changed on disk
// by looking at the time stamp
bool GLEFileInfo::hasChanged()
{
	QFileInfo fi;
	QDateTime modifiedTime;
	QString mainFile = primaryFile();
	if (mainFile.isEmpty())
	{
		return false;
	}

	fi.setFile(mainFile);

	if (!fi.isReadable())
		return false;

	modifiedTime = fi.lastModified();

	if (modifiedTime > lastModified)
		return true;
	else
		return false;
}

	
