/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include <QtGui>
#include "qgle_statics.h"
#include "qgle_definitions.h"

#include <math.h>
#ifdef Q_WS_WIN
	#include <windows.h>
#endif

/*******************************************
 * Static members used in a lot of classes *
 *******************************************/

void QGLE::flushIO()
{
	// Flush the IO to make debug messages appear quickly
	QTextStream out(stdout);
	out.flush();
	QTextStream err(stderr);
	err.flush();
}

QString QGLE::prettyDate(QString datestr)
{
	// Create a QDate object based on the
	// provided date
	QDate dt = QDate::fromString(datestr, "dd/MM/yyyy");
	QString prettyString = dt.toString("d MMMM yyyy");
	QRegExp rx("^(\\d?)(\\d)(?= )");
	QString suffix;
	if(rx.indexIn(prettyString) > -1)
	{
		if (rx.capturedTexts()[1].toInt() == 1)
			suffix = "th";
		else
			switch(rx.capturedTexts()[2].toInt())
			{
				case 1:
					suffix = "st";
					break;
				case 2:
					suffix = "nd";
					break;
				case 3:
					suffix = "rd";
					break;
				default:
					suffix = "th";
			}
		prettyString.replace(rx, rx.capturedTexts()[0] + suffix);
	}
	return(prettyString);

}

int QGLE::sign(double n)
{
	if (n > 0.0)
		return(1);
	else if (n < 0.0)
		return(-1);
	else
		return(0);
}

int QGLE::sign(int n)
{
	if (n > 0)
		return(1);
	else if (n < 0)
		return(-1);
	else
		return(0);
}

int QGLE::quadrant(double angle)
{
	double degrees = radiansToDegrees(angle);
	int quadrant;
	
	// Make the number positive
	while(degrees < 0.0)
		degrees += 360.0;

	// Make the number less than 360
	while(degrees > 360.0)
		degrees -= 360.0;

	if (degrees > 270.0)
		quadrant = 4;
	else if (degrees > 180.0)
		quadrant = 3;
	else if (degrees > 90.0)
		quadrant = 2;
	else
		quadrant = 1;
	
	return(quadrant);
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::absGLEToQt(double gleX, double gleY, double dpi, int areaHeight)
{
	double qtX, qtY;

	QPointF relative = relGLEToQt(gleX,gleY,dpi);
	qtX = relative.x() + (GS_OFFSET*dpi);
	qtY = areaHeight - (relative.y() + (GS_OFFSET*dpi));

	return(QPointF(qtX, qtY));
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::absGLEToQt(QPointF gle, double dpi, int areaHeight)
{
	return(absGLEToQt(gle.x(), gle.y(), dpi, areaHeight));
}

QSizeF QGLE::qtSizeToGLE(QSizeF qt, double dpi)
{
	// DO NOT RELY ON THIS FOR ACCURACY!
	double gleX, gleY;
	qt.setWidth(qt.width() - 2*GS_OFFSET*dpi);
	qt.setHeight(qt.height() - 2*GS_OFFSET*dpi);
	QPointF rel = relQtToGLE(qt.width(), qt.height(), dpi);
	gleX = rel.x();
	gleY = rel.y();

	return(QSizeF(gleX,gleY));
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::relGLEToQt(double gleX, double gleY, double dpi)
{
	double qtX, qtY;

	qtX = gleX*dpi/CM_PER_INCH;
	qtY = gleY*dpi/CM_PER_INCH;

	return(QPointF(qtX, qtY));
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::relGLEToQt(QPointF gle, double dpi)
{
	return(relGLEToQt(gle.x(), gle.y(), dpi));
}

// Static member used to convert betweeen coordinate systems
QSizeF QGLE::relGLEToQt(QSizeF gle, double dpi)
{
	QPointF qt;
	qt = relGLEToQt(gle.width(), gle.height(), dpi);
	return(QSizeF(qt.x(),qt.y()));
}

double QGLE::relGLEToQt(double gle, double dpi)
{
	return(gle*dpi/CM_PER_INCH);
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::relQtToGLE(double qtX, double qtY, double dpi)
{
	double gleX, gleY;

	gleX = qtX*CM_PER_INCH/dpi;
	gleY = qtY*CM_PER_INCH/dpi;

	return(QPointF(gleX, gleY));
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::relQtToGLE(QPointF qt, double dpi)
{
	return(relQtToGLE(qt.x(), qt.y(), dpi));
}

// Static member used to convert betweeen coordinate systems
QSizeF QGLE::relQtToGLE(QSizeF qt, double dpi)
{
	QPointF gle;
	gle = relQtToGLE(qt.width(), qt.height(), dpi);
	return(QSizeF(gle.x(),gle.y()));
}

double QGLE::relQtToGLE(double qt, double dpi)
{
	return(qt*CM_PER_INCH/dpi);
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::absQtToGLE(double qtX, double qtY, double dpi, int areaHeight)
{
	double gleX, gleY;

	gleX = ((qtX/dpi)-GS_OFFSET)*CM_PER_INCH;
	gleY = (((areaHeight - qtY)/dpi)-GS_OFFSET)*CM_PER_INCH;

	return(QPointF(gleX, gleY));
}

// Static member used to convert betweeen coordinate systems
QPointF QGLE::absQtToGLE(QPointF qt, double dpi, int areaHeight)
{
	return(absQtToGLE(qt.x(), qt.y(), dpi, areaHeight));
}

//! Static member used for converting from GLE coordinates to a GLE coordinate string
QString QGLE::GLEToStr(QPointF gle)
{
	return(GLEToStr(gle.x(), gle.y()));
}
//! Static member used for converting from GLE coordinates to a GLE coordinate string
QString QGLE::GLEToStr(double gleX, double gleY)
{
	QString x, y;

	x.setNum(gleX, 'f', GLE_NUMBER_MAX_DP);
	y.setNum(gleY, 'f', GLE_NUMBER_MAX_DP);

	x = ZeroCrop(x);
	y = ZeroCrop(y);

	return(QString("%1 %2").arg(x).arg(y));
}

//! Static member used for converting from GLE distance to a GLE string
QString QGLE::GLEToStr(double gle)
{
	QString x;

	x.setNum(gle, 'f', GLE_NUMBER_MAX_DP);

	x = ZeroCrop(x);

	return(QString("%1").arg(x));
}

//! Static member used to crop unnecessary zeros from the end of a number string
QString QGLE::ZeroCrop(QString str)
{
	int last0;
	int i;

	last0 = str.size();
	for(i=str.size()-1;i>0;i--)
	{
		if (str.at(i) == '0')
		{
			last0 = i;
		}
		else if (str.at(i) == '.')
		{
			last0++;
			break;
		}
		else
		{
			break;
		}
	}
	if (last0 != str.size())
		str.chop(str.size()-last0);

	return(str);
}

//! Static member used for finding the distance between two points
double QGLE::distance(QPointF one, QPointF two)
{
	return(sqrt(pow(one.x()-two.x(),2) + pow(one.y()-two.y(),2)));
}

double QGLE::angleBetweenTwoPoints(QPointF one, QPointF two)
{
	return(atan2(two.y()-one.y(),two.x()-one.x()));
}

double QGLE::radiansToDegrees(double rads)
{
	return(rads*180.0/M_PI);
}

double QGLE::degreesToRadians(double degrees)
{
	return(degrees*M_PI/180.0);
}

//! Static member used to compute the minimum of two double values
double QGLE::min(double a, double b) {
	return a < b ? a : b;
}

//! Static member comverting from centimeters to PostScript points
double QGLE::cmToPt(double cm) {
	return cm*72.0/CM_PER_INCH;
}

//! Static member used to compute the DPI value based on the display and image size (in cm)
int QGLE::computeAutoScaleDPIFromCm(QSize& dispSize, double imgWd, double imgHi) {
	// GLE always adds two points to the size of a drawing
	return computeAutoScaleDPIFromPts(dispSize, cmToPt(imgWd)+2.0, cmToPt(imgHi)+2.0);
}
	
//! Static member used to compute the DPI value based on the display and image size (in pt)
int QGLE::computeAutoScaleDPIFromPts(QSize& dispSize, double imgWd, double imgHi) {
	double dpi_wd = 72.0 * (double)(dispSize.width()-3) / imgWd - 1.0;
	double dpi_hi = 72.0 * (double)(dispSize.height()-3) / imgHi - 1.0;
	return (int)floor(min(dpi_wd, dpi_hi));
}

QString QGLE::GetExeName(const char* appname) 
{
	QString result = QString::null;	
#ifdef Q_WS_WIN
	char name[1024];
	DWORD res = GetModuleFileNameA(NULL, name, 1023);
	if (res > 0) 
	{
		name[res] = 0;
		result = name;
	}
#elif defined Q_WS_X11
	return(QFileInfo("/proc/self/exe").readLink());
#else
	#warning QGLE has not been tested on Mac OS X
	
	/* We could also try something with QApplication::applicationFilePath(),
	 * which uses argv[0] and the current directory.  Not sure how reliable
	 * that is.  Also not sure whether it'll work on Mac OS X.  Does GLE run on
	 * Mac OS X?
	 */

	/* According to:
	 * http://doc.trolltech.com/4.1/qcoreapplication.html#applicationFilePath
	 * the use of argv[0] and the current directory is only on Unix (and 
	 * presumably Linux.  Therefore, the following should work on Windows or
	 * Mac OS X
	 */
	return(QApplication::applicationFilePath());
#endif
	return result;
}

QString QGLE::GetDirName(QString fname) 
{
	QString result = fname;
	result.replace('\\', '/');
	int i = result.lastIndexOf('/');
	if (i != -1) 
	{
		result.truncate(i+1);
	}
	else
	{
		result = "";
	}
	return result;
	// // Could alternatively be achieved with
	// // something like:
	// QFileInfo fi(fname);
	// return(fi.absolutePath());
}

QRegExp QGLE::fileRegExp()
{
#ifdef Q_WS_WIN
	QRegExp rx("(file:///)?(.*\\.(?:gle|eps))");
#else
//#elif defined(Q_WS_X11)
	QRegExp rx("(file://)?(.*\\.(?:gle|eps))");
#endif
	return(rx);
}

QString QGLE::gsLibFileName()
{
#ifdef Q_WS_WIN
	return(QObject::tr("gsdll32.dll"));
#elif defined(Q_WS_X11)
	return(QObject::tr("libgs.so"));
#elif defined(Q_WS_MAC)
	return(QObject::tr("libgs.dylib"));
#else
#error "What operating system are you using?"
#endif
}

QString QGLE::libraryFilter()
{
#ifdef Q_WS_WIN
	return(QObject::tr("DLLs (*.dll)"));
#elif defined(Q_WS_X11)
	return(QObject::tr("Shared objects (*.so)"));
#elif defined(Q_WS_MAC)
	return(QObject::tr("Dynamic libraries (*.dylib)"));
#else
#error What operating system are you using?
#endif
}

QString QGLE::gleExecutableName()
{
#ifdef Q_WS_WIN
	return(QString("gle.exe"));
#else
//#elif defined(Q_WS_X11)
	return(QString("gle"));
#endif
}

QString QGLE::executableFilter()
{
#ifdef Q_WS_WIN
	return(QObject::tr("Executables (*.exe)"));
#else
//#elif defined(Q_WS_X11)
	return(QObject::tr("All files (*)"));
#endif
}

//! Static member to draw a box around a point
void QGLE::drawBox(QPainter *p, QPointF origin, double half_size, QPen pen)
{
	p->setPen(pen);
	p->drawLine(QLineF(origin.x() - half_size,
				origin.y() - half_size,
				origin.x() - half_size,
				origin.y() + half_size));
	p->drawLine(QLineF(origin.x() - half_size,
				origin.y() + half_size,
				origin.x() + half_size,
				origin.y() + half_size));
	p->drawLine(QLineF(origin.x() + half_size,
				origin.y() + half_size,
				origin.x() + half_size,
				origin.y() - half_size));
	p->drawLine(QLineF(origin.x() + half_size,
				origin.y() - half_size,
				origin.x() - half_size,
				origin.y() - half_size));
}

//! Static member to draw a cross at a point
void QGLE::drawCross(QPainter *p, QPointF origin, double half_size, QPen pen)
{
	p->setPen(pen);
	p->drawLine(QLineF(origin.x()-half_size,
				origin.y()-half_size,
				origin.x()+half_size,
				origin.y()+half_size));
	p->drawLine(QLineF(origin.x()-half_size,
				origin.y()+half_size,
				origin.x()+half_size,
				origin.y()-half_size));
}

void QGLE::drawPerpMark(QPainter *p, QPointF origin, QPen pen)
{
	p->setPen(pen);
	double half_size = OSNAP_BOX_SIZE;
	p->drawLine(QLineF(origin.x()-2*half_size,
				origin.y(),
				origin.x()+2*half_size,
				origin.y()));
	p->drawLine(QLineF(origin.x(),origin.y(),
				origin.x(),origin.y()-4*half_size));
	p->drawLine(QLineF(origin.x()-half_size,
				origin.y(),
				origin.x()-half_size,
				origin.y()-half_size));
	p->drawLine(QLineF(origin.x()-half_size,
				origin.y()-half_size,
				origin.x(),
				origin.y()-half_size));
}

void QGLE::drawTangentMark(QPainter *p, QPointF origin, QPen pen)
{
	p->setPen(pen);
	double half_size = OSNAP_BOX_SIZE;
	p->drawLine(QLineF(origin.x()-2*half_size,
				origin.y(),
				origin.x()+2*half_size,
				origin.y()));
	p->drawArc(QRectF(origin.x()-2*half_size,
				origin.y(),
				4*half_size,
				4*half_size), 0,180*16);

}

void QGLE::drawOSnap(QPainter *p, QPair<QPointF, int> snap)
{
	p->setPen(osnapPen());
	// Switch on snap.second
	switch(snap.second)
	{
		case PerpendicularSnap:
			drawPerpMark(p,snap.first,osnapPen());
			break;

		case TangentSnap:
			drawTangentMark(p,snap.first,osnapPen());
			break;

		default:
			drawBox(p,snap.first,OSNAP_BOX_SIZE,osnapPen());
	}
}

bool QGLE::inOrder(double x, double y, double z)
{
	if ((y >= x) && (z >= y))
		return(true);
	else if ((y < x) && (z < y))
		return(true);
	else
		return(false);
}
QPen QGLE::osnapPen()
{
	QPen pen;
	pen.setColor(Qt::darkYellow);
	pen.setWidth(1);
	return(pen);
}

QPen QGLE::osnapLinePen()
{
	QPen pen = osnapPen();
	pen.setStyle(Qt::DashLine);
	return(pen);
}
