/*
 *  glutBasics.cpp
 *  GLUTTest
 *
 *  Created by GGS on June 17 2003.
 *  Copyright (c) 2003 Apple. All rights reserved.
 *
 
 */
 
#include <stdlib.h> //for exit()
#include <stdio.h>  //for sprintf()

#if  (__APPLE__) || defined(MACOSX)
#include <GLUT/glut.h>
#include <OpenGL/gl.h>
#include <OpenGL/glu.h>
#else
#include <GL/glut.h>
#include <GL/gl.h>
#include <GL/glu.h>
#endif
 
#include <cmath>
#include <iostream>
#include <fstream>
using namespace std;

#include "trackball.h"
#include "Lsys.h"
#include "GL_Ldraw.h"
#include "Lparse.h"
#include "mutprobs.h"


#define DTOR 0.0174532925
#define CROSSPROD(p1,p2,p3) \
   p3.x = p1.y*p2.z - p1.z*p2.y; \
   p3.y = p1.z*p2.x - p1.x*p2.z; \
   p3.z = p1.x*p2.y - p1.y*p2.x
   
#ifndef MIN
	#define MIN(a, b) ((a) < (b) ? (a) : (b))
#endif

#define DOUBLE_CLICK_INTERVAL_MS 500
#define ANIM_STEPS 0.02
#define TABSPACING 16

enum {
	kTextureSize = 256
};

enum {
	OFF=0,
	DOLLY_SINGLE=1,
	DOLLY_ALL=2,
	PAN_SINGLE=3,
	PAN_ALL=4,
	TRACK_SINGLE=5,
	TRACK_ALL=6
};

enum {
  NO_CHANGE=0,
  TOGGLE,
  SINGLE,
  MULTIPLE
};

typedef struct {
	GLdouble x,y,z;
} recVec;

typedef struct {
	recVec viewPos; // View position
	recVec viewDir; // View direction vector
	recVec viewUp; // View up direction
	recVec rotPoint; // Point to rotate about
	GLdouble aperture; // gCamera aperture
	GLint screenWidth,screenHeight; // current window/screen height and width
} recCamera;

typedef struct {
	GLint DollyPanStartPoint[2];
	GLfloat TrackBallRotation [4];
	int movement;
	GLfloat WorldRotation[4];
	recCamera Camera;
	recVec Origin;
} ViewingVariables;

typedef struct {
  Lsys      *lsystem;
  GLuint    DisplayList;
  int       draw_state;
  GLboolean compiled;
  unsigned long rand_seed;
  float     grow_val;
  GLfloat   max_dim;
} ObjectVariables;

typedef struct sw {
  int number, x, y, w, h, x_index, y_index, parent;
  GLuint BlankList;
  int ShowLsys;
  int ShowInfo;
  int clicktime;
  bool hidden;
  ViewingVariables viewVar;
  ObjectVariables objVar;
  struct sw **selected;
} subWindow;
	
#define SUBWINS 9
#define COLUMNS 3
#define ROWS 3
#define MID_COLUMN 1
#define MID_ROW 1
#define FRAMERATE 20
#define FRAMES 20

subWindow gSubWindow[SUBWINS];
const char *filename=NULL;
unsigned long max_vertices=0;
bool save_on_quit=true;

void swap( int &m, int &n)
{
    int o;
    o = m;
    m = n;
    n = o;
}

#pragma mark ---- Window Order ----
int index_to_position(int parent_width, int parent_height, int x_index, int y_index, int *x, int *y, int *w, int *h) 
{
	if ((x_index < 0) || (x_index >= COLUMNS))
		return 1;
	if ((y_index < 0) || (y_index >= ROWS))
		return 2;
	(*x) = (x_index*parent_width)/COLUMNS;
	(*w) = ((x_index+1)*parent_width)/COLUMNS - (*x);
	(*y) = (y_index*parent_height)/ROWS;
	(*h) = ((y_index+1)*parent_height)/ROWS - (*y);
	return 0;
}



#pragma mark ---- Various init routines ----

void CameraReset(recCamera *c, recVec *o)
{
   c->aperture = 40;
   c->rotPoint = *o;
   
   c->viewPos.x = 0.0;
   c->viewPos.y = 0.0;
   c->viewPos.z = -20.0f;
   c->viewDir.x = -c->viewPos.x; 
   c->viewDir.y = -c->viewPos.y; 
   c->viewDir.z = -c->viewPos.z;

   c->viewUp.x = 0;  
   c->viewUp.y = 1;
   c->viewUp.z = 0;
}

void initWindowVariables(subWindow *win) {
  win->viewVar.DollyPanStartPoint[0] = 0; win->viewVar.DollyPanStartPoint[0] = 0;
  win->viewVar.TrackBallRotation [0] = 0.0; win->viewVar.TrackBallRotation [1] = 0.0;
  win->viewVar.TrackBallRotation [2] = 0.0; win->viewVar.TrackBallRotation [3] = 0.0;
  win->viewVar.movement=OFF;
  win->viewVar.WorldRotation [0] = 90.0;
  win->viewVar.WorldRotation [1] = -1.0;
  win->viewVar.WorldRotation [2] = 0.0;
  win->viewVar.WorldRotation [3] = 0.0;
  CameraReset(&(win->viewVar.Camera), &(win->viewVar.Origin));
  win->viewVar.Origin.x = 0.0; win->viewVar.Origin.y= 0.0; win->viewVar.Origin.z = 0.0;
};

#pragma mark ---- Utilities ----

int findSubWindow(int WindowNumber)
{
  for(int i=0;i<SUBWINS;i++)
    if (gSubWindow[i].number == WindowNumber) return i;
  return 0;
}

//a function used to get around a glut fullscreen bug (possibly only needed in Mac OS X)

void toggle_toggle(int num) {
  int win=glutGetWindow();
  for(int i=0;i<SUBWINS;i++) {
    glutSetWindow(gSubWindow[i].number);
    if (gSubWindow[i].hidden==false) {
      glutHideWindow();
      gSubWindow[i].hidden=true;
    } else {
      glutShowWindow();
      gSubWindow[i].hidden=false;
    };
    glutPostRedisplay();
  };
  glutSetWindow(win);
  if (--num) glutTimerFunc(0, toggle_toggle, num);
};

void change_shape(int state, bool double_refresh=false, int w=glutGet(GLUT_WINDOW_WIDTH), int h=glutGet(GLUT_WINDOW_HEIGHT)) {
  static bool single=false;
  bool changed=false;
  if (state==TOGGLE) {
    single=!single;
    changed=true;
  } else if (state==SINGLE) {
    if (single==false) changed=true;
    single=true;
  } else if (state==MULTIPLE) {
    if (single==true) changed=true;
    single=false;
  };

  glViewport(0,0,(GLsizei)w,(GLsizei)h);
  if (single==true) {
    for(int i=0;i<SUBWINS; i++) {
      subWindow *win = &(gSubWindow[i]);
      glutSetWindow(win->number);
      if (changed && (win!=*(win->selected))) {
	if ((win->hidden)==false) {
	  glutHideWindow();
	  win->hidden=true;
	};
      };
      win->x=win->y=0;
      win->w=w;
      win->h=h;
      glutReshapeWindow(win->w, win->h);
      glutPositionWindow(win->x, win->y);
      win->viewVar.Camera.screenWidth = win->w;
      win->viewVar.Camera.screenHeight = win->h;		
      glutPostRedisplay();
    };
    glutSetWindow(gSubWindow[0].parent);
    
  } else {
    for(int i=0;i<SUBWINS; i++) {
      subWindow *win = &(gSubWindow[i]);
      glutSetWindow(win->number);
      if (changed) {
	if ((win->hidden)==true) {
	  glutShowWindow();
	  win->hidden=false;
	};
      };
      index_to_position(w, h, win->x_index, win->y_index, &(win->x), &(win->y), &(win->w), &(win->h));
      glutReshapeWindow(win->w, win->h);
      glutPositionWindow(win->x, win->y);
      win->viewVar.Camera.screenWidth = win->w;
      win->viewVar.Camera.screenHeight = win->h;	
      glutPostRedisplay();
    };
    glutSetWindow(gSubWindow[0].parent);

    if ((changed) && (double_refresh)) glutTimerFunc(0,toggle_toggle,2); //work around redisplay bug
  };
}

void
drawGLString(GLfloat *x, GLfloat *y, const char *string, GLfloat line_spacing)
{
  int len, i;
	GLfloat p[4];
  glRasterPos2f(*x, *y);
  glGetFloatv(GL_CURRENT_RASTER_POSITION, p);
  len = (int) strlen(string);
  for (i = 0; i < len; i++) {
	  if (string[i]=='\n') {
		  *y+=line_spacing;
		  glRasterPos2f(*x, *y);
	  } else if (string [i]=='\t') {
		  glGetFloatv(GL_CURRENT_RASTER_POSITION, p);
		  glRasterPos2f(p[0]+(i%TABSPACING), *y);
	  } else {
		  glutBitmapCharacter(GLUT_BITMAP_HELVETICA_10, string[i]);
	  };
  };
}

void BlankList (GLuint * blankList)
{
	*blankList = glGenLists (1);
	glNewList(*blankList, GL_COMPILE);
	glEndList ();
}

#pragma mark ---- Drawing ----


void SetLighting(unsigned int mode)
{
	GLfloat mat_specular[] = {1.0, 1.0, 1.0, 1.0};
	GLfloat mat_shininess[] = {90.0};

	GLfloat position[4] = {7.0,-7.0,12.0,0.0};
	GLfloat ambient[4]  = {0.2,0.2,0.2,1.0};
	GLfloat diffuse[4]  = {1.0,1.0,1.0,1.0};
	GLfloat specular[4] = {1.0,1.0,1.0,1.0};
	
	glMaterialfv (GL_FRONT_AND_BACK, GL_SPECULAR, mat_specular);
	glMaterialfv (GL_FRONT_AND_BACK, GL_SHININESS, mat_shininess);
	
	glEnable(GL_COLOR_MATERIAL);
	glColorMaterial(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE);

	switch (mode) {
		case 0:
			break;
		case 1:
			glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,GL_FALSE);
			glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER,GL_FALSE);
			break;
		case 2:
			glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,GL_FALSE);
			glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER,GL_TRUE);
			break;
		case 3:
			glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,GL_TRUE);
			glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER,GL_FALSE);
			break;
		case 4:
			glLightModeli(GL_LIGHT_MODEL_TWO_SIDE,GL_TRUE);
			glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER,GL_TRUE);
			break;
	}
	
	glLightfv(GL_LIGHT0,GL_POSITION,position);
	glLightfv(GL_LIGHT0,GL_AMBIENT,ambient);
	glLightfv(GL_LIGHT0,GL_DIFFUSE,diffuse);
	glLightfv(GL_LIGHT0,GL_SPECULAR,specular);
	glEnable(GL_LIGHT0);
}

void draw_select_box(subWindow *win) {
	GLint matrixMode;
	GLint window_width = win->viewVar.Camera.screenWidth;
	GLint window_height = win->viewVar.Camera.screenHeight;
	
	glGetIntegerv(GL_MATRIX_MODE, &matrixMode);
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glScalef(2.0f / window_width, -2.0f / window_height, 1.0f);
	glTranslatef(-window_width / 2.0f, -window_height / 2.0f, 0.0f);
	glBegin(GL_LINE_LOOP);
	glColor3f(1.0f,0.0f,0.0f);
	glVertex2i(1,1);
	glVertex2i(window_width,1);
	glVertex2i(window_width,window_height);
	glVertex2i(1,window_height);
	glEnd();

	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(matrixMode);
};

void drawGLText(subWindow *win)
{
	char outString [256] = "";
	GLint matrixMode;
	GLint vp[4];
	GLfloat lineSpacing = 9.0f;
	GLfloat xpos=10, ypos;
	GLint startOffset = 7;
	GLint window_width = win->viewVar.Camera.screenWidth;
	GLint window_height = win->viewVar.Camera.screenHeight;
	
	glutSetWindow(win->number);
	
	glGetIntegerv(GL_VIEWPORT, vp);
	glViewport(0, 0, window_width, window_height);
	
	glGetIntegerv(GL_MATRIX_MODE, &matrixMode);
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();
	glScalef(2.0f / window_width, -2.0f / window_height, 1.0f);
	glTranslatef(-window_width / 2.0f, -window_height / 2.0f, 0.0f);
	
	// draw 
	glDisable(GL_LIGHTING);
	glColor3f (1.0, 1.0, 1.0);
	if (win->ShowInfo) {
		ypos=window_height - startOffset;
		sprintf (outString, "Camera Position: (%0.1f, %0.1f, %0.1f)", win->viewVar.Camera.viewPos.x, win->viewVar.Camera.viewPos.y, win->viewVar.Camera.viewPos.z);
		drawGLString (&xpos, &ypos, outString, lineSpacing);
		ypos-=lineSpacing;
		sprintf (outString, "Trackball Rotation: (%0.1f, %0.2f, %0.2f, %0.2f)", win->viewVar.TrackBallRotation[0], win->viewVar.TrackBallRotation[1], win->viewVar.TrackBallRotation[2], win->viewVar.TrackBallRotation[3]);
		drawGLString (&xpos, &ypos, outString, lineSpacing);
		ypos-=lineSpacing;
		sprintf (outString, "World Rotation: (%0.1f, %0.2f, %0.2f, %0.2f)", win->viewVar.WorldRotation[0], win->viewVar.WorldRotation[1], win->viewVar.WorldRotation[2], win->viewVar.WorldRotation[3]);
		drawGLString (&xpos, &ypos, outString, lineSpacing);
		ypos-=lineSpacing;
		sprintf (outString, "Aperture: %0.1f", win->viewVar.Camera.aperture);
		drawGLString (&xpos, &ypos, outString, lineSpacing);
	};
	
	if (win->ShowLsys) {
		ypos=startOffset+lineSpacing;
		drawGLString (&xpos, &ypos, win->objVar.lsystem->get_Lstring().c_str(), lineSpacing);
	}
	
	glPopMatrix();
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(matrixMode);
	
	glViewport(vp[0], vp[1], vp[2], vp[3]);
	
	
}

#pragma mark ---- various utility routines ----

void sync_views(subWindow *win) {
  for(int i=0;i<SUBWINS; i++)
    if (&(gSubWindow[i]) != win) {
      glutSetWindow(gSubWindow[i].number);
      gSubWindow[i].viewVar=win->viewVar;
      glutPostRedisplay();
    };
  glutSetWindow(win->number);
      
}

subWindow *gBreeding=NULL;
subWindow *gGrowing=NULL;

void breed(subWindow *Parent) {
  for(int i=0;i<SUBWINS;i++) {
    subWindow *current_win=&(gSubWindow[i]);
    glutSetWindow(current_win->number);
    
    if (current_win != Parent) {
      current_win->objVar.lsystem->derive_from(*(Parent->objVar.lsystem));
      current_win->objVar.compiled=GL_FALSE;
      current_win->objVar.rand_seed=0;
      glutPostRedisplay();
    };
  };
}

void hide_others(subWindow *chosen_win) {
  GLuint todel, win=glutGetWindow();
  for(int i=0;i<SUBWINS;i++) {
    subWindow *current_win=&(gSubWindow[i]);
    glutSetWindow(current_win->number);
    
    if (current_win != chosen_win) {
      current_win->viewVar=chosen_win->viewVar;
      todel=current_win->objVar.DisplayList;
      current_win->objVar.DisplayList = current_win->BlankList;
      glDeleteLists(todel,1);
      if (current_win->hidden==false) {
	glutHideWindow();
	current_win->hidden=true;
      };
      glutPostRedisplay();
    };
  };
  glutSetWindow(win);
};

void breeding_animation(int value) {
  if (gBreeding == NULL) {
    fprintf(stderr, "no selected window set\n");
    exit(1);
  };
  
  glutSetWindow(gBreeding->number);
  
  int target=0;
  while ((gSubWindow[target].x_index != MID_COLUMN) || (gSubWindow[target].y_index != MID_ROW)) {
    target++;
  };
  if ((value>=0) && (gBreeding != &(gSubWindow[target]))) {
    glutReshapeWindow(((gSubWindow[target].w * (FRAMES-value))+(gBreeding->w * value))/FRAMES, ((gSubWindow[target].h * (FRAMES-value))+(gBreeding->h * value))/FRAMES);
    glutPositionWindow(((gSubWindow[target].x * (FRAMES-value))+(gBreeding->x * value))/FRAMES, ((gSubWindow[target].y * (FRAMES-value))+(gBreeding->y * value))/FRAMES);
    glutPostRedisplay();
    glutTimerFunc(FRAMERATE, breeding_animation, value-1);
  } else {
    GLuint win=gBreeding->number;
    swap(gSubWindow[target].x_index,gBreeding->x_index);
    swap(gSubWindow[target].y_index,gBreeding->y_index);
    gSubWindow[target].ShowLsys = gBreeding->ShowLsys;
    gSubWindow[target].ShowInfo = gBreeding->ShowInfo;
    *(gBreeding->selected)=gBreeding;
    breed(gBreeding);
    gBreeding=NULL;
    
    //show the windows again, and put them in the right place
    for(int i=0;i<SUBWINS;i++) {
      if (gSubWindow[i].hidden) {
	glutSetWindow(gSubWindow[i].number);
	glutShowWindow();
	gSubWindow[i].hidden=false;
      };
    };
    glutSetWindow(gSubWindow[0].parent);
    change_shape(MULTIPLE, true);
    glutSetWindow(win);
  };
}

void animate_breeding(subWindow *win) {
  if ((gBreeding==NULL) && (win!=NULL))
    if ((win >= gSubWindow) && (win < gSubWindow+SUBWINS)) {
      //this is a valid subwindow - breed from it.
      gBreeding=win;
      hide_others(win);
      glutTimerFunc(FRAMERATE, breeding_animation, FRAMES);
    };
};

void growing_animation(int value) {
  GLuint i,todel;
  if (gGrowing == NULL) {
    fprintf(stderr, "no selected window set\n");
    exit(1);
  };

  i=glutGetWindow();
  todel=gGrowing->objVar.DisplayList;
  glutSetWindow(gGrowing->number);
  
  gGrowing->objVar.DisplayList = gGrowing->BlankList;
  glDeleteLists(todel,1);
  gGrowing->objVar.compiled=GL_FALSE;
  if (value<=FRAMES) {
    gGrowing->objVar.grow_val = gGrowing->objVar.lsystem->get_rec() * value/FRAMES;
    glutTimerFunc(FRAMERATE, growing_animation, value+1);
  } else {
    gGrowing->objVar.grow_val=-1.0f;
    gGrowing=NULL;
  };
  glutPostRedisplay();
  glutSetWindow(i);
  
}

void animate_growth(subWindow *win) {
  if ((gGrowing==NULL) && (win!=NULL))
    if ((win >= gSubWindow) && (win < gSubWindow+SUBWINS)) {
      gGrowing=win;
      glutTimerFunc(FRAMERATE, growing_animation, 0);
    };
};

#pragma mark ---- GLUT callbacks ----

void display()
{
  static unsigned long static_seed=123;
  unsigned long seed=static_seed;
  subWindow *win = &(gSubWindow[findSubWindow(glutGetWindow())]);
  Lsys *L=win->objVar.lsystem;
  if (win->objVar.compiled==GL_FALSE) {
    
    //New l-system
    float recursion_level;
    GLfloat *bounds;
    if (win->objVar.rand_seed==0) //we need a new seed for this L-system 
      win->objVar.rand_seed=static_seed;
    seed=win->objVar.rand_seed;
    if (win->objVar.grow_val < -0.5f) {
      recursion_level=L->get_rec();
      bounds = L->bounds;
    } else {
      recursion_level=win->objVar.grow_val;
      bounds=NULL;
    };
    
    //openGL initialize
    glMatrixMode (GL_MODELVIEW);
    glLoadIdentity ();
    glEnable(GL_DEPTH_TEST);
    glShadeModel(GL_SMOOTH);    
    glFrontFace(GL_CCW);
    glPolygonOffset (1.0, 1.0);
    SetLighting(4);
    
    win->objVar.DisplayList = glGenLists (1);
    glNewList(win->objVar.DisplayList, GL_COMPILE_AND_EXECUTE); //have to compile_and_execute to get vertices
    glEnable(GL_LIGHTING);
    win->objVar.draw_state=Ldraw_GL(Lparse(recursion_level, L->get_axiom().c_str(), L->get_rules_as_one_string().c_str(), -1.0f, 0, &(seed)), L->get_ang(), L->get_thick(), max_vertices, &(seed), bounds);
    glEndList ();
    static_seed=seed; //keep the static_seed random
    win->objVar.compiled=GL_TRUE;
    glutPostRedisplay();
    if (win->objVar.grow_val < -0.5f) {
      win->objVar.max_dim=sqrt(
			       (L->bounds[0] - L->bounds[1])*(L->bounds[0] - L->bounds[1]) +
			       (L->bounds[2] - L->bounds[3])*(L->bounds[2] - L->bounds[3]) +
			       (L->bounds[4] - L->bounds[5])*(L->bounds[4] - L->bounds[5]));
    };
    } else {
    GLdouble xmin, xmax, ymin, ymax;
    // far frustum plane
    GLdouble zFar = - win->viewVar.Camera.viewPos.z + win->objVar.max_dim;
    // near frustum plane clamped at 1.0
    GLdouble zNear = 0.1f;//MIN (- win->viewVar.Camera.viewPos.z - max_dim * 0.5, 1.0f);
    // window aspect ratio
    GLdouble aspect = win->viewVar.Camera.screenWidth / (GLdouble) win->viewVar.Camera.screenHeight; 
    // draw the display list

    if (win->objVar.draw_state==0)
      glClearColor (0.2f, 0.2f, 0.4f, 1.0f);	// clear the surface
    else
      glClearColor (0.4f, 0.2f, 0.2f, 1.0f);
    
    glClear (GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    if (win->objVar.draw_state < 0) {
      char *str="";
      GLfloat x=0.0f, y=0.0f;
      switch(win->objVar.draw_state) {
      case -3: str="Error: Bad function call"; break;
      case -2: str="OpenGL modelview stack exceeded"; break;
      case -1: str="Out of Memory"; break;
      };
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity ();
      glMatrixMode (GL_MODELVIEW);
      glLoadIdentity ();
      drawGLString(&x,&y, str, 14);
    } else {
      glMatrixMode (GL_PROJECTION);
      glLoadIdentity ();
      if (aspect > 1.0) {
	ymax = zNear * tan (win->viewVar.Camera.aperture * 0.5 * DTOR);
	ymin = -ymax;
	xmin = ymin * aspect;
	xmax = ymax * aspect;
      } else {
	xmax = zNear * tan (win->viewVar.Camera.aperture * 0.5 * DTOR);
	xmin = -xmax;
	ymin = xmin / aspect;
	ymax = xmax / aspect;
      }
      glFrustum(xmin, xmax, ymin, ymax, zNear, zFar);
      
      glMatrixMode (GL_MODELVIEW);
      glLoadIdentity ();
      gluLookAt (win->viewVar.Camera.viewPos.x, win->viewVar.Camera.viewPos.y, win->viewVar.Camera.viewPos.z,
		 win->viewVar.Camera.viewPos.x + win->viewVar.Camera.viewDir.x,
		 win->viewVar.Camera.viewPos.y + win->viewVar.Camera.viewDir.y,
		 win->viewVar.Camera.viewPos.z + win->viewVar.Camera.viewDir.z,
		 win->viewVar.Camera.viewUp.x, win->viewVar.Camera.viewUp.y, win->viewVar.Camera.viewUp.z);
      glRotatef (win->viewVar.TrackBallRotation[0], win->viewVar.TrackBallRotation[1], win->viewVar.TrackBallRotation[2], win->viewVar.TrackBallRotation[3]);
      glRotatef (win->viewVar.WorldRotation[0], win->viewVar.WorldRotation[1], win->viewVar.WorldRotation[2], win->viewVar.WorldRotation[3]);
      glTranslatef(-(L->bounds[0]+L->bounds[1])/2.0f, -(L->bounds[2]+L->bounds[3])/2.0f, -(L->bounds[4]+L->bounds[5])/2.0f);
      glCallList (win->objVar.DisplayList);
      drawGLText (win);
    }; //end switch
    if (*(win->selected)==win) draw_select_box(win);
    glutSwapBuffers();
  };
}

void key(unsigned char inkey, int px, int py) {
  static int x,y,w,h,fullscreen=0;
  int col,row;
  subWindow *win = *(gSubWindow[0].selected);
  glutSetWindow(win->number);
  
  switch (inkey) {
  case 9: //tab
    row=win->y_index;
    col=win->x_index+1;
    if (col>=COLUMNS) {
      col=0;
      row++;
      if (row>=ROWS) row=0;
    };
    for(int i=0;i<SUBWINS;i++) 
      if ((gSubWindow[i].y_index == row) && (gSubWindow[i].x_index == col)) {
	if (gSubWindow[i].hidden)
	  if (win->hidden==false) {
	    glutHideWindow(); //hide current window
	    win->hidden=true;
	  };
	glutPostRedisplay();
	*(win->selected)=win=&gSubWindow[i];
	glutSetWindow(win->number);
	if (win->hidden==true) {
	  glutShowWindow();
	  win->hidden=false;
	};
	glutPostRedisplay();
	break;
      };
    break;
    
  case 13: //carriage return
    animate_breeding(win);
    break;
    
  case 27: //escape
    if (fullscreen) {
      glutSetWindow(win->parent);
      glutPositionWindow(x,y);
      glutReshapeWindow(w,h);
      change_shape(NO_CHANGE,false,w,h); //shouldn't be needed, but is.
      glutPostRedisplay();
      glutSetWindow(win->number);
      fullscreen=0;
      
      toggle_toggle(2); //Work around glut fullscreen redisplay bug
    };
    break;
  case 'f': // fullscreen
  case 'F':
    if (fullscreen==0) {
      glutSetWindow(win->parent);
      x=glutGet(GLUT_WINDOW_X);
      y=glutGet(GLUT_WINDOW_Y);
      w=glutGet(GLUT_WINDOW_WIDTH);
      h=glutGet(GLUT_WINDOW_HEIGHT);
      glutFullScreen();
      glutSetWindow(win->number);
      fullscreen=1;
      
      toggle_toggle(2); //Work around glut fullscreen redisplay bug
    };
    break;
  case 'g':
  case 'G':
    animate_growth(win);
    break;
  case 'l': // show L-system code
  case 'L':
    win->ShowLsys =  1 - win->ShowLsys;
    if (glutGetModifiers() & GLUT_ACTIVE_SHIFT) {
      for(int i=0;i<SUBWINS;i++) {
	glutSetWindow(gSubWindow[i].number);
	gSubWindow[i].ShowLsys=win->ShowLsys;
	glutPostRedisplay();
      };
      glutSetWindow(win->number);
    } else {
      glutPostRedisplay();
    };
    break;
  case 'i': // info
  case 'I':
    win->ShowInfo =  1 - win->ShowInfo;
    if (glutGetModifiers() & GLUT_ACTIVE_SHIFT) {
      for(int i=0;i<SUBWINS;i++) {
	glutSetWindow(gSubWindow[i].number);
	gSubWindow[i].ShowInfo=win->ShowInfo;
	glutPostRedisplay();
      };
      glutSetWindow(win->number);
    } else {
      glutPostRedisplay();
    };
    break;

  case 32://space
    glutSetWindow(win->parent);
    change_shape(TOGGLE, true);
    glutSetWindow(win->number);
    break;

  case 8: //backspace
  case 127: //backspace
    initWindowVariables(win);
    if (glutGetModifiers() & GLUT_ACTIVE_SHIFT)
      sync_views(win);
    glutPostRedisplay();
    break;
  case 'q':
  case 'Q': //quit
    if (!(glutGetModifiers() & GLUT_ACTIVE_CTRL))
      save_on_quit=false;
    exit(0);
  };
}

void special(int key, int px, int py)
{
  subWindow *win = *(gSubWindow[0].selected);
  glutSetWindow(win->number);
  
   switch (key) {
    case GLUT_KEY_UP: // arrow up, smaller aperture
      win->viewVar.Camera.aperture -= 0.5f;
      if (win->viewVar.Camera.aperture < 0.0f)
	win->viewVar.Camera.aperture = 0.0f;
      if (glutGetModifiers() & GLUT_ACTIVE_SHIFT)
	sync_views(win);
      glutPostRedisplay();
      break;
    case GLUT_KEY_DOWN: // arrow down, larger aperture
      win->viewVar.Camera.aperture += 0.5f;
      if (glutGetModifiers() & GLUT_ACTIVE_SHIFT)
	sync_views(win);
      glutPostRedisplay();
      break;
    }
}

void mouseDolly (subWindow *win, int x, int y)
{
	if ((win->viewVar.movement == DOLLY_ALL) || (win->viewVar.movement == DOLLY_SINGLE)) {
		GLfloat dolly = (win->viewVar.DollyPanStartPoint[1] - y) * -win->viewVar.Camera.viewPos.z / 200.0f;
		win->viewVar.Camera.viewPos.z += dolly;
		if (win->viewVar.Camera.viewPos.z == 0.0) // do not let z = 0.0
		  win->viewVar.Camera.viewPos.z = 0.0001;
		win->viewVar.DollyPanStartPoint[0] = x;
		win->viewVar.DollyPanStartPoint[1] = y;
		glutPostRedisplay();
		if (win->viewVar.movement == DOLLY_ALL)
		  sync_views(win);
	};
}

void mouseDollyCallback (int x, int y) 
{mouseDolly(*(gSubWindow[0].selected), x,y);}

void mousePan(subWindow *win, int x, int y)
{
	if ((win->viewVar.movement == PAN_ALL) || (win->viewVar.movement == PAN_SINGLE)) {	
		GLfloat panX = (win->viewVar.DollyPanStartPoint[0] - x) / (900.0f / -win->viewVar.Camera.viewPos.z);
		GLfloat panY = (win->viewVar.DollyPanStartPoint[1] - y) / (900.0f / -win->viewVar.Camera.viewPos.z);
		win->viewVar.DollyPanStartPoint[0] = x;
		win->viewVar.DollyPanStartPoint[1] = y;
		win->viewVar.Camera.viewPos.x -= panX;
		win->viewVar.Camera.viewPos.y -= panY;
		glutPostRedisplay();
		if (win->viewVar.movement == PAN_ALL)
			sync_views(win);
		};
}

void mousePanCallback (int x, int y) 
{mousePan(*(gSubWindow[0].selected), x,y);}

void mouseTrackball (subWindow *win, int x, int y, GLboolean stopping)
{
	if ((win->viewVar.movement == TRACK_ALL) || (win->viewVar.movement == TRACK_SINGLE)) {
		rollToTrackball (x, y, win->viewVar.TrackBallRotation);
		if (stopping) {
			if (win->viewVar.TrackBallRotation[0] != 0.0)
				addToRotationTrackball (win->viewVar.TrackBallRotation, win->viewVar.WorldRotation);
			win->viewVar.TrackBallRotation [0] = win->viewVar.TrackBallRotation [1] = win->viewVar.TrackBallRotation [2] = win->viewVar.TrackBallRotation [3] = 0.0f;
		};
		glutPostRedisplay();
		if (win->viewVar.movement == TRACK_ALL)
			sync_views(win);
	};
}

void mouseTrackballCallback (int x, int y)
{ mouseTrackball(&(gSubWindow[findSubWindow(glutGetWindow())]), x,y, GL_FALSE);}

void mouse (int button, int state, int x, int y)
{
  subWindow *win=&(gSubWindow[findSubWindow(glutGetWindow())]);
  if ((state == GLUT_DOWN) && (*(win->selected)!=win)) { //selection has changed
    *(win->selected)=win;
    for(int i=0;i<SUBWINS;i++) {
      glutSetWindow(gSubWindow[i].number);
      glutPostRedisplay();
      };
    glutSetWindow(win->number);
  };

 if (button == GLUT_LEFT_BUTTON) { //TRACKBALL
    if (state == GLUT_DOWN) {
      int Timediff=glutGet(GLUT_ELAPSED_TIME)-win->clicktime;
       win->clicktime+=Timediff;
      glutMotionFunc (mouseTrackballCallback);
      if (win->viewVar.movement == DOLLY_SINGLE || win->viewVar.movement == DOLLY_ALL) { // if we are currently dollying, end dolly
	mouseDolly (win, x, y);
      } else if ((win->viewVar.movement == PAN_SINGLE) || (win->viewVar.movement == PAN_ALL)) {
	mousePan (win, x, y);
      }
      if (Timediff < DOUBLE_CLICK_INTERVAL_MS) {
	animate_breeding(win);
	return;
      };
      if (glutGetModifiers() & GLUT_ACTIVE_SHIFT)
	win->viewVar.movement=TRACK_ALL;
      else
	win->viewVar.movement=TRACK_SINGLE;
      startTrackball(x, y, 0, 0, win->viewVar.Camera.screenWidth, win->viewVar.Camera.screenHeight);
    } else if (state == GLUT_UP) {
      mouseTrackball(win, x, y, GL_TRUE);
      win->viewVar.movement=OFF;
    };
  } else if (button == GLUT_MIDDLE_BUTTON) { //DOLLY
    if (state == GLUT_DOWN) {
      win->viewVar.DollyPanStartPoint[0] = x;
      win->viewVar.DollyPanStartPoint[1] = y;
      glutMotionFunc (mouseDollyCallback);
      if (win->viewVar.movement == TRACK_SINGLE || win->viewVar.movement == TRACK_ALL) {// if we are currently trackballing, end trackball
	mouseTrackball(win, x, y, GL_TRUE);
      } else if ((win->viewVar.movement == PAN_SINGLE) || (win->viewVar.movement == PAN_ALL)) {
	mousePan (win, x, y);
      }
      if (glutGetModifiers() & GLUT_ACTIVE_SHIFT)
	win->viewVar.movement=DOLLY_ALL;
      else
	win->viewVar.movement=DOLLY_SINGLE;
    } else if (state == GLUT_UP) {
      mouseDolly (win, x, y);
      win->viewVar.movement=OFF;
    };
  } else if (button == GLUT_RIGHT_BUTTON) { //PAN
    if (state == GLUT_DOWN) {
      win->viewVar.DollyPanStartPoint[0] = x;
      win->viewVar.DollyPanStartPoint[1] = y;
      glutMotionFunc (mousePanCallback);
      if ((win->viewVar.movement == TRACK_SINGLE) || (win->viewVar.movement == TRACK_ALL)) {// if we are currently trackballing, end trackball
	mouseTrackball(win, x, y, GL_TRUE);
      } else if ((win->viewVar.movement == DOLLY_SINGLE) || (win->viewVar.movement == DOLLY_ALL)) {
	mouseDolly (win, x, y);
	win->viewVar.TrackBallRotation [0] = win->viewVar.TrackBallRotation [1] = win->viewVar.TrackBallRotation [2] = win->viewVar.TrackBallRotation [3] = 0.0f;
      }
      if (glutGetModifiers() & GLUT_ACTIVE_SHIFT)
	win->viewVar.movement=PAN_ALL;
      else
	win->viewVar.movement=PAN_SINGLE;
    } else if (state == GLUT_UP) {
      mousePan (win, x, y);
      win->viewVar.movement = OFF;
    };
  };
}

void spaceballmotion (int x, int y, int z)
{
  subWindow *win = &(gSubWindow[findSubWindow(glutGetWindow())]);
  long deadZone = 105;
	float scale = -win->viewVar.Camera.viewPos.z * 0.00000001f;
	if (abs (x) > deadZone) {
		GLfloat panX = abs (x) * x * scale;
		win->viewVar.Camera.viewPos.x += panX;
	}
	if (abs (y) > deadZone) {
		GLfloat panY = abs (y) * y * scale;
		win->viewVar.Camera.viewPos.y -= panY;
	}
	if (abs (z) > deadZone) {
		GLfloat dolly = abs (z) * z * scale;
		win->viewVar.Camera.viewPos.z += dolly;
		if (win->viewVar.Camera.viewPos.z == 0.0) // do not let z = 0.0
			win->viewVar.Camera.viewPos.z = 0.0001;
	}
	glutPostRedisplay();
}

void spaceballrotate (int rx, int ry, int rz)
{
	subWindow *win = &(gSubWindow[findSubWindow(glutGetWindow())]);
	long deadZone = 60;
	float rotation[4] = {0.0f, 0.0f, 0.0f, 0.0f};
	// handle rotations about each respective axis
	if (abs (rx) > deadZone) {
		rotation[0] = abs (rx) * -rx * 0.0000008f;
		rotation[1] = 1.0f;
		rotation[2] = 0.0f;
		rotation[3] = 0.0f;
		addToRotationTrackball (rotation, win->viewVar.WorldRotation);
	}
	if (abs (ry) > deadZone) {
		rotation[0] = abs (ry) * ry * 0.0000008f;
		rotation[1] = 0.0f;
		rotation[2] = 1.0f;
		rotation[3] = 0.0f;
		addToRotationTrackball (rotation, win->viewVar.WorldRotation);
	}
	if (abs(rz) > deadZone) {
		rotation[0] = abs (rz) * -rz * 0.0000008f;
		rotation[1] = 0.0f;
		rotation[2] = 0.0f;
		rotation[3] = 1.0f;
		addToRotationTrackball (rotation, win->viewVar.WorldRotation);
	}
	glutPostRedisplay();
}
	
void reshape(int w, int h) {
  glViewport(0,0,w,h);
  glutPostRedisplay();
};

void maindisplay() {
  glClear(GL_COLOR_BUFFER_BIT);
  glutSwapBuffers();
}

void mainreshape (int w, int h)
{
  change_shape(NO_CHANGE, false,w,h);
}


#pragma mark ---- inits ----


void initMainWindow(int mainwin) 
{
  glutSetWindow(mainwin);
  
}

void initSubWindow(subWindow *win, subWindow **sel, int parent, int x_index, int y_index, int w, int h, mut_mut_class *mmc)
{
	win->objVar.compiled=GL_FALSE;
	win->objVar.rand_seed = 0;
	win->objVar.draw_state = 0;
	win->objVar.max_dim = 20.0f;
	win->objVar.grow_val = -1.0f;
	win->objVar.lsystem=new Lsys(mmc);
	win->x_index=x_index; win->y_index=y_index;
	win->viewVar.Camera.screenWidth=w; win->viewVar.Camera.screenHeight=h;
	if (index_to_position(w, h, x_index, y_index, &(win->x), &(win->y), &(win->w), &(win->h))) {
		fprintf(stderr, "bad index number");
		exit(1);
	};

	win->parent=parent;
	win->selected=sel;
	win->ShowLsys = 0;
	win->ShowInfo = 0;
	win->clicktime = 0;
	win->number=glutCreateSubWindow(parent, win->x, win->y, win->w, win->h);
	

	initWindowVariables(win);
	BlankList(&(win->BlankList));
	win->objVar.DisplayList=win->BlankList;
		
	//set callback	
	glutDisplayFunc (display);
	glutReshapeFunc (reshape);
	glutKeyboardFunc (key);
	glutSpecialFunc (special);
	glutMouseFunc (mouse);
	glutMotionFunc(mouseTrackballCallback);
	glutSpaceballMotionFunc(spaceballmotion);
	glutSpaceballRotateFunc(spaceballrotate);
}

void quit() {
  if (filename && save_on_quit){	
    int i=0;
    while ((gSubWindow[i].x_index != MID_COLUMN) || (gSubWindow[i].y_index != MID_ROW))
      i++;
    if (i<SUBWINS) {
      fstream filestr(filename, fstream::out);
      if (filestr.is_open()) {
	filestr << gSubWindow[i].objVar.lsystem->get_Lstring();
	filestr.close();
      };
    };
  };
}

#pragma mark ---- main ----

int GLUTLBreeder(int argc, char* argv[])
{
  static subWindow *initial;
  mut_mut_class base_mutation_params;
  int mainwin, i, j;

  glutInit(&argc, (char **)argv);
  glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH); // non-stereo for main window
  glutInitWindowPosition (300, 50);
  glutInitWindowSize (900, 600);
  mainwin = glutCreateWindow("L-Breeder");
  
  
  initMainWindow(mainwin);
  glutReshapeFunc (mainreshape);
  glutDisplayFunc(maindisplay);
  glutKeyboardFunc (key);
  glutSpecialFunc (special);
  
  subWindow *win;
  initial=win=gSubWindow;
  for(i=0;i<COLUMNS;i++) for(j=0;j<ROWS;j++) {
    initSubWindow(win, &initial, mainwin, i,j,900,600, &base_mutation_params);
    if ((i==MID_COLUMN) && (j==MID_ROW))
      initial=win;
    win++;
  };
  glutSetWindow(initial->number);
  
  if (argc>2) {
    max_vertices=atol(argv[2]);
  };

  if (argc>1) {
    filename=argv[1];
    fstream filestr(filename, fstream::in);
    if (filestr.is_open()) {
      *(initial->objVar.lsystem) << filestr;
      filestr.close();
    };
  } else {
    *(initial->objVar.lsystem) << Lsys::basic_tree;
  };
  
  glutPostRedisplay();
  
  breed(initial);
  
  atexit(quit);
  glutMainLoop();
  
  return 0;
}





