/*
 * Decompiled with CFR 0.152.
 */
package org.limewire.ui.swing.player;

import ca.odell.glazedlists.EventList;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import javax.media.ControllerEvent;
import javax.media.ControllerListener;
import javax.media.EndOfMediaEvent;
import javax.media.IncompatibleSourceException;
import javax.media.Player;
import javax.media.StartEvent;
import javax.media.StopEvent;
import javax.media.Time;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.Timer;
import net.sf.fmj.concurrent.ExecutorServiceManager;
import org.limewire.concurrent.ExecutorsHelper;
import org.limewire.concurrent.ThreadPoolListeningExecutor;
import org.limewire.core.api.file.CategoryManager;
import org.limewire.core.api.library.LocalFileItem;
import org.limewire.player.api.PlayerState;
import org.limewire.setting.evt.SettingEvent;
import org.limewire.setting.evt.SettingListener;
import org.limewire.ui.swing.library.navigator.LibraryNavItem;
import org.limewire.ui.swing.player.PlayerMediator;
import org.limewire.ui.swing.player.PlayerMediatorListener;
import org.limewire.ui.swing.player.VideoDisplayDirector;
import org.limewire.ui.swing.player.VideoPlayerFactory;
import org.limewire.ui.swing.settings.SwingUiSettings;
import org.limewire.ui.swing.util.GuiUtils;
import org.limewire.ui.swing.util.I18n;
import org.limewire.ui.swing.util.NativeLaunchUtils;
import org.limewire.ui.swing.util.SwingUtils;

@Singleton
class VideoPlayerMediator
implements PlayerMediator {
    private final String playlistsNotSupported = I18n.tr("Playlists not supported in video");
    private Player player;
    private File currentVideo;
    private final VideoDisplayDirector displayDirector;
    private final List<PlayerMediatorListener> listenerList;
    private volatile Timer updateTimer;
    private final CategoryManager categoryManager;
    private boolean isSeeking;
    private final PlayerInitializer playerInitializer = new PlayerInitializer();
    private final ControllerListener controllerListener = new VideoControllerListener();

    @Inject
    VideoPlayerMediator(VideoDisplayDirector displayDirector, CategoryManager categoryManager) {
        this.displayDirector = displayDirector;
        this.categoryManager = categoryManager;
        this.listenerList = new ArrayList<PlayerMediatorListener>();
        ThreadPoolListeningExecutor tpe = new ThreadPoolListeningExecutor(1, 1, 5L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), ExecutorsHelper.daemonThreadFactory("Video ThreadPool"));
        ExecutorServiceManager.setExecutorService(ExecutorsHelper.unconfigurableExecutorService(tpe));
    }

    @Inject
    void register() {
        SwingUiSettings.PLAYER_ENABLED.addSettingListener(new SettingListener(){

            @Override
            public void settingChanged(SettingEvent evt) {
                SwingUtilities.invokeLater(new Runnable(){

                    @Override
                    public void run() {
                        boolean enabled = SwingUiSettings.PLAYER_ENABLED.getValue();
                        if (!enabled && VideoPlayerMediator.this.player != null) {
                            VideoPlayerMediator.this.closeVideo();
                        }
                    }
                });
            }
        });
    }

    @Override
    public void addMediatorListener(PlayerMediatorListener listener) {
        this.listenerList.add(listener);
    }

    @Override
    public void removeMediatorListener(PlayerMediatorListener listener) {
        this.listenerList.remove(listener);
    }

    @Override
    public File getCurrentMediaFile() {
        return this.currentVideo;
    }

    @Override
    public PlayerState getStatus() {
        if (this.player == null) {
            return PlayerState.UNKNOWN;
        }
        return this.convertControllerState(this.player.getState());
    }

    private PlayerState convertControllerState(int controllerState) {
        if (this.isSeeking) {
            return PlayerState.SEEKING;
        }
        switch (controllerState) {
            case 600: {
                return PlayerState.PLAYING;
            }
            case 300: {
                return PlayerState.PAUSED;
            }
        }
        return PlayerState.UNKNOWN;
    }

    @Override
    public boolean isActivePlaylist(LibraryNavItem navItem) {
        throw new UnsupportedOperationException(this.playlistsNotSupported);
    }

    @Override
    public boolean isPaused(File file) {
        return file.equals(this.currentVideo) && this.player != null && this.player.getState() != 600;
    }

    @Override
    public boolean isPlaying(File file) {
        return this.player.getState() == 600 && file.equals(this.currentVideo) && this.player != null;
    }

    @Override
    public boolean isPlaylistSupported() {
        return false;
    }

    @Override
    public boolean isSeekable() {
        return this.isDurationMeasurable();
    }

    @Override
    public boolean isShuffle() {
        return false;
    }

    @Override
    public boolean hasScrollingTitle() {
        return false;
    }

    @Override
    public void nextSong() {
        throw new UnsupportedOperationException(this.playlistsNotSupported);
    }

    @Override
    public void pause() {
        if (this.player != null) {
            this.player.stop();
        }
    }

    @Override
    public void play(File file) {
        if (file.equals(this.currentVideo)) {
            return;
        }
        this.initializePlayerOrNativeLaunch(file, null, false, true);
    }

    private void initializePlayerOrNativeLaunch(File file, Time time, boolean isFullScreen, boolean autoStart) {
        GuiUtils.getMainFrame().setCursor(Cursor.getPredefinedCursor(3));
        this.currentVideo = file;
        this.playerInitializer.initialize(file, time, isFullScreen, autoStart);
    }

    @Override
    public void play(LocalFileItem localFileItem) {
        this.play(localFileItem.getFile());
    }

    @Override
    public void prevSong() {
        throw new UnsupportedOperationException(this.playlistsNotSupported);
    }

    @Override
    public void resume() {
        if (this.player != null && this.player.getState() != 600) {
            this.player.start();
        }
    }

    @Override
    public void setActivePlaylist(LibraryNavItem navItem) {
        throw new UnsupportedOperationException(this.playlistsNotSupported);
    }

    @Override
    public void setPlaylist(EventList<LocalFileItem> fileList) {
        throw new UnsupportedOperationException(this.playlistsNotSupported);
    }

    @Override
    public void setShuffle(boolean shuffle) {
        throw new UnsupportedOperationException(this.playlistsNotSupported);
    }

    @Override
    public void setVolume(double value) {
        if (this.player.getGainControl() != null) {
            this.player.getGainControl().setLevel((float)value);
        }
    }

    @Override
    public boolean isVolumeSettable() {
        return this.player != null && this.player.getGainControl() != null;
    }

    @Override
    public void skip(double percent) {
        if (!this.isDurationMeasurable()) {
            throw new IllegalStateException("Can not skip when duration is unmeasurable");
        }
        this.isSeeking = true;
        this.player.setMediaTime(new Time(percent * this.player.getDuration().getSeconds()));
        this.isSeeking = false;
    }

    @Override
    public void stop() {
        if (this.player != null) {
            this.player.stop();
        }
    }

    private void firePlayerStateChanged(PlayerState state) {
        for (PlayerMediatorListener listener : this.listenerList) {
            listener.stateChanged(state);
        }
    }

    private void fireProgressUpdated() {
        this.fireProgressUpdated((float)(this.player.getMediaTime().getSeconds() / this.player.getDuration().getSeconds()));
    }

    private void fireProgressUpdated(float progress) {
        for (PlayerMediatorListener listener : this.listenerList) {
            listener.progressUpdated(progress);
        }
    }

    private void fireSongChanged(String name) {
        for (PlayerMediatorListener listener : this.listenerList) {
            listener.songChanged(name);
        }
    }

    public void closeVideo() {
        if (this.playerInitializer.isInitializing()) {
            this.playerInitializer.cancel();
        }
        this.displayDirector.close();
        this.killTimer();
        this.killPlayer();
        this.currentVideo = null;
    }

    private void killTimer() {
        if (this.updateTimer != null) {
            if (this.updateTimer.isRunning()) {
                this.updateTimer.stop();
            }
            this.updateTimer = null;
        }
    }

    private void killPlayer() {
        if (this.player != null) {
            this.player.stop();
            this.player.removeControllerListener(this.controllerListener);
            this.player.close();
            this.player.deallocate();
            this.player = null;
        }
    }

    public void setFullScreen(boolean isFullScreen) {
        if (this.player == null) {
            return;
        }
        if (this.displayDirector.isFullScreen() == isFullScreen) {
            return;
        }
        if (this.playerInitializer.isInitializing()) {
            return;
        }
        boolean isPlaying = this.player.getState() == 600;
        this.reInitializePlayer(isFullScreen, isPlaying);
    }

    private void reInitializePlayer(boolean isFullScreen, boolean isPlaying) {
        if (this.player == null) {
            throw new IllegalStateException("Video player not initialized");
        }
        this.killTimer();
        Time time = this.isDurationMeasurable() ? this.player.getMediaTime() : null;
        this.killPlayer();
        this.initializePlayerOrNativeLaunch(this.currentVideo, time, isFullScreen, isPlaying);
    }

    public boolean isFullScreen() {
        return this.displayDirector.isFullScreen();
    }

    private boolean isDurationMeasurable() {
        if (this.player == null) {
            return false;
        }
        long time = this.player.getDuration().getNanoseconds();
        return Player.DURATION_UNBOUNDED.getNanoseconds() != time && Player.DURATION_UNKNOWN.getNanoseconds() != time && Time.TIME_UNKNOWN.getNanoseconds() != time;
    }

    private void setEndOfMedia() {
        this.player.stop();
        this.player.setMediaTime(new Time(0L));
        this.updateTimer.stop();
        this.firePlayerStateChanged(PlayerState.EOM);
        this.fireProgressUpdated(100.0f);
    }

    private class PlayerInitializer {
        private PlayerInitalizationWorker initializationWorker;

        private PlayerInitializer() {
        }

        public boolean isInitializing() {
            return this.initializationWorker != null;
        }

        public void initialize(File file, Time time, boolean isFullScreen, boolean autoStart) {
            if (this.isInitializing()) {
                this.cancel();
            }
            this.initializationWorker = new PlayerInitalizationWorker(VideoPlayerMediator.this.currentVideo, time, isFullScreen, autoStart);
            this.initializationWorker.execute();
        }

        public void cancel() {
            this.initializationWorker.cancelInitialization();
            this.initializationWorker = null;
        }

        private void finish(Player newPlayer, Time time, boolean isFullScreen, boolean autoStart) {
            this.initializationWorker = null;
            if (newPlayer == null) {
                VideoPlayerMediator.this.displayDirector.close();
                return;
            }
            if (time != null) {
                newPlayer.setMediaTime(time);
            }
            newPlayer.addControllerListener(VideoPlayerMediator.this.controllerListener);
            VideoPlayerMediator.this.player = newPlayer;
            VideoPlayerMediator.this.displayDirector.show();
            VideoPlayerMediator.this.fireSongChanged(VideoPlayerMediator.this.currentVideo.getName());
            if (VideoPlayerMediator.this.updateTimer == null) {
                VideoPlayerMediator.this.updateTimer = new Timer(100, new TimerAction());
            }
            VideoPlayerMediator.this.updateTimer.start();
            this.startVideo(autoStart);
        }

        private void startVideo(final boolean startVideo) {
            ActionListener listener = new ActionListener(){

                @Override
                public void actionPerformed(ActionEvent e) {
                    if (VideoPlayerMediator.this.player != null) {
                        VideoPlayerMediator.this.player.start();
                        VideoPlayerMediator.this.firePlayerStateChanged(PlayerState.PLAYING);
                        if (VideoPlayerMediator.this.isDurationMeasurable()) {
                            VideoPlayerMediator.this.fireProgressUpdated();
                        }
                        if (!startVideo) {
                            VideoPlayerMediator.this.player.stop();
                        }
                    }
                }
            };
            Timer timer = new Timer(100, listener);
            timer.setRepeats(false);
            timer.start();
        }

        private class PlayerInitalizationWorker
        extends SwingWorker<Player, Void> {
            private final File mediaFile;
            private final Time time;
            private final boolean autoStart;
            private final boolean isFullScreen;
            private boolean canceled = false;
            private final Container renderPanel = new JPanel(new BorderLayout());

            public PlayerInitalizationWorker(File mediaFile, Time time, boolean isFullScreen, boolean autoStart) {
                this.mediaFile = mediaFile;
                this.time = time;
                this.autoStart = autoStart;
                this.isFullScreen = isFullScreen;
                VideoPlayerMediator.this.displayDirector.initialize(this.renderPanel, isFullScreen);
            }

            public void cancelInitialization() {
                this.canceled = true;
            }

            @Override
            protected Player doInBackground() throws Exception {
                try {
                    return new VideoPlayerFactory().createVideoPlayer(this.mediaFile, this.renderPanel);
                }
                catch (IncompatibleSourceException e) {
                    this.nativeLaunch(this.mediaFile);
                    return null;
                }
            }

            private void nativeLaunch(File file) {
                NativeLaunchUtils.safeLaunchFile(file, VideoPlayerMediator.this.categoryManager);
            }

            @Override
            protected void done() {
                GuiUtils.getMainFrame().setCursor(Cursor.getDefaultCursor());
                Player newPlayer = null;
                try {
                    newPlayer = (Player)this.get();
                }
                catch (InterruptedException e) {
                }
                catch (ExecutionException e) {
                    throw new RuntimeException(e);
                }
                if (this.canceled) {
                    if (newPlayer != null) {
                        newPlayer.close();
                        newPlayer.deallocate();
                    }
                    return;
                }
                PlayerInitializer.this.finish(newPlayer, this.time, this.isFullScreen, this.autoStart);
            }
        }
    }

    private class TimerAction
    implements ActionListener {
        private TimerAction() {
        }

        @Override
        public void actionPerformed(ActionEvent e) {
            if (!VideoPlayerMediator.this.isDurationMeasurable()) {
                VideoPlayerMediator.this.updateTimer.stop();
                return;
            }
            if (VideoPlayerMediator.this.player.getMediaTime().getSeconds() >= VideoPlayerMediator.this.player.getDuration().getSeconds()) {
                VideoPlayerMediator.this.setEndOfMedia();
            } else {
                VideoPlayerMediator.this.fireProgressUpdated();
            }
        }
    }

    private class VideoControllerListener
    implements ControllerListener {
        private VideoControllerListener() {
        }

        @Override
        public void controllerUpdate(final ControllerEvent controllerEvent) {
            SwingUtils.invokeNowOrLater(new Runnable(){

                @Override
                public void run() {
                    if (controllerEvent instanceof EndOfMediaEvent) {
                        VideoPlayerMediator.this.setEndOfMedia();
                    } else if (controllerEvent instanceof StartEvent || controllerEvent.getSourceController().getState() == 600) {
                        VideoPlayerMediator.this.firePlayerStateChanged(PlayerState.OPENED);
                        VideoPlayerMediator.this.firePlayerStateChanged(PlayerState.PLAYING);
                        if (VideoPlayerMediator.this.updateTimer == null) {
                            VideoPlayerMediator.this.updateTimer = new Timer(100, new TimerAction());
                        }
                        if (!VideoPlayerMediator.this.updateTimer.isRunning()) {
                            VideoPlayerMediator.this.updateTimer.start();
                        }
                    } else if (controllerEvent instanceof StopEvent) {
                        VideoPlayerMediator.this.firePlayerStateChanged(PlayerState.STOPPED);
                        if (VideoPlayerMediator.this.updateTimer != null) {
                            VideoPlayerMediator.this.updateTimer.stop();
                        }
                    }
                }
            });
        }
    }
}

