// Copyright (c) 2002-2010 Wieger Wesselink
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

/// \file sat/buddy.h
/// \brief Contains bdd functions

#ifndef SAT_BUDDY_H
#define SAT_BUDDY_H

#include <iostream>
#include <sstream>
#include <map>
#include <string>
#include <vector>
#include <stdexcept>
#include <boost/tokenizer.hpp>
#include "bdd.h" // from Buddy
#include "sat/variable_set.h"
#include "sat/math_util.h"

namespace buddy {

struct size_threshold
{
  static int& value()
  {
    static int value;
    return value;
  }
};

//---------------------------------------------------------//
//                     size
//---------------------------------------------------------//
// returns the number of nodes of the bdd
inline
int size(bdd b)
{
// REORDERING
  int result = bdd_nodecount(b);

#ifdef REORDERING
  if (result > size_threshold::value())
  {
    size_threshold::value() = 2 * size_threshold::value();
    std::cout << "Size threshold          = " << size_threshold::value() << std::endl;
    std::cout << "Before reordering: size = " << result << std::endl;
    bdd_reorder(BDD_REORDER_SIFT);
    result = bdd_nodecount(b);
    std::cout << "After reordering : size = " << size(b) << std::endl;
  }
#endif // REORDERING
    
  return result;
}

//---------------------------------------------------------//
//                     satcount
//---------------------------------------------------------//
// returns the number of satisfying variable assignments
inline
double satcount(bdd b, int n)
{
  return bdd_satcount(b) / math::pow2(bdd_varnum() - n);
}

//---------------------------------------------------------//
//                     one
//---------------------------------------------------------//
// returns the constant value 1
inline
bdd one()
{
  return bddtrue;
}

//---------------------------------------------------------//
//                     zero
//---------------------------------------------------------//
// returns the constant value 0
inline
bdd zero()
{
  return bddfalse;
}

//---------------------------------------------------------//
//              exists
//---------------------------------------------------------//
// remove all occurrences in the bdd (left & right) of variables
// in the set var by existential quantification
inline
bdd exists(bdd left, bdd right, bdd var)
{
  return bdd_appex(left, right, bddop_and, var);
}

//---------------------------------------------------------//
//              forall
//---------------------------------------------------------//
// remove all occurrences in the bdd (left & right) of variables
// in the set var by universal quantification
inline
bdd forall(bdd left, bdd right, bdd var)
{
  return bdd_appall(left, right, bddop_and, var);
}

//---------------------------------------------------------//
//                     find_sat_solution
//---------------------------------------------------------//
// finds one satisfying variable assignment; returns bdd_zero if no solution exists
inline
bdd find_sat_solution(bdd b, const variable_set& variables)
{
  bdd result = bdd_satone(b);

  // workaround for a serious bug
  if (satcount(b, variables.size()) > 1)
  {
//    std::cerr << "ERROR in find_sat_solution" << std::endl;
    for (unsigned int i = 0; i < variables.size(); i++)
    {
      int index = variables.index(i);
      bdd x = result & bdd_ithvar(index);
      if (x != zero())
        result = x;
    }
  }

  return result;
}

// prototype
static inline
void AllSatHandler(char* varset, int size);

class AllSat
{
 public:
    static std::ostringstream& stream()
    {
      static std::ostringstream os;
      return os;
    }
    
    static std::string all_sat(const bdd& b, const variable_set& variables)
    {
      stream().str("");
      bdd_allsat(b, AllSatHandler);
      std::string allsat = stream().str();
      typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
      boost::char_separator<char> sep("\n");
      Tokenizer t(allsat, sep);

      std::string result;
      for(Tokenizer::iterator i = t.begin(); i != t.end(); ++i)
      {
        std::string s = *i;
        for (unsigned int j = 0; j < variables.size(); j++)
          result = result + s[variables.index(j)];
        result = result + "\n";
      }
      return result;
    }
};

static inline
void AllSatHandler(char* varset, int size)
{
  for (int v = 0; v < size; ++v)
  {
    AllSat::stream() << (varset[v] < 0 ? 'X' : (char)('0' + varset[v]));
  }
  AllSat::stream() << std::endl;
}

//---------------------------------------------------------//
//                 bdd_allsat
//---------------------------------------------------------//
// returns a string representation of all solutions
inline
std::string bdd_allsat(const bdd& b, const variable_set& variables)
{
  return AllSat::all_sat(b, variables);
}

//---------------------------------------------------------//
//                 load
//---------------------------------------------------------//
// loads a bdd from a file
inline
bdd load(const std::string& filename)
{
  bdd result;
  bdd_fnload(const_cast<char*>(filename.c_str()), result);
  return result;
}

//---------------------------------------------------------//
//                 save
//---------------------------------------------------------//
// saves a bdd to a file
inline
void save(const std::string& filename, const bdd& b)
{
  bdd_fnsave(const_cast<char*>(filename.c_str()), b);
}

//---------------------------------------------------------//
//                 replace
//---------------------------------------------------------//
// replace variables without taking care of clashes like x0 -> x1; x1 -> x0
inline
bdd replace(bdd b, const std::map<int, int>& permutation)
{
  bdd result = b;
  for (std::map<int, int>::const_iterator i = permutation.begin(); i != permutation.end(); ++i)
  {
    int i0 = i->first;
    int i1 = i->second;
    result = bdd_compose(result, bdd_ithvar(i0), i1);
  }
  return result;
}

//---------------------------------------------------------//
//                 permute
//---------------------------------------------------------//
// permute variables; scratch must be the index of a variable that doesn't
// occur in b
inline
bdd permute(bdd b, const std::map<int, int>& permutation, int scratch)
{ 
  bdd result = b;

  for (std::map<int, int>::const_iterator i = permutation.begin(); i != permutation.end(); ++i)
  {
    int i0 = i->first;
    int i1 = i->second;
    if (i0 < i1)
    {
      result = bdd_compose(result, bdd_ithvar(scratch) , i0);
      result = bdd_compose(result, bdd_ithvar(i0) , i1);
      result = bdd_compose(result, bdd_ithvar(i1), scratch);
    }
  }

  return result;
}

//---------------------------------------------------------//
// returns a string representation of a bdd solution
template <class BDD>
std::string solution_string(const BDD& b, const variable_set& variables)
{
  std::ostringstream out;
  int n = variables.size();

  // print validation (very inefficient code!)
  if (satcount(b, n) > 0)
  {
    BDD s = find_sat_solution(b, variables);
    for (int i = 0; i < n; i++)
    {
      BDD x = bdd_ithvar(variables.index(i));
      out << (((s & x) == zero()) ? "0" : "1");
    }
  }
  return out.str();
}

//---------------------------------------------------------//
//                     print_solution
//---------------------------------------------------------//
// prints a solution of a bdd
template <class BDD>
void print_solution(const BDD& b, const variable_set& variables)
{
  int n = variables.size();

  // print validation (very inefficient code!)
  if (satcount(b, n) > 0)
  {
    BDD s = find_sat_solution(b, variables);
    std::cout << solution_string(b, variables);
  }
}

//---------------------------------------------------------//
//                     print_variables
//---------------------------------------------------------//
// prints info about a bdd
void print_variables(const variable_set& variables)
{
  std::string line;
  for (unsigned int i = 0; i < variables.size(); i++)
  { 
    std::string s = line + (line.size() == 0 ? "" : " ") + variables.name(i);
    if (line.size() > 70)
    {
      std::cout << s << "\n";
      line = "";
    }
    else
    {
      line = s;
    }
  }
  if (line.size() > 0)
  {
    std::cout << line << "\n";
  }
}

//---------------------------------------------------------//
//                     bdd_print
//---------------------------------------------------------//
// prints info about a bdd
template <class BDD>
void print(const BDD& b, const variable_set& variables)
{
  int n = variables.size();

  std::cout << "number of variables   : " << n << std::endl;
  std::cout << "number of nodes       : " << size(b) << "\n";
  std::cout << "number of validations : " << math::print_as_int(satcount(b, n)) << "\n\n";
  std::cout << "variables:\n";
  print_variables(variables);

  if (satcount(b, n) > 0)
  {
    std::cout << "\nvalidation:\n";
    print_solution(b, variables);
    std::cout << "\n\nSATISFIABLE";
  }
  else
  {
    std::cout << "\nUNSATISFIABLE";
  }
}

void print(bdd b, const variable_set& variables)
{
  int n = variables.size();
  std::cout << "bdd: " << n << " " << size(b) << " " << math::print_as_int(satcount(b, n)) << std::endl;
}

} // namespace buddy

#endif // SAT_BUDDY_H
