//-----------------------------------------------------------------------------
// Memory management
//-----------------------------------------------------------------------------

#ifndef __MEM_H__
#define __MEM_H__

#include "definitions.h"

#define DESCRIPTION_MAXLENGTH 64

#ifdef _DEBUG
  #if 0
    #define DEBUG_MEM     // Activate the memory debug manager (very slow !)
  #endif
#endif

/**
 * The memory management provides internal methods for memory allocation
 * functions. The cake allocation is by default disabled, but it could be
 * used for debugging.
 */
#ifdef DEBUG_MEM
  #include <stddef.h>

  typedef char strDescription[DESCRIPTION_MAXLENGTH+1];
  typedef struct
  {
    void *mem;          /**< Pointer to memory zone */
    size_t len;         /**< Memory zone length */
    strDescription descr;   /**< Description string (for debug help) */
  } memZone;

  /**
   * Adds a new reference into register.
   * @param m The start of memory zone to register.
   * @param l The length of memory zone.
   * @param d A description for the memory zone (to help for debug)
   */
  void add_reference(void *m, int l, strDescription d);
  
  /**
   * Updates an existing reference.
   * @param m The start of memory zone to update.
   * @param new_m The new start memory zone.
   * @param new_l The new length of memory zone.
   * @param new_d The new description of memory zone.
   */
  void update_reference(void *m, void *new_m, int new_l, strDescription new_d);
  
  /**
   * Removes an existing reference.
   * @param m The start of memory zone to remove from register.
   */
  void remove_reference(void *m);
  
  /**
   * Generates a break in program execution.
   * This should allow to interrupt the program to perform debugging.
   */
  void force_break(void);

  /**
   * Free the allocated memory automatically.
   * Attention: Use this function if you are sure that what you are doing !
   */
  void auto_freemem(void);

  /**
   * Destroys the register.
   */
  void delete_memregister(void);

  /**
   * Writes a report on current memory allocation in logfile.
   */
  void log_mem_report(void);

  /**
  * Gets the number of references in memory register
  * @return the number of allocations
  */
  int get_references(void);

  //-------------------------------------------------------------------------

  /**
   * Allocate memory space
   * @param size the memory size to allocate
   * @return a pointer at the beginning of memory zone or a NULL pointer
   *         if the requested memory couldn't be allocated
   */
  void *cake_malloc(size_t size, strDescription description = "");

  /**
   * Resize an allocated memory zone
   * @param mem the start of memory zone
   * @param size the new size of memory zone
   */
  void *cake_realloc(void *mem, size_t size, strDescription description = "");

  /**
   * Free a memory zone
   * @param mem the start of memory zone
   */
  void cake_free(void *mem);

#else
  #include <stdlib.h>

  #define cake_malloc(l, d) malloc(l)         /**< standard malloc function */
  #define cake_realloc(m, l, d) realloc(m, l)     /**< standard realloc function */
  #define cake_free(m) free(m)            /**< standard free function */
#endif

#endif  /* __MEM_H__ */
