 //-----------------------------------------------------------------------------
// Cake App
//-----------------------------------------------------------------------------

#include "app.h"
#include "alias.h"
#include "commands.h"
#include "console.h"
#include "definitions.h"
#include "framework.h"
#include "logfile.h"
#include "overlay.h"
#include "render.h"
#include "timer.h"
#include "vars.h"
#include "system.h"
#include "math.h"
#include "mem.h"
#include "sound.h"

#ifdef WIN32
  #include <process.h>          // for _getpid
  #include <direct.h>           // for _getcwd
  #include <string.h>           // for _strlwr
  #include <time.h>
#else
  #include <time.h>
  #include <unistd.h>           // for getcwd
  #define _getcwd getcwd
#endif

LogFile*    gLogFile = NULL;  // Log file
Console*    gConsole = NULL;  // Console
FrameWork*    gFramework = NULL;  // FrameWork
Vars*     gVars = NULL;   // Variables
Alias*      gAlias = NULL;    // Aliases
Commands*   gCommands = NULL; // Commands
Render*     gRender = NULL;   // Render
Overlay*    gOver = NULL;   // Overlay

#define FILENAME_LENGTH 1025    // Filenames length

Var r_path("r_path", "base;baseq3", VF_PERSISTENT | VF_LATCH | VF_SYSTEM);
Var r_mapsubdir("r_mapsubdir", "maps/", VF_PERSISTENT | VF_SYSTEM);
Var cg_drawFPS("cg_drawFPS", 0, VF_PERSISTENT);
Var cg_FPSrefreshrate("cg_FPSrefreshrate", 250.f, VF_PERSISTENT);
Var cg_drawtime("cg_drawtime", 0, VF_PERSISTENT);
Var cg_drawflush("cg_drawflush", 0);
Var cg_drawtris("cg_drawtris", 0);
Var cg_drawverts("cg_drawverts", 0);
Var cg_apicalls("cg_apicalls", 0);
Var cg_showpos("cg_showpos", 0);
Var cg_showrot("cg_showrot", 0);
Var r_debug("r_debug", 0);
Var sys_appstate("sys_appstate", INACTIVE, VF_SYSTEM);
Var sys_soundsupport("sys_soundsupport", 0, VF_SYSTEM);
Var r_targetlocationdist("r_targetlocationdist", 32768.f);

float lastfps = 0;          // fps value for last fps refresh interval
float timelastfps = 0;        // time of last fps display
float totallastfps = 1;       // sum of frames fps during current fps refresh interval
unsigned int frameslastfps = 0;   // number of frames during current fps refresh interval
char demomsg[64];         // message display when demo is recording/playing
int demofilenameindex = 0;      // variable used for demo management

/**
 * Return the length of a string without color codes
 */
int _strlen(const char *s)
{
  int i = 0, r, l = (int) strlen(s);
  r = l;
  
  while (l--)
  {
    if (s[i++] == '^') r-=2;
  }

  return r;
}

//-----------------------------------------------------------------------------
// Predefined commands
//-----------------------------------------------------------------------------

void cmd_bsplist(int argc, char *argv[])
{
  getPAKContent(NULL, ".bsp", argc>1?argv[1]:NULL);
}

void cmd_wavlist(int argc, char *argv[])
{
  getPAKContent(NULL, ".wav", argc>1?argv[1]:NULL);
}

void cmd_md3list(int argc, char *argv[])
{
  getPAKContent(NULL, ".md3", argc>1?argv[1]:NULL);
}

void cmd_listfiles(int argc, char *argv[])
{
  if (argc > 1)
  {
    getPAKContent(NULL, argv[1], argc>2?argv[2]:NULL);
  }
  else gConsole->Insertln("usage: %s <string> [<string>]", argv[0]);
}

void cmd_logpreviews(int argc, char *argv[])
{
  gConsole->Insertln("Writing previews into logfile...");
  logPreviews();
}

App::App(void)
{
  sys_appstate = INACTIVE;
  
  _getcwd(app_path, PATH_LENGTH); // get current path
  strcat(app_path, "\\");
  
  world   = NULL;

  clients = NULL;
  demos = NULL;
  numclients = 0;
  curr_client = 0;

  memset(curr_map, '\0', 32*sizeof(char));
  reloading = false;

  if (!(gCommands = new Commands))  ThrowException(ALLOCATION_ERROR, "App:App.gCommands");
  if (!(gVars = new Vars))      ThrowException(ALLOCATION_ERROR, "App:App.gVars");
  if (!(gAlias = new Alias))      ThrowException(ALLOCATION_ERROR, "App:App.gAlias");
  if (!(gLogFile = new LogFile))    ThrowException(ALLOCATION_ERROR, "App:App.gLogFile");
  if (!(gRender = new Render))    ThrowException(ALLOCATION_ERROR, "App:App.gRender");
  if (!(gConsole = new Console))    ThrowException(ALLOCATION_ERROR, "App:App.gConsole");
  if (!(gFramework = new FrameWork))  ThrowException(ALLOCATION_ERROR, "App:App.gFramework");
  if (!(gOver = new Overlay))     ThrowException(ALLOCATION_ERROR, "App:App.gOver");
  gOver->Init();

  gConsole->Insertln("<br/><b>^6----- Starting cake engine ------------------------------------</b>");

  #ifdef _DEBUG
    gConsole->Insertln("^1%s ^0debug version of %s (%s)", _VERSION_, __DATE__, __TIME__);
  #else
    gConsole->Insertln("^2%s ^0release version of %s (%s)", _VERSION_, __DATE__, __TIME__);
  #endif

  // Register variables
  gVars->RegisterVar(r_path);
  gVars->RegisterVar(r_mapsubdir);
  gVars->RegisterVar(cg_drawFPS);
  gVars->RegisterVar(cg_FPSrefreshrate);
  gVars->RegisterVar(cg_drawtime);
  gVars->RegisterVar(cg_drawflush);
  gVars->RegisterVar(cg_drawtris);
  gVars->RegisterVar(cg_drawverts);
  gVars->RegisterVar(cg_apicalls);
  gVars->RegisterVar(cg_showpos);
  gVars->RegisterVar(cg_showrot);
  gVars->RegisterVar(r_debug);
  gVars->RegisterVar(sys_appstate);
  gVars->RegisterVar(sys_soundsupport);
  gVars->RegisterVar(r_targetlocationdist);

  // Add commands
  gCommands->AddCommand("bsplist", cmd_bsplist, "display the list of bsp map names (if an argument is present, only names containing this argument will be displayed)");
  gCommands->AddCommand("wavlist", cmd_wavlist, "display the list of wav (music and sounds) files (if an argument is present, only names containing this argument will be displayed)");
  gCommands->AddCommand("md3list", cmd_md3list, "display the list of md3 files (if an argument is present, only names containing this argument will be displayed)");
  gCommands->AddCommand("listfiles", cmd_listfiles, "display the list of all files matching with a substring");
  gCommands->AddCommand("logpreviews", cmd_logpreviews, "write the map list with corresponding levelshot into the logfile");
}

void App::Init(void)
{
  sys_appstate = INITIALIZING;

  gCommands->AddToHistory = false;

  // init the file environment
  FileEnvironment::Init();
  FileEnvironment::AddPath("./");

  // Read the config file
  ConfigFile(CONFIG_FILENAME);

  Timer::Init();

  #ifdef WIN32
  {
    // OS Info
    OSVERSIONINFO vinfo;
    vinfo.dwOSVersionInfoSize = sizeof(vinfo);

    gLogFile->OpenFile();

    if (!GetVersionEx (&vinfo))
      gConsole->Insertln("^5WARNING: Couldn't get OS info");

    gConsole->Insertln("<b>System infos :</b>");
    gConsole->Insertln("\tWin32 platform");
    gConsole->Insertln("\tBuild number : %d", vinfo.dwBuildNumber);
    gConsole->Insertln("\tMajor version : %d", vinfo.dwMajorVersion);
    gConsole->Insertln("\tMinor version : %d", vinfo.dwMinorVersion);
    gConsole->Insertln("\tPlatform ID : %d", vinfo.dwPlatformId);
    gConsole->Insertln("\tCSD version : %s", vinfo.szCSDVersion);
    gConsole->Insertln("\tCurrent process ID : %d", _getpid());
    gConsole->Insertln("\tCPU vendor name : %s", GetVendorString());
    gConsole->Insertln("\tCPU speed : %d MHz", GetCpuSpeed());
    gConsole->Insertln("\tCPU cache size : %d", GetCacheInfo());
    if (GetCPUCaps(HAS_3DNOW)) gConsole->Insertln("\t3DNow! detected");
    struct tm *newtime; time_t aclock; time(&aclock); newtime = localtime(&aclock);
    gConsole->Insertln("\tSystem date and time : %s", asctime(newtime));

    gLogFile->CloseFile();
  }
  #else
  {
    gConsole->Insertln("\tNot a Win32 Platform");
  }
  #endif

  #if 0
  { // in this case, last path is more important
    // for instance, with "../base;../missionpack", the more
    // important path is "../missionpack"
    int i, j, l;
    i = 0;
    j = 0;
    l = (int) strlen(r_path.svalue);
    char buffer[256];
    while (i < l)
    {
      memset(buffer, 0, 256);
      j = 0;
      while (i < l && r_path.svalue[i] != ';') buffer[j++] = r_path.svalue[i++];
      if (buffer[j] != '/') buffer[j] = '/';
      FileEnvironment::AddPath(buffer);
      ++i;
    }
  }
  #else
  { // in this case, first path is more important
    // for instance, with "../base;../missionpack", the more
    // important path is "../base"
    int i, j;
    i = (int) strlen(r_path.svalue) - 1;
    j = i;
    char buffer[256];
    while (i >= 0)
    {
      memset(buffer, 0, 256);
      while (i >= 0 && r_path.svalue[i] != ';') --i;
      if (i > 0) strncpy(buffer, &r_path.svalue[i+1], j-i);
      else strncpy(buffer, r_path.svalue, j-i);
      if (buffer[j-i] != '/') buffer[j-i] = '/';
      FileEnvironment::AddPath(buffer);
      j = --i;
    }
  }
  #endif
  FileEnvironment::dumpEnvironment();
  
  // Initialize sound system
  if (!initSoundSystem())
  {
    gConsole->Insertln("^1Could not initialize sound system!");
    sys_soundsupport = 0;
  }
  else sys_soundsupport = 1;

  // Loads packages
  FileEnvironment::LoadPacks();

  #ifdef _DEBUG
    getPAKContent(NULL, ".bsp");  // print a list of bsp files in log file
    cg_drawFPS = 1;
    cg_drawtime = 1;
    cg_showpos = 1;
    cg_showrot = 1;
    cg_drawflush = 1;
    cg_drawtris = 1;
    cg_drawverts = 1;
    cg_apicalls = 1;
  #endif

  // Seed the random-number generator with current time so that
  // the numbers will be different every time we run.
  srand((unsigned)time(NULL));

  InitializeShaders();

  gCommands->AddToHistory = true;
}

App::~App(void)
{
  if (sys_appstate.ivalue != INACTIVE)
  {
    sys_appstate = INACTIVE;
    gLogFile->Insert("^5WARNING: cake was not shut correctly\n");
  }

  // Creating new config file
/*  FILE *fp;
  if (!(fp = fopen(CONFIG_FILENAME, "w")))
  {
    gLogFile->Insert("^1ERROR: Unable to create config file for writing\n");
  }
  else
  {
    fclose(fp);
    gVars->SaveToFile(CONFIG_FILENAME);
  }
*/

  gVars->UnregisterVar(cg_showrot);
  gVars->UnregisterVar(cg_showpos);
  gVars->UnregisterVar(cg_drawtime);
  gVars->UnregisterVar(cg_drawFPS);
  gVars->UnregisterVar(cg_FPSrefreshrate);
  gVars->UnregisterVar(cg_drawflush);
  gVars->UnregisterVar(cg_drawtris);
  gVars->UnregisterVar(cg_drawverts);
  gVars->UnregisterVar(cg_apicalls);
  gVars->UnregisterVar(r_debug);
  gVars->UnregisterVar(r_mapsubdir);
  gVars->UnregisterVar(r_path);
  gVars->UnregisterVar(sys_appstate);
  gVars->UnregisterVar(sys_soundsupport);
  gVars->UnregisterVar(r_targetlocationdist);

  gCommands->RemoveCommand("bsplist");
  gCommands->RemoveCommand("wavlist");
  gCommands->RemoveCommand("md3list");
  gCommands->RemoveCommand("listfiles");
  gCommands->RemoveCommand("logpreviews");

  delete gOver; gOver = NULL;
  delete gFramework; gFramework = NULL;
  delete gConsole; gConsole = NULL;
  delete gRender; gRender = NULL;
  delete gVars; gVars = NULL;
  delete gAlias; gAlias = NULL;
  delete gCommands; gCommands = NULL;

  #ifdef DEBUG_MEM
    if (get_references()) log_mem_report();
    else gLogFile->Insert("No memory leak detected.\n");
    delete_memregister();
  #endif

  delete gLogFile; gLogFile = NULL;
}

void App::Run(void)
{
  sys_appstate = RUNNING;
}

void App::Shut(void)
{
  sys_appstate = SHUTING;

  UnloadMap();

  if (clients) delete [] clients;
  clients = NULL;
  numclients = 0;

  DestroyShaderList();

  gFramework->Shut();
  gConsole->Shut();
  gRender->Shut();
  gOver->Shut();
  
  sys_appstate = INACTIVE;

  gLogFile->Insert("<br/><b>^6----- cake engine shuting down --------------------------------</b>\n");
  Timer::Refresh();
  char time_text[64] =  { '\0' };
  int minutes = (int)Timer::fTime/60;
  int seconds = (int)fmod(Timer::fTime,60);
  sprintf(time_text, "%d:", minutes);
  if (seconds < 10) strcat(time_text, "0");
  sprintf(time_text, "%s%d", time_text, seconds);
  gLogFile->Insert("Execution time: %s\n", time_text);

  shutdownSoundSystem();
}

void App::DrawWorld(void)
{
  if (!gRender->GetWidth() || !gRender->GetHeight() ||
    !world || sys_appstate.ivalue != RUNNING) return;

  int i;

  // Updating demo
  for (i = 0; i < numclients; ++i)
  {
    if (demos[i].isPlaying)
    {
      float demopercent = demos[i].NewFrame();
      sprintf(demomsg, "playing demo (%d %%)", (int)(demopercent));
    }
    else if (demos[i].isRecording)
    {
      demos[i].NewFrame();
      sprintf(demomsg, "recording");
    }
  }

  world->Render(clients, numclients);

  for (i = 0; i < numclients; ++i)
  {
    if ((demos[i].isRecording && (long)(Timer::fTime*2)%2) || demos[i].isPlaying)
      gOver->String(demomsg, gFramework->GetFont(NORMAL_CHAR),
        (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*strlen(demomsg))/2.f,
        (float) gRender->GetHeight()-SMALL_CHAR_HEIGHT);
  }

  // Draws the camera position
  if (cg_showpos.ivalue && cg_showpos.ivalue <= numclients)
  {
    Client *client = &clients[cg_showpos.ivalue-1];

    char pos_text[128] = { '\0' };
    float x, y;
    sprintf(pos_text, "pos (%.3f;%.3f;%.3f)",
      client->cam.pos[0],
      client->cam.pos[1],
      client->cam.pos[2]);

    x = (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(pos_text));
    y = (float) gRender->GetHeight()-SMALL_CHAR_HEIGHT;

    gOver->String(pos_text, gFramework->GetFont(NORMAL_CHAR), x, y);
  }

  // Draws the camera rotation
  if (cg_showrot.ivalue && cg_showrot.ivalue <= numclients)
  {
    Client *client = &clients[cg_showrot.ivalue-1];

    char rot_text[128];
    float x, y;

    // rotation
    memset(rot_text, '\0', 128*sizeof(char));
    sprintf(rot_text, "rot (%.3f;%.3f;%.3f)",
      client->cam.rot[PITCH],
      client->cam.rot[YAW],
      client->cam.rot[ROLL]);

    x = (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(rot_text));
    y = (float) gRender->GetHeight()-SMALL_CHAR_HEIGHT;
    if (cg_showpos.ivalue) y -= (float) (SMALL_CHAR_HEIGHT);

    gOver->String(rot_text, gFramework->GetFont(NORMAL_CHAR), x, y);

    // forward vector
    memset(rot_text, '\0', 128*sizeof(char));
    sprintf(rot_text, "forward (%.3f;%.3f;%.3f)",
      client->forward[0],
      client->forward[1],
      client->forward[2]);

    x = (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(rot_text));
    y -= (float) (SMALL_CHAR_HEIGHT);

    gOver->String(rot_text, gFramework->GetFont(NORMAL_CHAR), x, y);

    // right
    memset(rot_text, '\0', 128*sizeof(char));
    sprintf(rot_text, "right (%.3f;%.3f;%.3f)",
      client->right[0],
      client->right[1],
      client->right[2]);

    x = (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(rot_text));
    y -= (float) (SMALL_CHAR_HEIGHT);

    gOver->String(rot_text, gFramework->GetFont(NORMAL_CHAR), x, y);

    // up
    memset(rot_text, '\0', 128*sizeof(char));
    sprintf(rot_text, "up (%.3f;%.3f;%.3f)",
      client->up[0],
      client->up[1],
      client->up[2]);

    x = (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(rot_text));
    y -= (float) (SMALL_CHAR_HEIGHT);

    gOver->String(rot_text, gFramework->GetFont(NORMAL_CHAR), x, y);
  }

  gOver->Render(&gFramework->shaders);
}

void App::DrawInterface(void)
{
  // Draws FPS
  if (cg_drawFPS.ivalue && totallastfps)
  {
    char fps_text[64] = { '\0' };
    if (1000.f*(Timer::fTime-(float)timelastfps) > cg_FPSrefreshrate.fvalue)
    {
      lastfps = (float)frameslastfps/totallastfps;
      timelastfps = (float) Timer::fTime;
      totallastfps = 0;
      frameslastfps = 0;
    }
    totallastfps += (float)Timer::frametime;
    ++frameslastfps;
    sprintf(fps_text, "^%d%4.2f^0 fps", lastfps<25?1:lastfps>30?0:5, lastfps);
    gOver->String(fps_text, gFramework->GetFont(NORMAL_CHAR), (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(fps_text)), 0);
  }

  // Draw time
  if (cg_drawtime.ivalue)
  {
    char time_text[64] =  { '\0' };
    float y = 0;
    int minutes = (int)Timer::fTime/60;
    int seconds = (int)fmod(Timer::fTime,60);
    if (cg_drawFPS.ivalue) y += SMALL_CHAR_HEIGHT;
    sprintf(time_text, "%d:", minutes);
    if (seconds < 10) strcat(time_text, "0");
    sprintf(time_text, "%s%d", time_text, seconds);
    gOver->String(time_text, gFramework->GetFont(NORMAL_CHAR), (float) (gRender->GetWidth()-SMALL_CHAR_WIDTH*_strlen(time_text)), y);
  }

  // display rendering infos (bottom left)
  if (cg_drawflush.ivalue)
  {
    char msg[64] =  { '\0' };
    sprintf(msg, "flush calls: %d", (int) gRender->num_flush);
    gOver->String(msg, gFramework->GetFont(NORMAL_CHAR), 0, (float)gRender->GetHeight()-SMALL_CHAR_HEIGHT);
  }

  if (cg_drawtris.ivalue)
  {
    char msg[64] =  { '\0' };
    float y = (float)gRender->GetHeight()-SMALL_CHAR_HEIGHT;
    if (cg_drawflush.ivalue) y -= SMALL_CHAR_HEIGHT;
    sprintf(msg, "rendered tris: %d", (int) gRender->num_tris);
    gOver->String(msg, gFramework->GetFont(NORMAL_CHAR), 0, y);
  }

  if (cg_drawverts.ivalue)
  {
    char msg[64] =  { '\0' };
    float y = (float)gRender->GetHeight()-SMALL_CHAR_HEIGHT;
    if (cg_drawflush.ivalue) y -= SMALL_CHAR_HEIGHT;
    if (cg_drawtris.ivalue) y -= SMALL_CHAR_HEIGHT;
    sprintf(msg, "rendered verts: %d", (int) gRender->num_verts);
    gOver->String(msg, gFramework->GetFont(NORMAL_CHAR), 0, y);
  }

  if (cg_apicalls.ivalue)
  {
    char msg[64] =  { '\0' };
    float y = (float)gRender->GetHeight()-SMALL_CHAR_HEIGHT;
    if (cg_drawflush.ivalue) y -= SMALL_CHAR_HEIGHT;
    if (cg_drawtris.ivalue) y -= SMALL_CHAR_HEIGHT;
    if (cg_drawverts.ivalue) y -= SMALL_CHAR_HEIGHT;
    sprintf(msg, "api calls: %d", (int) gRender->num_apicalls);
    gOver->String(msg, gFramework->GetFont(NORMAL_CHAR), 0, y);
  }

  // target locations
  /*
  if (world)
  {
    EntityManager* ent = world->GetBSP()->GetEntities();
    vec3_t vbuffer;
    for (int i = 0; i < ent->num_target_location[0]; ++i)
    {
      VectorSub(ent->first_target_speaker[i].origin, gRender->current_camera->pos, vbuffer);
      if (vbuffer[0]*vbuffer[0]+vbuffer[1]*vbuffer[1]+vbuffer[2]+vbuffer[2] < r_targetlocationdist.fvalue)
      {
        gOver->String(ent->first_target_location[i].message,
        gFramework->GetFont(NORMAL_CHAR),
        (gRender->GetWidth()-(float)(SMALL_CHAR_WIDTH*(int)strlen(ent->first_target_location[i].message))/2),
        gRender->GetHeight()/2+(float)SMALL_CHAR_WIDTH);
      }
    }
  }
  */

  gOver->Render(&gFramework->shaders);

  gConsole->Update();
  gConsole->Render();
}

double App::GetFPS(void)
{
  return 1/Timer::frametime;
}

World* App::GetWorld(void)
{
  return world;
}

const char* App::GetCurrentMapName(void)
{
  return curr_map;
}

int App::LoadMap(const char* map_name)
{
  int reload = !stricmp(curr_map, map_name) && reloading;
  reloading = false;

  // Get a start position for client
  if (!reload)
  {
    // Create new clients
    if (!CreateClients(1))
    {
      gConsole->Insertln("^1Currently unable to create new clients. Loading aborted.");
      return 0;
    }
  }

  UnloadMap();

  strncpy(curr_map, map_name, 32);

  Timer::Refresh();
  float startloadingtime = (float) Timer::fTime;

  gConsole->Insertln("Loading map %s...", map_name);

  world = new World;
  if (!world) ThrowException(ALLOCATION_ERROR, "App::LoadMap.world");

  // load a map, if one was specified:
  // (objects must be created before shader loading because here we add the shader reference)
  // TODO: objects should be loaded throug the world, the world should be an object factory
  if (!world->AddMap(map_name))
  {
    delete world;
    world = NULL;
    gConsole->Insertln("Map %s not found.", map_name);
    return 0;
  }

  gConsole->SetState(CLOSED);
  gConsole->addToMessages = false;

  world->Init();

  // Get a start position for client
  if (!reload)
  {
    if (world->GetNumStartPos())
    {
      // get a random start pos
      world->SetStartPos(&clients[0], rand()%world->GetNumStartPos());
    }
    else gConsole->Insertln("^1ERROR: Couldn't find a spawnpoint.");
  }
  else
  {
    // If map is reloaded, clients stay at same place as they were before
    // unloading the map. The internal cluster and leaf camera values are
    // allways defined but are not valid any more for new map. Following
    // function call will reinit these values. If they are not reinitialized
    // screen (and current leaf/cluster) won't be updated until camera
    // changes leaf/cluster.
    for (int i = 0; i < numclients; ++i) clients[i].UpdateCam();
  }

  Timer::Refresh();
  gConsole->Insertln("Map loaded successfully in %f sec.", (float)Timer::fTime-startloadingtime);
  gConsole->addToMessages = true;

  return 1;
}

int App::ReloadMap(void)
{
  if (strlen(curr_map))
  {
    reloading = true;
    return LoadMap(curr_map);
  }
  else
  {
    gConsole->Insertln("^5No map running.");
    return 0;
  }
}

void App::UnloadMap(void)
{
  if (world)
  {
    gConsole->Insertln("Unloading map...");
    world->Shut();
    delete world;
    world = NULL;

    gConsole->Resize(-1, gRender->GetHeight());
  }
}

int App::CreateClients(int num)
{
  if (!num) return 0;

  // Check if already one demo is recording
  for (int i = 0; i < numclients; ++i)
  {
    Demo *demo = &demos[i];
    if (demos[i].isRecording || demos[i].isPlaying)
    {
      gConsole->Insertln("^1Unable to change number of clients when a demo is recording or playing.");
      return 0;
    }
  }

  if (clients) delete [] clients;
  clients = new Client[num];
  if (!clients) ThrowException(ALLOCATION_ERROR, "World::CreateClients.clients");
  numclients = num;
  curr_client = 0;
  InitClients();

  if (demos) delete [] demos;
  demos = new Demo[num];
  if (!demos) ThrowException(ALLOCATION_ERROR, "World::CreateClients.demos");

  return 1;
}

Client* App::GetClient(int num)
{
  if (!clients) return NULL;
  if (num < 0 || num >= numclients) return NULL;
  return &clients[num];
}

int App::GetNumClients(void)
{
  return numclients;
}

void App::SetCurrentClient(int num)
{
  if (num < 0 || num >= numclients) return;
  curr_client = num;

  for (int i = 0; i < numclients; ++i) clients[i].isCurrent = false;
  clients[num].isCurrent = true;
}

Client* App::GetCurrentClient(void)
{
  if (clients) return &clients[curr_client];
  else return NULL;
}

int App::GetCurrentClientIndex(void)
{
  if (!numclients) return -1;
  return curr_client;
}

// Initialize the clients
void App::InitClients(void)
{
  if (!numclients) return;
  int a = numclients, b = 1, c = numclients, i, j;

  // Get the possible values
  for (i = 1; i <= c; ++i)
  {
    for (j = i; j <= c; ++j)
    {
      if (c == i*j)
      {
        a = j;
        b = i;
      }
    }
  }

  // Create the windows
  int l, t, w, h;
  w = gRender->GetWidth()/a;
  h = gRender->GetHeight()/b;
  l = 0, t = 0;

  for (i = 0; i < numclients; ++i)
  {
    clients[i].Init(l, t, w, h);
    l += w;
    if (!((i+1) % a))
    {
      l = 0;
      t += h;
    }
  }
  curr_client = 0;

  for (i = 0; i < numclients; ++i)
    clients[i].isCurrent = (i==curr_client)?true:false;
}

void App::RecordDemo(float framesinterval)
{
  if (!numclients)
  {
    gConsole->Insertln("^5WARNING: No available client !");
    return;
  }

  // Check if already one demo is recording
  for (int i = 0; i < numclients; ++i)
  {
    if (demos[i].isRecording)
    {
      gConsole->Insertln("^5Already recording a demo.");
      return;
    }
  }

  Demo* demo = &demos[curr_client];

  if (demo->StartDemoRecording(&clients[curr_client], curr_map, framesinterval))
  {
    gConsole->Close();
    gConsole->Insertln("Start recording...");
    gConsole->Insertln("Stop recording by entering \"stopdemo <filename>\" or pressing ESC (default demo filename will be used).");
  }
  else ThrowException(DEFAULT_EXCEPTION, "Invalid demo state !");
}

int App::StopDemo(const char *destfile)
{
  if (!numclients)
  {
    gConsole->Insertln("^5WARNING: No available client !");
    return 0;
  }

  Demo* demo = &demos[curr_client];

  if (demo->isRecording)
  {
    char demofilename[32] = { '\0' };

    // stop recording
    if (destfile)
    {
      strncpy(demofilename, destfile, 31);
    }
    else
    {
      // stop recording demo and save to default file
      int c, d, u;
      do {  // skip existing files
        c = demofilenameindex/100;
        d = (demofilenameindex-c*100)/10;
        u = (demofilenameindex-c*100-d*10);
        memset(demofilename, 0, 32);
        sprintf(demofilename, "cake%d%d%d.demo", c, d, u);
        ++demofilenameindex;
      } while(fopen(demofilename, "r"));
    }

    if (demo->StopDemoRecording(demofilename))
      gConsole->Insertln("Demo saved into \"%s\"", demofilename);
    else
      gConsole->Insertln("^1Demo saving abordted.");

    return 1;
  }
  else if (demo->isPlaying)
  {
    // stop playing
    demo->StopDemoPlaying();
    gConsole->Insertln("Demo stopped.");
    return 2;
  }
  else return 0;
}

void App::PlayDemo(const char *demofile, bool loop, enum_InterpolationMode mode)
{
  Demo* curr_demo;

  if (numclients)
  {
    curr_demo = &demos[curr_client];

    if (curr_demo->isPlaying)
    {
      gConsole->Insertln("^1Demo is currently playing");
      return;
    }

    if (curr_demo->isRecording)
    {
      gConsole->Insertln("^1Demo is currently recording");
      return;
    }
  }

  // if no world, we need to create a temporal demo until world is created
  if (!world || !numclients)
  {
    curr_demo = new Demo;
    if (!curr_demo) ThrowException(ALLOCATION_ERROR, "App::PlayDemo.curr_demo");
  }

  // Get the map to be loaded for demo running
  char mapname[32] = { '\0' };
  strcpy(mapname, curr_demo->LoadDemo(demofile, mode));

  if (!strlen(mapname))
  {
    // Wrong mapname
    gConsole->Insertln("Demo loading aborted.");
    delete curr_demo;
    return;
  }

  // if current map is not same as needed map
  if (!world || stricmp(curr_map, mapname))
  {
    if (!world) delete curr_demo; // demo will be re-created
    if (! LoadMap(mapname)) return;

    // if is reloaded, demo is destroyed and re-created, then we
    // need to reload the demo for default client
    curr_demo = &demos[curr_client];
    curr_demo->LoadDemo(demofile, mode);
  }

  gConsole->Insertln("Starting demo...");
  curr_demo->StartDemoPlaying(&clients[curr_client], loop);
}
