//-----------------------------------------------------------------------------
// Console
//-----------------------------------------------------------------------------

#include "cake.h"
#include "console.h"
#include "definitions.h"
#include "alias.h"
#include "commands.h"
#include "logfile.h"
#include "overlay.h"
#include "render.h"
#include "timer.h"
#include "vars.h"
#include "mem.h"
#include "system.h"

#ifdef WIN32
  #include <tchar.h>        // pour _vsntprintf
#else
  #include <stdlib.h>
  #include <stdio.h>
  #include <stdarg.h>
  //#include <sys/va_list.h>
  #define _vsntprintf vsnprintf
#endif

float opening_or_closing_progress = 1.f;
int currently_used_message_lines = 0;
int prompt_len = 0, cons_i, cons_j;
char final_buffer[N_LINES_MAX][CONSOLE_LINELENGTH] = { { '\0'} };

Var v_enablelog("v_enablelog", 1);

//-----------------------------------------------------------------------------------------

void ConsLine::Init(void)
{
  memset(content, '\0', CONSOLE_LINELENGTH*sizeof(char));
  length = 0;
  nlines = 1;
  breakPos = (int*) cake_malloc(sizeof(int), "ConsLine::Init.breakPos");
  if (!breakPos) ThrowException(ALLOCATION_ERROR, "ConsLine::Init.breakPos");
  breakPosSize = 1;
  breakPos[0] = 0;
}

void ConsLine::Shut(void)
{
  cake_free(breakPos);
  breakPos = NULL;
}

void ConsLine::ReInit(void)
{
  memset(content, '\0', CONSOLE_LINELENGTH*sizeof(char));
  length = 0;
  nlines = 1;
  breakPosSize = 1;
  breakPos[0] = 0;
}

int ConsLine::Update(int linelen)
{
  nlines = 1;

  int llen = 0, i = 0;
  while (i < length)
  {
    if (content[i] == TEXT_STYLE_KEYWORD)
    {
      ++i;
      if (content[i] == 'a' || content[i] == 'c')
        while (i < length && content[i] != '}') ++i;
      else if (content[i] == 'i' || content[i] == 'b') ++i;
      continue;
    }
    else if (content[i] == '^')
    {
      if (i < length-1) i += 2;
      continue;
    }

    if (llen >= linelen)  // line break
    {
      ++nlines;
      if (nlines > breakPosSize)
      {
        breakPos = (int*) cake_realloc(breakPos, nlines*sizeof(int), "ConsLine::Update.breakPos");
        breakPosSize = nlines;
      }

      // try to find a better place for cutting
      int z = i-1;

      if (nlines > 1)
      {
        while (z > breakPos[nlines-2] && content[z] != ' ') --z;
        if (z <= breakPos[nlines-2]) z = i;
        else ++z;
      }
      else
      {
        while (z > 0 && content[z] != ' ') --z;
        if (z <= 0) z = i;
        else ++z;
      }
      i = z;
      breakPos[nlines-1] = i;
      llen = 1;
    }
    else
    {
      ++llen;
    }
    ++i;
  }

#if 0
  printf("------------------------------------\n");
  for (int a = 0; a < nlines; ++a) printf("%d : %d\n", a, breakPos[a]);
#endif

  return nlines;
}

void ConsLine::SetContent(const char *c)
{
  memset(content, '\0', CONSOLE_LINELENGTH*sizeof(char));
  strcpy(content, c);
  length = (int) strlen(content);
}

int ConsLine::AddContent(const char *c)
{
  if ((int) strlen(content) + (int) strlen(c) < CONSOLE_LINELENGTH) strcat(content, c);
  else return 0;
  length = (int) strlen(content);
  return length;
}

char* ConsLine::GetContent(int p)
{
  memset(buffer, '\0', CONSOLE_LINELENGTH*sizeof(char));
  if (p < 0 || p >= nlines) return content;
  else
  {
    if (p == nlines - 1) memcpy(buffer, &content[breakPos[p]], (length-breakPos[p])*sizeof(char));
    else memcpy(buffer, &content[breakPos[p]], (breakPos[p+1]-breakPos[p])*sizeof(char));
  }

  // Recopie les derniers paramtres de couleur en dbut de string
  if (p > 0)
  {
    char tmp[CONSOLE_LINELENGTH];
    memset(tmp, 0, CONSOLE_LINELENGTH*sizeof(char));

    for (int k = 0, t = 0; k < breakPos[p];)
    {
      if (content[k] == TEXT_STYLE_KEYWORD)
      {
        tmp[t++] = content[k++];  // TEXT_STYLE_KEYWORD
        if (k >= length) return buffer;
        if (content[k] == 'c' || content[k] == 'a')
        {
          while (k < length && content[k] != '}') tmp[t++] = content[k++];
          if (k < length) tmp[t++] = content[k++];
        }
        else if (content[k] == 'i' || content[k] == 'b')
        {
          tmp[t++] = content[k++];  // i || b
          tmp[t++] = content[k++];  // + || -
        }
      }
      else if (content[k] == '^')
      {
        tmp[t++] = content[k++];  // '^'
        if (k >= length) return buffer;
        tmp[t++] = content[k++];  // valeur
      }
      else ++k;
    }

    strcat(tmp, buffer);
    strcpy(buffer, tmp);

#if 0
    printf("%s\n", buffer);
#endif
  }

  return buffer;
}

//-----------------------------------------------------------------------------------------

void cmd_close(int argc, char *argv[])
{
  gConsole->Close();
}

void cmd_messages(int argc, char *argv[])
{
  if (argc > 1)
  {
    if (!stricmp(argv[1],"on") || !stricmp(argv[1],"1")) gConsole->showMessages = true;
    else if (!stricmp(argv[1],"off") || !stricmp(argv[1],"0")) gConsole->showMessages = false;
    else gConsole->Insertln("^5Argument not valid");
  }
  else gConsole->Insertln("usage: %s <value>", argv[0]);
}

void cmd_messagetimelife(int argc, char *argv[])
{
  if (argc > 1) gConsole->messageMaxLife = (float) atof(argv[1]);
  else gConsole->Insertln("usage: %s <value>", argv[0]);
}

void cmd_clear(int argc, char *argv[])
{
  gConsole->Clear();
}

Console::Console(void)
{
  font = -1;
  back = -1;
  close = -1;
  reduce = -1;
  maximise = -1;
  titlebar = -1;
  scrollUp = -1;
  scrollDown = -1;
  scroll = -1;
  resize = -1;

  cursorSpeed = 2;
  prompt = NULL;
  cursorSymbol = '_';
  SetPrompt("]");

  movingConsole = false;
  topPos = 0;
  leftPos = 0;
  width = 640-2*leftPos;
  height = 240;
  
  state = CLOSED;

  Coeff = 1;
  SetFontSize(8, 16, 0);
  fontRows = 16;
  fontCols = 16;

  showMessages = true;
  addToMessages = true;
  messageMaxLife = 2.5f;
  startNewMessageLine = true;

  autoCut = true;

  openSpeed = .25f;
  closeSpeed = .25f;
  enableOpeningClosingAnimations = true;

  cursorPos = 0;
  scrollVal = 0;

  ActiveBorderColor[0]  = 1.f;
  ActiveBorderColor[1]  = 1.f;
  ActiveBorderColor[2]  = 0.f;
  ActiveBorderColor[3]  = 1.f;

  InactiveBorderColor[0]  = 0.4f;
  InactiveBorderColor[1]  = 0.4f;
  InactiveBorderColor[2]  = 0.4f;
  InactiveBorderColor[3]  = 1.f;

  showVScroll = true;
  showTitleBar = true;
  titleBarHeight = 18;

  int i;

  startNewLine      = true;
  NbrAllocatedLines   = LINESBLOC;
  NbrUsedLines      = 1;
  NbrTrueLines      = 1;
  ConsoleLines      = (ConsLine*) cake_malloc(NbrAllocatedLines*sizeof(ConsLine), "Console::Console.ConsoleLines");
  for (i = 0; i < NbrAllocatedLines; ++i) ConsoleLines[i].Init();
  memset(ConsoleLines, '\0', CONSOLE_LINELENGTH);

  memset(MessageLines, '\0', sizeof(MessageLines));
  for (i = 0; i < C_NMESSAGELINES; ++i) MessageLife[i] = 0.f;

  if (movingConsole)
  {
    Resize(gRender->GetWidth() - 40, gRender->GetHeight()/2);
    SetTopPos(20+titleBarHeight);
    SetLeftPos(20);
  }
  else
  {
    Resize(gRender->GetWidth(), gRender->GetHeight()/2);
    SetTopPos(0);
    SetLeftPos(0);    
  }

  Recalculate_NLines();

  gCommands->AddCommand("close", cmd_close, "closes the console");
  gAlias->SetAlias("hide", "close");
  gCommands->AddCommand("showmessages", cmd_messages, "activates or disactivates the display of console messageswhenconsole is closed");
  gCommands->AddCommand("msglife", cmd_messagetimelife, "sets the life time for console messages when console is closed");
  gCommands->AddCommand("clear", cmd_clear, "clears the console content");
  gAlias->SetAlias("clr", "clear");

  gVars->RegisterVar(v_enablelog);
}

Console::~Console(void)
{
  for (int i = 0; i < NbrAllocatedLines; ++i) ConsoleLines[i].Shut();
  cake_free(ConsoleLines); ConsoleLines = NULL;

  if (prompt) cake_free(prompt);
  prompt = NULL;

  gVars->UnregisterVar(v_enablelog);
}

void Console::Init(void)
{
  // load shader that have been referenced, and load also his textures
  // (shaders must be loaded before any other object)
  gConsole->Insertln("Updating console shaders...");
  shaders.Update();
}

void Console::Shut(void)
{
  gConsole->Insertln("Shuting console...");
  shaders.ResetAll();
  shaders.Update();
}

void Console::Clear(void)
{
  int i;
  for (i = 0; i < NbrAllocatedLines; ++i) ConsoleLines[i].Shut();
  cake_free(ConsoleLines);
  startNewLine      = true;
  NbrUsedLines      = 1;
  cursorPos       = 0;
  scrollVal       = 0;
  NbrAllocatedLines   = LINESBLOC;
  ConsoleLines = (ConsLine*) cake_malloc(NbrAllocatedLines*sizeof(ConsLine), "Console::Clear.ConsoleLines");
  for (i = 0; i < NbrAllocatedLines; ++i) ConsoleLines[i].Init();
}

void Console::Insert(const char* s, ...)
{
  if (!s || !strlen(s)) return;

  va_list msg;
  char buffer[CONSOLE_LINELENGTH] = { '\0' };

  va_start(msg, s);
  _vsntprintf(buffer, CONSOLE_LINELENGTH - 1, s, msg);
  va_end(msg);

  if (strlen(buffer) <= 0) return;

  if (v_enablelog.ivalue) gLogFile->Insert(buffer);

  char backupCommand[CONSOLE_LINELENGTH] = { '\0' };  // backup current command line
  if (ConsoleLines[NbrUsedLines-1].length)
    strncpy(backupCommand, ConsoleLines[NbrUsedLines-1].content, CONSOLE_LINELENGTH);

  scrollVal = 0;
  char final[CONSOLE_LINELENGTH] = { '\0' };
  int l = (int) strlen(buffer);

  // Parse the input string and
  //  -> replace tabulations with spaces
  //  -> remove HTML syntax
  //  -> recopy other characters
  for (int i = 0, j = 0; i < l; ++i)
  {
    if (buffer[i] == '\n')
    {
      if (startNewLine)
      {
        AddLine();
        ConsoleLines[NbrUsedLines-2].Init();
      }

      startNewLine = true;

      if (!strlen(final)) continue;

      if (addToMessages) AddMessageLine(final);

      if (!ConsoleLines[NbrUsedLines-2].AddContent(final))
      {
        AddLine();
        ConsoleLines[NbrUsedLines-2].Init();
        if (strlen(backupCommand)) ConsoleLines[NbrUsedLines-1].SetContent(backupCommand);
        ConsoleLines[NbrUsedLines-2].SetContent(final);
      }

      j = 0;
      memset(final, '\0', CONSOLE_LINELENGTH);
    }
    else if (buffer[i] == '\t')
    {
      for (int k = 0; k < TAB_LEN && j < CONSOLE_LINELENGTH-1; ++k)
        final[j++] = ' ';
    }
    else if (buffer[i] == '<')
    {
      // TODO: don't allow \t and \n here
      // analyse string to find if it is a valid tag
      int backup_pos = i;
      while (i < l && buffer[i] != '>') ++i;
      if (i < l)
      {
        if (buffer[i-1] != '/' && buffer[backup_pos+1] != '/')
        {
          // search for closing tag
          // get tag
          char tag[CONSOLE_LINELENGTH] = { '\0' };
          strncpy(tag, &buffer[backup_pos+1], i-backup_pos-1);
          int taglen = (int) strlen(tag);
          while (++i < l)
          {
            if (i < l-taglen-2 && buffer[i] == '<' && buffer[i+1] == '/' &&
              !strnicmp(&buffer[i+2], tag, taglen) && buffer[i+2+taglen] == '>')
            {
              // closing tag found
              i = backup_pos+taglen+1;  // skip '/', '>' and tag length
              break;
            }
          }
          if (i == l)
          {
            // no closing tag found, recopy the full string
            i = backup_pos;
            if (j < CONSOLE_LINELENGTH-1) final[j++] = buffer[i];
          }
        }
      }
      else
      {
        // no closing tag (it is a single "smaller" (<) character)
        i = backup_pos;
        if (j < CONSOLE_LINELENGTH-1) final[j++] = buffer[i];
      }
    }
    else if (j < CONSOLE_LINELENGTH-1) final[j++] = buffer[i];  // default char, recopy it
  }

  if (l = (int) strlen(final))
  {
    // clamp string into valid size
    if (l >= CONSOLE_LINELENGTH)
      final[CONSOLE_LINELENGTH-1] = '\0';

    if (addToMessages)
      AddToLastMessageLine(final);

    if (startNewLine)
    {
      AddLine();
      ConsoleLines[NbrUsedLines-2].Init();
    }

    if (!ConsoleLines[NbrUsedLines-2].AddContent(final))
    {
      AddLine();
      ConsoleLines[NbrUsedLines-2].Init();
      ConsoleLines[NbrUsedLines-2].SetContent(final);
    }

    startNewLine = false;
  }

  if (strlen(backupCommand)) ConsoleLines[NbrUsedLines-1].SetContent(backupCommand);
  Recalculate_NLines();

  //  if (gRender->ready && State::curr_shader && gTimer && Timer::frametime < 0.04)    // Ne le fait pas si les fps sont trop basses
  //  {
  //    Update();
  //    Render();
  //  }
}

void Console::Insertln(const char* s, ...)
{
  if (!s || !strlen(s)) return;

  va_list msg;
  char buffer[CONSOLE_LINELENGTH] = { '\0' };
  
  va_start(msg, s);
  _vsntprintf(buffer, CONSOLE_LINELENGTH - 1, s, msg);
  va_end(msg);

  if (strlen(buffer) <= 0) return;

  Insert("%s\n", buffer);
}

// Command line management
int Console::GetNbrUsedLines(void)
{
  return NbrUsedLines;
}

// Ajoute un caractre  la commande
void Console::AddChar(char c)
{
  if ((int) ConsoleLines[NbrUsedLines-1].length > CONSOLE_LINELENGTH)
  {
    beep();
    return;
  }

  if (c == TEXT_STYLE_KEYWORD) return;

  if (c != DELETE_CHAR)
  {
    if (cursorPos > 0)
    {
      char begin[CONSOLE_LINELENGTH] = { '\0' };
      char end[CONSOLE_LINELENGTH] = { '\0' };
      
      strncpy(begin, ConsoleLines[NbrUsedLines-1].content, ConsoleLines[NbrUsedLines-1].length-cursorPos);
      strncpy(end, &ConsoleLines[NbrUsedLines-1].content[ConsoleLines[NbrUsedLines-1].length-cursorPos], CONSOLE_LINELENGTH);

      strcat(begin, " ");
      begin[strlen(begin)-1] = c;
      strncat(begin, end, CONSOLE_LINELENGTH);
      ConsoleLines[NbrUsedLines-1].SetContent(begin);
    }
    else
    {
      strcat(ConsoleLines[NbrUsedLines-1].content, " ");
      ConsoleLines[NbrUsedLines-1].content[ConsoleLines[NbrUsedLines-1].length++] = c;
    }
  }
  else  // Supprime un caractre de la commande par Delete
  {
    if (cursorPos)
    {
      char begin[CONSOLE_LINELENGTH] = { '\0' };
      char end[CONSOLE_LINELENGTH] = { '\0' };
      
      strncpy(begin, ConsoleLines[NbrUsedLines-1].content, ConsoleLines[NbrUsedLines-1].length-cursorPos);
      strncpy(end, &ConsoleLines[NbrUsedLines-1].content[ConsoleLines[NbrUsedLines-1].length-cursorPos+1], CONSOLE_LINELENGTH);
      
      strncat(begin, end, CONSOLE_LINELENGTH);
      ConsoleLines[NbrUsedLines-1].SetContent(begin);

      --cursorPos;
    }
    else beep();
  }
}

// Supprime un caractre de la commande par retour chariot
void Console::DelChar(void)
{
  if (ConsoleLines[NbrUsedLines-1].length == 0 || ConsoleLines[NbrUsedLines-1].length == cursorPos)
  {
    beep();
    return;
  }

  if (cursorPos > 0)
  {
    if (cursorPos < (int) ConsoleLines[NbrUsedLines-1].length)
    {
      char begin[CONSOLE_LINELENGTH] = { '\0' };
      char end[CONSOLE_LINELENGTH] = { '\0' };
      
      strncpy(begin, ConsoleLines[NbrUsedLines-1].content, ConsoleLines[NbrUsedLines-1].length-cursorPos-1);
      strncpy(end, &ConsoleLines[NbrUsedLines-1].content[ConsoleLines[NbrUsedLines-1].length-cursorPos], CONSOLE_LINELENGTH);
      
      strncat(begin, end, CONSOLE_LINELENGTH);
      ConsoleLines[NbrUsedLines-1].SetContent(begin);
    }
  }
  else
  {
    if (ConsoleLines[NbrUsedLines-1].length)
    {
      ConsoleLines[NbrUsedLines-1].content[ConsoleLines[NbrUsedLines-1].length-1] = '\0';
      --ConsoleLines[NbrUsedLines-1].length;
    }
  }
}

// Rinitialise la commande
void Console::ReInitCurrentCommand(void)
{
  cursorPos = 0;
  ConsoleLines[NbrUsedLines-1].ReInit();
}

// Retourne la commande
char* Console::GetCurrentCommand(void)
{
  return ConsoleLines[NbrUsedLines-1].content;
}

// Dfinit la commande courante
void Console::SetCurrentCommand(char* s, ...)
{
  va_list msg;
  char buffer[CONSOLE_LINELENGTH] = { '\0' };
  
  va_start (msg, s);
  _vsntprintf (buffer, CONSOLE_LINELENGTH - 1, s, msg);
  va_end (msg);

  ConsoleLines[NbrUsedLines-1].SetContent(buffer);
}

// Create a new line in console
void Console::AddLine(void)
{
  // Rem : Allocation is performed by blocs of size LINESBLOC, 
  //       this avoids to do frequent reallocations
  if (NbrUsedLines == NbrAllocatedLines)
  {
    ConsoleLines = (ConsLine*) cake_realloc(ConsoleLines, (NbrAllocatedLines+LINESBLOC)*sizeof(ConsLine), "Console::AddLine.ConsoleLines");
    if (!ConsoleLines) ThrowException(ALLOCATION_ERROR, "Console::AddLine.ConsoleLines");
    for (int i = NbrAllocatedLines; i < NbrAllocatedLines+LINESBLOC; ++i) ConsoleLines[i].Init();
    NbrAllocatedLines += LINESBLOC;
  }
  ++NbrUsedLines;
  cursorPos = 0;
}

// Add a line in messages list
void Console::AddMessageLine(char* s)
{
  if (currently_used_message_lines < C_NMESSAGELINES) ++currently_used_message_lines;
  else
  {
    for (int i = 0; i < currently_used_message_lines; ++i)
    {
      memset(MessageLines[i], '\0', CONSOLE_LINELENGTH);
      strncpy(MessageLines[i], MessageLines[i+1], CONSOLE_LINELENGTH);
      MessageLife[i] = MessageLife[i+1];
    }
  }

  if (autoCut)
  {
    // Count length of text invisible elements (color italic, bold tags)
    unsigned long acceptable_len = (unsigned long) ((float)gRender->GetWidth()/(float)MiniConsoleFontSizeX);
    unsigned long real_pos = 0, llen = 0;
    while (llen < acceptable_len && real_pos < strlen(s))
    {
      if (s[real_pos] == '^') ++real_pos;
      else ++llen;
      ++real_pos;
    }
    
    if (strlen(s) > real_pos)
    {
      // Search for a breaking position in string
      while (real_pos > 0 && s[real_pos] != 32) --real_pos;
      if (real_pos > 0)
      {
        char tmp_buff[CONSOLE_LINELENGTH] = { '\0' };
        strncpy(tmp_buff, s, real_pos);
        
        MessageLife[currently_used_message_lines-1] = 0.f;
        strcpy(MessageLines[currently_used_message_lines-1], tmp_buff);

        // Following variable is used to recopy the font parameters on next lines
        char settings_s[CONSOLE_LINELENGTH] = { '\0' };
        for (unsigned long i = 0; i < real_pos; ++i)
        {
          if (s[i] == '^')
          {
            settings_s[strlen(settings_s)] = s[i++];
            settings_s[strlen(settings_s)] = s[i++];
          }
        }
        strcat(settings_s, &s[real_pos+1]);
        AddMessageLine(settings_s);
        return;
      }
    }
  }

  MessageLife[currently_used_message_lines-1] = 0.f;
  sprintf(MessageLines[currently_used_message_lines-1], "%s^0", s);
  startNewMessageLine = true;
}

void Console::AddToLastMessageLine(char *s)
{
  if (currently_used_message_lines == 0 || startNewMessageLine)
  {
    AddMessageLine(s);
    startNewMessageLine = false;
    return;
  }

  int l;
  char buffer[CONSOLE_LINELENGTH] = { '\0' };
  strncpy(buffer, MessageLines[currently_used_message_lines-1], CONSOLE_LINELENGTH);
  if ((l = (int) strlen(buffer)) < CONSOLE_LINELENGTH-1) strncat(buffer, s, CONSOLE_LINELENGTH-l-1);

  if (autoCut)
  {
    // Count length of text invisible elements (color italic, bold tags)
    unsigned long acceptable_len = (unsigned long) ((float)gRender->GetWidth()/(float)MiniConsoleFontSizeX);
    unsigned long real_pos = 0, llen = 0;
    while (llen < acceptable_len && real_pos < strlen(buffer))
    {
      if (buffer[real_pos] == '^') ++real_pos;
      else ++llen;
      ++real_pos;
    }

    if (strlen(buffer) > real_pos)
    {
      // Search for a breaking position in string
      while (real_pos > 0 && buffer[real_pos] != 32) --real_pos;
      if (real_pos > 0)
      {
        char tmp_buff[CONSOLE_LINELENGTH] = { '\0' };
        strncpy(tmp_buff, buffer, real_pos);

        MessageLife[currently_used_message_lines-1] = 0.f;
        strcpy(MessageLines[currently_used_message_lines-1], tmp_buff);

        // Following variable is used to recopy the font parameters on next lines
        char settings_s[CONSOLE_LINELENGTH] = { '\0' };
        for (unsigned long i = 0; i < real_pos; ++i)
        {
          if (buffer[i] == '^')
          {
            settings_s[strlen(settings_s)] = buffer[i++];
            settings_s[strlen(settings_s)] = buffer[i++];
          }
        }
        strcat(settings_s, &buffer[real_pos+1]);
        AddMessageLine(settings_s);
        startNewMessageLine = false;
        return;
      }
    }
  }

  MessageLife[currently_used_message_lines-1] = 0.f;
  strncpy(MessageLines[currently_used_message_lines-1], buffer, CONSOLE_LINELENGTH);
}

void Console::Render(void)
{

  if (state != CLOSED)
  {
    gRender->InitializeViewPort();        // set up 2d viewport

    // Draw the console background + text
    gOver->Render(&shaders);          // render overlay
    gRender->ForceFlush();

    // Draw the console border
    GLboolean lighting, texture;

    glGetBooleanv(GL_TEXTURE_2D, &texture);
    glGetBooleanv(GL_LIGHTING, &lighting);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_LIGHTING);

    State::setPolygonMode(GL_LINE);

    glColor4fv(ActiveBorderColor);
    glLineWidth(2);
    glBegin(GL_LINES);
      glVertex2f(0, opening_or_closing_progress*(float)(topPos+height));
      glVertex2f((float) width, opening_or_closing_progress*(float)(topPos+height));
    glEnd();
    glLineWidth(1);

    State::setPolygonMode(GL_FILL);

    gRender->CheckGLError(11);
    if (texture) { glEnable(GL_TEXTURE_2D); gRender->CheckGLError(); }
    if (lighting) { glEnable(GL_LIGHTING); gRender->CheckGLError(); }
  }
}

void Console::Update(void)
{
  // Update messages life time
  for (cons_i = 0; cons_i < currently_used_message_lines; ++cons_i)
  {
    MessageLife[cons_i] += (float) Timer::frametime;
  }
  
  for (cons_i = 0; cons_i < currently_used_message_lines; ++cons_i)
  {
    // Remove too old messages
    if (MessageLife[cons_i] >= messageMaxLife)
    {
      for (cons_j = cons_i; cons_j < currently_used_message_lines; ++cons_j)
      {
        memset(MessageLines[cons_j], '\0', CONSOLE_LINELENGTH);
        strncpy(MessageLines[cons_j], MessageLines[cons_j+1], CONSOLE_LINELENGTH);
        MessageLife[cons_j] = MessageLife[cons_j+1];
      }
      memset(MessageLines[--currently_used_message_lines], '\0', CONSOLE_LINELENGTH);
      --cons_i;
    }
    if (!currently_used_message_lines || cons_i < 0) break;
  }

  // Display messages if console is collapsed
  if ((state == CLOSED) && showMessages && currently_used_message_lines)
  {
    // Display messages only
    for (cons_i = 0; cons_i < currently_used_message_lines; ++cons_i)
    {
      gOver->String(MessageLines[cons_i],
        font,
        0,
        (float) (cons_i*MiniConsoleFontSizeY),
        (float) MiniConsoleFontSizeX,
        (float) MiniConsoleFontSizeY,
        (float) fontRows,
        (float) fontCols);
    }
  }

  if (state == CLOSED) return;

  char tmp_line[CONSOLE_LINELENGTH] = { '\0' };

  // Opening/Closing animation
  if (!movingConsole && enableOpeningClosingAnimations)
  {
    if (state == OPENING) opening_or_closing_progress += (float)Timer::frametime/openSpeed;
    if (state == CLOSING) opening_or_closing_progress -= (float)Timer::frametime/closeSpeed;
    if (opening_or_closing_progress >= 1.f && state == OPENING) { state = OPEN; opening_or_closing_progress = 1.f; } 
    if (opening_or_closing_progress <= 0.f && state == CLOSING) { state = CLOSED; opening_or_closing_progress = 0.f;}
  }
  else
  {
    if (state == OPENING) { state = OPEN; opening_or_closing_progress = 1.f; } 
    if (state == CLOSING) { state = CLOSED; opening_or_closing_progress = 0.f;}   
  }

  // Draw background
  if (back >= 0)
  {
    gOver->Quad(back, (float) leftPos, opening_or_closing_progress*(float)(topPos+height)-height, (float) width, (float) height);
    char version[64] = { '\0' };
    sprintf(version, "^2%s", _VERSION_);
    #ifdef _DEBUG
      strcat(version, " (debug)");
    #else
      strcat(version, " (release)");
    #endif
    gOver->String(version, GetFont(), (float)(width-ConsoleFontSizeX*(strlen(version)-2)), opening_or_closing_progress*(float)(topPos+height) - 1.5f*ConsoleFontSizeY);
  }

  // Display console lines starting from bottom
  if (NbrUsedLines < NbrMaxLines) cons_j = NbrUsedLines - 1;
  else cons_j = NbrMaxLines - 1;
  int pos_j = topPos+(int)((height+topPos)*(opening_or_closing_progress))-height+(cons_j)*ConsoleFontSizeY;

  // Display the command line
  cons_i = NbrUsedLines - 1;
  if ((long) ConsoleLines[cons_i].length < MaxTextLineLength - (prompt_len+1))
  {
    sprintf(tmp_line, "^0%s%s", prompt, ConsoleLines[cons_i].content);
    gOver->String(tmp_line, font, (float) leftPos, (float) pos_j, (float) ConsoleFontSizeX, (float) ConsoleFontSizeY, (float) fontRows, (float) fontCols);
  }
  else
  {
    char buffer[CONSOLE_LINELENGTH] = { '\0' };

    if ((int) ConsoleLines[cons_i].length > cursorPos + MaxTextLineLength - (prompt_len+1))
      strncpy(buffer, &(ConsoleLines[cons_i].content[ConsoleLines[cons_i].length-cursorPos-(MaxTextLineLength - (prompt_len+1))]), MaxTextLineLength-(prompt_len+1));
    else
      strncpy(buffer, ConsoleLines[cons_i].content, MaxTextLineLength - (prompt_len+1));

    sprintf(tmp_line, "^0%s%s", prompt, buffer);
    gOver->String(tmp_line, font, (float) leftPos, (float) pos_j, (float) ConsoleFontSizeX, (float) ConsoleFontSizeY, (float) fontRows, (float) fontCols);
  }

  // Cursor
  if ((long)(Timer::fTime*cursorSpeed)%2) //  -> effet de clignotement du curseur
  {
    sprintf(tmp_line, "^0%c", cursorSymbol);
    if ((long) ConsoleLines[NbrUsedLines - 1].length < MaxTextLineLength - prompt_len - 1)
    {
      gOver->String(tmp_line, font, (float) (leftPos+ConsoleFontSizeX*(ConsoleLines[NbrUsedLines-1].length+prompt_len-cursorPos)),
        (float) pos_j, (float) ConsoleFontSizeX, (float) ConsoleFontSizeY, (float) fontRows, (float) fontCols);
    }
    else
    {
      if ((int) ConsoleLines[NbrUsedLines - 1].length > cursorPos + MaxTextLineLength - prompt_len - 1)
      {
        gOver->String(tmp_line, font, (float) (leftPos+ConsoleFontSizeX*(MaxTextLineLength-1)), (float) pos_j, (float) ConsoleFontSizeX, (float) ConsoleFontSizeY, (float) fontRows, (float) fontCols);
      }
      else
      {
        gOver->String(tmp_line, font, (float) (leftPos+ConsoleFontSizeX*(ConsoleLines[NbrUsedLines-1].length-cursorPos+prompt_len)),
          (float) pos_j, (float) ConsoleFontSizeX, (float) ConsoleFontSizeY, (float) fontRows, (float) fontCols);
      }
    }
  }

  pos_j -= ConsoleFontSizeY;
  --cons_j;
  --cons_i;

  // Other lines
  int t = scrollVal;
  while (cons_j >= 0 && cons_i >= 0)
  {
    for (int k = ConsoleLines[cons_i].nlines - 1; k >= 0; --k)
    {
      if (t <= 0)
      {
        gOver->String(ConsoleLines[cons_i].GetContent(k), font, (float) leftPos, (float) pos_j,
           (float) ConsoleFontSizeX, (float) ConsoleFontSizeY, (float) fontRows, (float) fontCols);

        pos_j -= ConsoleFontSizeY;
        --cons_j;
      }
      else --t;
      if (cons_j < 0) break;
    }
    --cons_i;
  }
}

void Console::Recalculate_NLines(void)
{
  NbrMaxLines = height/ConsoleFontSizeY;

  // Calculates the real number of used lines
  NbrTrueLines = 1;
  MaxTextLineLength = (int) ((float)width/(float)ConsoleFontSizeX);
  for (int i = 0; i < NbrUsedLines - 1; ++i)
    NbrTrueLines += ConsoleLines[i].Update(MaxTextLineLength);
}

// Title, prompt
void Console::SetTitle(char *t)
{
  memset(title, '\0', CONSOLE_LINELENGTH);
  strncpy(title, t, CONSOLE_LINELENGTH);
}

void Console::SetPrompt (char *s)
{
  if (prompt) cake_free(prompt);

  bool must_readapt_len = false;

  if (strlen(s) > MAXPROMPTLEN)
  {
    must_readapt_len = true;
    while (strlen(s) > MAXPROMPTLEN) ++s;
  }
  
  prompt_len = (int) (strlen(s)>MAXPROMPTLEN?MAXPROMPTLEN:strlen(s));
  prompt = (char*) cake_malloc((prompt_len+1)*sizeof(char), "Console::SetPrompt.prompt");
  memset(prompt, '\0', prompt_len+1);
  strcpy(prompt, s);

  if (must_readapt_len)
    prompt[0] = prompt[1] = prompt[2] = '.';

  prompt_len = (int) strlen(prompt);
}

char* Console::GetPrompt(void)
{
  return prompt;
}

// Opening and closing
void Console::Open(void)
{
  if (enableOpeningClosingAnimations) state = OPENING;
  else state = OPEN;
  isActive = true;
}

void Console::Close(void)
{
  if (enableOpeningClosingAnimations) state = CLOSING;
  else state = CLOSED;
  isActive = false;
}

enum_ConsoleStates Console::GetState(void)
{
  return state;
}

enum_ConsoleStates Console::ToggleState(void)
{
  if (state == CLOSING || state == CLOSED) Open();
  else Close();

  return state;
}

void Console::SetState(enum_ConsoleStates s)
{
  state = s;

  if (!enableOpeningClosingAnimations)
  {
    if (state == OPENING) state = OPEN;
    if (state == CLOSING) state = CLOSED;
  }

  if (state == OPEN)
  {
    opening_or_closing_progress = 1.f;
    isActive = true;
  }
  else if (state == CLOSED)
  {
    opening_or_closing_progress = 0.f;
    isActive = false;
  }
}

// Scrolling
void Console::ScrollConsole(enum_ConsoleScrollDir dir)
{
  Recalculate_NLines();
  if (NbrTrueLines < NbrMaxLines)
  {
    beep();
    return;
  }

  if (dir == UP)
  {
    if ((scrollVal + 1) > (NbrTrueLines - NbrMaxLines)) { beep(); return; }
    ++scrollVal;    
  }
  else if (dir == DOWN)
  {
    if ((scrollVal - 1) < 0) { beep(); return; }
    --scrollVal;  
  }
  else if (dir == TOP)  scrollVal = NbrTrueLines - NbrMaxLines;
  else if (dir == BOTTOM) scrollVal = 0;
}

void Console::SetVScrollYPos (int y, bool center)
{
  scrollVal = (int) ((float)(center?y-RESIZE_SQUARE/2:y)*(float)(NbrMaxLines-NbrUsedLines)/
             (float)(height-RESIZE_SQUARE))+(int)((float)(NbrUsedLines-NbrMaxLines)*
             (float)(topPos+height-RESIZE_SQUARE)/(float)(height-RESIZE_SQUARE));
  if (scrollVal < 0) scrollVal = 0;
  if (scrollVal > NbrUsedLines - NbrMaxLines) scrollVal = NbrUsedLines - NbrMaxLines;
  if (NbrUsedLines < NbrMaxLines) scrollVal = 0;
}

// Type management
void Console::ToggleType (void)
{
  movingConsole = !movingConsole;
}
  
// Change d'tat de console et prend automatiquement en compte la maximisation
void Console::SetType(bool console_type)
{
  if (movingConsole && !console_type)
  {
    if (!ConsoleIsMaximized)
    {
      HBackup = height;
      WBackup = width;
      LBackup = leftPos;
      TBackup = topPos;
    }
    
    topPos = 0;
    leftPos = 0;
    Resize(gRender->GetWidth(), gRender->GetHeight()/2);

    movingConsole = console_type;
  }
  else if (!movingConsole && console_type)
  {
    movingConsole = console_type;

    if (ConsoleIsMaximized)
    {
      topPos = titleBarHeight;
      leftPos = 0;
      Resize(gRender->GetWidth(), gRender->GetHeight()-titleBarHeight);
    }
    else
    {
      topPos = TBackup;
      leftPos = LBackup;
      Resize(WBackup, HBackup);
    }
  }
}

// Resizing
void Console::Resize(int w, int h)
{
  if (w > 0 &&
    (showVScroll && (w-RESIZE_SQUARE) > (prompt_len>MAXPROMPTLEN?prompt_len:MAXPROMPTLEN+2)*ConsoleFontSizeX+2) ||
    !showVScroll && w > (prompt_len>MAXPROMPTLEN?prompt_len:MAXPROMPTLEN+2)*ConsoleFontSizeX+2)
  {
    width = w;
  //  if (BackGround != NULL) BackGround->SetSize(width, -1);
  }

  if (h > 0 && h > 3*ConsoleFontSizeY)
  {
    height = h;
    Recalculate_NLines();
    if (scrollVal > (NbrUsedLines - NbrMaxLines)) scrollVal = (NbrUsedLines - NbrMaxLines);
    if (scrollVal < 0) scrollVal = 0;
  //  if (BackGround != NULL) BackGround->SetSize(-1, height);
  }
}

void Console::Maximize(void)
{
  if (ConsoleIsMaximized || !movingConsole) return;

  HBackup = height;
  WBackup = width;
  LBackup = leftPos;
  TBackup = topPos;

  topPos = titleBarHeight;
  leftPos = 0;
  Resize(gRender->GetWidth(), gRender->GetHeight()-titleBarHeight);

  ConsoleIsMaximized = true;
}

void Console::Unmaximize(void)
{
  if (!ConsoleIsMaximized || !movingConsole) return;

  ConsoleIsMaximized = false;
  
  topPos = TBackup;
  leftPos = LBackup;
  Resize(WBackup, HBackup);
}

bool Console::IsMaximized(void)
{
  return ConsoleIsMaximized;
}

void Console::ToggleMaximisation (void)
{
  if (ConsoleIsMaximized) Unmaximize();
  else Maximize();
}

// Position management
void Console::SetTopPos(int top, int test)
{
  if (test >= 1 && (ConsoleIsMaximized || !movingConsole)) return;

  topPos = top;

  if (test >= 0 && test <= 1)
  {
    if (topPos < 0) topPos = 0;
    if ((topPos + height) > gRender->GetHeight()) topPos = gRender->GetHeight() - height;
  }
}

// Dfinit la position de gauche de la console
void Console::SetLeftPos(int left, int test)
{
  if (test >= 1 && (ConsoleIsMaximized || !movingConsole)) return;

  leftPos = left;

  if (test >= 0 && test <= 1)
  {
    if (leftPos < 0) leftPos = 0;
    if ((leftPos + width) > gRender->GetWidth()) leftPos = gRender->GetWidth() - width;
  }
}

int Console::GetWidth(void)
{
  return width;
}

int Console::GetHeight(void)
{
  return height;
}

int Console::GetLeft(void)
{
  return leftPos;
}

int Console::GetTop(void)
{
  return topPos;
}

// Cursor management
void Console::MoveCursor(enum_ConsoleCursorScroll d)
{
  cursorPos -= d;
  if (cursorPos < 0)
  {
    cursorPos = 0;
    if (d != C_EXTREM_RIGHT) beep();
  }
  if (cursorPos >= (int) ConsoleLines[NbrUsedLines-1].length)
  {
    cursorPos = (int) ConsoleLines[NbrUsedLines-1].length;
    if (d != C_EXTREM_LEFT) beep();
  }
}

void Console::SetCursorSpeed(float f)
{
  cursorSpeed = f;
}

void Console::SetCursorSymbol(char c)
{
  cursorSymbol = c;
}

// Font resizing
void Console::SetFontSize (int xsize, int ysize, int n)
{
  if (xsize < 1 && ysize < 1) return;

  if (n == 1)
  {
    if (xsize > 0) ConsoleFontSizeX = xsize;
    if (ysize > 0) ConsoleFontSizeY = ysize;
  }
  else if (n == 2)
  {
    if (xsize > 0) MiniConsoleFontSizeX = xsize;
    if (ysize > 0) MiniConsoleFontSizeY = ysize;
  }
  else 
  {
    if (xsize > 0) ConsoleFontSizeX = MiniConsoleFontSizeX = xsize;
    if (ysize > 0) ConsoleFontSizeY = MiniConsoleFontSizeY = ysize;
  }

  titleBarHeight = 3*ConsoleFontSizeY/2;
  if (titleBarHeight <= RESIZE_SQUARE) titleBarHeight = RESIZE_SQUARE+RESIZE_SQUARE/2;

  Recalculate_NLines();
}

int Console::GetFontSizeX(int n)
{
  if (n == 2) return MiniConsoleFontSizeX;
  return ConsoleFontSizeX;
}

int Console::GetFontSizeY(int n)
{
  if (n == 2) return MiniConsoleFontSizeY;
  return ConsoleFontSizeY;
}

void Console::SetCoeff(float c)
{
  Coeff = c;
}

// Console Shaders
void Console::SetBack(int b)    { back = b; }
void Console::SetClose(int c)   { close = c; }
void Console::SetMaximise(int m)  { maximise = m; }
void Console::SetReduce(int r)    { reduce = r; }
void Console::SetTitleBar(int t)  { titlebar = t; }
void Console::SetScrollUp(int s)  { scrollUp = s; }
void Console::SetScrollDown(int s)  { scrollDown = s; }
void Console::SetScroll(int s)    { scroll = s; }
void Console::SetResize(int r)    { resize = r; }

void Console::SetFont(int f, int rows, int cols)
{
  font = f;
  if (rows > 0) fontRows = rows;
  if (cols > 0) fontCols = cols;
}

int Console::GetFont(void)      { return font; }

