#ifndef __SYSTEM_H__
#define __SYSTEM_H__

/**
 * Detected CPU Vendors - returned by GetCPUCaps(CPU_VENDOR_STRING);
 */
typedef enum CPU_VENDORS
{
    VENDOR_UNKNOWN,
    VENDOR_AMD,
    VENDOR_INTEL,
    VENDOR_CYRIX,
    VENDOR_CENTAUR
} CPU_VENDORS;

/**
 * Detected CPU models - returned by GetCPUCaps(CPU_TYPE);
 */
typedef enum CPU_TYPES
{
    UNKNOWN,
    AMD_Am486,
    AMD_K5,
    AMD_K6,
    AMD_K6_MMX,
    AMD_K6_2,
    AMD_K6_3,
    AMD_K7,
    AMD_ATHLON,

    INTEL_486DX,
    INTEL_486SX,
    INTEL_486DX2,
    INTEL_486SL,
    INTEL_486SX2,
    INTEL_486DX2E,
    INTEL_486DX4,
    INTEL_Pentium,
    INTEL_Pentium_MMX,
    INTEL_Pentium_Pro,
    INTEL_Pentium_II,
    INTEL_Celeron,
    INTEL_Pentium_III,
} CPU_TYPES;


/**
 * Detected CPU capabilities - used as input to the GetCPUCaps() function.
 */
typedef enum CPUCAPS
{
    // Synthesized values
    CPU_VENDOR,       /**< Manufacturer (returns enum CPU_VENDORS) */
    CPU_TYPE,       /**< CPU type (return enum CPU_TYPES) */
    CPU_VENDOR_STRING,    /**< CPU vendor name string (return const char *) */
    CPU_NAME_STRING,    /**< CPU Processor string (extended functions 0x80000002 - 0x80000004, return const char *) */
    CPU_MFG,        /**< Manufacturer (returns enum CPU_MFGS) */

  // Processor Features - returned as boolean values
    HAS_CPUID,        /**< Supports CPUID instruction */
    HAS_FPU,        /**< FPU present */
    HAS_VME,        /**< Virtual Mode Extensions */
    HAS_DEBUG,        /**< Debug extensions */
    HAS_PSE,        /**< Page Size Extensions */
    HAS_TSC,        /**< Time Stamp Counter */
    HAS_MSR,        /**< Model Specific Registers */
    HAS_PAE,        /**< Page Address Extensions */
    HAS_MCE,        /**< Machine Check Extensions */
    HAS_CMPXCHG8,     /**< CMPXCHG8 instruction */
    HAS_APIC,       /**< APIC */
    HAS_SYSENTER,     /**< SYSENTER/SYSEXIT instruction */
    HAS_MTRR,       /**< Memory Type Range Registers */
    HAS_GPE,        /**< Global Paging Extensions */
    HAS_MCA,        /**< Machine Check Architecture */
    HAS_CMOV,       /**< CMOV instruction */
    HAS_PAT,        /**< Page Attribue Table */
    HAS_PSE36,        /**< PSE36 (Page Size Extensions) */

    HAS_MMX_EXT,      /**< MMX Extensions */
    HAS_MMX,        /**< MMX support */
    HAS_FXSAVE,       /**< FXSAVE/FXRSTOR instruction */

    HAS_3DNOW_EXT,      /**< Extended 3DNow! support */
    HAS_3DNOW,        /**< 3DNow! support */

    HAS_SSE_MMX,      /**< SSE MMX support (same as HAS_MMXEXT) */
    HAS_SSE,        /**< SSE */
    HAS_SSE_FP,       /**< SSE FP support */

    // Cache parameters (not all values apply to all cpus)
    CPU_L1_DTLB_ASSOC,      /**< L1 Data Cache TLB Associativity */
    CPU_L1_DTLB_ENTRIES,    /**< L1 Data Cache TLB Entries */
    CPU_L1_ITLB_ASSOC,      /**< L1 Instruction Cache TLB Associativity (0xff = full associativity) */
    CPU_L1_ITLB_ENTRIES,    /**< L1 Instruction Cache TLB Entries */

    CPU_L1_EDTLB_ASSOC,     /**< Extended (2/4 Mbyte) L1 Data Cache TLB Associativity (0xff = full associativity) */
    CPU_L1_EDTLB_ENTRIES,   /**< Extended (2/4 Mbyte) L1 Data Cache TLB Entries */
    CPU_L1_EITLB_ASSOC,     /**< Extended (2/4 Mbyte) L1 Instruction Cache TLB Associativity */
    CPU_L1_EITLB_ENTRIES,   /**< Extended (2/4 Mbyte) L1 Instruction Cache TLB Entries */

    CPU_L1_DCACHE_SIZE,     /**< L1 Data Cache Size (kbytes) */
    CPU_L1_DCACHE_ASSOC,    /**< L1 Data Cache Associativity (0xff = full associativity) */
    CPU_L1_DCACHE_LINES,    /**< L1 Data Cache Lines */
    CPU_L1_DCACHE_LSIZE,    /**< L1 Data Cache Line Size (bytes) */

    CPU_L1_ICACHE_SIZE,     /**< L1 Instruction Cache Size (kbytes) */
    CPU_L1_ICACHE_ASSOC,    /**< L1 Instruction Cache Associativity (0xff = full associativity) */
    CPU_L1_ICACHE_LINES,    /**< L1 Instruction Cache Lines */
    CPU_L1_ICACHE_LSIZE,    /**< L1 Instruction Cache Line Size (bytes) */

    CPU_L2_CACHE_SIZE,      /**< L2 Unified Cache Size (Kbytes) */
    CPU_L2_CACHE_ASSOC,     /**< L2 Unified Cache Associativity (0xf = full associativity) */
    CPU_L2_CACHE_LINES,     /**< L2 Unified Cache Lines (lines per tag) */
    CPU_L2_CACHE_LSIZE,     /**< L2 Unified Cache Line Size (bytes) */

    CPU_L2_DTLB_ASSOC,      /**< L2 Data Cache TLB Associativity */
    CPU_L2_DTLB_ENTRIES,    /**< L2 Data Cache TLB Entries */
    CPU_L2_UTLB_ASSOC,      /**< L2 Instruction or Unified Cache TLB Associativity (0xf = full associativity) */
    CPU_L2_UTLB_ENTRIES,    /**< L2 Instruction or Unified Cache TLB Entries */

    CPU_L2_EDTLB_ASSOC,     /**< Extended (2/4 Mbyte) L2 Data Cache TLB Associativity (0xf = full associativity) */
    CPU_L2_EDTLB_ENTRIES,   /**< Extended (2/4 Mbyte) L2 Data Cache TLB Entries */
    CPU_L2_EUTLB_ASSOC,     /**< Extended (2/4 Mbyte) L2 Instruction or Unified Cache TLB Associativity */
    CPU_L2_EUTLB_ENTRIES,   /**< Extended (2/4 Mbyte) L2 Instruction or Unified Cache TLB Entries */

} CPUCAPS;

/**
 * Detected CPU Manufacturers - returned by GetCPUCaps(CPU_MFG)
 */
typedef enum CPU_MFGS
{
    MFG_UNKNOWN,
    MFG_AMD,
    MFG_INTEL,
    MFG_CYRIX,
    MFG_CENTAUR
} CPU_MFGS;

/**
 * Get the CPU vendor name. The function is specially written for AMD
 * processors and may return incorrect values with other CPUs.
 * @return a 48 byte char table that contains
 */
unsigned char* GetVendorString(void);

/**
 * Get the CPU cache size. The function is specially written for AMD
 * processors and may return incorrect values with other CPUs.
 * @return the CPU cache size
 */
int GetCacheInfo(void);

/**
 * Get the CPU speed. The function is specially written for AMD
 * processors and may return incorrect values with other CPUs.
 * @return the CPU speed
 */
unsigned int GetCpuSpeed(void);

/**
 * Looks for a CPU caps.
 */
unsigned long GetCPUCaps(CPUCAPS);

/**
 * Get an argument in a string
 */
void GetArg(const char* command, int i, char *dest);

/**
 * Gets the number of arguments of a string.
 */
int GetNArgs(const char* command);

//-----------------------------------------------------------------------------
// Exceptions
//-----------------------------------------------------------------------------

typedef enum
{
  NO_EXCEPTION      = 0,
  DEFAULT_EXCEPTION   = 1,
  ALLOCATION_ERROR    = 2,
  OVERFLOW_ERROR      = 3,
  OUT_OF_BOUNDS_VALUE   = 4,
  NULL_POINTER      = 5,
  DIVISION_BY_ZERO    = 6,
  ERROR_OPENING_FILE    = 7,
  FATAL_ERROR       = 8
} CakeException;

/**
 * Return a string containing the exception description
 * @param val The exception number.
 * @return The exception description.
 */
const char *GetExceptionDescription(CakeException val);

/**
 * Throws an exception and stops the application.
 * @param val The exception number.
 * @param msg An eventual message
 */
void ThrowException(CakeException val, const char *msg = NULL);

#endif
