/*
 * Parse tree representation types.
 * reference: Holub, "Compiler Design in C"
 */

#ifndef _H_IL_
#define _H_IL_

#ifndef _H_HOSTINFO_
#include "hostinfo.h"
#endif

/*
 * Symbol names are prefixed by the
 * following for enums, structs and unions
 */
#define ENUM_PREFIX				'1'
#define STRUCT_PREFIX			'2'
#define UNION_PREFIX			'3'

/*
 * Type Modifiers
 */
#define TYPEMOD_EXTERN			0x0001
#define TYPEMOD_AUTO			0x0002
#define TYPEMOD_REGISTER		0x0004
#define TYPEMOD_SHORT			0x0008
#define TYPEMOD_STATIC			0x0010
#define TYPEMOD_SIGNED			0x0020
#define TYPEMOD_UNSIGNED		0x0040
#define TYPEMOD_VOLATILE		0x0080
#define TYPEMOD_TYPEDEF			0x0100
#define TYPEMOD_CONST			0x0200
#define TYPEMOD_LONG			0x1000

/*
 * Our enums ordinals don't overlap nor do
 * they start with zero so that it's easy
 * to trap pointers to trash when debugging
 */
typedef enum {
	pointer_decl = 1,
	int_decl,
	fp_decl,
	field_decl,
	func_decl,
	enum_decl,
	array_decl,
	struct_decl
} decl_class_t;

/*
 * expression kinds
 */
typedef enum {
	_Error = 32,
	_Elipsis,
	_FP_Number,
	_Int_Number,
	_Type,
	_Sym,
	_Ident,
	_String,
	_List,						/* first binary */
	_Bit_Field,
	_Dot_Selected,
	_Arrow_Selected,
	_Array_Index,
	_Func_Call,
	_Type_Cast,
	_Assign,
	_Mul_Assign,
	_Div_Assign,
	_Mod_Assign,
	_Add_Assign,
	_Sub_Assign,
	_Shl_Assign,
	_Shr_Assign,
	_Band_Assign,
	_Xor_Assign,
	_Bor_Assign,
	_Eq,
	_Ne,
	_Lt,
	_Le,
	_Gt,
	_Ge,
	_Land,
	_Lor,
	_Band,
	_Bor,
	_Xor,
	_Add,
	_Sub,
	_Mul,
	_Div,
	_Rem,
	_Shl,
	_Shr,						/* last binary */
	_Sizeof,					/* first unary */
	_Pre_Inc,
	_Pre_Dec,
	_Post_Inc,
	_Post_Dec,
	_Addrof,
	_Unary_Plus,
	_Unary_Minus,
	_Ones_Complement,
	_Not,
	_Aggregate,
	_Indirect,					/* last unary */
	_Cond
} node_kind_t;


typedef struct node_t {
	node_kind_t	node_kind;
	file_pos_t	node_def;
	union {
		struct{struct node_t *bool,*tru,*fals;}	cond;
		struct{char *form; int len;}	str;
		struct{struct node_t *l,*r;}	binary;
		struct{char *name; char *cmnt;}	id;
		struct node_t 					*unary;
		struct symbol_t					*sym;
		struct typeinfo_t				*typ;
		host_int_t						ival;
		host_float_t					fval;
	} node;
} node_t;

/*
 * Macros to get the file name and line number
 * info from the original C source
 */
#define NODE_FNAME(n)	file_name(n->node_def)
#define NODE_FLINE(n)	line_number(n->node_def)

/*
 * Possible declaration types
 */
typedef enum {
	pointer_to = 128,
	array_of,
	struct_of,
	union_of,
	field_type,
	int_type,
	float_type,
	void_type,
	function_type,
	enum_type,
	typemodifier
} typekind_t;

typedef struct typeinfo_t {
	typekind_t			type_kind;

	unsigned int		_unsigned:1;
	unsigned int		_signed:1;
	unsigned int		_short:1;
	unsigned int		_long:1;
	unsigned int		_long_long:1;
	unsigned int		_volatile:1;
	unsigned int		_constant:1;
	unsigned int		_extern:1;
	unsigned int		_static:1;
	unsigned int		_auto:1;
	unsigned int		_register:1;
	unsigned int		_typedef:1;

	unsigned int		_builtin:1;				/* An intrinsic type */
	unsigned int		_anonymous:1;

	union {
		int					array_elements;
		struct symbol_t		*subp_params;
	}					type_info;

	unsigned int		_sizeof;
	unsigned int		_alignof;

	unsigned int		type_hash; 				/* Comparison heuristic */

	struct symbol_t		*type_base;

	struct typeinfo_t	*type_anonymous_list;
	struct typeinfo_t	*type_next;
} typeinfo_t;

/*
 * Possible symbol declarations
 */
typedef enum {
	type_symbol = 256,
	func_symbol,
	param_symbol,
	var_symbol,
	enum_literal
} sym_kind_t;

typedef struct symbol_t {
	sym_kind_t			sym_kind;
	unsigned char		sym_scope;

	unsigned int		intrinsic:1; 	/* intrinsic/builtin symbol */

	unsigned int		_volatile:1;
	unsigned int		_const:1;
	unsigned int		_created_name:1;
	unsigned int		_created_by_reference:1;

	unsigned int		var_params:1;	/* Variable number of params */

	unsigned int		_assume_int:1;
	unsigned int		_eret:1;		/* returns a value */

	unsigned int		has_initializer:1;

	unsigned int		gened:1; 		/* Symbol has been passed to gen routines */
	unsigned int		cleared:1; 		/* Symbol can be queued */
	unsigned int		stored:1; 		/* Symbol has been added to symbol table */

	int					traversal_unit;	/* heuristic to eliminate redundant unit traversals */

	node_t				*sym_ident;		/* C name - node saved for comments */
	char				*sym_ada_name;	/* Generated Ada name */
	file_pos_t			sym_def;		/* File,line definition */
	typeinfo_t			*sym_type; 		/* Symbols type info */
	int					bitoffset; 		/* For fields only */

	union {
		node_t			*initializer;
		host_int_t		intval;
	} sym_value;

	struct symbol_t		*sym_tags; 		/* List of struct/union fields or enum literals */

	struct symbol_t		*sym_parse_list;/* List used by front end */
	struct symbol_t		*sym_scope_list;/* List used by front end */
	struct symbol_t		*sym_gen_list; 	/* List used by back end */

	/* hash table info */
	hash_t				sym_hash; 		/* Comparison heuristic */
	struct symbol_t 	*sym_hash_list;
} symbol_t;

#endif
