package net.psammead.util;

/** static helper class: diff Strings in classic and side-by-side format - see man diff */
public class StringDiff {
	private static final String	LEFT_PREFIX			= "< ";
	private static final String	RIGHT_PREFIX		= "> ";
	private static final String	CHANGE_SEPARATOR	= "---";
	private static final String	NUMBER_SEPARATOR	= ",";
	private static final String	DELETE_OP			= "d";
	private static final String	APPEND_OP			= "a";
	private static final String	CHANGE_OP			= "c";
	private	static final String	LINE_SEPARATOR		= "\n";
	
	/** instances of this class cannot be built */
	private StringDiff() {}

	/** classic diff */
	public static String classic(String left, String right) {
		String[]		leftLines	= left.split(LINE_SEPARATOR);
		String[]		rightLines	= right.split(LINE_SEPARATOR);
		
		Diff			diff		= new Diff(leftLines, rightLines);
		Diff.Change		script		= diff.diff2(false);
		
		StringBuilder	out			= new StringBuilder();
		for (Diff.Change hunk=script; hunk!=null; hunk=hunk.link) {
			// determine left and right line ranges
			int deletes		= hunk.deleted;
			int inserts		= hunk.inserted;
			if (deletes == 0 && inserts == 0)	continue;
			int leftFirst	= hunk.line0;
			int rightFirst	= hunk.line1;
			int leftLast	= hunk.line0 + deletes - 1;
			int rightLast	= hunk.line1 + inserts - 1;
			
			// Write out the line number header for this hunk	
			// left range
			if (deletes > 1) {
				out.append(Integer.toString(leftFirst + 1)).append(NUMBER_SEPARATOR);
			}
			out.append(Integer.toString(leftLast + 1));
			// change letter
				 if (inserts == 0)	out.append(DELETE_OP);
			else if (deletes == 0)	out.append(APPEND_OP);
			else					out.append(CHANGE_OP);
			// right range
			if (inserts > 1) {
				out.append(Integer.toString(rightFirst + 1)).append(NUMBER_SEPARATOR);
			}
			out.append(Integer.toString(rightLast + 1)).append(LINE_SEPARATOR);
	
			// Write the lines that the left file has.
			if (deletes != 0) {
				for (int i=leftFirst; i<=leftLast; i++) {
					out.append(LEFT_PREFIX).append(leftLines[i]).append(LINE_SEPARATOR);
				}
			}
	
			// Write a Separator between lines
			if (inserts != 0 && deletes != 0) {
				out.append(CHANGE_SEPARATOR).append(LINE_SEPARATOR);
			}
	
			// Write the lines that the right file has.
			if (inserts != 0) {
				for (int i=rightFirst; i<=rightLast; i++) {
					out.append(RIGHT_PREFIX).append(rightLines[i]).append(LINE_SEPARATOR);
				}
			}
		}
		return out.toString();
	}
	
	/** side-by-side diff */
	public static String sideBySide(String left, String right, String leftPrefix, String rightPrefix, String commonPrefix, String lineSeparator) {
		String[]		leftLines	= left.split(lineSeparator);
		String[]		rightLines	= right.split(lineSeparator);
		Diff			diff		= new Diff(leftLines, rightLines);
		Diff.Change		script		= diff.diff2(false);
		StringBuilder	out			= new StringBuilder();
		
		int	line;
		int	cursor	= 0;
		for (Diff.Change hunk=script; hunk!=null; hunk=hunk.link) {
			int	leftPos		= hunk.line0;
			int deletes		= hunk.deleted;
			int rightPos	= hunk.line1;
			int inserts		= hunk.inserted;
			
			// output common lines
			for (line=cursor; line<leftPos; line++) {
				out.append(commonPrefix).append(leftLines[line]).append(lineSeparator);
			}
			
			// output left lines
			for (line=leftPos; line<leftPos+deletes; line++) {
				out.append(leftPrefix).append(leftLines[line]).append(lineSeparator);
			}
			
			// output right lines
			for (line=rightPos; line<rightPos+inserts; line++) {
				out.append(rightPrefix).append(rightLines[line]).append(lineSeparator);
			}

			cursor	= leftPos	+ deletes;
		}
		// append common lines
		for (line=cursor; line<leftLines.length; line++) {
			out.append(commonPrefix).append(leftLines[line]).append(lineSeparator);
		}
		return out.toString();
	}
}
