package net.psammead.util.ui;

import java.awt.Color;
import java.awt.Container;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;

import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.WindowConstants;

// example:
// c.add(label, new GridBagConstraintsBuilder().pos(0,0).size(1,1).weight(1,1).done()));

/** incrementally builds a GridBagConstraints object */
public final class GridBagConstraintsBuilder {
	public static void main(String[] args) {
		JLabel	label1	= new JLabel("hallo");
		JLabel	label2	= new JLabel("welt");
		label1.setBackground(Color.red);
		label2.setBackground(Color.green);
		label1.setOpaque(true);
		label2.setOpaque(true);
		
		JFrame	frame	= new JFrame("GridBagConstraintsBuilder");
		
		Container	content	= frame.getContentPane();
		content.setLayout(new GridBagLayout());
		content.add(label1, new GridBagConstraintsBuilder()
				.pos(0, 0).size(1, 1).weight(0, 0).fillNone().insetsLeft(5).done());
		content.add(label2, new GridBagConstraintsBuilder()
				.pos(1, 0).size(1, 1).weight(1, 0).fillHorizontal().insetsLeft(5).insetsRight(5).done());

		frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
		frame.pack();
		frame.setVisible(true);
	}
	
	//-------------------------------------------------------------------------
	//## object

	private GridBagConstraints c;
	
	public GridBagConstraintsBuilder() {
		c	= new GridBagConstraints();
	}
	
	public GridBagConstraintsBuilder(GridBagConstraints constraints) {
		c	= constraints;
	}
	
	public GridBagConstraints done() {
		return c;
	}
	
	//-------------------------------------------------------------------------
	//## all
	
	public GridBagConstraintsBuilder reset() {
		reset(c);
		return this;
	}
	
	public GridBagConstraintsBuilder load(GridBagConstraints constraints) {
		copy(constraints, c);
		return this;
	}
	
	public GridBagConstraintsBuilder store(GridBagConstraints constraints) {
		copy(c, constraints);
		return this;
	}
	
	//-------------------------------------------------------------------------
	//## gridx/gridy
	
	public GridBagConstraintsBuilder pos(int x, int y) {
		return posX(x).posY(y);
	}
	
	public GridBagConstraintsBuilder posRelative() {
		return posRelativeX().posRelativeY();
	}
	
	public GridBagConstraintsBuilder posX(int x) {
		if (x < 0)	throw new IllegalArgumentException("expected x >= 0");
		c.gridx	= x;
		return this;
	}
	
	public GridBagConstraintsBuilder posY(int y) {
		if (y < 0)	throw new IllegalArgumentException("expected y >= 0");
		c.gridy	= y;
		return this;
	}
	
	public GridBagConstraintsBuilder posRelativeX() {
		c.gridx	= GridBagConstraints.RELATIVE;
		return this;
	}
	
	public GridBagConstraintsBuilder posRelativeY() {
		c.gridy	= GridBagConstraints.RELATIVE;
		return this;
	}

	//-------------------------------------------------------------------------
	//## width/height
	
	public GridBagConstraintsBuilder size(int x, int y) {
		return sizeX(x).sizeY(x);
	}
	
	public GridBagConstraintsBuilder sizeRelative() {
		return sizeRelativeX().sizeRelativeY();
	}
	
	public GridBagConstraintsBuilder sizeRemainder() {
		return sizeRemainderX().sizeRemainderY();
	}
	
	public GridBagConstraintsBuilder sizeX(int x) {
		if (x < 0)	throw new IllegalArgumentException("expected x >= 0");
		c.gridwidth	= x;
		return this;
	}
	
	public GridBagConstraintsBuilder sizeY(int y) {
		if (y < 0)	throw new IllegalArgumentException("expected y >= 0");
		c.gridheight	= y;
		return this;
	}
	
	public GridBagConstraintsBuilder sizeRelativeX() {
		c.gridwidth		= GridBagConstraints.RELATIVE;
		return this;
	}
	
	public GridBagConstraintsBuilder sizeRelativeY() {
		c.gridheight	= GridBagConstraints.RELATIVE;
		return this;
	}
	
	public GridBagConstraintsBuilder sizeRemainderX() {
		c.gridwidth		= GridBagConstraints.REMAINDER;
		return this;
	}
	
	public GridBagConstraintsBuilder sizeRemainderY() {
		c.gridheight	= GridBagConstraints.REMAINDER;
		return this;
	}
	
	//-------------------------------------------------------------------------
	//## weightx/weighty
	
	public GridBagConstraintsBuilder weight(double x, double y) {
		return weightX(x).weightY(y);
	}
	
	public GridBagConstraintsBuilder weightX(double x) {
		if (x < 0 || x > 1)	throw new IllegalArgumentException("expected 0 <= x <= 1");
		c.weightx	= x;
		return this;
	}
	
	public GridBagConstraintsBuilder weightY(double y) {
		if (y < 0 || y > 1)	throw new IllegalArgumentException("expected 0 <= y <= 1");
		c.weighty	= y;
		return this;
	}
	
	//-------------------------------------------------------------------------
	//## ipadx/y
	
	public GridBagConstraintsBuilder ipad(int x, int y) {
		return ipadX(x).ipadY(y);
	}
	
	public GridBagConstraintsBuilder ipadX(int x) {
		if (x < 0)	throw new IllegalArgumentException("expected x >= 0");
		c.ipadx	= x;
		return this;
	}
	
	public GridBagConstraintsBuilder ipadY(int y) {
		if (y < 0)	throw new IllegalArgumentException("expected y >= 0");
		c.ipady	= y;	
		return this;
	}

	//-------------------------------------------------------------------------
	//## insets
	
	public GridBagConstraintsBuilder insets(int top, int left, int bottom, int right) {
		return insetsTop(top).insetsLeft(left).insetsBottom(bottom).insetsRight(right);
	}

	public GridBagConstraintsBuilder insetsTop(int top) {
		if (top < 0)	throw new IllegalArgumentException("expected top >= 0");
		Insets	old	= c.insets;
		c.insets	= new Insets(top, old.left, old.bottom, old.right);
		return this;
	}
	
	public GridBagConstraintsBuilder insetsLeft(int left) {
		if (left < 0)	throw new IllegalArgumentException("expected left >= 0");
		Insets	old	= c.insets;
		c.insets	= new Insets(old.top, left, old.bottom, old.right);
		return this;
	}
	
	public GridBagConstraintsBuilder insetsBottom(int bottom) {
		if (bottom < 0)	throw new IllegalArgumentException("expected bottom >= 0");
		Insets	old	= c.insets;
		c.insets	= new Insets(old.top, old.left, bottom, old.right);
		return this;
	}
	
	public GridBagConstraintsBuilder insetsRight(int right) {
		if (right < 0)	throw new IllegalArgumentException("expected top >= 0");
		Insets	old	= c.insets;
		c.insets	= new Insets(old.top, old.left, old.bottom, right);
		return this;
	}
	
	//-------------------------------------------------------------------------
	//## insets
	
	public GridBagConstraintsBuilder anchorCenter() {
		c.anchor	= GridBagConstraints.CENTER;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorNorth() {
		c.anchor	= GridBagConstraints.NORTH;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorSouth() {
		c.anchor	= GridBagConstraints.SOUTH;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorEast() {
		c.anchor	= GridBagConstraints.EAST;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorWest() {
		c.anchor	= GridBagConstraints.WEST;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorNorthEast() {
		c.anchor	= GridBagConstraints.NORTHEAST;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorNorthWest() {
		c.anchor	= GridBagConstraints.NORTHWEST;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorSouthEast() {
		c.anchor	= GridBagConstraints.SOUTHEAST;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorSouthWest() {
		c.anchor	= GridBagConstraints.SOUTHWEST;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorPageStart() {
		c.anchor	= GridBagConstraints.PAGE_START;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorPageEnd() {
		c.anchor	= GridBagConstraints.PAGE_END;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorLineStart() {
		c.anchor	= GridBagConstraints.LINE_START;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorLineEnd() {
		c.anchor	= GridBagConstraints.LINE_END;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorFirstLineStart() {
		c.anchor	= GridBagConstraints.FIRST_LINE_START;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorFirstLineEnd() {
		c.anchor	= GridBagConstraints.FIRST_LINE_END;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorLastLineStart() {
		c.anchor	= GridBagConstraints.LAST_LINE_START;
		return this;
	}
	
	public GridBagConstraintsBuilder anchorLastLineEnd() {
		c.anchor	= GridBagConstraints.LAST_LINE_START;
		return this;
	}
	
	//-------------------------------------------------------------------------
	//## fill
	
	public GridBagConstraintsBuilder fillNone() {
		c.fill	= GridBagConstraints.NONE;
		return this;
	}

	public GridBagConstraintsBuilder fillHorizontal() {
		c.fill	= GridBagConstraints.HORIZONTAL;
		return this;
	}

	public GridBagConstraintsBuilder fillVertical() {
		c.fill	= GridBagConstraints.VERTICAL;
		return this;
	}
	
	public GridBagConstraintsBuilder fillBoth() {
		c.fill	= GridBagConstraints.BOTH;
		return this;
	}
	
	//-------------------------------------------------------------------------
	//## helper
	
	/** resets GridBagConstraints to their default value */
	private static void reset(GridBagConstraints c) {
		c.gridx			= GridBagConstraints.RELATIVE;
		c.gridy			= GridBagConstraints.RELATIVE;
		c.gridwidth		= 1;
		c.gridheight	= 1;
		c.weightx		= 0;
		c.weighty		= 0;
		c.anchor		= GridBagConstraints.CENTER;	
		c.fill			= GridBagConstraints.NONE;
		c.insets		= new Insets(0,0,0,0);
		c.ipadx			= 0;
		c.ipady			= 0;
	}
	
	/** copies all values of a GridBagConstraints */
	private static void copy(GridBagConstraints from, GridBagConstraints to) {
		to.gridx		= from.gridx;
		to.gridy		= from.gridy;
		to.gridwidth	= from.gridwidth;
		to.gridheight	= from.gridheight;
		to.weightx		= from.weightx;
		to.weighty		= from.weighty;
		to.anchor		= from.anchor;
		to.fill			= from.fill;
		to.insets		= from.insets;
		to.ipadx		= from.ipadx;
		to.ipady		= from.ipady;
	}
}
