/*
Copyright (c) 2009-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	dkfigw.h	Include file for fig writer module.

This module is mainly intended as the backend for the
DKrause::FigWriter Perl module but can be used
standalone too.

A graphics is constructed in memory which can be saved as
a Fig file.
*/



#ifndef DKFIGW_H_INCLUDED



/**	Avoid multiple inclusions of this file.
*/
#define DKFIGW_H_INCLUDED 1



#include <dk.h>
#include <dkmem.h>
#include <dksto.h>



/**	Color cell.
*/
typedef struct _dk_fig_cc_t_ {
  unsigned short	ccn;	/**< Color cell number. */
  int			r;	/**< Red. */
  int			g;	/**< Green. */
  int			b;	/**< Blue. */
} dk_fig_cc_t;



/**	Arrowhead style.
*/
typedef struct _dk_fig_ah_t_ {
  unsigned char		sha;	/**< Arrowhead shape. */
  unsigned char		fil;	/**< Fill style. */
  double		wid;	/**< Arrow width. */
  double		hei;	/**< Arrow height. */
} dk_fig_ah_t;


/**	Style structure.
*/
typedef struct _dk_fig_style_t_ {
  unsigned short	lay;	/**< Layer number. */
  short			pco;	/**< Pen color. */
  short			fco;	/**< Fill color. */
  short			liw;	/**< Line width. */
  unsigned char		lst;	/**< Line style. */
  unsigned char		ljn;	/**< Line join. */
  unsigned char		lcp;	/**< Line cap. */
  unsigned char		fil;	/**< Fill style. */
  unsigned char		ffl;	/**< Font flags. */
  unsigned char		fno;	/**< Font number. */
  unsigned char		tal;	/**< Text alignment. */
  unsigned char		ahs;	/**< Arrowheads. */
  double		sva;	/**< Style value. */
  double		fsz;	/**< Font size. */
  dk_fig_ah_t		fwa;	/**< Forward arrowhead. */
  dk_fig_ah_t		bwa;	/**< Backward arrowhead. */
} dk_fig_style_t;



/**	Polyline/polygon type.
*/
typedef struct _dk_fig_pl_pt_t_ {
  unsigned long		npt;	/**< Number of point. */
  double		x;	/**< X position. */
  double		y;	/**< Y position. */
} dk_fig_pl_pt_t;



/**	Spline point type.
*/
typedef struct _dk_fig_sp_pt_t_ {
  unsigned long		npt;	/**< Number of point. */
  double		x;	/**< X position. */
  double		y;	/**< Y position. */
  double		s;	/**< Approximation/interpolation coefficient. */
} dk_fig_sp_pt_t;


/**	Fig object.
*/
typedef struct _dk_fig_object_t_ {
  unsigned long			obn;	/**< Object number. */
  int				otp;	/**< Object type. */
  int				stp;	/**< Subtype. */
  dk_fig_style_t		sty;	/**< Object style. */
  union {
    struct {
      double			xm;	/**< Center point x. */
      double			ym;	/**< Center point y. */
      double			ra;	/**< Radius. */
      double			a0;	/**< Alpha start. */
      double			a1;	/**< Alpha end. */
    }				arc;	/**< Arc object details. */
    struct {
      double			xm;	/**< Center x. */
      double			ym;	/**< Center y. */
      double			rx;	/**< Radius x. */
      double			ry;	/**< Radius y. */
      double			ro;	/**< Rotation. */
    }				ell;	/**< Ellipse. */
    struct {
      dk_storage_t		*spt;	/**< Points. */
      dk_storage_iterator_t	*ipt;	/**< Iterator through points. */
      unsigned long		npn;	/**< Next point number. */
      char			*fn;	/**< File name for image. */
      double			x0;	/**< First point, x position. */
      double			y0;	/**< First point, y position. */
      double			x1;	/**< Second point, x position. */
      double			y1;	/**< Second point, y position. */
      double			cr;	/**< Corner radius for arc boxes. */
    }				pll;	/**< Polyline/polygon/spline object details. */
    struct {
      double			x;	/**< X position. */
      double			y;	/**< Y position. */
      double			rot;	/**< Rotation. */
      char			*txt;	/**< Text. */
    }				txt;	/**< Text object details. */
  }				det;	/**< Object details. */
} dk_fig_object_t;



/**	Writer object for entire drawing.
*/
typedef struct _dk_fig_writer_t_ {
  dk_storage_t		*ost;	/**< Object storage. */
  dk_storage_iterator_t	*osi;	/**< Object storage iterator. */
  dk_storage_t		*ccs;	/**< Color cell storage. */
  dk_storage_iterator_t	*cci;	/**< Color cell storage iterator. */
  dk_fig_object_t	*cob;	/**< Current object. */
  dk_fig_object_t	*sob;	/**< Selected object. */
  unsigned long		non;	/**< Next object number. */
  unsigned short	ncc;	/**< Next color cell number. */
  short			tco;	/**< Transparent color. */
  dk_fig_style_t	sty;	/**< Style for new objects. */
  unsigned char		pap;	/**< Paper size. */
  unsigned char		olt;	/**< Flag: Origin on left top (1) or left bottom (0). */
  unsigned char		ori;	/**< Orientation (0=landscape, 1=portrait). */
  unsigned char		jus;	/**< Justification (0=center, 1=left). */
  unsigned char		fun;	/**< Fig file units (0=metric, 1=inches). */
  unsigned char		utf;	/**< Flag: UTF-8 encoded strings. */
  unsigned char		ver;	/**< Flag: Verbose. */
  int			uni;	/**< Units for image size (0=Fig, 1=inch, 2=cm). */
  unsigned char		ang;	/**< Unit for angles (0=radians, 1=degree). */
  double		imw;	/**< Image width. */
  double		imh;	/**< Image height. */
  double		iwf;	/**< Image width in Fig units. */
  double		ihf;	/**< Image height in Fig units. */
  double		xsh;	/**< X shift in Fig units. */
  double		ysh;	/**< Y shift in Fig units. */
  int			erc;	/**< Error code. */
  int			mer;	/**< Mathematical error. */
} dk_fig_writer_t;

/**	Data type for use with Perl.  */
typedef dk_fig_writer_t		*DKrause__FigWriter;


#ifdef EXTERN
#undef EXTERN
#endif
#if DKFIGW_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

/**	Compare two polyline/polygon points by number.
	This function is used to build the internal
	sorted storage of points for a polygon.
	@param	l	Left point.
	@param	r	Right point.
	@param	c	Comparison criteria (ignored).
	@return	Comparison result.
*/
EXTERN int
dkfigw_compare_pl_points DK_PR((void *l, void *r, int c));

/**	Compare two spline points by number.
	This function is used to build the internal
	sorted storage of points for a spline.
	@param	l	Left point.
	@param	r	Right point.
	@param	c	Comparison criteria (ignored).
	@return	Comparison result.
*/
EXTERN int
dkfigw_compare_sp_points DK_PR((void *l, void *r, int c));

/**	Compare two color cells.
	This function is used to build the internal sorted
	storage of color cells.
	@param	l	Left color cell.
	@param	r	Right color cell.
	@param	c	Comparison criteria (0=cell/cell, 1=cell/number).
	@return	Comparison result.
*/
EXTERN int
dkfigw_compare_color_cells DK_PR((void *l, void *r, int c));

/**	Compare two objects.
	This function is used to build the internal
	sorted storage of Fig objects.
	@param	l	Left object.
	@param	r	Right object.
	@param	c	Comparison criteria (0=object/object, 1=object/number).
	@return	Comparison result.
*/
EXTERN int
dkfigw_compare_objects DK_PR((void *l, void *r, int c));

/**	Create new writer structure.
	@return	Pointer to new writer structure on success, NULL on error.
*/
EXTERN dk_fig_writer_t *
dkfigw_new DK_PR((void));

/**	Remove writer structure, release memory.
	@param	fwp	Pointer to structure to release.
*/
EXTERN void
dkfigw_delete DK_PR((dk_fig_writer_t *fwp));

/**	Enable/disable verbose mode.
	In verbose mode the module prints warnings to stderr
	for illegal names in dkfigw_set_xxx_by_name() functions
	and suggests correct value.
	Verbose mode is turned off by default for new Fig
	writer structures.
	@param	fwp	Fig writer structure.
	@param	fl	Flag to enable/disable.
*/
EXTERN void
dkfigw_set_verbose DK_PR((dk_fig_writer_t *fwp, int fl));

/**	Enable/disable usage of UTF-8.
	When creating a new Fig writer structure the LANG
	environment variable is inspected to set the default
	value.
	@param	fwp	Fig writer structure.
	@param	fl	Flag to enable/disable.
*/
EXTERN void
dkfigw_set_utf8 DK_PR((dk_fig_writer_t *fwp, int fl));

/**	Set image size.
	@param	fwp	Writer structure.
	@param	uni	Size unit (0=Fig, 1=inch, 2=cm).
	@param	w	Image width.
	@param	h	Image height.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfigw_set_image_size DK_PR((dk_fig_writer_t *fwp,int uni,double w,double h));

/**	Set image origin.
	@param	fwp	Writer structure.
	@param	olt	Flag: Origin at left top (1) or left bottom (0).
*/
EXTERN void
dkfigw_set_image_origin DK_PR((dk_fig_writer_t *fwp, int olt));

/**	Set text alignment for all following texts added.
	@param	fwp	Writer structure.
	@param	tal	Alignment (FIG_SUB_TEXT_xxx).
*/
EXTERN void
dkfigw_set_text_align DK_PR((dk_fig_writer_t *fwp, unsigned char tal));

/**	Set/reset rigid flag for all following texts added.
	@param	fwp	Writer structure.
	@param	fl	Set (non-zero) or reset (0) flag.
*/
EXTERN void
dkfigw_set_text_rigid DK_PR((dk_fig_writer_t *fwp, int fl));

/**	Set/reset special flag for all following texts added.
	@param	fwp	Writer structure.
	@param	fl	Set (non-zero) or reset (0) flag.
*/
EXTERN void
dkfigw_set_text_special DK_PR((dk_fig_writer_t *fwp, int fl));

/**	Set PS font index for all following texts added.
	@param	fwp	Writer structure.
	@param	fno	Font index (0 to 34).
*/
EXTERN void
dkfigw_set_text_psfont DK_PR((dk_fig_writer_t *fwp, unsigned char fno));

/**	Set PS font for all following texts added, choose font by name.
	@param	fwp	Writer structure.
	@param	n	Font name:
	"Times Roman", "Times Italic", "Times Bold", "Times Bold Italic",
	"Avantgarde Book", "Avantgarde Book Oblique", "Avantgarde Demi",
	"Avantgarde Demi Oblique", "Bookman Light", "Bookman Light Italic",
	"Bookman Demi", "Bookman Demi Italic", "Courier", "Courier Oblique",
	"Courier Bold", "Courier Bold Oblique", "Helvetica",
	"Helvetica Oblique", "Helvetica Bold", "Helvetica Bold Oblique",
	"Helvetica Narrow", "Helvetica Narrow Oblique", "Helvetica Narrow Bold",
	"Helvetica Narrow Bold Oblique", "New Century Schoolbook Roman",
	"New Century Schoolbook Italic",
	"New Century Schoolbook Bold",
	"New Century Schoolbook Bold Italic",
	"Palatino Roman", "Palatino Italic", "Palatino Bold",
	"Palatino Bold Italic", "Symbol", "Zapf Chancery Medium Italic" or
	"Zapf Dingbats".
*/
void
dkfigw_set_text_psfont_by_name DK_PR((dk_fig_writer_t *fwp, char *n));

/**	Set LaTeX font index for all following texts added.
	@param	fwp	Writer structure.
	@param	fno	Font index (0=default, 1=roman, 2=bold, 3=italic,
	4=sans-serif, 5=typewriter).
*/
EXTERN void
dkfigw_set_text_latexfont DK_PR((dk_fig_writer_t *fwp, unsigned char fno));

/**	Set text size fo all following texts added.
	@param	fwp	Writer structure.
	@param	fsz	Font size in pt.
*/
EXTERN void
dkfigw_set_text_size DK_PR((dk_fig_writer_t *fwp, double fsz));

/**	Set layer for all following objects.
	@param	fwp	Writer structure.
	@param	lay	Layer number.
*/
EXTERN void
dkfigw_set_layer DK_PR((dk_fig_writer_t *fwp, unsigned short lay));

/**	Move one layer upwards.
	@param	fwp	Writer structure.
	@return	Non-zero number on success, 0 on error.
*/
EXTERN int
dkfigw_layer_up DK_PR((dk_fig_writer_t *fwp));

/**	Define a new color.
	@param	fwp	Fig writer structure.
	@param	r	Red.
	@param	g	Green.
	@param	b	Blue.
	@return	The new color cell number on success, 0 on error.
*/
EXTERN int
dkfigw_define_color DK_PR((dk_fig_writer_t *fwp, int r, int g, int b));

/**	Set pen color for all following objects.
	@param	fwp	Writer structure.
	@param	col	Color cell number (0 to 31 for predefined colors,
	32 and above for self-defined colors).
*/
EXTERN void
dkfigw_set_pen_color DK_PR((dk_fig_writer_t *fwp, short col));

/**	Set pen color for all following objects by color name.
	@param	fwp	Writer structure.
	@param	n	Color name:
	"black", "blue", "green", "cyan", "red", "magenta", "yellow",
	"white", "blue.darkest", "blue.dark", "blue.light", "blue.lightest",
	"green.dark", "green.normal", "green.light", "cyan.dark", "cyan.normal",
	"cyan.light", "red.dark", "red.normal", "red.light", "magenta.dark",
	"magenta.normal", "magenta.light", "brown.dark", "brown.normal",
	"brown.light", "pink.darkest", "pink.dark", "pink.light",
	"pink.lightest" or "gold".
*/
EXTERN void
dkfigw_set_pen_color_by_name DK_PR((dk_fig_writer_t *fwp, char *n));

/**	Set fill color for all following objects.
	@param	fwp	Writer structure.
	@param	col	Color cell number, either a user defined
	color (32 and above) or a predefined color:
	FIG_COLOR_BLACK, FIG_COLOR_BLUE, FIG_COLOR_GREEN, FIG_COLOR_CYAN,
	FIG_COLOR_RED, FIG_COLOR_MAGENTA, FIG_COLOR_YELLOW, FIG_COLOR_WHITE,
	FIG_COLOR_DARKEST_BLUE, FIG_COLOR_DARK_BLUE, FIG_COLOR_LIGHT_BLUE,
	FIG_COLOR_LIGHTEST_BLUE, FIG_COLOR_DARK_GREEN,
	FIG_COLOR_NORMAL_GREEN, FIG_COLOR_LIGHT_GREEN, FIG_COLOR_DARK_CYAN,
	FIG_COLOR_NORMAL_CYAN, FIG_COLOR_LIGHT_CYAN, FIG_COLOR_DARK_RED,
	FIG_COLOR_NORMAL_RED, FIG_COLOR_LIGHT_RED, FIG_COLOR_DARK_MAGENTA,
	FIG_COLOR_NORMAL_MAGENTA, FIG_COLOR_LIGHT_MAGENTA, FIG_COLOR_DARK_BROWN,
	FIG_COLOR_NORMAL_BROWN, FIG_COLOR_LIGHT_BROWN, FIG_COLOR_DARKEST_PINK,
	FIG_COLOR_DARK_PINK, FIG_COLOR_LIGHT_PINK, FIG_COLOR_LIGHTEST_PINK or
	FIG_COLOR_GOLD.
*/
EXTERN void
dkfigw_set_fill_color DK_PR((dk_fig_writer_t *fwp, short col));

/**	Set fill color for all following objects by color name.
	@param	fwp	Writer structure.
	@param	n	Color name:
	"black", "blue", "green", "cyan", "red", "magenta", "yellow",
	"white", "blue.darkest", "blue.dark", "blue.light", "blue.lightest",
	"green.dark", "green.normal", "green.light", "cyan.dark", "cyan.normal",
	"cyan.light", "red.dark", "red.normal", "red.light", "magenta.dark",
	"magenta.normal", "magenta.light", "brown.dark", "brown.normal",
	"brown.light", "pink.darkest", "pink.dark", "pink.light",
	"pink.lightest" or "gold".
*/
EXTERN void
dkfigw_set_fill_color_by_name DK_PR((dk_fig_writer_t *fwp, char *n));

/**	Set transparent color
	@param	fwp	Writer structure.
	@param	col	Index of the transparent color.
*/
EXTERN void
dkfigw_set_transparent_color DK_PR((dk_fig_writer_t *fwp, short col));

/**	Set fill style for all following objects.
	@param	fwp	Writer structure.
	@param	fil	Fill style, one from:
	FIG_FILL_NONE, FIG_FILL_BLACK, FIG_FILL_005, FIG_FILL_010, FIG_FILL_015,
	FIG_FILL_020, FIG_FILL_025, FIG_FILL_030, FIG_FILL_035, FIG_FILL_040,
	FIG_FILL_045, FIG_FILL_050, FIG_FILL_055, FIG_FILL_060, FIG_FILL_065,
	FIG_FILL_070, FIG_FILL_075, FIG_FILL_080, FIG_FILL_085, FIG_FILL_090,
	FIG_FILL_095, FIG_FILL_100, FIG_FILL_PURE, FIG_FILL_105, FIG_FILL_110,
	FIG_FILL_115, FIG_FILL_120, FIG_FILL_125, FIG_FILL_130, FIG_FILL_135,
	FIG_FILL_140, FIG_FILL_145, FIG_FILL_150, FIG_FILL_155, FIG_FILL_160,
	FIG_FILL_165, FIG_FILL_170, FIG_FILL_175, FIG_FILL_180, FIG_FILL_185,
	FIG_FILL_190, FIG_FILL_195, FIG_FILL_200, FIG_FILL_WHITE,
	FIG_FILL_LEFT_DIAGONAL_30, FIG_FILL_RIGHT_DIAGONAL_30,
	FIG_FILL_CROSSHATCH_30, FIG_FILL_LEFT_DIAGONAL_45,
	FIG_FILL_RIGHT_DIAGONAL_45, FIG_FILL_CROSSHATCH_45,
	FIG_FILL_BRICKS_HORIZONTAL, FIG_FILL_BRICKS_VERTICAL,
	FIG_FILL_HORIZONTAL_LINES, FIG_FILL_VERTICAL_LINES,
	FIG_FILL_CROSSHATCH, FIG_FILL_HORIZONTAL_SHINGLES_RIGHT,
	FIG_FILL_HORIZONTAL_SHINGLES_LEFT, FIG_FILL_VERTICAL_SHINGLES_1,
	FIG_FILL_VERTICAL_SHINGLES_2, FIG_FILL_LARGE_FISH_SCALES,
	FIG_FILL_SMALL_FISH_SCALES, FIG_FILL_CIRCLES,
	FIG_FILL_HEXAGONS, FIG_FILL_OCTAGONS,
	FIG_FILL_HORIZONTAL_TIRE_TREADS or FIG_FILL_VERTICAL_TIRE_TREADS.
*/
EXTERN void
dkfigw_set_fill_style DK_PR((dk_fig_writer_t *fwp, unsigned char fil));

/**	Set line style for all following objects.
	@param	fwp	Writer structure.
	@param	lst	Line style, one from:
	FIG_LS_SOLID, FIG_LS_DASHED, FIG_LS_DOTTED, FIG_LS_DASH_DOTTED,
	FIG_LS_DASH_DOUBLE_DOTTED or FIG_LS_DASH_TRIPLE_DOTTED.
*/
EXTERN void
dkfigw_set_line_style DK_PR((dk_fig_writer_t *fwp, unsigned char lst));

/**	Set line join style for all following objects.
	@param	fwp	Writer structure.
	@param	ljn	Line join style, one from:
	FIG_LJ_MITER, FIG_LJ_ROUND, or FIG_LJ_BEVEL.
*/
EXTERN void
dkfigw_set_line_join DK_PR((dk_fig_writer_t *fwp, unsigned char ljn));

/**	Set line width for all following objects.
	@param	fwp	Writer structure.
	@param	lw	Line width in Fig linewidth units (1/80 inch).
*/
EXTERN void
dkfigw_set_line_width DK_PR((dk_fig_writer_t *fwp, short lw));

/**	Set "angle-in-degree" flag in writer.
	@param	fwp	Writer structure.
	@param	fl	Set (non-zero value) or reset (0) the flag.
*/
EXTERN void
dkfigw_set_angle_degree DK_PR((dk_fig_writer_t *fwp, int fl));

/**	Set line cap style for all following objects.
	@param	fwp	Writer structure.
	@param	lcp	Line cap style, one from:
	FIG_CS_BUTT, FIG_CS_ROUND or FIG_CS_PROJECTING.
*/
EXTERN void
dkfigw_set_line_cap DK_PR((dk_fig_writer_t *fwp, unsigned char lcp));

/**	Set forward arrowhead style for all following objects.
	@param	fwp	Writer structure.
	@param	sha	Arrowhead shape, one from:
	FIG_AS_STICK, FIG_AS_TRIANGLE, FIG_AS_CLOSED_INDENTED or
	FIG_AS_CLOSED_POINTED.
	@param	fil	Arrowhead fill style, one from:
	FIG_AF_WHITE (white filled) or FIG_AF_PC (pen color filled).
	@param	wid	Arrowhead width.
	@param	hei	Arrowhead height.
*/
EXTERN void
dkfigw_set_arrowhead_f DK_PR((dk_fig_writer_t *fwp, unsigned char sha, unsigned char fil, double wid, double hei));

/**	Set backward arrowhead style for all following objects.
	@param	fwp	Writer structure.
	@param	sha	Arrowhead shape, one from:
	FIG_AS_STICK, FIG_AS_TRIANGLE, FIG_AS_CLOSED_INDENTED or
	FIG_AS_CLOSED_POINTED.
	@param	fil	Arrowhead fill style, one from:
	FIG_AF_WHITE (white filled) or FIG_AF_PC (pen color filled).
	@param	wid	Arrowhead width.
	@param	hei	Arrowhead height.
*/
EXTERN void
dkfigw_set_arrowhead_b DK_PR((dk_fig_writer_t *fwp, unsigned char sha, unsigned char fil, double wid, double hei));

/**	Set arrowheads directions.
	@param	fwp	Fig structure.
	@param	ahs	Arrowheads, or-combination of 0x01 (forward) and 0x02 (backward).
*/
EXTERN void
dkfigw_set_arrowheads DK_PR((dk_fig_writer_t *fwp, unsigned char ahs));

/**	Initial settings for arrowheads.
	@param	fwp	Fig structure.
*/
EXTERN void
dkfigw_set_suggested_arrow_settings DK_PR((dk_fig_writer_t *fwp));

/**	Add rectangle.
	@param	fwp	Writer structure.
	@param	x0	Point 1 x.
	@param	y0	Point 1 y.
	@param	x1	Point 2 x.
	@param	y1	Point 2 y.
	@return	Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_rectangle DK_PR((dk_fig_writer_t *fwp, double x0, double y0, double x1, double y1));

/**	Add arc box.
	@param	fwp	Writer structure.
	@param	x0	Point 1 x.
	@param	y0	Point 1 y.
	@param	x1	Point 2 x.
	@param	y1	Point 2 y.
	@param	r	Corner radius.
	@return	Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_arc_box DK_PR((dk_fig_writer_t *fwp, double x0, double y0, double x1, double y1, double r));

/**	Add image.
	@param	fwp	Writer structure.
	@param	x0	Point 1 x.
	@param	y0	Point 1 y.
	@param	x1	Point 2 x.
	@param	y1	Point 2 y.
	@param	fn	File name.
	@return	Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_image DK_PR((dk_fig_writer_t *fwp, double x0, double y0, double x1, double y1, char *fn));

/**	Add polygon. After creating the polygon use dkfigw_point() to add points.
	@param	fwp	Writer structure.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_polygon DK_PR((dk_fig_writer_t *fwp));

/**	Add polyline. After creating the polyline use dkfigw_point() to add points.
	@param	fwp	Writer structure.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_polyline DK_PR((dk_fig_writer_t *fwp));

/**	Add point to the current polygon or polyline object.
	@param	fwp	Writer structure.
	@param	x	X position.
	@param	y	Y position.
	@return	Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_point DK_PR((dk_fig_writer_t *fwp, double x, double y));

/**	Add open spline.
	After creating the spline use dkfigw_spline_point() to add points.
	@param	fwp	Writer structure.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_open_spline DK_PR((dk_fig_writer_t *fwp));

/**	Add closed spline.
	After creating the spline use dkfigw_spline_point() to add points.
	@param	fwp	Writer structure.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_closed_spline DK_PR((dk_fig_writer_t *fwp));

/**	Add point to current spline object.
	@param	fwp	Writer structure.
	@param	x	X position.
	@param	y	Y position.
	@param	s	Approximation/interpolation parameter.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_spline_point DK_PR((dk_fig_writer_t *fwp, double x, double y, double s));

/**	Add open arc.
	@param	fwp	Writer structure.
	@param	x	Center x.
	@param	y	Center y.
	@param	r	Radius.
	@param	a0	Start angle in radians.
	@param	a1	End angle in radians.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_open_arc DK_PR((dk_fig_writer_t *fwp, double x, double y, double r, double a0, double a1));

/**	Add closed arc (pie wedge).
	@param	fwp	Writer structure.
	@param	x	Center x.
	@param	y	Center y.
	@param	r	Radius.
	@param	a0	Start angle in radians.
	@param	a1	End angle in radians.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_closed_arc DK_PR((dk_fig_writer_t *fwp, double x, double y, double r, double a0, double a1));

/**	Add circle.
	@param	fwp	Writer structure.
	@param 	x	Center x.
	@param	y	Center y.
	@param	r	Radius.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_circle DK_PR((dk_fig_writer_t *fwp, double x, double y, double r));

/**	Add ellipse.
	@param	fwp	Writer structure.
	@param	x	Center x.
	@param	y	Center y.
	@param	rx	Horizontal radius.
	@param	ry	Vertical radius.
	@param	rot	Rotation in radians.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_ellipse DK_PR((dk_fig_writer_t *fwp, double x, double y, double rx, double ry, double rot));

/**	Add text.
	@param	fwp	Writer structure.
	@param	x	X position.
	@param	y	Y position.
	@param	t	Text.
	@param	a	Rotation in radians.
	@return Positive number on success, 0 on error.
*/
EXTERN unsigned long
dkfigw_text DK_PR((dk_fig_writer_t *fwp, double x, double y, char *t, double a));

/**	Write Fig file to stream.
	@param	os	Output stream.
	@param	fwp	Fig structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfigw_write DK_PR((dk_stream_t *os, dk_fig_writer_t *fwp));

/**	Write Fig file to named file.
	@param	fn	File name to write to.
	@param	fwp	Fig structure.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkfigw_file DK_PR((char *fn, dk_fig_writer_t *fwp));

#ifdef __cplusplus
};
#endif



/** Color: Default color.
*/
#define	FIG_COLOR_DEFAULT		(-1)

/** Color: Black.
*/
#define FIG_COLOR_BLACK			0

/** Color: Blue.
*/
#define FIG_COLOR_BLUE			1

/** Color: Green.
*/
#define FIG_COLOR_GREEN			2

/** Color: Cyan.
*/
#define FIG_COLOR_CYAN			3

/** Color: Red.
*/
#define FIG_COLOR_RED			4

/** Color: Magenta.
*/
#define FIG_COLOR_MAGENTA		5

/** Color: Yellow.
*/
#define FIG_COLOR_YELLOW		6

/** Color: White.
*/
#define FIG_COLOR_WHITE			7

/** Color: Darkest blue.
*/
#define FIG_COLOR_DARKEST_BLUE		8

/** Color: Dark blue.
*/
#define FIG_COLOR_DARK_BLUE		9

/** Color: Light blue.
*/
#define FIG_COLOR_LIGHT_BLUE		10

/** Color: Lightest blue.
*/
#define FIG_COLOR_LIGHTEST_BLUE		11

/** Color: Dark green.
*/
#define FIG_COLOR_DARK_GREEN		12

/** Color: Green.
*/
#define FIG_COLOR_NORMAL_GREEN		13

/** Color: Light green.
*/
#define FIG_COLOR_LIGHT_GREEN		14

/** Color: Dark cyan.
*/
#define FIG_COLOR_DARK_CYAN		15

/** Color: Cyan.
*/
#define FIG_COLOR_NORMAL_CYAN		16

/** Color: Light cyan.
*/
#define FIG_COLOR_LIGHT_CYAN		17

/** Color: Dark red.
*/
#define FIG_COLOR_DARK_RED		18

/** Color: Red.
*/
#define FIG_COLOR_NORMAL_RED		19

/** Color: Light red.
*/
#define FIG_COLOR_LIGHT_RED		20

/** Color: Dark magenta.
*/
#define FIG_COLOR_DARK_MAGENTA		21

/** Color: Magenta.
*/
#define FIG_COLOR_NORMAL_MAGENTA	22

/** Color: Light magenta.
*/
#define FIG_COLOR_LIGHT_MAGENTA		23

/** Color: Dark brown.
*/
#define FIG_COLOR_DARK_BROWN		24

/** Color: Brown.
*/
#define FIG_COLOR_NORMAL_BROWN		25

/** Color: Light brown.
*/
#define FIG_COLOR_LIGHT_BROWN		26

/** Color: Darkest pink.
*/
#define FIG_COLOR_DARKEST_PINK		27

/** Color: Dark pink.
*/
#define FIG_COLOR_DARK_PINK		28

/** Color: Light pink.
*/
#define FIG_COLOR_LIGHT_PINK		29

/** Color: Lightest pink.
*/
#define FIG_COLOR_LIGHTEST_PINK		30

/** Color: Gold.
*/
#define FIG_COLOR_GOLD			31



/** Fill style: Not filled.
*/
#define FIG_FILL_NONE			0xFF

/** Fill style: Black filled.
*/
#define FIG_FILL_BLACK			0x00

/** Fill style: Saturation   5 percent.
*/
#define FIG_FILL_005			0x01

/** Fill style: Saturation  10 percent.
*/
#define FIG_FILL_010			0x02

/** Fill style: Saturation  15 percent.
*/
#define FIG_FILL_015			0x03

/** Fill style: Saturation  20 percent.
*/
#define FIG_FILL_020			0x04

/** Fill style: Saturation  25 percent.
*/
#define FIG_FILL_025			0x05

/** Fill style: Saturation  30 percent.
*/
#define FIG_FILL_030			0x06

/** Fill style: Saturation  35 percent.
*/
#define FIG_FILL_035			0x07

/** Fill style: Saturation  40 percent.
*/
#define FIG_FILL_040			0x08

/** Fill style: Saturation  45 percent.
*/
#define FIG_FILL_045			0x09

/** Fill style: Saturation  50 percent.
*/
#define FIG_FILL_050			0x0A

/** Fill style: Saturation  55 percent.
*/
#define FIG_FILL_055			0x0B

/** Fill style: Saturation  60 percent.
*/
#define FIG_FILL_060			0x0C

/** Fill style: Saturation  65 percent.
*/
#define FIG_FILL_065			0x0D

/** Fill style: Saturation  70 percent.
*/
#define FIG_FILL_070			0x0E

/** Fill style: Saturation  75 percent.
*/
#define FIG_FILL_075			0x0F

/** Fill style: Saturation  80 percent.
*/
#define FIG_FILL_080			0x10

/** Fill style: Saturation  85 percent.
*/
#define FIG_FILL_085			0x11

/** Fill style: Saturation  90 percent.
*/
#define FIG_FILL_090			0x12

/** Fill style: Saturation  95 percent.
*/
#define FIG_FILL_095			0x13

/** Fill style: Fill with pure color.
*/
#define FIG_FILL_100			0x14

/** Fill style: Fill with pure color.
*/
#define FIG_FILL_PURE			0x14

/** Fill style: Saturation 105 percent.
*/
#define FIG_FILL_105			0x15

/** Fill style: Saturation 110 percent.
*/
#define FIG_FILL_110			0x16

/** Fill style: Saturation 115 percent.
*/
#define FIG_FILL_115			0x17

/** Fill style: Saturation 120 percent.
*/
#define FIG_FILL_120			0x18

/** Fill style: Saturation 125 percent.
*/
#define FIG_FILL_125			0x19

/** Fill style: Saturation 130 percent.
*/
#define FIG_FILL_130			0x1A

/** Fill style: Saturation 135 percent.
*/
#define FIG_FILL_135			0x1B

/** Fill style: Saturation 140 percent.
*/
#define FIG_FILL_140			0x1C

/** Fill style: Saturation 145 percent.
*/
#define FIG_FILL_145			0x1D

/** Fill style: Saturation 150 percent.
*/
#define FIG_FILL_150			0x1E

/** Fill style: Saturation 155 percent.
*/
#define FIG_FILL_155			0x1F

/** Fill style: Saturation 160 percent.
*/
#define FIG_FILL_160			0x20

/** Fill style: Saturation 165 percent.
*/
#define FIG_FILL_165			0x21

/** Fill style: Saturation 170 percent.
*/
#define FIG_FILL_170			0x22

/** Fill style: Saturation 175 percent.
*/
#define FIG_FILL_175			0x23

/** Fill style: Saturation 180 percent.
*/
#define FIG_FILL_180			0x24

/** Fill style: Saturation 185 percent.
*/
#define FIG_FILL_185			0x25

/** Fill style: Saturation 190 percent.
*/
#define FIG_FILL_190			0x26

/** Fill style: Saturation 195 percent.
*/
#define FIG_FILL_195			0x27

/** Fill style: White filled.
*/
#define FIG_FILL_200			0x28

/** Fill style: White filled.
*/
#define FIG_FILL_WHITE			0x28

/** Fill style: 30 degree lines to the left.
*/
#define FIG_FILL_LEFT_DIAGONAL_30	0x29

/** Fill style: 30 degree lines to the right.
*/
#define FIG_FILL_RIGHT_DIAGONAL_30	0x2A

/** Fill style: Diagonal (30 degree) crosshatch.
*/
#define FIG_FILL_CROSSHATCH_30		0x2B

/** Fill style: Diagonal lines to the left.
*/
#define FIG_FILL_LEFT_DIAGONAL_45	0x2C

/** Fill style: Diagonal lines to the right.
*/
#define FIG_FILL_RIGHT_DIAGONAL_45	0x2D

/** Fill style: Diagonal (45 degree) crosshatch.
*/
#define FIG_FILL_CROSSHATCH_45		0x2E

/** Fill style: Horizontal bricks.
*/
#define FIG_FILL_BRICKS_HORIZONTAL	0x2F

/** Fill style: Vertical bricks.
*/
#define FIG_FILL_BRICKS_VERTICAL	0x30

/** Fill style: Horizontal lines.
*/
#define FIG_FILL_HORIZONTAL_LINES	0x31

/** Fill style: Vertical lines.
*/
#define FIG_FILL_VERTICAL_LINES		0x32

/** Fill style: Crosshatch.
*/
#define FIG_FILL_CROSSHATCH		0x33

/** Fill style: Horizontal shingles, going to the right.
*/
#define FIG_FILL_HORIZONTAL_SHINGLES_RIGHT	0x34

/** Fill style: Horizontal shingles, going to the left.
*/
#define FIG_FILL_HORIZONTAL_SHINGLES_LEFT	0x35

/** Fill style: Vertical shingles, version 1.
*/
#define FIG_FILL_VERTICAL_SHINGLES_1		0x36

/** Fill style: Vertical shingles, version 2.
*/
#define FIG_FILL_VERTICAL_SHINGLES_2		0x37

/** Fill style: Large fish scales.
*/
#define FIG_FILL_LARGE_FISH_SCALES	0x38

/** Fill style: Small fish scales.
*/
#define FIG_FILL_SMALL_FISH_SCALES	0x39

/** Fill style: Circles.
*/
#define FIG_FILL_CIRCLES		0x3A

/** Fill style: Hexagons.
*/
#define FIG_FILL_HEXAGONS		0x3B

/** Fill style: Octagons.
*/
#define FIG_FILL_OCTAGONS		0x3C

/** Fill style: Horizontal tire treads.
*/
#define FIG_FILL_HORIZONTAL_TIRE_TREADS	0x3D

/** Fill style: Vertical tire treads.
*/
#define FIG_FILL_VERTICAL_TIRE_TREADS	0x3E



/** Line style: Default.
*/
#define FIG_LS_DEFAULT			0xFF

/** Line style: Solid.
*/
#define FIG_LS_SOLID			0x00

/** Line style: Dashed.
*/
#define FIG_LS_DASHED			0x01

/** Line style: Dots.
*/
#define FIG_LS_DOTTED			0x02

/** Line style: Dash, dot.
*/
#define FIG_LS_DASH_DOTTED		0x03

/** Line style: Dash, two dots.
*/
#define FIG_LS_DASH_DOUBLE_DOTTED	0x04

/** Line style: Dash, three dots.
*/
#define FIG_LS_DASH_TRIPLE_DOTTED	0x05



/** Line join style: Mitered.
*/
#define FIG_LJ_MITER			0x00

/** Line join style: Rounded.
*/
#define FIG_LJ_ROUND			0x01

/** Line join style: Beveled.
*/
#define FIG_LJ_BEVEL			0x02



/** Line cap style: Butted.
*/
#define FIG_CS_BUTT			0x00

/** Line cap style: Rounded.
*/
#define FIG_CS_ROUND			0x01

/** Line cap style: Projecting.
*/
#define FIG_CS_PROJECTING		0x02



/** Arrowhead shape: Stick.
*/
#define FIG_AS_STICK			0x00

/** Arrowhead shape: Triangle.
*/
#define FIG_AS_TRIANGLE			0x01

/** Arrowhead shape: Indented.
*/
#define FIG_AS_CLOSED_INDENTED		0x02

/** Arrowhead shape: Pointed.
*/
#define FIG_AS_CLOSED_POINTED		0x03



/** Arrowhead fill: White.
*/
#define FIG_AF_WHITE			0x00

/** Arrowhead fill: Use pen color.
*/
#define FIG_AF_PC			0x01



/** Text flag: Rigid text.
*/
#define FIG_TEXTFLAG_RIGID		0x01

/** Text flag: LaTeX special text.
*/
#define FIG_TEXTFLAG_SPECIAL		0x02

/** Text flag: Use PS font.
*/
#define FIG_TEXTFLAG_PS			0x04

/** Text flag: Hidden text.
*/
#define FIG_TEXTFLAG_HIDDEN		0x08


/** Font: LaTeX default font.
*/
#define FIG_FONT_LATEX_DEFAULT		0x00

/** Font: Roman (LaTeX).
*/
#define FIG_FONT_LATEX_ROMAN		0x01

/** Font: Bold (LaTeX).
*/
#define FIG_FONT_LATEX_BOLD		0x02

/** Font: Italic (LaTeX).
*/
#define FIG_FONT_LATEX_ITALIC		0x03

/** Font: Sans serif (LaTeX).
*/
#define FIG_FONT_LATEX_SANS_SERIF	0x04

/** Font: Typewriter (LaTeX).
*/
#define FIG_FONT_LATEX_TYPEWRITER	0x05



/** Font: Default PS font.
*/
#define FIG_FONT_PS_DEFAULT					0xFF

/** Font: Times (PS).
*/
#define FIG_FONT_PS_TIMES_ROMAN					0x00

/** Font: Times italic (PS).
*/
#define FIG_FONT_PS_TIMES_ITALIC				0x01

/** Font: Times bold (PS).
*/
#define FIG_FONT_PS_TIMES_BOLD					0x02

/** Font: Times bold italic (PS).
*/
#define FIG_FONT_PS_TIMES_BOLD_ITALIC				0x03

/** Font: Avantgarde Book (PS).
*/
#define FIG_FONT_PS_AVANTGARDE_BOOK				0x04

/** Font: Avantgarde Book oblique (PS).
*/
#define FIG_FONT_PS_AVANTGARDE_BOOK_OBLIQUE			0x05

/** Font: Avantgarde Book demi (PS).
*/
#define FIG_FONT_PS_AVANTGARDE_BOOK_DEMI			0x06

/** Font: Avantgarde Book demi oblique (PS).
*/
#define FIG_FONT_PS_AVANTGARDE_BOOK_DEMI_OBLICQUE		0x07

/** Font: Bookman light (PS).
*/
#define FIG_FONT_PS_BOOKMAN_LIGHT				0x08

/** Font: Bookman light italic (PS).
*/
#define FIG_FONT_PS_BOOKMAN_LIGHT_ITALIC			0x09

/** Font: Bookman demi (PS).
*/
#define FIG_FONT_PS_BOOKMAN_DEMI				0x0A

/** Font: Bookman demi italic (PS).
*/
#define FIG_FONT_PS_BOOKMAN_DEMI_ITALIC				0x0B

/** Font: Courier (PS).
*/
#define FIG_FONT_PS_COURIER					0x0C

/** Font: Courier oblique (PS).
*/
#define FIG_FONT_PS_COURIER_OBLIQUE				0x0D

/** Font: Courier bold (PS).
*/
#define FIG_FONT_PS_COURIER_BOLD				0x0E

/** Font: Courier bold oblique (PS).
*/
#define FIG_FONT_PS_COURIER_BOLD_OBLIQUE			0x0F

/** Font: Helvetica (PS).
*/
#define FIG_FONT_PS_HELVETICA					0x10

/** Font: Helvetica oblique (PS).
*/
#define FIG_FONT_PS_HELVETICA_OBLIQUE				0x11

/** Font: Helvetica bold (PS).
*/
#define FIG_FONT_PS_HELVETICA_BOLD				0x12

/** Font: Helvetica bold oblique (PS).
*/
#define FIG_FONT_PS_HELVETICA_BOLD_OBLIQUE			0x13

/** Font: Helvetica Narrow (PS).
*/
#define FIG_FONT_PS_HELVETICA_NARROW				0x14

/** Font: Helvetica Narrow oblique (PS).
*/
#define FIG_FONT_PS_HELVETICA_NARROW_OBLIQUE			0x15

/** Font: Helvetica Narrow bold (PS).
*/
#define FIG_FONT_PS_HELVETICA_NARROW_BOLD			0x16

/** Font: Helvetica Narrow bold oblique (PS).
*/
#define FIG_FONT_PS_HELVETICA_NARROW_BOLD_OBLIQUE		0x17

/** Font: New Century Schoolbook (PS).
*/
#define FIG_FONT_PS_NEW_CENTURY_SCHOOLBOOK_ROMAN		0x18

/** Font: New Century Schoolbook italic (PS).
*/
#define FIG_FONT_PS_NEW_CENTURY_SCHOOLBOOK_ITALIC		0x19

/** Font: New Century Schoolbook bold (PS).
*/
#define FIG_FONT_PS_NEW_CENTURY_SCHOOLBOOK_BOLD			0x1A

/** Font: New Century Schoolbook bold italic (PS).
*/
#define FIG_FONT_PS_NEW_CENTURY_SCHOOLBOOK_BOLD_ITALIC		0x1B

/** Font: Palatino roman (PS).
*/
#define FIG_FONT_PS_PALATINO_ROMAN				0x1C

/** Font: Palatino italic (PS).
*/
#define FIG_FONT_PS_PALATINO_ITALIC				0x1D

/** Font: Palatino bold (PS).
*/
#define FIG_FONT_PS_PALATINO_BOLD				0x1E

/** Font: Palatino bold italic (PS).
*/
#define FIG_FONT_PS_PALATINO_BOLD_ITALIC			0x1F

/** Font: Symbold (PS).
*/
#define FIG_FONT_PS_SYMBOL					0x20

/** Font: Chancery (PS).
*/
#define FIG_FONT_PS_ZAPF_CHANCERY_MEDIUM_ITALIC			0x21

/** Font: Zapf Dingbats (PS).
*/
#define FIG_FONT_PS_ZAPF_DINGBATS				0x22



/** Fig object type: Color definition.
*/
#define FIG_OBJECT_COLORCELL	0x00



/** Fig object type: Arc.
*/
#define FIG_OBJECT_ARC		0x05

/** Arc type: Open arc.
*/
#define FIG_SUB_ARC_OPEN	0x01

/** Arc type: Closed arc.
*/
#define FIG_SUB_ARC_CLOSED	0x02




/** Fig object type: Ellipse.
*/
#define FIG_OBJECT_ELLIPSE	0x01

/** Ellipse type: Ellipse specified by radii.
*/
#define FIG_SUB_ELLIPSE_RADII	0x01

/** Ellipse type: Ellipse specified by diameters.
*/
#define FIG_SUB_ELLIPSE_DIA	0x02

/** Ellipse type: Circle specified by radius.
*/
#define FIG_SUB_CIRCLE_RADIUS	0x03

/** Ellipse type: Circle specified by diameter.
*/
#define FIG_SUB_CIRCLE_DIA	0x04




/** Fig object type: Polyline/polygon/box/image.
*/
#define FIG_OBJECT_POLYLINE	0x02

/** Polyline type: Polyline.
*/
#define FIG_SUB_POLYLINE	0x01

/** Polyline type: Box (rectangle).
*/
#define FIG_SUB_BOX		0x02

/** Polyline type: Polygon.
*/
#define FIG_SUB_POLYGON		0x03

/** Polyline type: Arc box.
*/
#define FIG_SUB_ARC_BOX		0x04

/** Polyline type: Included image.
*/
#define FIG_SUB_IMAGE		0x05




/** Fig object type: Spline.
*/
#define FIG_OBJECT_SPLINE	0x03

/** Spline type: Open approximated spline.
*/
#define FIG_SUB_SPLINE_OPEN_APPROXIMATED	0x00

/** Spline type: Closed approximated spline.
*/
#define FIG_SUB_SPLINE_CLOSED_APPROXIMATED	0x01

/** Spline type: Open interpolated spline.
*/
#define FIG_SUB_SPLINE_OPEN_INTERPOLATED	0x02

/** Spline type: Closed interpolated spline.
*/
#define FIG_SUB_SPLINE_CLOSED_INTERPOLATED	0x03

/** Spline type: Open X spline.
*/
#define FIG_SUB_SPLINE_OPEN_X			0x04

/** Spline type: Closed X spline.
*/
#define FIG_SUB_SPLINE_CLOSED_X			0x05



/** Fig object type: text.
*/
#define FIG_OBJECT_TEXT		0x04

/** Text align: Left.
*/
#define FIG_SUB_TEXT_LEFT	0x00

/** Text align: Centered.
*/
#define FIG_SUB_TEXT_CENTER	0x01

/** Text align: Right.
*/
#define FIG_SUB_TEXT_RIGHT	0x02



/** Orientation: Landdscape.
*/
#define FIG_ORIENTATION_LANDSCAPE	0x00

/** Orientation: Portrait.
*/
#define FIG_ORIENTATION_PORTRAIT	0x01



/** Units: Metric.
*/
#define FIG_UNITS_METRIC		0x00

/** Units: Inches.
*/
#define FIG_UNITS_INCHES		0x01



/** Paper size: Letter.
*/
#define FIG_PAPER_LETTER		0x00

/** Paper size: Legal.
*/
#define FIG_PAPER_LEGAL			0x01

/** Paper size: Ledger.
*/
#define FIG_PAPER_LEDGER		0x02

/** Paper size: Tabloid.
*/
#define FIG_PAPER_TABLOID		0x03

/** Paper size: A.
*/
#define FIG_PAPER_A			0x04

/** Paper size: B.
*/
#define FIG_PAPER_B			0x05

/** Paper size: C.
*/
#define FIG_PAPER_C			0x06

/** Paper size: D.
*/
#define FIG_PAPER_D			0x07

/** Paper size: E.
*/
#define FIG_PAPER_E			0x08

/** Paper size: A4.
*/
#define FIG_PAPER_A4			0x09

/** Paper size: A3.
*/
#define FIG_PAPER_A3			0x0A

/** Paper size: A2.
*/
#define FIG_PAPER_A2			0x0B

/** Paper size: A1.
*/
#define FIG_PAPER_A1			0x0C

/** Paper size: A0.
*/
#define FIG_PAPER_A0			0x0D

/** Paper size: B5.
*/
#define FIG_PAPER_B5			0x0E



/** Single page graphics.
*/
#define FIG_PAGES_SINGLE		0x00

/** Multipage graphics.
*/
#define FIG_PAGES_MULTIPLE		0x01



/** Image justification: Centered.
*/
#define FIG_JUSTIFICATION_CENTER	0x00

/** Image justification: Flush left.
*/
#define FIG_JUSTIFICATION_LEFT		0x01



/** Width and height units: Fig units.
*/
#define FIG_WH_FIG			0

/** Width and height units: inches.
*/
#define FIG_WH_INCHES			1

/** Width and height units: cm.
*/
#define FIG_WH_CM			2



#endif
/* ifndef DKFIGW_H_INCLUDED */


