/*
Copyright (c) 2004-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file dkle.h Encoding changes for use with LaTeX
	This module converts 32-bit characters into the appropriate
	LaTeX sequences to show the glyphs.

	A dk_le_t structure is needed for the conversions, this structure
	can be created using dkle_open(). After usage release the memory
	by calling dkle_close().
	Conversion tables can be loaded using the dkle_load() function.
	The dkle_get_encoding() function can be used to get a string
	containing the LaTeX sequence for a character.

	Unless otherwise stated, int functions in this module return a positive
	number to indicate success or a true condition, 0 to indicate an error
	or an unfullfilled condition.
	Pointer functions return valid pointers on success, NULL on error.

	See
	<a href="../../latex.html">"dklibs - LaTeX encodings"</a>
	for details about encoding table files.
*/

#ifndef DK_LATSUP_INC
#define DK_LATSUP_INC 1

#include <dk.h>
#include <dktypes.h>


#if defined(EXTERN)
#undef EXTERN
#endif
#ifndef DK_LATSUP_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif

#if defined(__cplusplus)
extern "C" {
#endif



/**	Create encoding structure.
	The encoding structure is created in dynamically allocated memory.
	Use dkle_open() to destory the structure and release the memory
	when the structure is no longer needed.
	@param	d	A base directory containing the data files.
	@return	Pointer to encoding structure on success, NULL on error.
*/
EXTERN dk_le_t *
dkle_open DK_PR((char *d));



/**	Attempt to load table for character \a c from file (if not yet
	loaded).
	@param	l	Pointer to encoding structure.
	@param	c	The 32 bit character to convert.
	@return	Flag to indicate the table is loaded.
*/
EXTERN int
dkle_load DK_PR((dk_le_t *l, dk_udword c));



/**	Get encoding for character (LaTeX sequence).
	@param	l	Pointer to encoding structure.
	@param	c	32 bit character to convert.
	@param	m	Flag to indicate math mode (1) or text mode (0).
	@return	Pointer to LaTex sequence string on success, NULL on error.
*/
EXTERN char *
dkle_get_encoding DK_PR((dk_le_t *l, dk_udword c, int m));



/**	Retrieve (and optionally reset) the error code for the last
	error occured in the encoding structure.
	@param	l	Pointer to encoding structure.
	@param	r	Flag to reset error code.
	@return	Key to indicate type of last error.
*/
EXTERN int
dkle_get_error_code DK_PR((dk_le_t *l, int r));



/**	Retrieve file name in which the last error occured.
	@param	l	Pointer to encoding structure.
	@return	File name of last error or NULL.
*/
EXTERN char *
dkle_get_filename DK_PR((dk_le_t *l));



/**	Retrieve line number in which the last error occured.
	@param	l	Pointer to encoding structure.
	@return	Line number.
*/
EXTERN unsigned long
dkle_get_error_lineno DK_PR((dk_le_t *l));



/**	Find necessary result buffer length to convert plain text string
	into LaTeX sequence.
	@param	l	Pointer to encoding structure.
	@param	s	Plain text string.
	@return	The length needed for the result buffer (you must
		add 1 for the final 0x00 byte). A result of 0 indicates
		an error.
*/
EXTERN size_t
dkle_length_plain_to_latex DK_PR((dk_le_t *l, char *s));



/**	Convert plain text string into LaTeX sequence.
	@param	l	Pointer to encoding structure.
	@param	d	Destination buffer to store result.
	@param	s	Length of \a d in bytes.
	@param	p	Plaint text string.
	@result	Flag to indicate success.
*/
EXTERN int
dkle_put_plain_to_latex DK_PR((dk_le_t *l, char *d, size_t s, char *p));



/**	Find necessary result buffer length to convert UTF-8 encoded text string
	into LaTeX sequence.
	@param	l	Pointer to encoding structure.
	@param	u	UTF-8 encoded text string.
	@return	The length needed for the result buffer (you must
		add 1 for the final 0x00 byte). A result of 0 indicates
		an error.
*/
EXTERN size_t
dkle_length_utf8_to_latex DK_PR((dk_le_t *l, char *u));



/**	Convert UTF-8 encoded text string into LaTeX sequence.
	@param	l	Pointer to encoding structure.
	@param	d	Destination buffer to store result.
	@param	s	Length of \a d in bytes.
	@param	u	UTF-8 encoded text string.
	@result	Flag to indicate success.
*/
EXTERN int
dkle_put_utf8_to_latex DK_PR((dk_le_t *l, char *d, size_t s, char *u));



/**	Destroy encoding structure obtained from dkle_open() and
	release the memory.
	@param	l	Pointer to encoding structure.
*/
EXTERN void
dkle_close DK_PR((dk_le_t *l));

EXTERN void
dkle_pkg_reset DK_PR((dk_le_t *le));

EXTERN dk_le_pkg_t *
dkle_pkg_next DK_PR((dk_le_t *le));

#if defined(__cplusplus)
}
#endif

#define DK_LE_FLAGS_UTF8	1

#endif


