/*
Copyright (c) 2007-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file	dksdbii.h	DK SDBI internals.
The int functions in this module return a positive value on success,
0 on error unless otherwise noted.
*/



#ifndef DK_SDBII_H_INCLUDED
#define DK_SDBII_H_INCLUDED	1

#include <dk.h>

#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if DK_HAVE_ALLOC_H
#include <alloc.h>
#endif
#if DK_HAVE_MALLOC_H
#include <malloc.h>
#endif


/**	Database access API structure.
 */
typedef struct {
  char	*buffer;	/**< File name buffer. */
  char	*fn;		/**< File name, part of buffer. */
  int	tp;		/**< Database type:  BDB, NDBM or GDBM. */
  int	acc;		/**< Access type:  READ, WRITE, RDWR. */
  int	fm;		/**< File permissions. */
  int	did_truncate;	/**< Flag: file was already truncated. */
  mode_t	tfm;	/**< Translated file permissions. */
  int	blksize;	/**< Block size (for GDBM). */
  void	*dbptr;		/**< Pointer to the real database. */
} dk_sdbi_db;



/*
	BDB module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if DKSDBI_BDB
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if DK_HAVE_DB_H
#if defined(__cplusplus)
extern "C" {
#endif

/**	Close Berkely DB database.
 * 	@param	p	SDBI API structure.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_bdb_close DK_PR((dk_sdbi_db *p));

/**	Open Berekeley DB database.
 * 	@param	p	SDBI API structure.
 * 	@return Success flag.
 */
EXTERN int
dksdbi_bdb_open DK_PR((dk_sdbi_db *p));

/**	Store a value in a Berkeley DB database.
 * 	@param	p	SDBI API structure.
 * 	@param	kp	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@param 	vp	Value pointer.
 * 	@param	vl	Value length (bytes).
 * 	@param	insmod	0 (replace allowed) or DK_SDBI_INSMOD_NO_REPLACE (do not eplace existing entries.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_bdb_store DK_PR((dk_sdbi_db *p, void *kp, size_t kl, void *vp, size_t vl, int insmod));

/**	Remove entry from Berkeley database.
 * 	@param	p	SDBI API structure.
 * 	@param	k	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_bdb_delete DK_PR((dk_sdbi_db *p, char *k, size_t kl));

/**	Fetch value from Berkeley database.
 * 	@param	p	SDBI API structure.
 * 	@param	kp	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@param	vp	Value buffer pointer.
 * 	@param	vl	In: bytes available in buffer, out: bytes used.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_bdb_fetch DK_PR((dk_sdbi_db *p, void *kp, size_t kl, void *vp, size_t *vl));

/**	Fetch string from Berekeley database.
 * 	@param	p	SDBI API structure.
 * 	@param	k	Key string.
 * 	@param	v	Value buffer pointer.
 *	@param	s	Size of \a v.
 *	@return Success flag.
 */
EXTERN int
dksdbi_bdb_string_fetch DK_PR((dk_sdbi_db *p, char *k, char *v, size_t s));

/**	Traverse Berkeley database.
 * 	@param	p	SDBI API structure.
 * 	@param	d	Traversal support record.
 * 	@param	f	Traversal function (invoked for each entry).
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_bdb_traverse DK_PR((dk_sdbi_db *p, void *d, dk_sdbi_fct_t *f));

/**	Synchronize Berkeley database to disk.
 * 	@param	p	SDBI API structure.
 * 	@return Success flag.
 */
EXTERN int
dksdbi_bdb_sync DK_PR((dk_sdbi_db *p));
#if defined(__cplusplus)
}
#endif
#endif



/*
	NDBM module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if DKSDBI_NDBM
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if DK_HAVE_NDBM_H
#if defined(__cplusplus)
extern "C" {
#endif

/**	Close NDBM database.
 * 	@param	p	SDBI API structure.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_ndbm_close DK_PR((dk_sdbi_db *p));

/**	Open NDBM database.
 * 	@param	p	SDBI API structure.
 * 	@return Success flag.
 */
EXTERN int
dksdbi_ndbm_open DK_PR((dk_sdbi_db *p));

/**	Store entry to NDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	kp	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@param 	vp	Value pointer.
 * 	@param	vl	Value length (bytes).
 * 	@param	insmod	0 (replace allowed) or DK_SDBI_INSMOD_NO_REPLACE (do not eplace existing entries.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_ndbm_store DK_PR((dk_sdbi_db *p, void *kp, size_t kl, void *vp, size_t vl, int insmod));

/**	Delete entry from NDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	k	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_ndbm_delete DK_PR((dk_sdbi_db *p, char *k, size_t kl));

/**	Fetch entry from NDBM database
 * 	@param	p	SDBI API structure.
 * 	@param	kp	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@param	vp	Value buffer pointer.
 * 	@param	vl	In: bytes available in buffer, out: bytes used.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_ndbm_fetch DK_PR((dk_sdbi_db *p, void *kp, size_t kl, void *vp, size_t *vl));

/**	Fetch string from NDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	k	Key string.
 * 	@param	v	Value buffer pointer.
 *	@param	s	Size of \a v.
 *	@return Success flag.
 */
EXTERN int
dksdbi_ndbm_string_fetch DK_PR((dk_sdbi_db *p, char *k, char *v, size_t s));

/**	Traverse NDBM database
 * 	@param	p	SDBI API structure.
 * 	@param	d	Traversal support record.
 * 	@param	f	Traversal function (invoked for each entry).
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_ndbm_traverse DK_PR((dk_sdbi_db *p, void *d, dk_sdbi_fct_t *f));

/**	Synchronize NDBM database to disk.
 * 	@param	p	SDBI API structure.
 * 	@return Success flag.
 */
EXTERN int
dksdbi_ndbm_sync DK_PR((dk_sdbi_db *p));
#if defined(__cplusplus)
}
#endif
#endif



/*
	GDBM module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if DKSDBI_GDBM
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if DK_HAVE_GDBM_H
#if defined(__cplusplus)
extern "C" {
#endif

/**	Close GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_gdbm_close DK_PR((dk_sdbi_db *p));

/**	Open GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@return Success flag.
 */
EXTERN int
dksdbi_gdbm_open DK_PR((dk_sdbi_db *p));

/**	Store a value in a GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	kp	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@param 	vp	Value pointer.
 * 	@param	vl	Value length (bytes).
 * 	@param	insmod	0 (replace allowed) or DK_SDBI_INSMOD_NO_REPLACE (do not eplace existing entries.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_gdbm_store DK_PR((dk_sdbi_db *p, void *kp, size_t kl, void *vp, size_t vl, int insmod));

/**	Remove entry from GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	k	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_gdbm_delete DK_PR((dk_sdbi_db *p, char *k, size_t kl));

/**	Fetch value from GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	kp	Key pointer.
 * 	@param	kl	Key length (bytes).
 * 	@param	vp	Value buffer pointer.
 * 	@param	vl	In: bytes available in buffer, out: bytes used.
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_gdbm_fetch DK_PR((dk_sdbi_db *p, void *kp, size_t kl, void *vp, size_t *vl));

/**	Fetch string from GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	k	Key string.
 * 	@param	v	Value buffer pointer.
 *	@param	s	Size of \a v.
 *	@return Success flag.
 */
EXTERN int
dksdbi_gdbm_string_fetch DK_PR((dk_sdbi_db *p, char *k, char *v, size_t s));

/**	Traverse GDBM database.
 * 	@param	p	SDBI API structure.
 * 	@param	d	Traversal support record.
 * 	@param	f	Traversal function (invoked for each entry).
 * 	@return	Success flag.
 */
EXTERN int
dksdbi_gdbm_traverse DK_PR((dk_sdbi_db *p, void *d, dk_sdbi_fct_t *f));

/**	Synchronize GDBM database to disk.
 * 	@param	p	SDBI API structure.
 * 	@return Success flag.
 */
EXTERN int
dksdbi_gdbm_sync DK_PR((dk_sdbi_db *p));
#if defined(__cplusplus)
}
#endif
#endif

#endif

