/*
Copyright (c) 2000-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file dksignal.h
	Portable signal handling.
	This module provides data types and functions for signal
	handling.
	There are at least 3 signal handling function groups
	on Unix/Linux systems:
	- sigaction() - the preferred mechanism,
	- sigset() and
	- signal().

	The code example below shows how to use the functions in this module:
	\code
	volatile int sigint_received = 0;

	dk_signal_ret_t
	sigint_handler(int signo) {
	  dksignal_refresh(signo, sigint_handler);
	  sigint_received = 1;
	  dksignal_return(0);
	}

	int main(int argc, char *argv[]) {
	  dk_signal_disp_t	disp_int = NULL;
	  switch(dksignal_available()) {
	    case DK_SIGNAL_NONE: {
	      printf("Warning: No signal handling available.\n");
	    } break;
	    case DK_SIGNAL_SIGNAL: {
	      printf("Warning: Signal handling uses signal() function.\n");
	    } break;
	  }
	  disp_int = dksignal_set(SIGINT, sigint_handler);
	  ... do something taking a while ...
	  if(disp_int) dksignal_set(SIGINT, disp_int);
	}
	\endcode
*/

#ifndef DK_SIGNAL_INCLUDED
#define DK_SIGNAL_INCLUDED 1

#include <dk.h>
#include <dktypes.h>

/** dksignal_available() result: sigaction() available. */
#define DK_SIGNAL_SIGACTION	3

/** dksignal_available() result: sigset() available. */
#define DK_SIGNAL_TYPE_SIGSET	2

/** dksignal_available() result: signal() available. */
#define DK_SIGNAL_SIGNAL	1

/** dksignal_available() result: No signal handling available. */
#define DK_SIGNAL_NONE		0

#if defined(EXTERN)
#undef EXTERN
#endif
#ifndef DK_SIGNAL_C
#if !DK_HAVE_PROTOTYPES
#define EXTERN extern
#else
#define EXTERN /* nix */
#endif
#else
#define EXTERN /* nix */
#endif



/** Refresh signal handler.
    The first instruction of a signal handler should be
    dksignal_refresh(signo, handler_fct).
*/
#if DK_HAVE_SIGACTION
#define dksignal_refresh(i,disp)	/* nix */
#else
#if DK_HAVE_SIGSET
#define dksignal_refresh(i,disp)        /* nix */
#else
#if DK_HAVE_SIGNAL
#define dksignal_refresh(i,disp)        signal(i,disp)
#else
#define dksignal_refresh(i,disp)        /* nix */
#endif
#endif
#endif



/** Return from signal handler.
    The last instruction of a signal handler should be
    dksignal_return(0).
*/
#if DK_HAVE_SIGACTION
#define dksignal_return(i)		/* nix */
#else
#if DK_HAVE_SIGSET
#define dksignal_return(i)              /* nix */
#else
#if DK_HAVE_SIGNAL
#define dksignal_return(i)              /* nix */
#else
#define dksignal_return(i)              /* nix */
#endif
#endif
#endif



#if defined(__cplusplus)
extern "C" {
#endif



/**	Set signal handler.
	The function assigns the handler \a disp to signal \a signo
	and returns a pointer to the previous handler.
*/
EXTERN
dk_signal_disp_t dksignal_set DK_PR((int signo, dk_signal_disp_t disp));



/**
	Check available signal handling mechanisms.
	@return	Key number to indicate used signal handling mechanism:
	- DK_SIGNAL_SIGACTION if sigaction() is used,
	- DK_SIGNAL_SIGSET if sigset() is used,
	- DK_SIGNAL_SIGNAL if signal() is used or
	- DK_SIGNAL_NONE if no signal handling mechanism is available.
*/
EXTERN
int		 dksignal_available DK_PR((void));



#if defined(__cplusplus)
}
#endif
#endif

