/*
Copyright (c) 2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/


/**	@file	fsnmplog.c	The fsnmplog module in the fsnmp program.
*/



/**	Inside the fsnmplog.c module.
*/
#define FSNMPLOG_C	1
#include "fsnmp.h"


#line 49 "fsnmplog.ctr"




/**	Keyword for output filter.
*/
static char kw_of[] = { "OF: " };



/**	Keywords for log entry priority.
*/
static char *kw_prio[] = {
  "",
  "FATAL: ",
  "ERROR: ",
  "Warning: ",
};



/**	Write timestamp to log file.
	@param	fc	Fsnmp job
	@param	f	Log file.
*/
static
void
log_time DK_P2(FC *,fc, FILE *,f)
{
  time_t ct;
  struct tm *tm;
  time(&ct);
  if(ct > fc->last_timestamp) {
    tm = localtime(&ct);
    if(tm) {
      fc->last_timestamp = ct;
      fprintf(
        f,
	"# %04d-%02d-%02d %02d:%02d:%02d\n",
	(1900 + tm->tm_year),
	(1 + tm->tm_mon),
	tm->tm_mday,
	tm->tm_hour,
	tm->tm_min,
	tm->tm_sec
      );
    }
  }
}



/**	Log a message.
	@param	fc	Fsnmp job.
	@param	prio	Log priority.
	@param	format	Format string.
*/
#if DK_HAVE_PROTOTYPES
void fsnmplog(FC *fc, int prio, char *format, ...)
#else
void fsnmplog(fc, prio, format, ...) FC *fc; int prio; char *format;
#endif
{
  FILE *f = NULL;
  int must_close_file = 0;
  va_list arg_list;
  /* open file or choose stderr */
  if((prio < PRIO_DEBUG) || ((fc->flags) & FSNMP_FLAG_DEBUG)) {
#if FORCE_OF_LOGGING_TO_STDERR
    if((fc->flags) & FSNMP_FLAG_OF) {
      f = stderr;
    } else {
#endif
      if(fsnmpcmd_get_argv(fc, 's')) {
        
        
        f = dksf_fopen(
	  fsnmpcmd_get_argv(fc, 's'), 
	  ((fc->flags) & FSNMP_FLAG_HAVE_LOG) ? "a" : "w"
	);
	if(f) {
	  must_close_file = 1;
	  fc->flags |= FSNMP_FLAG_HAVE_LOG;
	} else {
	  f = stderr;
	}
      } else {
        f = stderr;
      }
#if FORCE_OF_LOGGING_TO_STDERR
    }
#endif
  }

  /* write to file */
  if(f) {
    log_time(fc, f);
    if((fc->flags) & FSNMP_FLAG_OF) {
      fputs(kw_of, f);
    }
    if(prio <= PRIO_WARNING) {
      fputs(kw_prio[prio], f);
    }
    va_start(arg_list, format);
    vfprintf(f, format, arg_list);
    va_end(arg_list);
    fputc('\n', f);
    fflush(f);
  }

  /* close file if necessary */
  if(f) { if(must_close_file) { fclose(f); } } f = NULL;
}



/**	Log one printer response.
	@param	fc	Fsnmp job.
	@param	bytes	Number of bytes returned by printer.
*/
void
fsnmplog_response DK_P2(FC *,fc, int,bytes)
{
  FILE *f = NULL;
  int must_close_file = 0;
  int i;
  /* open file or choose stderr */
  if(fsnmpcmd_get_argv(fc, 's')) {
    f = dksf_fopen(
      fsnmpcmd_get_argv(fc, 's'),
      ((fc->flags) & FSNMP_FLAG_HAVE_LOG) ? "a" : "w"
    );
    if(f) {
      must_close_file = 1;
      fc->flags |= FSNMP_FLAG_HAVE_LOG;
    } else {
      f = stderr;
    }
  } else {
    f = stderr;
  }

  /* write to file */
  if(f) {
    log_time(fc, f);
  }
  fputs(fsnmp_kw[1], f);
  for(i = 0; i < bytes; i++) {
    fputc((fc->buffer_in)[i], f);
  }
  if((fc->buffer_in)[bytes - 1] != '\n') {
    fputc('\n', f);
  }
  
  /* close file */
  if(f) { if(must_close_file) { fclose(f); } } f = NULL;
}



