/*
Copyright (c) 2008-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/** @file prqdpj.c	Module for PJ related functions. */


/**	Inside the prqdpj module. */
#define PRQDPJ_C	1
#include "prqd.h"

#line 46 "prqdpj.ctr"




/** Possible responses if a print job is denied. */
static char *deny_actions[] = {
  "REMOVE\n", "HOLD\n", "FAIL\n"
};



/** Array of request types. */
static char *request_types[] = {
  "jobstart",
  "filestart",
  "start",
  "end",
  "fileend",
  "jobend",
  "info",
  "control",
  NULL
};



/**	Initialize a PJ structure.
	@param	pj	Pointer to prqd job structure.
*/
void
prqd_pj_init DK_P1(PJ *,pj)
{
  pj->ec = 0;
#if DEBUG
  pj->deb = 1;
#else
  pj->deb = 0;
#endif
  pj->last_log_time = (time_t)0UL;
  pj->prqdc = NULL;
  pj->e1 = 0; pj->e2 = 0;
  pj->backlog = 5;
  (pj->dbbe).tp = DATABASE_TYPE_UNKNOWN;
  pj->args = NULL;
  pj->rqt = 0;
  pj->rqtstr = NULL;
  pj->b1 = NULL; pj->b2 = NULL; pj->b3 = NULL;
  pj->sz_b1 = 0; pj->sz_b2 = 0; pj->sz_b3 = 0;
  pj->dbkey = NULL; pj->sz_dbkey = 0;
  pj->dbval = NULL; pj->sz_dbval = 0;
  pj->uname = NULL; pj->pname = NULL; pj->rname = NULL;
  pj->tname = NULL; pj->pages = NULL; pj->cname = NULL;
  pj->hname = NULL;
  pj->limit_type = 0;
  pj->limit_pages = 0UL;
  pj->cr_cname = NULL; pj->cr_uname = NULL; pj->cr_pages = NULL;
  pj->cr_pname = NULL; pj->cr_deltap = 0L;
  pj->cr_st = NULL;
  pj->cr_sti = NULL;
  pj->allow_file_logging = 0;
}



/**	Retrieve value of command line argument.
	If -- i.e. one command line argument was -Pmyprinter --
	an attempt to retrieve the value for command line argument
	'P' will return ``myprinter''.
	@param	pj	Pointer to print job structure.
	@param	c	Command line argument to find value for.
	@return	String pointer or NULL.
*/
char *
prqd_pj_get_arg DK_P2(PJ *,pj, char,c)
{
  char *back = NULL;
  
  if(pj) {
    if((c >= 'a') && (c <= 'z')) {
      back = (pj->args)[c - 'a'];
    } else {
      if((c >= 'A') && (c <= 'Z')) {
        back = (pj->args)[c - 'A' + 26];
      }
    }
  } 
  return back;
}



/**	Set command line argument value.
	@param	pj	Pointer to print job structure.
	@param	c	Command line argument.
	@param	s	Value assigned to command line argument.
*/
void
prqd_pj_set_arg DK_P3(PJ *,pj, char,c, char *,s)
{
  
  if(pj) {
    if((c >= 'a') && (c <= 'z')) {
      (pj->args)[c - 'a'] = s;
    } else {
      if((c >= 'A') && (c <= 'Z')) {
        (pj->args)[c - 'A' + 26] = s;
      }
    }
  } 
}



/**	Retrieve all command line argument values from
	an input line and store them in the PJ.
	@param	pj	Pointer to prqd job structure.
	@param	il	Input line to process.
*/
void
prqd_pj_set_all_args DK_P2(PJ *,pj, char *,il)
{
  char *p1, *p2;
  p2 = il;
  while(p2) {
    p1 = dkstr_chr(p2, '\'');
    if(p1) {
      p1++;
      p2 = dkstr_chr(p1, '\'');
      if(p2) {
        *(p2++) = '\0';
	if(*p1 == '-') {
	  p1++;
	  if(*p1) {
	    prqd_pj_set_arg(pj, *p1, &(p1[1]));
	  }
	}
      }
    } else {
      p2 = NULL;
    }
  }
}



/**	Find request type.
	@param	pj	Pointer to print job structure.
	@param	s	First word in an input line.
*/
void
prqd_pj_get_rq_type DK_P2(PJ *,pj, char *,s)
{
  pj->rqt = dkstr_array_index(request_types, s, 0);
  
}



/**	Check whether a character c is an acceptable
	character in names.
	@param	c	The character to check.
	@return	1 on success, 0 on error.
*/
static
int
is_acceptable_character DK_P1(char,c)
{
  int back = 0;
  if((c >= 'a') && (c <= 'z')) {
    back = 1;
  } else {
    if((c >= 'A') && (c <= 'Z')) {
      back = 1;
    } else {
      if((c >= '1') && (c <= '9')) {
        back = 1;
      } else {
        switch(c) {
	  case '0':
          case '.': case '_': case '-':
	  case ' ': case '\t':
	  back = 1; break;
        }
      }
    }
  }
  return back;
}



/**	Check whether a name consists of acceptable
	characters only.
	@param	s	String containing the name.
	@return	1 on success, 0 on error.
*/
int
prqd_pj_is_acceptable_name DK_P1(char *,s)
{
  int back = 0;
  char *ptr;
  
  if(s) {
    back = 1;
    ptr = s;
    while(*ptr) {
      if(!is_acceptable_character(*ptr)) {
        back = 0;
      }
      ptr++;
    }
  } 
  return back;
}



/**	Copy deny action to response buffer.
	@param	pj	Pointer to prqd job structure.
*/
void
prqd_pj_use_deny_action DK_P1(PJ *,pj)
{
  PR *pr; PC *pc;
  int act = DENY_ACTION_REMOVE;
  char *pn = NULL;
  
  if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'P'))) {
    pn = prqd_pj_get_arg(pj, 'P');
  }
  if(!(pn)) {
    if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'Q'))) {
      pn = prqd_pj_get_arg(pj, 'Q');
    }
  }
  if(pn) {
    pr = dksto_it_find_like((pj->prqdc)->pri, pn, 1);
    if(pr) {
      while(pr->alt) { pr = pr->alt; }
      if(pr->pc) {
        pc = pr->pc;
	act = pc->da;
      }
    }
  }
  switch(act) {
    case DENY_ACTION_HOLD : {
      strcpy(pj->b2, deny_actions[1]);
    } break;
    case DENY_ACTION_FAIL : {
      strcpy(pj->b2, deny_actions[2]);
    } break;
    default: {
      strcpy(pj->b2, deny_actions[0]);
    } break;
  } 
}



/**	Check command line arguments for information.
	@param	pj	Pointer to prqd job structure.
*/
void
prqd_pj_get_names DK_P1(PJ *,pj)
{
  PR *pr;
  pj->uname = NULL;
  pj->pname = NULL;
  pj->tname = NULL;
  pj->pages = NULL;
  pj->cname = NULL;
  pj->rname = NULL;
  pj->hname = NULL;
  
  /* uname */
  if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'n'))) {
    pj->uname = prqd_pj_get_arg(pj, 'n');
  }
  if(!(pj->uname)) {
    if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'L'))) {
      pj->uname = prqd_pj_get_arg(pj, 'L');
    }
  }
  /* pname */
  if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'P'))) {
    pj->pname = prqd_pj_get_arg(pj, 'P');
  }
  if(!(pj->pname)) {
    if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'Q'))) {
      pj->pname = prqd_pj_get_arg(pj, 'P');
    }
  }
  /* hname */
  if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'h'))) {
    pj->hname = prqd_pj_get_arg(pj, 'h');
  }
  if(!(pj->hname)) {
    if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'H'))) {
      pj->hname = prqd_pj_get_arg(pj, 'H');
    }
  }
  /* tname */
  pj->tname = prqd_pj_get_arg(pj, 'J');
  if(!(pj->tname)) {
    pj->tname = prqd_pj_get_arg(pj, 'f');
  }
  if(!(pj->tname)) {
    pj->tname = prqd_pj_get_arg(pj, 'N');
  }
  /* pages */
  if(prqd_pj_is_acceptable_name(prqd_pj_get_arg(pj, 'p'))) {
    pj->pages = prqd_pj_get_arg(pj, 'p');
  }
  /* rname and cname */
  if(pj->pname) {
    pj->rname = pj->pname;
    pr = dksto_it_find_like((pj->prqdc)->pri, pj->pname, 1);
    if(pr) {
      while(pr->alt) { pr = pr->alt; }
      pj->rname = pr->n;
      if(pr->pc) pj->cname = (pr->pc)->n;
    }
  }
  
  
  
  
  
  
  
}



/**	Find limit for current user/printer combination.
	@param	pj	Pointer to prqd job structure.
*/
void
prqd_pj_find_limit DK_P1(PJ *,pj)
{
  PR *pr; PC *pc; LE *le;
  pj->limit_type = LIMIT_VALUE_DENIED;
  pj->limit_pages = 0UL;
  struct passwd *pw;
  struct group *gr;
  gid_t g;
  uid_t u;
  int is_bigger;
  int use_this_limit;
  char **lfdptr;
  
  if((pj->uname) && (pj->pname)) {
    (pj->b3)[0] = '\0';
    pw = getpwnam(pj->uname); gr = NULL;
    if(pw) {
      u = pw->pw_uid;
      g = pw->pw_gid;
      gr = getgrgid(g);
      if(gr) {
        if(gr->gr_name) {
	  if(strlen(gr->gr_name) < pj->sz_b3) {
	    strcpy(pj->b3, gr->gr_name);
	  }
	}
      }
      pr = dksto_it_find_like((pj->prqdc)->pri, pj->pname, 1);
      if(pr) {
        while(pr->alt) { pr = pr->alt; }
        pc = pr->pc;
        if(pc) {
          dksto_it_reset(pc->li);
	  while((le = (LE *)dksto_it_next(pc->li)) != NULL) {
	    
	    is_bigger = 0;
	    if(le->val > pj->limit_type) { is_bigger = 1; }
	    if((le->val == LIMIT_VALUE_PAGES) && (pj->limit_type == LIMIT_VALUE_PAGES)) {
	      if(le->p > pj->limit_pages) {
	        is_bigger = 1;
	      }
	    }
	    if(is_bigger) {
	      
	      use_this_limit = 0;
	      switch(le->tp) {
	        case LIMIT_ENTRY_TYPE_DEFAULT: {
		  use_this_limit = 1;	
		} break;
		case LIMIT_ENTRY_TYPE_GROUP: {
		  if(le->who) {
		    if(strcmp(le->who, pj->b3) == 0) {
		      use_this_limit = 1; 
		    } else {
		      gr = getgrnam(le->who);
		      if(gr) {
		        if(gr->gr_mem) {
			  lfdptr = gr->gr_mem;
			  while((*lfdptr) && (use_this_limit == 0)) {
			    if(strcmp(pj->uname, *lfdptr) == 0) {
			      use_this_limit = 1; 
			    }
			    lfdptr++;
			  }
			}
		      }
		    }
		  }
		} break;
		case LIMIT_ENTRY_TYPE_USER: {
		  if(le->who) {
		    if(strcmp(le->who, pj->uname) == 0) {
		      use_this_limit = 1; 
		    }
		  }
		} break;
	      }
	      if(use_this_limit) { 
	        pj->limit_type = le->val;
		pj->limit_pages = le->p;
	      }
	    }
	  }
        }
      }
    } else {			
      prqdlog(pj, PRQD_PRIO_ERROR, prqd_get_kw(45), pj->uname);
    }
  } 
}




