/*
Copyright (c) 2000-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/


#ifndef BMEPSI_H_INCLUDED
#define BMEPSI_H_INCLUDED 1

/**	@file	bmepsi.h	Internal data types and function
	declarations.
*/


#ifndef GROUPNAME
/**	Bmeps is in the application group "dktools"
*/
#define GROUPNAME "dktools"
#endif

#include <dk.h>

#if DK_HAVE_ZLIB_H
#include <zlib.h>
#if DK_HAVE_PNG_H
#include <png.h>
#endif
#endif
#if DK_HAVE_JPEGLIB_H
#include <jpeglib.h>
#endif
#if DK_HAVE_TIFF_H
#include <tiff.h>
#include <tiffio.h>
#endif

#include <stdio.h>
#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if DK_HAVE_UNISTD_H
#include <unistd.h>
#endif
#if DK_HAVE_MATH_H
#include <math.h>
#endif
#if DK_HAVE_STRING_H
#include <string.h>
#else
#if DK_HAVE_STRINGS_H
#include <strings.h>
#endif
#endif

#include <dkerror.h>
#include <dklogc.h>
#include <dkstr.h>
#include <dkmem.h>
#include <dkapp.h>
#include <dksto.h>
#include <dklic.h>
#include <dksf.h>
#include <dkstream.h>
#include <dkof.h>
#include <dkbifi.h>
#include <dkbif.h>
#include <dkcp.h>


/**	PDF object position.
*/
typedef struct {
  unsigned long objno;	/**< Object number. */
  unsigned long pos;	/**< Byte position in file. */
} PDFOBJPOS;




/**	Draft info.
*/
typedef struct {
  double d;		/**< Line width. */
  double deltax;	/**< X offset. */
  double deltay;	/**< Y offset. */
} DI;



/**	Media (paper) size.
*/
typedef struct {
  double	px0;	/**< Paper: lower left x. */
  double	py0;	/**< Paper: lower left y. */
  double	px1;	/**< Paper: upper right x. */
  double	py1;	/**< Paper: upper right yx. */
  double	bx0;	/**< Bounding box: lower left x. */
  double	by0;	/**< Bounding box: lower left y. */
  double	bx1;	/**< Bounding box: upper right x. */
  double	by1;	/**< Bounding box: upper right y. */
} MEDIASIZE;



/**	Media size table entry.
*/
typedef struct {
  char		*name;	/**< Entry name. */
  MEDIASIZE	ms;	/**< Size information. */
} MSENTRY;



/**	Bmeps options set
*/
typedef struct {
  int		ot;	/**< Output type BMEPS_OUTPUT_TYPE_xxx. */
  int		l;	/**< EPS/PDF level. */
  int		enc;	/**< Encodings BMEPS_ENCODING_xxx. */
  dk_bif_sof_t	jfif_s;	/**< Allowed JFIF SOF types. */
  unsigned long	opt;	/**< Options. */
  unsigned long pdfopt;	/**< PDF-specific options. */
  double	dbgr;	/**< Default background red. */
  double	dbgg;	/**< Default background green. */
  double	dbgb;	/**< Default background blue. */
  double	mtl;	/**< Image mask trigger level. */
  MEDIASIZE	ms;	/**< Media size. */
  int		pred;	/**< Predictor. */
} BO;



/**	Bmeps options set table entry.
*/
typedef struct {
	char	*n;	/**< Case-insensitive name. */
	BO	*bo;	/**< Options data. */
} BOENTRY;



/**	Bmeps job data.
*/
typedef struct {
  /*
  	used from b2dvips module
  */
  BO		*bo;		/**< Options set (application wide). */
  BO		*bo2;		/**< Options set (image-specific). */
  dk_stream_t	*os;		/**< Output stream, writes to outf. */
  FILE		*outf;		/**< Output file. */
  FILE		*inf;		/**< Input file. */
  char		*infname;	/**< Input file name. */
  dk_app_t	*a;		/**< Application. */
  long		frame_s;	/**< Start frame (-1=all frames, 0=first, 1=second. */
  long		frame_e;	/**< End frame. */
  long		frame_n;	/**< Number of frames. */
  char		*tmp1;		/**< Temporary file to save standard input. */
  char		*tmp2;		/**< Page graphics instructions temporary file. */
  char		*tmp3;		/**< Image data stream (red/gray) temporary file. */
  char		*tmp4;		/**< Image data stream (green) temporary file. */
  char		*tmp5;		/**< Image data stream (blue) temporary file. */
  char		*tmp6;		/**< Alpha/mask data stream temporary file. */
  char		*tmp7;		/**< Temporary copy of TIFF file. */
  dk_bif_t	*bif;		/**< Interface to source image file. */
  dk_storage_t	*pdfobjs;	/**< PDF object positions. */
  dk_storage_iterator_t	*pdfobjsi;	/**< Iterator for object positions. */
  unsigned long	 nexto;		/**< Next object to handle .*/
  unsigned long	 psp;		/**< Pages start position. */
  unsigned long	 olo;		/**< Outlines object number. */
  /*
  	used from bmeps application
	(and possibly the b2dvips module)
  */
  int		it;		/**< Image type. */
  char		**msg;		/**< Messages, the program prints. */
  int		exval;		/**< Exit value of the application program. */
  int		cmd;		/**< Command, what to do. */
  char		*i1;		/**< Input file name from argv. */
  char		*o1;		/**< Output file name from argv. */
  char		*lang;		/**< Language from argv. */
  unsigned char	mm;		/**< Flag: make mode. */
  unsigned char frame_c;	/**< Number of frames configured. */
  struct {
    int must_rotate;		/**< Flag: must rotate the image. */
    int must_scale;		/**< Flag: must scale image. */
    double	sfx;		/**< Scale factor for x. */
    double	sfy;		/**< Scale factor for y. */
      struct {
        double x0;		/**< Lower left corner x. */
        double y0;		/**< Lower left corner y. */
        double x1;		/**< Upper right corner x. */
        double y1;		/**< Upper right corner y. */
      } hrbb;			/**< Paper size. */
      struct {
        double x0;		/**< Lower left corner x. */
        double y0;		/**< Lower left corner y. */
        double x1;		/**< Upper right corner x. */
        double y1;		/**< Upper right corner y. */
      } ia;			/**< Image area available. */
      struct {
        double x0;		/**< Lower left corner x. */
        double y0;		/**< Lower left corner y. */
        double x1;		/**< Upper right corner x. */
        double y1;		/**< Upper right corner y. */
      } iu;			/**< Image area really used. */
      struct {
        long x0;		/**< Lower left corner x. */
        long y0;		/**< Lower left corner y. */
        long x1;		/**< Upper right corner x. */
        long y1;		/**< Upper right corner y. */
      } bb;			/**< Bounding box. */
  } trans;			/**< Image translation data. */
  int	me;			/**< Mathematic error. */
  unsigned char	af;				/**< Flag: alpha avail. */
  unsigned char alpha_channel_transferred;	/**< Flag: Alpha trans.*/
  unsigned char suppress_alpha_info;		/**< Flag: Suppress alpha. */
} BJ;




/**	Options entry.
*/
typedef struct {
  unsigned long number;	/**< Number in order of appearance. */
  char *contents;	/**< Option text. */
} OPTENTRY;




/**	Input type table entry.
*/
typedef struct {
  int t;		/**< Input type. */
  char *s;		/**< File name suffix. */
} INPUTTYPEENTRY;




/** Pointer definition. */

typedef char *PCHAR;



/** Output type: EPS.
*/
#define BMEPS_OUTPUT_TYPE_EPS	0

/** Output type: PDF.
*/
#define BMEPS_OUTPUT_TYPE_PDF	1

/** Output type: Bounding box.
*/
#define BMEPS_OUTPUT_TYPE_BB	2




/** Encoding: ASCII-85.
*/
#define BMEPS_ENCODING_ASCII85		1

/** Encoding: Run-length.
*/
#define BMEPS_ENCODING_RUNLENGTH	2

/** Encoding: DCT.
*/
#define BMEPS_ENCODING_DCT		4

/** Encoding: Flate.
*/
#define BMEPS_ENCODING_FLATE		16

/** Encoding: LZW
*/
#define BMEPS_ENCODING_LZW		32


/** Encoding: Nonde.
*/
#define BMEPS_ENCODING_BINARY		128



/** Predictor: None.
*/
#define BMEPS_PREDICTOR_NONE		0

/** Predictor: TIFF
*/
#define BMEPS_PREDICTOR_TIFF		1


/** Predictor: Find automatically (use same as input file).
*/
#define BMEPS_PREDICTOR_INPUT		2

/** Predictor: PNG sub.
*/
#define BMEPS_PREDICTOR_PNG_SUB		3

/** Predictor: PNG up.
*/
#define BMEPS_PREDICTOR_PNG_UP		4

/** Predictor: PNG average.
*/
#define BMEPS_PREDICTOR_PNG_AVERAGE	5

/** Predictor: PNG paeth.
*/
#define BMEPS_PREDICTOR_PNG_PAETH	6

/** PS level 1.
*/
#define BMEPS_PS_LEVEL_1		1

/** PS level 2
*/
#define BMEPS_PS_LEVEL_2		2

/** PS level 3
*/
#define BMEPS_PS_LEVEL_3		3



/** PDF level 1.2
*/
#define BMEPS_PDF_LEVEL_12		11

/** PDF level 1.3
*/
#define BMEPS_PDF_LEVEL_13		12

/** PDF level 1.4
*/
#define BMEPS_PDF_LEVEL_14		13



/** Option: Use dictionary image operator.
*/
#define BMEPS_OPT_OPERATOR_DICTIONARY	0x00000001UL

/** Option: Write separated data streams for R, G and B.
*/
#define BMEPS_OPT_SEPARATED_DATA	0x00000002UL

/** Option: Write colored output.
*/
#define BMEPS_OPT_COLOR_OUTPUT		0x00000004UL

/** Option: Use alpha channel for mixing.
*/
#define BMEPS_OPT_ALPHA_MIX		0x00000008UL

/** Option: Always use background color from command line.
*/
#define BMEPS_OPT_PREFER_SPEC_BG	0x00000010UL

/** Option: Convert alpha channel to image mask.
*/
#define BMEPS_OPT_IMAGE_MASK		0x00000020UL

/** Option: Write DSC comments.
*/
#define BMEPS_OPT_DSC			0x00000040UL

/** Option: Use showpage operator.
*/
#define BMEPS_OPT_SHOWPAGE		0x00000080UL

/** Option: Use resolution chunk from input file.
*/
#define BMEPS_OPT_RESOLUTION		0x00000100UL

/** Option: Use temporary dictionary for image.
*/
#define BMEPS_OPT_DICTIONARY		0x00000200UL

/** Option: Use vmreclaim operator.
*/
#define BMEPS_OPT_VMRECLAIM		0x00000400UL

/** Option: Transfer alpha channel to output file.
*/
#define BMEPS_OPT_TRANSFER_ALPHA	0x00000800UL

/** Option: Use specified paper size.
*/
#define BMEPS_OPT_PAPER_SIZE		0x00001000UL

/** Option: Create draft only.
*/
#define BMEPS_OPT_DRAFT			0x00002000UL

/** Option: Set interpolation flag.
*/
#define BMEPS_OPT_INTERPOLATE		0x00004000UL

/** Option: Set interpolation flag for DCT data from JPEG files too.
*/
#define BMEPS_OPT_JPEG_INTERPOLATE	0x00008000UL

/** Option: Pass-through of DCT-encoded data always.
*/
#define BMEPS_OPT_FORCE_DCT		0x00010000UL



/** Page fitting: Fit width.
*/
#define BMEPS_PDFOPT_FIT_WIDTH			1UL

/** Page fitting: Fit height.
*/
#define BMEPS_PDFOPT_FIT_HEIGHT			2UL

/** Page fitting: Fit page.
*/
#define BMEPS_PDFOPT_FIT_PAGE	(BMEPS_PDFOPT_FIT_WIDTH | BMEPS_PDFOPT_FIT_HEIGHT)

/** Mask for page fitting bits.
*/
#define BMEPS_PDFOPT_FIT_MASK	(BMEPS_PDFOPT_FIT_WIDTH | BMEPS_PDFOPT_FIT_HEIGHT)

/** PDF output option: Allow page attributes.
*/
#define BMEPS_PDFOPT_ALLOW_PDF_PAGE_ATTRIBUTES	4UL



/** Command: Run normally.
*/
#define BMEPS_CMD_RUN			0

/** Command: Show help.
*/
#define BMEPS_CMD_HELP			1

/** Command: Show version.
*/
#define BMEPS_CMD_VERSION		2

/** Command: Save new configuration.
*/
#define BMEPS_CMD_CONFIGURE		4

/** Command: Forget save configuration.
*/
#define BMEPS_CMD_UNCONFIGURE		8

/** Command: Show configuration.
*/
#define BMEPS_CMD_SHOWCONF		16

/** Command: Show error message for wrong command line options.
*/
#define BMEPS_CMD_ERROR			32

/** Command: Show configuration file contents.
*/
#define BMEPS_CMD_CONFIGURATION_FILE	64

/** Command: Save standard input to new configuration file.
*/
#define BMEPS_CMD_WRITE_CONFIGURATION_FILE	128

/** Command mask: All dkapp-specific commands.
*/
#define BMEPS_CMD_APP (BMEPS_CMD_HELP\
|BMEPS_CMD_VERSION\
|BMEPS_CMD_CONFIGURE\
|BMEPS_CMD_UNCONFIGURE\
|BMEPS_CMD_SHOWCONF\
|BMEPS_CMD_CONFIGURATION_FILE\
|BMEPS_CMD_WRITE_CONFIGURATION_FILE\
)

#if defined(EXTERN)
#undef EXTERN
#endif

/* b2bj module */
/** Definition of extern functions of the b2bj module. */
#if B2BJ_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Intialize bmeps job.
	@param	bj	Bmeps job to initialize.
*/
EXTERN void
bmeps_bj_init DK_PR((BJ *bj));

#if defined(__cplusplus)
}
#endif



/* b2bo module */
/** Definition of extern functions of the b2bo module. */
#if B2BO_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Get defaults.
	@param	bo	Bmeps options set to fill with defaults.
*/
EXTERN void
bmeps_bo_get_defaults DK_PR((BO *bo));

/**	Add one configuration line to options set.
	@param	bj	Bmeps job.
	@param	bo	Bmeps options set.
	@param	s	Line from configuration file.
	@return	1 on success, 0 on error.
*/
EXTERN int
bmeps_bo_add_line DK_PR((BJ *bj, BO *bo, char *s));

/**	Add language definition and options to bmeps job.
	@param	bj	Bmeps job.
	@param	langdef	Language and options, i.e. "ps,color=y".
*/
EXTERN void bmeps_bo_get_configuration DK_PR((BJ *bj, char *langdef, int v));

/**	Retrieve preferred language from preferences system.
	@param	bj	Bmeps job.
	@param	buffer	Buffer for language string.
	@param	sz	Buffer size.
	@return	1 on success, 0 on error.
*/
EXTERN int bmeps_bo_get_pref_lang DK_PR((BJ *bj, char *buffer, size_t sz));

/**	Show configuration file contents.
	@param	bj	Bmeps job.
	@return	1 on success, 0 on error.
*/
EXTERN int bmeps_bo_show_configuration_file DK_PR((BJ *bj));

/**	Save standard input into new configuration file.
	@param	bj	Bmeps job.
	@return	1 on success, 0 on error.
*/
EXTERN int bmeps_bo_write_configuration_file DK_PR((BJ *bj));

#if defined(__cplusplus)
}
#endif

/* b2str module */
/** Definition of extern functions of the b2str module. */
#if B2STR_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Retrieve one string.
	@param	s	String number.
	@return	Valid pointer on success, NULL on error.
*/
EXTERN char *bmeps_str_get DK_PR((size_t s));

/**	Get maximum string index.
	@return	Maximum index.
*/
EXTERN size_t bmeps_get_num DK_PR((void));

/**	Return pointer to array of strings used by bmeps.
	@return	Array pointer.
*/
EXTERN char **bmeps_str_base DK_PR((void));

#if defined(__cplusplus)
}
#endif



/* b2conv module */
/** Definition of extern functions of the b2conv module. */
#if B2CONV_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Run one conversion.
	@param	bj	Bmeps job.
*/
EXTERN void bmeps_convert DK_PR((BJ *bj));

#if defined(__cplusplus)
}
#endif



/* b2eps module */
/** Definition of extern functions of the b2eps module. */
#if B2EPS_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Convert to PS/EPS, use traditional encoding (no DCT).
	@param	bj	Bmeps job.
*/
EXTERN void
bmeps_eps_conventional DK_PR((BJ *bj));

/**	Convert to PS/EPS, use DCT-encoded data from
	JPEG file directly.
	@param	bj	Bmeps job.
*/
EXTERN void
bmeps_eps_dct DK_PR((BJ *bj));

/**	Create PS/EPS draft.
	@param	bj	Bmeps job.
*/
EXTERN void
bmeps_eps_draft DK_PR((BJ *bj));

/**	Show bounding box.
	@param	bj	Bmeps job.
*/
EXTERN void
bmeps_eps_bb DK_PR((BJ *bj));
#if defined(__cplusplus)
}
#endif

/* b2pdf module */
/** Definition of extern functions of the b2pdf module. */
#if B2PDF_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Convert to PDF.
	@param	bj	Bmeps job.
	@param	how	Conversion type (0=normal, 1=DCT-passthrough, 2=draft.
*/
EXTERN void
bmeps_pdf DK_PR((BJ *bj, int how));

#if defined(__cplusplus)
}
#endif


/* b2tool1 module */
/** Definition of extern functions of the b2tool1 module. */
#if B2TOOL1_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Find out how to handle image size.
	@param	bj	Bmeps job.
	@return	0= use width and height as is,
	1= use pHY chunk (resolution chunk) in image,
	2= use specified media size.
*/
EXTERN int
bmeps_tool_how_to_handle_bb DK_PR((BJ *bj));

/**	Get number of EPS bits per component for a specified
	input number of bits per component.
	@param	b	Bits per component from input file.
	@return	Bits per component for PS/EPS output.
*/
EXTERN unsigned short
bmeps_tool_eps_output_bits DK_PR((unsigned short b));

/**	Get number of PDF bits per component for a specified
	input number of bits per component.
	@param	b	Bits per component from input file.
	@return	Bits per component for PDF output.
*/
EXTERN unsigned short
bmeps_tool_pdf_output_bits DK_PR((unsigned short b));

/**	Correct inconsitencies (if any) in bmeps options set.
	@param	bo	Bmeps options set.
*/
EXTERN void
bmeps_tool_correct_bo DK_PR((BO *bo));

/**	Check whether interpolation flag must be enabled.
	@param	bj	Bmeps job.
	@return	Value of interpolation flag.
*/
EXTERN int
bmeps_tool_interpolate DK_PR((BJ *bj));

/**	Find file name suffix for output type.
	@param	t	Output type.
	@return	File name suffix.
*/
EXTERN char *
bmeps_tool_suffix_for_output_type DK_PR((int t));

/**	Calculate dimensions (line width, angle) for draft.
	@param	bj	Bmeps job.
	@param	dp1	Inner lines.
	@param	dp2	Outer lines.
	@param	f1	Inner lines factor.
	@param	f2	Border lines factor.
	@return	1 on success, 0 on error.
*/
EXTERN int
bmeps_tool_calc_di DK_PR((BJ *bj, DI *dp1, DI *dp2, double f1, double f2));

/**	Calculate used areas, shifting and rotation.
	@param	bj	Bmeps job.
*/
EXTERN void
bmeps_tool_calculate_areas DK_PR((BJ *bj));

/**	Show error message: Not enough memory.
	@param	bj	Bmeps job.
*/
EXTERN void
bmeps_tool_error_memory DK_PR((BJ *bj));

/**	Show simple error message.
	@param	bj	Bmeps job.
	@param	ll	Log level (DK_LOG_xxx).
	@param	n	Index in string array.
*/
EXTERN void
bmeps_tool_msg_1 DK_PR((BJ *bj, int ll, size_t n));


/**	Show  error message, one custom string:
	bmeps_str_get(n1) s bmeps_str_get(n2).
	@param	bj	Bmeps job.
	@param	ll	Log level (DK_LOG_xxx).
	@param	n1	Index in string array.
	@param	n2	Index in string array.
	@param	s	Customized string.
*/
EXTERN void
bmeps_tool_msg_3 DK_PR((BJ *bj, int ll, size_t n1, size_t n2, char *s));

/**	Show  error message, two custom strings:
	bmeps_str_get(n1) s1 bmeps_str_get(n2) s2 bmeps_str_get(n3).
	@param	bj	Bmeps job.
	@param	ll	Log level (DK_LOG_xxx).
	@param	n1	Index in string array.
	@param	n2	Index in string array.
	@param	n3	Index in string array.
	@param	s1	1st customized string.
	@param	s2	2nd customized string.
*/
EXTERN void
bmeps_tool_msg_5 DK_PR((BJ *bj, int ll, size_t n1, size_t n2, size_t n3, char *s1, char *s2));

#if defined(__cplusplus)
}
#endif


/**	Buffer size for file names.
*/
#define NAME_BUFFER_SIZE	128

/**	Maximum configuration line length.
*/
#define CONFIG_LINE_SIZE	1024

/**	Epsilon for floating point calculations with dpi numbers.
*/
#define DPI_EPSILON		0.00001

#endif
/* BMEPS_H_INCLUDED */

