/*
Copyright (c) 2009-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  other contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	fchksize.c	The fchksize program.
*/



/*

-c	--check
	Compare checksums against list

-q	--quiet
	No output for succeeded tests when checking.

-s	--status
	No output when checking, set exit code only

-h	--help
	Show help text

-v	--version
	Show version information

-l	--list
	The specified file names (or standard input) are lists
	containing file names of files to inspect.

Output format:
<size> <name>
	<size>	File size in bytes.
	<name>	File name.

Exit codes:
0	success
1	Size mismatch
2	Syntax or option error
3	Attempt to handle a directory
4	Illegal file type
5	Stat failed (file does not exist/no permissions?)
*/


#include "fchksum.h"
#include "dktools-version.h"




#line 86 "fchksize.ctr"





/** Exit code returned by the program. */
static int exval = 0;



/** Flag: print help text. */
static unsigned char flag_help = 0x00;

/** Flag: Show version information. */
static unsigned char flag_version = 0x00;

/** Flag: Quiet, show failed tests only. */
static unsigned char flag_quiet = 0x00;

/** Flag: Set exit code only. */
static unsigned char flag_status = 0x00;

/** Flag: check file list. */
static unsigned char flag_check = 0x00;

/** Flag: File list. */
static unsigned char flag_list = 0x00;

/** Flag: Check directories. */
static unsigned char flag_dir = 0x00;

/** Number of files reporting wrong file size. */
static unsigned long files_failed = 0UL;

/** Number of files checked. */
static unsigned long files_checked = 0UL;

/** File size. */
static LLU file_size = ZERO_LONG_LONG_UNSIGNED;



/** Long options. */

static char *long_options[] = {
/*  0 */	"help",
/*  1 */	"version",
/*  2 */	"check",
/*  3 */	"quiet",
/*  4 */	"status",
/*  5 */	"directories",
/*  6 */	"list",
NULL
};


#if ON_A_WINDOWS_SYSTEM
#if DK_HAVE__STAT64
/** Bit size for version number. */
#define VERSION_SIZE_BITS "64"
#else
/** Bit size for version number. */
#define VERSION_SIZE_BITS "32"
#endif
#else
#if DK_HAVE_STAT64 && DK_HAVE_LARGEFILE64_SOURCE
/** Bit size for version number. */
#define VERSION_SIZE_BITS "64"
#else
/** Bit size for version number. */
#define VERSION_SIZE_BITS "32"
#endif
#endif


/** Text to show version number and license conditions. */

static char *version_text[] = {
"fchksize - File size check program, version "
VERSNUMB ", " VERSION_SIZE_BITS "-bit",
"Copyright (C) 2008-2010 - Dipl.-Ing. Dirk Krause",
"http://dktools.sourceforge.net/fchksum.html",
"",
"Redistribution and use in source and binary forms, with or without",
"modification, are permitted provided that the following conditions are met:",
"* Redistributions of source code must retain the above copyright notice, this",
"  list of conditions and the following disclaimer.",
"* Redistributions in binary form must reproduce the above copyright notice,",
"  this list of conditions and the following disclaimer in the documentation",
"  and/or other materials provided with the distribution.",
"* Neither the name of the Dirk Krause nor the names of other contributors may",
"  be used to endorse or promote products derived from this software without",
"  specific prior written permission.",
"THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''",
"AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE",
"IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE",
"ARE DISCLAIMED.",
"IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY",
"DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES",
"(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;",
"LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND",
"ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT",
"(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS",
"SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.",
"",
NULL
};



/** Help text. */

static char *help_text[] = {
"",
"fchksize -h",
"\t\tprints a help text.",
"",
"fchksize -v",
"\t\tprints version information.",
"",
"fchksize [-l] [...]",
"\t\tprints the size of the specified files.",
"\t\t-l    the specified files are lists of files.",
"",
"fchksize -c|--check [-q|--quiet|-s|--status] [<files>]",
"\t\tchecks messages digests for files in the list.",
"\t\t-c    check files against a check list",
"\t\t-q    quiet, do not write report lines for succeeded checks",
"\t\t-s    set exit status code only, do not write any report lines",
"",
NULL
};



/** Program name, used in messages. */

static char progname[] = { "fchksize" };



/** Default set of whitespaces. */

static char default_whitespace_set[] = { " \t\r\n" };



/** Messages printed by the program. */

static char *msg[] = {
/*  0 */
"<stdin>",

/*  1 */
"%s: OK\n",

/*  2 */
"%s: FAILED\n",

/*  3 */
"%s: WARNING %lu of %lu checked files did NOT match!\n",

/*  4 */
"%s:%s: ERROR - Can not handle directories!\n",

/*  5 */
"%s:%s: ERROR - Illegal file type!\n",

/*  6 */
"%s:%s: ERROR - Failed to obtain information about file!\n",

/*  7 */
"%s:%s: Warning - Creating checksum for a device!\n",

/*  8 */
"%s:%s: Pipe skipped.\n",

/*  9 */
"%s: ERROR - Illegal long option --%s\n",

/* 10 */
"%s: ERROR - Illegal option --%s\n",

/* 11 */
"%s:%s:%lu: ERROR - Syntax error in checklist!\n",

/* 12 */
"%s:%s: ERROR - Failed to read file!\n",

/* 13 */
"-",

NULL
};



/** Used to read lines from check list. */
static char inputline[1024];



#if !DK_HAVE_STRICMP
#if !DK_HAVE_STRCASECMP
/**	Fallback function if neither strcasecmp() nor stricmp() is available.
	@param	a	Left text in comparison.
	@param	b	Right text in comparison.
	@return	Text comparison result -1, 0 or 1 for a<b, a=b or a>b.
*/
static
int
portable_stricmp(char *a, char *b) {
  int back = 0;	/**< Return value. */
  int ende;	/**< Flag: finished. */
  char aval;	/**< Current left side character. */
  char bval;	/**< Current right side character. */
  char *aptr;	/**< Current left side pointer. */
  char *bptr;	/**< Current right side pointer. */
  
  if(a && b) {
    aptr = a; bptr = b;
    ende = 0;
    while((ende == 0) && (back == 0)) {
      aval = *aptr; bval = *bptr;
      if(aval && bval) {
	if(isascii(aval) && islower(aval)) aval = toupper(aval);
	if(isascii(bval) && islower(bval)) bval = toupper(bval);
	if(aval != bval) {
	  ende = 1;
	  if(aval > bval) back = 1;
	  else            back = -1;
	} else {
	  aptr++; bptr++;
	}
      } else {
	ende = 1;
	if(aval) { back = 1; }
	if(bval) { back = -1; }
      }
    }
  } 
  return back;
}
#endif
#endif



/**	Find string in array of strings, return index.
	@param	array	Array of pointers to strings.
	@param	str	String to search for in \a array.
	@param	cs	Flag: case-sensitive.
	@return	Index of \a str in \a array, or -1 on error (not found).
*/
int dkstr_array_index(char **array, char *str, int cs)
{
  int back = -1;	/**< Return value. */
  char **ptr;		/**< Current array element to test. */
  int i;		/**< Index of current element. */
  
  if(array && str) {
    i = 0; ptr = array;
    while((*ptr) && (back == -1)) {
      if(cs) {
	if(strcmp(*ptr,str) == 0) {
	  back = i;
	}
      } else {
	if(STRICMP(*ptr,str) == 0) {
	  back = i;
	}
      }
      if(back == -1) {
	ptr++; i++;
      }
    }
  } 
  return back;
}



/**	Fallback routine for systems without strchr().
	@param	str	The string to search.
	@param	c	The character to search for.
	@return	Pointer to character on success, NULL on error.
*/
char *
dkstr_chr(char *str, int c)
{
  char *back = NULL;	/**< Return value. */
  
  if(str) {
#if DK_HAVE_STRCHR
    back = strchr(str,c);
#else
    char *ptr;
    ptr = str;
    while((*ptr) && (!back)) {
      if(((int)(*ptr)) == c) {
	back = ptr;
      } else {
        ptr++;
      }
    }
#endif
  } 
  return back;
}



/**	Find start of string.
	@param	str	The text to find a string in.
	@param	whsp	Set of whitespaces.
	@return	Pointer to first text character on success, NULL on error.
*/
char *
dkstr_start(char *str, char *whsp)
{
  char *back = NULL;	/**< Return value. */
  char *ptr;		/**< Used to traverse string. */
  char *wh;		/**< Whitespace set. */
  
  if(str) {
    wh = (whsp ? whsp : default_whitespace_set);
    ptr = str;
    while((*ptr) && (!back)) {
      if(dkstr_chr(wh,((int)(*ptr)))) {
	ptr++;
      } else {
	back = ptr;
      }
    }
  } 
  return back;
}



/**	Find start of next string.
	@param	str	The text to find a string in.
	@param	whsp	Set of whitespaces.
	@return	Pointer to next text start on success, NULL on error.
*/
char    *
dkstr_next(char *str, char *whsp)
{
  char *back = NULL;	/**< Return value. */
  char *ptr;		/**< Used to traverse string. */
  char *wh;		/**< Whitespace set. */
  int state;		/**< Current state: 0=no text yet, 1=text, 2=done. */
  
  if(str) {
    ptr = str;
    wh = (whsp ? whsp : default_whitespace_set);
    state = 0;
    while((state < 2) && (*ptr)) {
      if(dkstr_chr(wh,((int)(*ptr)))) {
	if(state == 1) {
	  state = 2;
	  *(ptr++) = '\0';
	  back = dkstr_start(ptr,wh);
	}
	ptr++;
      } else {
	state = 1;
	ptr++;
      }
    }
  } 
  return back;
}



/**	Remove trailing whitespaces.
	@param	str	The string to modify.
	@param	whsp	Set of whitespaces.
*/
void
dkstr_chomp(char *str, char *whsp)
{
  char *wh;	/**< Whitespace set. */
  char *ptr;	/**< Used to traverse text. */
  char *x;	/**< First whitespace after last text. */
  
  if(str) {
    wh = (whsp ? whsp : default_whitespace_set);
    x = NULL; ptr = str;
    while(*ptr) {
      if(dkstr_chr(wh,((int)(*ptr)))) {
	if(!x) {
	  x = ptr;
	}
      } else {
	x = NULL;
      }
      ptr++;
    }
    if(x) { *x = '\0'; }
  }
  
}



/**	Error message: Directory.
	@param	s	File name.
*/
static
void
err_directory(char *s) {
  fprintf(stderr, msg[4], progname, s); fflush(stderr);
  if(exval != 1) exval = 3;
}



/**	Error message: File is of wrong type.
	@param	s	File name.
*/
static
void
err_illegal_file_type(char *s) {
  fprintf(stderr, msg[5], progname, s); fflush(stderr);
  if(exval != 1) exval = 4;
}



/**	Error message: stat() failed.
	@param	s	File name.
*/
static
void
err_stat_failed(char *s) {
  fprintf(stderr, msg[6], progname, s); fflush(stderr);
  if(exval != 1) exval = 5;
}



/**	Warning: This path name is a device.
	@param	s	File name.
*/
static
void
warn_device(char *s) {
  fprintf(stderr, msg[7], progname, s); fflush(stderr);
}



#if ON_A_WINDOWS_SYSTEM
/**	Warning: Skipping pipe.
	@param	s	File name.
*/
static
void
warn_pipe(char *s) {
  fprintf(stderr, msg[8], progname, s); fflush(stderr);
}
#endif



/**	Check whether we can find the size for the file type.
	Print error message if file type is not acceptable.
	@param	filename	File name.
	@return	1 for yes, 0 for no.
*/
static
int
is_acceptable_file_type(char *filename) {
  int back = 0;	/**< Return value. */
#if ON_A_WINDOWS_SYSTEM

#if DK_HAVE__STAT64
  struct __stat64 stbuf;
  
  file_size = ZERO_LONG_LONG_UNSIGNED;
  if(_stat64(filename, &stbuf) == 0) {
    if(((stbuf.st_mode) & _S_IFMT) == _S_IFREG) {
      back = 1;
      file_size = stbuf.st_size;
    } else {
      if(((stbuf.st_mode) & _S_IFMT) == _S_IFDIR) {
        if(flag_dir) {
	  back = 1;
	  file_size = stbuf.st_size;
	} else {
	  err_directory(filename);
	}
      } else {
        if(((stbuf.st_mode) & _S_IFMT) == _S_IFCHR) {
	  back = 1;
	  file_size = stbuf.st_size;
	  warn_device(filename);
	} else {
	  if(((stbuf.st_mode) & _S_IFMT) == _S_IFIFO) {
	    back = 0;
	    warn_pipe(filename);
	  } else {
	    err_illegal_file_type(filename);
	  }
	}
      }
    }
  } else {
    err_stat_failed(filename);
  }
#else
  struct _stat stbuf;
  
  file_size = ZERO_LONG_LONG_UNSIGNED;
  if(_stat(filename, &stbuf) == 0) {
    if(((stbuf.st_mode) & _S_IFMT) == _S_IFREG) {
      back = 1; file_size = stbuf.st_size;
    } else {
      if(((stbuf.st_mode) & _S_IFMT) == _S_IFDIR) {
        if(flag_dir) {
	  back = 1; file_size = stbuf.st_size;
	} else {
	  err_directory(filename);
	}
      } else {
        if(((stbuf.st_mode) & _S_IFMT) == _S_IFCHR) {
	  back = 1; file_size = stbuf.st_size;
	  warn_device(filename);
	} else {
	  if(((stbuf.st_mode) & _S_IFMT) == _S_IFIFO) {
	    back = 0;
	    warn_pipe(filename);
	  } else {
	    err_illegal_file_type(filename);
	  }
	}
      }
    }
  } else {
    err_stat_failed(filename);
  }
#endif

#else

#if DK_HAVE_STAT64 && DK_HAVE_LARGEFILE64_SOURCE
  struct stat64 stbuf;
  
  file_size = ZERO_LONG_LONG_UNSIGNED;
  if(stat64(filename, &stbuf) == 0) {
    if(S_ISREG(stbuf.st_mode)) {
      back = 1; file_size = stbuf.st_size;
    } else {
      if(S_ISDIR(stbuf.st_mode)) {
        if(flag_dir) {
	  back = 1; file_size = stbuf.st_size;
	} else {	
	  err_directory(filename);
	}
      } else {
        if(S_ISCHR(stbuf.st_mode)) {
	  back = 1; file_size = stbuf.st_size;
	  warn_device(filename);
	} else {
	  if(S_ISBLK(stbuf.st_mode)) {
	    warn_device(filename);
	    back = 1; file_size = stbuf.st_size;
	  } else {	
	    err_illegal_file_type(filename);
	  }
	}
      }
    }
  } else {		
    err_stat_failed(filename);
  }
#else
  struct stat stbuf;
  
  file_size = ZERO_LONG_LONG_UNSIGNED;
  if(stat(filename, &stbuf) == 0) {
    if(S_ISREG(stbuf.st_mode)) {
      back = 1; file_size = stbuf.st_size;
    } else {
      if(S_ISDIR(stbuf.st_mode)) {
        if(flag_dir) {
	  back = 1; file_size = stbuf.st_size;
	} else {	
	  err_directory(filename);
	}
      } else {
        if(S_ISCHR(stbuf.st_mode)) {
	  back = 1; file_size = stbuf.st_size;
	  warn_device(filename);
	} else {
	  if(S_ISBLK(stbuf.st_mode)) {
	    warn_device(filename);
	    back = 1; file_size = stbuf.st_size;
	  } else {	
	    err_illegal_file_type(filename);
	  }
	}
      }
    }
  } else {		
    err_stat_failed(filename);
  }
#endif

#endif
  return back;
}



/**	Process command line options.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
*/
static
void
process_arguments(int argc, char *argv[]) {
  char **lfdptr;	/**< Used to traverse \a argv. */
  char *ptr;		/**< Current cmd arg to process. */
  int i;		/**< Index of current arg. */
  lfdptr = argv; lfdptr++; i = 1;
  while(i < argc) {
    ptr = *lfdptr;
    if(*ptr == '-') {
      ptr++;
      switch(*ptr) {
        case '-': {	/* long option */
	  ptr++;
	  switch(dkstr_array_index(long_options, ptr, 0)) {
	    case 0: flag_help = 0x01; break;
	    case 1: flag_version = 0x01; break;
	    case 2: flag_check = 0x01; break;
	    case 3: flag_quiet = 0x01; break;
	    case 4: flag_status = 0x01; break;
	    case 5: flag_dir = 0x01; break;
	    case 6: flag_list = 0x01; break;
	    default: {
	      if(exval != 1) exval = 2;
	      flag_help = 0x01;
	      /* ERROR: Unknown long option. */
	      fprintf(stderr, msg[9], progname, ptr);
	    } break;
	  }
	} break;
	case 'h': flag_help = 0x01; break;
	case 'v': flag_version = 0x01; break;
	case 'l': flag_list = 0x01; break;
	case 'c': flag_check = 0x01; break;
	case 'q': flag_quiet = 0x01; break;
	case 's': flag_status = 0x01; break;
	case 'd': flag_dir = 0x01; break;
	default: {
	  flag_help = 0x01;
	  if(exval != 1) exval = 2;
	  /* ERROR: Unknown option. */
	  fprintf(stderr, msg[10], progname, ptr);
	} break;
      }
    }
    lfdptr++; i++;
  }
}



/**	Report size and name for one file.
	@param	n	File name.
*/
static
void
handle_one_file(char *n) {
  
  if(is_acceptable_file_type(n)) {
#if DK_HAVE_LONG_LONG_INT
#if defined(WIN32) || defined(_WIN32)
    
    printf("%I64u %s\n", file_size, n);
#else
    
    printf("%llu %s\n", file_size, n);
#endif
#else
    
    printf("%lu %s\n", file_size, n);
#endif
  }
  
}



/**	Handle one input file (file list).
	@param	f	Used to read the file.
	@param	n	File name, used for error reporting.
*/
static
void
run_for_file_list(FILE *f, char *n) {
  unsigned long lineno = 0UL;		/**< Line number for error report. */
  char *p;				/**< Start of line. */
  
  while(fgets(inputline, sizeof(inputline), f)) {
    lineno++;
    p = dkstr_start(inputline, NULL);
    if(p) {
      dkstr_chomp(p, NULL);
      handle_one_file(p);
    } else {
      /* empty input line */
    }
  } 
}



/**	Handle one input file (check list).
	@param	f	Used to read the file.
	@param	n	File name, used for error reporting.
*/
static
void
check_list(FILE *f, char *n) {
  unsigned long lineno = 0UL;		/**< Line number for error report. */
  char *p;				/**< Start of line. */
  char *x;				/**< Pointer to file name. */
  LLU fs;				/**< File size from check list. */
  int res;				/**< sscanf result. */
  
  while(fgets(inputline, sizeof(inputline), f)) {
    lineno++;
    p = dkstr_start(inputline, NULL);
    if(p) {
      x = dkstr_next(p, NULL);
      if(x) {
        dkstr_chomp(x, NULL);
#if DK_HAVE_LONG_LONG_INT
#if defined(WIN32) || defined(_WIN32)
        res = sscanf(p, "%I64u", &fs);
#else
        res = sscanf(p, "%llu", &fs);
#endif
#else
        res = sscanf(p, "%lu", &fs);
#endif
        if(res == 1) {
	  files_checked++;
	  if(is_acceptable_file_type(x)) {
	    if(fs != file_size) {
	      exval = 1;
	      if(!(flag_status)) {
	        /* Check failed */
		printf(msg[2], x);
		files_failed++;
		if(files_failed == 0UL) files_failed = 0xFFFFFFFFUL;
	      }
	    } else {
	      if(!((flag_status) || (flag_quiet))) {
	        /* OK line */
		printf(msg[1], x);
	      }
	    }
	  }
	} else {
	  /* ERROR Syntax */
	  fprintf(stderr, msg[11], progname, (n ? n : msg[0]), lineno);
	}
      } else {
        /* ERROR: Syntax */
	fprintf(stderr, msg[11], progname, (n ? n : msg[0]), lineno);
      }
    }
  } 
}



/**	Build checksum for standard input.
*/
static
void
run_for_stdin(void) {
    int bytes_read;
    LLU nv;
    int ec;
#if ON_A_WINDOWS_SYSTEM
    int oldmode;	/**< Old text/binary mode. */
    oldmode = _setmode(_fileno(stdin), _O_BINARY);
#endif
    ec = 0;
    nv = file_size = ZERO_LONG_LONG_UNSIGNED;
    do {
      bytes_read = fread(inputline,1,sizeof(inputline),stdin);
      if(bytes_read > 0) {
        nv = file_size + (LLU)bytes_read;
	if(nv < file_size) {
	  /* Overflow */
	  ec = 1;
	} else {
	  file_size = nv;
	}
      }
    } while((bytes_read > 0) && (ec == 0));
    if(ec == 0) {
#if DK_HAVE_LONG_LONG_INT
#if defined(WIN32) || defined(_WIN32)
    
    printf("%I64u %s\n", file_size, msg[13]);
#else
    
    printf("%llu %s\n", file_size, msg[13]);
#endif
#else
    
    printf("%lu %s\n", file_size, msg[13]);
#endif
    }
#if ON_A_WINDOWS_SYSTEM
    _setmode(_fileno(stdin), oldmode);
#endif
}



/**	Process the file on the command line.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
*/
static
void
run_for_files(int argc, char *argv[]) {
  char **lfdptr;		/**< Pointer to traverse argv. */
  char *ptr;			/**< Current argv element. */
  int i;			/**< Index of current argv element. */
  unsigned char fnf = 0x00;	/**< Flag: file name found. */
  FILE *f;			/**< File for check list/run list. */
  
  lfdptr = argv; lfdptr++; i = 1;
  while(i < argc) {
    ptr = *lfdptr;		
    if(*ptr != '-') {		
      fnf = 0x01;
      if(flag_check) {		
#if DK_HAVE_LARGEFILE64_SOURCE && DK_HAVE_FOPEN64
        f = fopen64(ptr, "r");
#else
	f = fopen(ptr, "r");
#endif
	if(f) {
	  check_list(f, ptr);
	  fclose(f);
	} else {
	  /* ERROR: Failed to read file ptr */
	  fprintf(stderr, msg[12], progname, (ptr ? ptr : msg[0]));
	  fflush(stderr);
	}
      } else {
        if(flag_list) {		
#if DK_HAVE_LARGEFILE64_SOURCE && DK_HAVE_FOPEN64
	  f = fopen64(ptr, "r");
#else
	  f = fopen(ptr, "r");
#endif
	  if(f) {
	    run_for_file_list(f, ptr);
	    fclose(f);
	  } else {
	    /* ERROR: Failed to read file ptr */
	    fprintf(stderr, msg[12], progname, (ptr ? ptr : msg[0]));
	    fflush(stderr);
	  }
	} else {		
	  handle_one_file(ptr);
	}
      }
    }
    i++; lfdptr++;
  }
  if(fnf == 0x00) {		/* Process standard input. */
    if(flag_check) {		/* stdin is file list to check. */
      check_list(stdin, msg[0]);
    } else {
      if(flag_list) {		/* stdin is list of files to report. */
        run_for_file_list(stdin, msg[0]);
      } else {			/* Report stdin. */
        run_for_stdin();
      }
    }
  }
  if(flag_check) {
    if(files_failed) {
      fprintf(stderr, msg[3], progname, files_failed, files_checked);
      fflush(stderr);
    }
  } 
}



/**	Show a text (NULL-termintated array of strings).
	@param	t	Pointer to start of array.
*/
static
void
show_text(char **t) {
  char **ptr;	/**< Pointer to traverse the array. */
  ptr = t;
  while(*ptr) {
    fputs(*(ptr++), stdout); fputc('\n', stdout);
  }
}



/**	The main function.
	@param	argc	Number of command line arguments.
	@param	argv	Command line arguments array.
	@return	0 on success, positive values on error.
*/
int main(int argc, char *argv[]) {
  
#line 1012 "fchksize.ctr"

  
  process_arguments(argc, argv);
  if(exval == 0) {
    if(flag_help || flag_version) {
      show_text(version_text);
      if(flag_help) {
        show_text(help_text);
      }
    } else {
      run_for_files(argc, argv);
    }
  } else {
    if(flag_help) {
      show_text(version_text);
      show_text(help_text);
    }
  }
  
  
#line 1031 "fchksize.ctr"

  exit(exval); return exval;
}

