/*
jlayout - A Java code generator for GUI layout
Copyright (c) 2007-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

/**	@file jl.h	Main include file for the jlayout program.
 */
#ifndef JL_H_INCLUDED
#define JL_H_INCLUDED

#include <dk.h>

#include <stdio.h>
#if DK_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if DK_HAVE_STRING_H
#include <string.h>
#endif
#if DK_HAVE_UNISTD_H
#include <unistd.h>
#endif
#if DK_HAVE_PROCESS_H
#include <process.h>
#endif

#include <dkmem.h>
#include <dkstr.h>
#include <dkapp.h>
#include <dklogc.h>
#include <dklic.h>
#include <dksf.h>
#include <dksto.h>
#include <dkbf.h>
#include <dkstream.h>

#ifndef GROUPNAME
/** Application group name .*/
#define GROUPNAME "dktools"
#endif



/**	Data for one source code line kept in memory.
 */
typedef struct _one_line_ {
  unsigned long lineno;		/**< Line number. */
  char		*t;		/**< Text in line without newline. */
} JSRC_LINE;



/** Event listener for an object.
*/
typedef struct _event_listener_ {
  unsigned long number;		/**< Number (in order of appearance). */
  int		tp;		/**< Listener type. */
  char		*t;		/**< Text. */
} JLISTENER;



/**	jlayout object
*/
typedef struct _jlayout_object_ {
  char		*on;		/**< Object name. */
  char		*oc;		/**< Object class. */
  int		 ocn;		/**< numeric object class (if exists). */
  struct _jlayout_object_	*parent;	/**< Parent object. */
  struct _jlayout_object_	*menubar;	/**< Menu bar in object. */
  unsigned long	obj_no;		/**< Object number (ID). */
  unsigned long obj_lvl;	/**< Level. */
  char		*border;	/**< Border definition. */
  int		fill;		/**< Fill. */
  int		anchor;		/**< Anchor. */
  int		border_type;	/**< Border type. */
  int		border_subtype;	/**< Sub type. */
  int		bo_top;		/**< Border top. */
  int		bo_left;	/**< Border left. */
  int		bo_right;	/**< Border right. */
  int		bo_bottom;	/**< Border bottom. */
  char		*c_border;	/**< Contents border. */
  int		c_fill;		/**< Contents fill. */
  int		c_anchor;	/**< Contents anchor. */
  unsigned long	l_creation;	/**< Line number on which object was created. */
  int		layout;		/**< Layout to use for contents. */
  long		rows;		/**< Number of rows in gridback layout. */
  long		columns;	/**< Number of columns in gridbag layout. */
  long		last_row_used;	/**< Last used row. */
  long		last_col_used;	/**< Last column used. */
  long		pos_x;		/**< Position (x) as specified. */
  long		pos_y;		/**< Position (y) as specified. */
  long		pos_w;		/**< Size (w) as specified. */
  long		pos_h;		/**< Size (h) as specified. */
  long		cor_x;		/**< Corrected x after jlcheck(). */
  long		cor_y;		/**< Corrected y after jlcheck(). */
  long		cor_w;		/**< Corrected w after jlcheck(). */
  long		cor_h;		/**< Corrected h after jlcheck(). */
  long		cor_mx;		/**< Corrected mx. */
  long		cor_my;		/**< Corrected my. */
  char		*constructor;	/**< Constructor arguments. */
  char		*action_cmd;	/**< Action cmd. */
  char		*text;		/**< Text. */
  char		*icon_file;	/**< Icon file name. */
  char		*border_text;	/**< text for text border. */
  dk_storage_t	*l_st;		/**< listeners in normal sort order. */
  dk_storage_iterator_t	*l_it;	/**< Iterator for listeners. */
  dk_storage_t	*li_st;		/**< listeners in inverted sort order. */
  dk_storage_iterator_t	*li_it;	/**< Iterators for listeners inverted. */
  unsigned long n_listeners;	/**< number of listeners. */
  dk_storage_t	*c_st;		/**< contents. */
  dk_storage_iterator_t *c_it;	/**< Iterator through contents elements. */
  unsigned long	n_c;		/**< number of contents elements. */
  char		*tf_width;	/**< width (for text fields). */
  char		*card_add_text;	/**< text for add() in CardLayout. */
  long		current_x;	/**< current x position for insertion. */
  long		current_y;	/**< current y position for insertion. */
  long		current_w;	/**< current w for insertion. */
  long		current_h;	/**< current h for insertion. */
  long		max_x;		/**< last column used in gridbag layout. */
  long		max_y;		/**< last row used in gridbag layout. */
  unsigned long	f2;		/**< flag set. */
  long	grid_layout_r;		/**< rows in grid/gridbag layout. */
  long  grid_layout_c;		/**< columns in grid/gridbag layout. */
  dk_bitfield_t	*distance_x;	/**< added distance columns. */
  dk_bitfield_t	*distance_y;	/**< added distance rows. */
  double	weightx;	/**< weight x for gridbag constraints. */
  double	weighty;	/**< weight y for gridbag constraints. */
  long		s_min_x;	/**< Minimum x. */
  long		s_min_y;	/**< Minimum y. */
  long		s_pref_x;	/**< Preferred x. */
  long		s_pref_y;	/**< Preferred y. */
  long		s_max_x;	/**< Maximum x. */
  long		s_max_y;	/**< Maximum y. */
  char		*tt_text;	/**< tooltip text. */
  char		*acd_text;	/**< accessible description. */
  int		h_align;	/**< horizontal alignment. */
  int		v_align;	/**< vertical alignent. */
  int		h_textpos;	/**< horizontal text position. */
  int		v_textpos;	/**< vertibal text position. */
  char		*title;		/**< title for frames. */
  int		def_close;	/**< default close operation. */
  int		enab;		/**< enabled or not. */
  char		*font;		/**< the font. */
  char		*foreground;	/**< foreground color. */
  char		*background;	/**< background color. */
  unsigned char	font_changes;	/**< font style changes. */
  double	weightl;	/**< left border weight. */
  double	weightr;	/**< right border weight. */
  double	weightt;	/**< top border weight. */
  double	weightb;	/**< bottom border weight. */
} JLO;


/**	Node to store contents element.
 */
typedef struct _contents_node_ {
  unsigned long n;	/**< Number in order of appearance. */
  int		t;	/**< Type. */
  JLO		*o;	/**< Object. */
} JCN;



/**	Size group / button group
*/
typedef struct _size_group_ {
  char			*n;	/**< Group name. */
  dk_storage_t		*s;	/**< Storage containing object pointers. */
  dk_storage_iterator_t	*i;	/**< Iterator on storage. */
} JSG;



/**	Jlayout job.
*/
typedef struct {
  dk_app_t	*a;		/**< Application structure. */
  int		ac;		/**< Command line argc. */
  char		**av;		/**< Command line argv. */
  int		exval;		/**< Exit code for the main program. */
  char		*ifnameorig;	/**< Original input file name. */
  char		*ofnameorig;	/**< Original output file name. */
  char		*ifname;	/**< Real input file name. */
  char		*ofname;	/**< Real output file name. */
  char		**msg;		/**< The messages to print out. */
  int		run_silently;	/**< Run silently (1) or as filter (2). */
  unsigned long	errlineno;	/**< Line number for error messages. */
  char		*errfilename;	/**< File name in which the error is located. */
  int		cmd;		/**< Command to execute. */
  dk_storage_t	*l_st;		/**< Lines from original Java source file. */
  dk_storage_iterator_t	*l_it;	/**< Iterator through lines. */
  unsigned long	ln_variables;	/**< Iine number in *.java for variables. */
  unsigned long ln_setup;	/**< Line number for setup procedure. */
  unsigned long ln_cleanup;	/**< Line number for cleanup procedure. */
  unsigned long ln_import;	/**< Line number for import instructions. */
  char		*input_buffer;	/**< Buffer for input line. */
  size_t	sz_input_buffer;	/**< Size of \a input_buffer. */
  unsigned short	f1;	/**< Flag set 1. */
  dk_storage_t	*o_st;		/**< Objects sorted by name. */
  dk_storage_iterator_t	*o_it;	/**< Objects iterator. */
  dk_storage_t	*g_st;		/**< Size groups sorted by name. */
  dk_storage_iterator_t	*g_it;	/**< Size group iterator. */
  dk_storage_t	*b_st;		/**< Button group storage. */
  dk_storage_iterator_t	*b_it;	/**< Button group iterator. */
  dk_storage_t	*uc_st;		/**< Used classes. */
  dk_storage_iterator_t *uc_it;	/**< Iterator through used classes. */
  JLO		*o_main;	/**< Main object. */
  JLO		*o_cur;		/**< Current object. */
  char*		def_border;	/**< Default border. */
  int		def_fill;	/**< Default fill. */
  int		def_anchor;	/**< Default anchor. */
  unsigned long	max_obj_no;	/**< Maximum object number. */
  unsigned long	max_level;	/**< Maximum level. */
  int		foldmt;		/**< Fold marker types. */
  unsigned char	use_dklibsj;	/**< Flag: Use dklibsj libraries. */
  unsigned char freeze;		/**< Freeze dklibsj analysis. */
} JLJ;



/*	jlmsg module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if JLMSG_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Retrieve key/value pairs for messages.
 * 	@return	Pointer to key/value pair array.
 */
EXTERN
dk_key_value_t *
jlmsg_msg DK_PR((void));

/**	Get size of key/value pairs array.
 * 	@return Size of the array (number of elements).
 */
EXTERN
size_t
jlmsg_size DK_PR((void));

/**	Log simple message.
 * 	@param	j	Jlayout job.
 * 	@param	l	Log level.
 * 	@param	msgno	Index of message in array.
 */
EXTERN
void
jlmsg_log1 DK_PR((JLJ *j, int l, size_t msgno));

/**	Log multipart message (3 parts).
 * 	@param	j	Jlayout job.
 * 	@param	l	Log level.
 * 	@param	msg1	Index of first message part in string array.
 * 	@param	fn	File name.
 * 	@param	msg2	Index of second message part in string array.
 */
EXTERN
void
jlmsg_log3 DK_PR((JLJ *j, int l, size_t msg1, char *fn, size_t msg2));

/**	Issue error message: Not enough memory.
 * 	@param	j	JLayout job.
 */
EXTERN
void
jlmsg_error_memory DK_PR((JLJ *j));
#if defined(__cplusplus)
};
#endif



/*	jsrc module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if JSRC_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Prepare to read Java source file.
 * 	@param	j	Jlayout job.
 * 	@return	1 on success, 0 on error.
 */
EXTERN
int
jsrc_setup DK_PR((JLJ *j));

/**	Read Java source file.
 * 	@param	j	Jlayout job.
 * 	@return	1 on success, 0 on error.
 */
EXTERN
int
jsrc_read DK_PR((JLJ *j));

/**	Write Java source file.
 * 	@param	j	JLayout job.
 * 	@return	1 on success, 0 on error.
 */
EXTERN
int
jsrc_write DK_PR((JLJ *j));

/**	Clean up jsrc modules usage of \a j.
 * 	@param	j	Jlayout job.
 */
EXTERN
void
jsrc_cleanup DK_PR((JLJ *j));

/**	Delete one source code line.
 * 	@param	l	Line to delete.
 */
EXTERN
void
jsrc_line_delete DK_PR((JSRC_LINE *l));

/**	Create one source code line.
 * 	@param	s	Input buffer contents.
 * 	@param	l	Line number.
 * 	@return	Pointer to stored line on success, NULL on error.
 */
EXTERN
JSRC_LINE *
jsrc_line_new DK_PR((char *s, unsigned long l));
#if defined(__cplusplus)
};
#endif



/*	jlread module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if JLREAD_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Read *jl file.
 * 	@param	j	Jlayout job.
 * 	@return	1 on success, 0 on error.
 */
EXTERN
int
jlread DK_PR((JLJ *j));

/**	Compare two objects by name.
 * 	@param	l	Left object.
 * 	@param	r	Right object.
 * 	@param	cr	Comparison criteria (ignored).
 * 	@return	Comparison result.
 */
EXTERN
int
jlread_compare_objects_by_name DK_PR((void *l, void *r, int cr));

/**	Delete contents node, release memory.
 * 	@param	j	Contents node to delete.
 */
EXTERN
void
jlread_cn_delete DK_PR((JCN *j));

/**	Create contents node.
 * 	@param	n	Node number.
 * 	@param	o	GUI object.
 * 	@return	Pointer to new contents node on success, NULL on error.
 */
EXTERN
JCN *
jlread_cn_new DK_PR((unsigned long n, JLO *o));
#if defined(__cplusplus)
};
#endif



/*	jlclean module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if JLCLEAN_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Clean up jlayout job.
 * 	@param	j	Jlayout job.
 */
EXTERN
void
jlcleanup DK_PR((JLJ *j));
#if defined(__cplusplus)
};
#endif



/*	jlo module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if JLO_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Delete GUI object.
 * 	@param	o	GUI object to delete.
 */
EXTERN
void
jlo_delete DK_PR((JLO *o));

/**	Create GUI object.
 * 	@param	on	Object name.
 * 	@return	Pointer to new object.
 */
EXTERN
JLO *
jlo_new DK_PR((char *on));

/**	Set classname.
 * 	@param	o	GUI object.
 * 	@param	cn	Class name.
 * 	@return	1 on success, 0 on error.
 */
EXTERN
int
jlo_set_classname DK_PR((JLO *o, char *cn));

/**	Retrieve class name from object.
 * 	@param	o	GUI object.
 * 	@return	Class name.
 */
EXTERN
char *
jlo_get_classname DK_PR((JLO *o));

/**	Create new size or button group.
 * 	@param	n	Group name.
 * 	@return	Pointer to new group on success, NULL on error.
 */
EXTERN
JSG *
jsg_new DK_PR((char *n));

/**	Delete size or button group.
 * 	@param	j	Group to delete.
 */
EXTERN
void
jsg_delete DK_PR((JSG *j));

/**	Delete event listener.
 * 	@param	jl	Listener to delete.
 */
EXTERN
void
jlo_listener_delete DK_PR((JLISTENER *jl));

/**	Create new event listener.
 * 	@param	tp	Event type.
 * 	@param	t	Text.
 * 	@param	n	listener ID.
 * 	@return	Pointer to new listener on success, NULL on error.
 */
EXTERN
JLISTENER *
jlo_listener_new DK_PR((int tp, char *t, unsigned long n));
#if defined(__cplusplus)
};
#endif



/*	jlcheck module
*/
#if defined(EXTERN)
#undef EXTERN
#endif
#if JLCHECK_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif
#if defined(__cplusplus)
extern "C" {
#endif

/**	Check (for completeness and consistency).
 * 	@param	j	Jlayout job.
 * 	@return	1 on success, 0 on error.
 */
EXTERN
int
jlcheck DK_PR((JLJ *j));
#if defined(__cplusplus)
};
#endif

#ifdef EXTERN
#undef EXTERN
#endif
#if JLCONF_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

/** Read configuration.
 *  @param j	Jlayout job structure.
 */
EXTERN
void
jlconf_read DK_PR((JLJ *j));

#ifdef __cplusplus
};
#endif


/** @defgroup f1_have	Flags to indicate known information about an object. */
/*@{*/
/**	Object is in a size group. */
#define F1_HAVE_SIZE_GROUP	0x00000001UL

/**	Gridbag layout. */
#define F1_HAVE_GRIDBAG		0x00000002UL

/**	Icon file names. */
#define F1_HAVE_ICON_FILE_NAMES	0x00000004UL

/**	Button groups. */
#define F1_HAVE_BUTTON_GROUPS	0x00000008UL

/**	Tool tip text as accessable description. */
#define F1_USE_TTT_AS_AD	0x00000010UL

/**	Font style. */
#define F1_HAVE_FONT_STYLE	0x00000020UL
/*@}*/


/**	Flag 2: Use content pane.
*/
#define F2_USE_CONTENT_PANE	0x00000001UL

/**	@defgroup	classes	Well-known class IDs. */
/*@{*/
/* number n corresponds to index i in jlread/known_classes: i = n -1 */

/**	No class specified yet. */
#define CLASS_NOT_SPECIFIED	(-1)

/**	User defined class. */
#define CLASS_USER_DEFINED	0

/**	Existing object. */
#define CLASS_EXISTING_OBJECT	1

/**	JFrame. */
#define CLASS_JFRAME		2

/**	JDialog. */
#define CLASS_JDIALOG		3

/**	JPanel. */
#define CLASS_JPANEL		4

/**	JApplet. */
#define CLASS_JAPPLET		5

/**	JButton. */
#define CLASS_JBUTTON		6

/**	JCheckBox. */
#define CLASS_JCHECKBOX		7

/**	JRadioButton. */
#define CLASS_JRADIOBUTTON	8

/**	JCheckBoxMenuItem. */
#define CLASS_JCHECKBOXMENUITEM	9

/**	JRadioButtonMenuItem. */
#define CLASS_JRADIOBUTTONMENUITEM	10

/**	JToggleButton. */
#define CLASS_JTOGGLEBUTTON	11

/**	JColorChooser. */
#define CLASS_JCOLORCHOOSER	12

/**	JComboBox. */
#define CLASS_JCOMBOBOX		13

/**	JOptionPane. */
#define CLASS_JOPTIONPANE	14

/**	JEditorPane. */
#define CLASS_JEDITORPANE	15

/**	JTextPane. */
#define CLASS_JTEXTPANE		16

/**	JFileChooser. */
#define CLASS_JFILECHOOSER	17

/**	JFormattedTextField. */
#define CLASS_JFORMATTEDTEXTFIELD	18

/**	JInternalFrame. */
#define CLASS_JINTERNALFRAME	19

/**	JDesktopPane. */
#define CLASS_JDESKTOPPANE	20

/**	JLayeredPane. */
#define CLASS_JLAYEREDPANE	21

/**	JLabel. */
#define CLASS_JLABEL		22

/**	JList. */
#define CLASS_JLIST		23

/**	JMenuBar. */
#define CLASS_JMENUBAR		24

/**	JMenu. */
#define CLASS_JMENU		25

/**	JMenuItem. */
#define CLASS_JMENUITEM		26

/**	JPasswordField. */
#define CLASS_JPASSWORDFIELD	27

/**	JProgressBar. */
#define CLASS_JPROGRESSBAR	28

/**	JScrollPane. */
#define CLASS_JSCROLLPANE	29

/**	JSeparator. */
#define CLASS_JSEPARATOR	30

/**	JSlider. */
#define CLASS_JSLIDER		31

/**	JSpinner. */
#define CLASS_JSPINNER		32

/**	JSplitPane. */
#define CLASS_JSPLITPANE	33

/**	JTabbedPane. */
#define CLASS_JTABBEDPANE	34

/**	JTable. */
#define CLASS_JTABLE		35

/**	JTextArea. */
#define CLASS_JTEXTAREA		36

/**	JTextField. */
#define CLASS_JTEXTFIELD	37

/**	JToolBar. */
#define CLASS_JTOOLBAR		38

/**	JComponent. */
#define CLASS_JCOMPONENT	39

/**	JTree. */
#define CLASS_JTREE		40

/**	JPopupMenu. */
#define CLASS_JPOPUPMENU	41
/*@}*/

/**	@defgroup	contentsnodes	Content node types. */
/*@{*/
/**	Node type not (yet) specified. */
#define JCN_NOT_SPECIFIED	(-1)

/**	Object. */
#define JCN_OBJECT		0

/**	Glue. */
#define JCN_GLUE		1

/**	Separator. */
#define JCN_SEPARATOR		2
/*@}*/


/**	@defgroup	layouts	Corresponds to layout_values in jlread.ctr */
/*@{*/
/**	No layout specified (yet). */
#define LAYOUT_NOT_SPECIFIED	(-1)

/**	No layout. */
#define LAYOUT_NONE		0

/**	Gridbag layout. */
#define LAYOUT_GRIDBAG		1

/**	Card layout. */
#define LAYOUT_CARD		2

/**	Border layout. */
#define LAYOUT_BORDER		3

/**	X box (line). */
#define LAYOUT_BOX_X		4

/**	Y box (column). */
#define LAYOUT_BOX_Y		5

/**	Page box layout (top to bottom). */
#define LAYOUT_BOX_PAGE		6

/**	Line layout (left to right). */
#define LAYOUT_BOX_LINE		7

/**	Flow layout. */
#define LAYOUT_FLOW		8

/**	Grid layout. */
#define LAYOUT_GRID		9

/**	Group layout. */
#define LAYOUT_GROUP		10

/**	Spring layout. */
#define LAYOUT_SPRING		11
/*@}*/


/**	@defgroup	borders	Border types. */
/*@{*/

/**	No border specified (yet). */
#define BORDER_NOT_SPECIFIED	(-1)

/**	Border entry is empty. */
#define BORDER_EMPTY		0

/**	Etched (lowered) border. */
#define BORDER_ETCHED_LOWERED	1

/**	Etched (raised) border. */
#define BORDER_ETCHED_RAISED	2

/**	Beveled (lowered) border. */
#define BORDER_BEVEL_LOWERED	3

/**	Beveled (raised) border. */
#define BORDER_BEVEL_RAISED	4

/**	Line border. */
#define BORDER_LINE		5

/**	Matted border. */
#define BORDER_MATTE		6

/**	Titled border. */
#define BORDER_TITLED		7
/*@}*/


/**	@defgroup	fillings	Fill types. */
/*@{*/

/**	No fill type specified (yet). */
#define FILL_NOT_SPECIFIED	(-1)

/**	Not filled. */
#define FILL_NONE		0

/**	Fill horizontally. */
#define FILL_HORIZONTAL		1

/**	Fill vertically. */
#define FILL_VERTICAL		2

/**	Fill both horizontally and vertically. */
#define FILL_BOTH		3
/*@}*/


/**	@defgroup	anchors	Anchor positionss. */
/*@{*/

/**	No anchor specified (yet). */
#define ANCHOR_NOT_SPECIFIED		(-1)

/**	Center. */
#define ANCHOR_CENTER			0

/**	North. */
#define ANCHOR_NORTH			1

/**	Northeast. */
#define ANCHOR_NORTHEAST		2

/**	East. */
#define ANCHOR_EAST			3

/**	Southeast. */
#define ANCHOR_SOUTHEAST		4

/**	South. */
#define ANCHOR_SOUTH			5

/**	Southwest. */
#define ANCHOR_SOUTHWEST		6

/**	West. */
#define ANCHOR_WEST			7

/**	Northwest. */
#define ANCHOR_NORTHWEST		8

/**	Start of page. */
#define ANCHOR_PAGE_START		9

/**	End of page. */
#define ANCHOR_PAGE_END			10

/**	Start of line. */
#define ANCHOR_LINE_START		11

/**	End of line. */
#define ANCHOR_LINE_END			12

/**	Start of first line. */
#define ANCHOR_FIRST_LINE_START		13

/**	End of first line. */
#define ANCHOR_FIRST_LINE_END		14

/**	Start of last line. */
#define ANCHOR_LAST_LINE_START		15

/**	End of last line. */
#define ANCHOR_LAST_LINE_END		16

/**	Baseline. */
#define ANCHOR_BASELINE			17

/**	Baseline leading. */
#define ANCHOR_BASELINE_LEADING		18

/**	Baseline trailing. */
#define ANCHOR_BASELINE_TRAILING	19

/**	Above baseline. */
#define ANCHOR_ABOVE_BASELINE		20

/**	Above baseline leading. */
#define ANCHOR_ABOVE_BASELINE_LEADING	21

/**	Above baseline trailing. */
#define ANCHOR_ABOVE_BASELINE_TRAILING	22

/**	Below baseline. */
#define ANCHOR_BELOW_BASELINE		23

/**	Below baseline leading. */
#define ANCHOR_BELOW_BASELINE_LEADING	24

/**	Below baseline trailing. */
#define ANCHOR_BELOW_BASELINE_TRAILING	25

/**	Maximum numeric value for anchors. */
#define ANCHOR_MAX			25
/*@}*/


/* commands decide what the program has to do */
/**	@defgroup	commands	Commands for jlayout. */
/*@{*/

/**	Run normally. */
#define RUN_CMD_RUN		0

/**	Show version. */
#define RUN_CMD_VERSION		1

/**	Show help. */
#define RUN_CMD_HELP		2

/**	Save configuration settings. */
#define RUN_CMD_CONFIGURE	4

/**	Unconfigur application (remove configuration settings). */
#define RUN_CMD_UNCONFIGURE	8

/**	Show current configuration. */
#define RUN_CMD_SHOW_CONFIG	16
/*@}*/


/**	@defgroup	eventlisteners	Event listeners. */
/*@{*/
/**	No listener specified (yet). */
#define LISTENER_NOT_SPECIFIED	(-1)

/**	Action listener. */
#define LISTENER_ACTION			0

/**	Window listener. */
#define LISTENER_WINDOW			1

/**	Item listener. */
#define LISTENER_ITEM			2

/**	Caret listener. */
#define LISTENER_CARET			3

/**	Change listener. */
#define LISTENER_CHANGE			4

/**	Component listener. */
#define LISTENER_COMPONENT		5

/**	Container listener. */
#define LISTENER_CONTAINER		6

/**	Document listener. */
#define LISTENER_DOCUMENT		7

/**	Focus listener. */
#define LISTENER_FOCUS			8

/**	Internal frame listener. */
#define LISTENER_INTERNAL_FRAME		9

/**	Key listener. */
#define LISTENER_KEY			10

/**	List data listener. */
#define LISTENER_LIST_DATA		11

/**	List selection listener. */
#define LISTENER_LIST_SELECTION		12

/**	Mouse listener. */
#define LISTENER_MOUSE			13

/**	Mouse motion listener. */
#define LISTENER_MOUSE_MOTION		14

/**	Mouse Wheel listener. */
#define LISTENER_MOUSE_WHEEL		15

/**	Property change listener. */
#define LISTENER_PROPERTY_CHANGE	16

/**	Table model listener. */
#define LISTENER_TABLE_MODEL		17

/**	Tree expansion listener. */
#define LISTENER_TREE_EXPANSION		18

/**	Tree model listener. */
#define LISTENER_TREE_MODEL		19

/**	Tree selection listener. */
#define LISTENER_TREE_SELECTION		20

/**	Tree will expand listener. */
#define LISTENER_TREE_WILL_EXPAND	21

/**	Undoable edit listener. */
#define LISTENER_UNDOABLE_EDIT		22

/**	Transfer handler listener. */
#define LISTENER_TRANSFER_HANDLER	23

/**	Maximum numeric value for listeners. */
#define LISTENER_MAX			23
/*@}*/


/**	@defgroup	objectattributes	Object attributes. */
/*@{*/
/**	Constructor. */
#define OBJ_ATT_CONSTRUCTOR		0x0001

/**	Action command. */
#define OBJ_ATT_ACTION_CMD		0x0002

/**	Text. */
#define OBJ_ATT_TEXT			0x0004

/**	Icon file. */
#define OBJ_ATT_ICON_FILE		0x0008

/**	Text field width. */
#define OBJ_ATT_TEXTFIELD_WIDTH		0x0010
/*@}*/


/**	@defgroup	horizontalalign	Horizontal align. */
/*@{*/
/**	Alignment not (yet) specified. */
#define HALIGN_NOT_SPECIFIED		(-1)

/**	Leading. */
#define HALIGN_LEADING			0

/**	Left. */
#define HALIGN_LEFT			1

/**	Centered. */
#define HALIGN_CENTER			2

/**	Right. */
#define HALIGN_RIGHT			3

/**	Trailing. */
#define HALIGN_TRAILING			4
/*@}*/


/**	@defgroup	verticalalign	Vertical alignment. */
/*@{*/
/**	Alignment not specified (yet). */
#define VALIGN_NOT_SPECIFIED		(-1)

/**	Top. */
#define VALIGN_TOP			0

/**	Centered. */
#define VALIGN_CENTER			1

/**	Bottom. */
#define VALIGN_BOTTOM			2
/*@}*/


/**	@defgroup	horizontaltextpos	Horizontal text positision. */
/*@{*/
/**	not specified (yet). */
#define HTEXTPOS_NOT_SPECIFIED		(-1)

/**	Left. */
#define HTEXTPOS_LEFT			0

/**	Centered. */
#define HTEXTPOS_CENTER			1

/**	Right. */
#define HTEXTPOS_RIGHT			2

/**	Leading. */
#define HTEXTPOS_LEADING		3

/**	Trailing. */
#define HTEXTPOS_TRAILING		4
/*@}*/


/**	@defgroup	verticaltextpos	Vertical text position. */
/*@{*/

/**	Not specified (yet). */
#define VTEXTPOS_NOT_SPECIFIED		(-1)

/**	Top. */
#define VTEXTPOS_TOP			0

/**	Centered. */
#define VTEXTPOS_CENTER			1

/**	Bottom. */
#define VTEXTPOS_BOTTOM			2
/*@}*/



/**	@defgroup	closeoperation	Close operation. */
/*@{*/
/**	Not specified (yet). */
#define CLOSE_OPERATION_NOT_SPECIFIED	(-1)

/**	Do nothing. */
#define CLOSE_OPERATION_NOTHING		0

/**	Hide window. */
#define CLOSE_OPERATION_HIDE		1

/**	Dispose window. */
#define CLOSE_OPERATION_DISPOSE		2

/**	Exit application. */
#define CLOSE_OPERATION_EXIT		3
/*@}*/

/**	Comparison criterium: Compare objects. */
#define CR_COMPARE_OBJECTS		0

/**	Comparison criterium: Compare objects and names. */
#define CR_COMPARE_OBJECT_AND_NAME	1

/**	Enabled attribute: Not specified. */
#define ENABLED_NOT_SPECIFIED		(-1)

/**	Enabled attribute: Disabled. */
#define ENABLED_NO			0

/**	Enabled attribute: Enabled. */
#define ENABLED_YES			1

/**	Fold markers: Use no fold markers. */
#define FOLD_MARKER_NONE		0

/**	Fold markers: Markers for vim. */
#define FOLD_MARKER_VIM			0x0001

/**	Fold markers: Markers for NetBeans. */
#define FOLD_MARKER_NETBEANS		0x0002

#endif
/* JL_H_INCLUDED */

