/*
Copyright (c) 2004-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	dkbsp.h	Bezier spline calculations.
\f{eqnarray*}
x(t)&=&{(1-t)}^3x_0+3t{(1-t)}^2x_{0+}+3t^2(1-t)x_{1-}+t^3x_1\\[0.2em]
{\left.\frac{\textrm{d}x}{\textrm{d}t}\right|}_{t=0}&=&-3x_0+3x_{0+}\\[0.2em]
{\left.\frac{\textrm{d}x}{\textrm{d}t}\right|}_{t=1}&=&-3x_{1-}+3x_1\\[0.2em]
x_{0+}&=&x_0+\frac{1}{3}\cdot{}{\left.\frac{\textrm{d}x}{\textrm{d}t}\right|}_{t=0}\\[0.2em]
x_{1-}&=&x_1-\frac{1}{3}\cdot{}{\left.\frac{\textrm{d}x}{\textrm{d}t}\right|}_{t=1}\\
\f}
*/



#ifndef DKBSP_INC



/** Protection from multiple inclusion. */
#define DKBSP_INC 1



#include <dk.h>



/**	Bezier spline segment.
*/
typedef struct {
  double x0;		/**< x at t=0. */
  double dxdt0;		/**< dt/dx at t=0. */
  double x1;		/**< x at t=1. */
  double dxdt1;		/**< dt/dx at t=1. */
  double xp0;		/**< x0+. */
  double xm1;		/**< x1-. */
  double min;		/**< Minimum x value. */
  double max;		/**< Maximum x value. */
  double xvalue;	/**< Value for a given t. */
  double dxdt;		/**< First derivative. */
} dk_bspline_t;

#if defined(EXTERN)
#undef EXTERN
#endif
#if DKBSP_C
#define EXTERN /* nix */
#else
#if DK_HAVE_PROTOTYPES
#define EXTERN /* nix */
#else
#define EXTERN extern
#endif
#endif



/**	Calculate Bezier spline control points from
	x_0, dx/dt at x_0, dx/dt at x_1 and x_1.
	The result is stored in the dk_bspline_t
	structure.
	@param	s	Bezier spline structure.
	@param	x0	x at start of interval.
	@param	d0	dx/dt at x0.
	@param	x1	x at end of interval.
	@param	d1	dx/dt at x1.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkbsp_calculate DK_PR((dk_bspline_t *s,double x0, double d0, double x1, double d1));



/**	Calculate minimum and maximum x value for a Bezier spline segment.
	@param	s	Bezier spline structure.
	@param	x0	X-value of "left" control point.
	@param	d0	X-value of second control point.
	@param	x1	X-value of "right" control point.
	@param	d1	X-value of third control point.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkbsp_minmax DK_PR((dk_bspline_t *s,double x0, double d0, double x1, double d1));



/**	Apply variable, calculate value and derivative to find minimum
	and maximum.
	@param	s	Bezier spline structure.
	@param	x0	Left end point.
	@param	xp	Left control point.
	@param	x1	Right end point.
	@param	xm	Right control point.
	@param	t	The t value.
	@return	1 on success, 0 on error.
*/
EXTERN int
dkbsp_for_t DK_PR((dk_bspline_t *s,double x0,double xp,double x1,double xm,double t));



#endif

