/*
Copyright (c) 2004-2010, Dirk Krause
All rights reserved.

Redistribution and use in source and binary forms,
with or without modification, are permitted provided
that the following conditions are met:

* Redistributions of source code must retain the above
  copyright notice, this list of conditions and the
  following disclaimer.
* Redistributions in binary form must reproduce the above 
  opyright notice, this list of conditions and the following
  disclaimer in the documentation and/or other materials
  provided with the distribution.
* Neither the name of the Dirk Krause nor the names of
  contributors may be used to endorse or promote
  products derived from this software without specific
  prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED.
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/



/**	@file	dkfigfnt.c	Fig fonts.
*/



/**	Inside the dkfigfnt module. */
#define DKFIGFNT_C 1



#include "dkfig.h"




#line 54 "dkfigfnt.ctr"




/**	Compare two font handling structure.
	Comparison by handling type, font number,
	font size, original font number and
	font flags.
	@param	pl	Left font handling structure.
	@param	pr	Right font handling structure.
	@param	cr	Comparison criteria (ignored).
	@return	The comparison result.
*/
int
dkfig_fnt_comp_fonth DK_P3(void *,pl, void *,pr, int,cr)
{
  int back = 0;
  dk_fig_fonth_t *fl, *fr;
  
  fl = (dk_fig_fonth_t *)pl; fr = (dk_fig_fonth_t *)pr;
  if(fl) {
    if(fr) {
      if((fl->handling) > (fr->handling)) {
        back = 1;
      } else {
        if((fl->handling) < (fr->handling)) {
	  back = -1;
	}
      }
      if(!back) {
        if((fl->fontno) > (fr->fontno)) {
	  back = 1;
	} else {
	  if((fl->fontno < fr->fontno)) {
	    back = -1;
	  }
	}
      }
      if(!back) {
        double d1, d2, d3;
	d1 = fabs(fl->fontsize); d2 = fabs(fr->fontsize);
	d3 = ((d1 < d2) ? (d1) : (d2));
	d3 = d3 * 0.0001;
	if(fabs((fl->fontsize) - (fr->fontsize)) > d3) {
        if((fl->fontsize) > (fr->fontsize)) {
	  back = 1;
	} else {
	  if((fl->fontsize) < (fr->fontsize)) {
	    back = -1; 
	  }
	}
	}
      }
      if(!back) {
        if(fl->ofontno < fr->ofontno) {
	  back = -1;
	} else {
	  if(fl->ofontno > fr->ofontno) {
	    back = 1;
	  }
	}
      }
      if(!back) {
        if(fl->oflags > fr->oflags) {
	  back = 1;
	} else {
	  if(fl->oflags < fr->oflags) {
	    back = -1;
	  }
	}
      }
    } else {
      back = 1;
    }
  } else {
    if(fr) {
      back = -1;
    }
  }
  
  return back;
}



/**	Translate LaTeX font selection from Fig file
	into index in PS fonts array.
	@param	i	Fig font index.
	@return	PS font name index.

*/
static int
figlatfont_to_number DK_P1(int, i)
{ 
  int back = 0;
  
  switch(i) {
    case 1: {			
      back = 0;
    } break;
    case 2: {			
      back = 2;
    } break;
    case 3: {			
      back = 1;
    } break;
    case 4: {			
      back = 16;
    } break;
    case 5: {			
      back = 12;
    } break;
    default: {			
      back = 0;
    } break;
  } 
  return back;
}



/**	Translate font LaTeX font selection in XFig into
	a bit set of font features
	(roman/sans-serif/typwriter, normal/bold, upright/italic).
	@param	i	Fig font index.
	@return	Feature flags.
*/
static int
figlatfont_to_features DK_P1(int, i)
{
  int back = 0;
  
  switch(i) {
    case 1: {			
      back = DKFIG_FONT_RM;
    } break;
    case 2: {			
      back = DKFIG_FONT_BD;
    } break;
    case 3: {			
      back = DKFIG_FONT_IT;
    } break;
    case 4: {			
      back = DKFIG_FONT_SF;
    } break;
    case 5: {			
      back = DKFIG_FONT_TT;
    } break;
    default: {			
      back = 0;
    } break;
  } 
  return back;
}



/**	Fill font handling structure to match requirements
	of a Fig text object.
	@param	f	Destination font handling structure.
	@param	n	Text handling type for normal texts.
	@param	s	Text handling type for special texts.
	@param	t	Fig text object.
*/
void
dkfig_fnt_fonth_for_text DK_P4(dk_fig_fonth_t *,f, int,n, int,s, dk_fig_text *,t)
{
  int th;
  int ofn;	/* original font number */
  
  f->handling = 0; f->fontno = 0; f->fontsize = 12.0;
  f->ofontno  = 0; f->oflags = 0;
  if(f && t) {
    f->ofontno = t->font;
    f->oflags  = t->font_flags;
    th = (((t->font_flags) & 2) ? s : n); 
    ofn = t->font; if(ofn < 0) ofn = 0; if(ofn >= 35) ofn = 34;
    f->handling = 0; f->fontno = ofn; f->fontsize = t->font_size;
    
    if(((t->font_flags) & 2) || (th & DKFIG_TH_HANDLING)) {	
      /* Options: text handled by LaTeX */
      if(th & DKFIG_TH_FONT_TEX) {		
        f->handling = 2;
      } else {					
        if(th & DKFIG_TH_FONT_SIMILAR) {	
	  if(th & DKFIG_TH_SIZE_FIG) {		
	    f->handling = 4;
	  } else {				
	    f->handling = 5;
	  }
	  if((t->font_flags) & 4) {		
	    /* f->fontno = fig_font_35[ofn].features; */
	    f->fontno = dkfont_get_features(ofn);
	  } else {				
	    f->fontno = figlatfont_to_features(ofn);
	  }
	} else {				
	  if((t->font_flags) & 4) {		
	    f->handling = 3;
	  } else {				
	    f->handling = 4;
	    f->fontno = figlatfont_to_features(ofn);
	  }
	}
      }
    } else {						
      /* Options: no LaTeX handling */
      if((t->font_flags) & 4) {			
        /* FIG: no LaTeX handling */
	if(th & DKFIG_TH_FONT_TEX) {		
	  f->handling = 0;
	} else {				
	  f->handling = 1;
	}
      } else {					
        /* FIG: text handled by LaTeX */
        /*
	  2004/05/13
	  As the options advice us not to use LaTeX we
	  convert the font number directly to use a
	  PS font.
	*/
	f->fontno = figlatfont_to_number(ofn);
	f->handling = ((th & DKFIG_TH_FONT_TEX) ? 0 : 1);
      }
    }
  }
  
}



/**	Create new font handling structure.
	@return	Pointer to new structure on success, NULL on error.
*/
dk_fig_fonth_t *
dkfig_fnt_new_fonth DK_P0()
{
  dk_fig_fonth_t *back = NULL;
  
  back = dk_new(dk_fig_fonth_t,1);
  if(back) {
    back->handling = 0;
    back->fontno   = 0;
    back->fontsize = 0.0;
    back->fonthno  = 0UL;
    back->ofontno  = 0;
    back->oflags   = 0;
  }
  
  return back;
}



/**	Create a copy of a font handling structure.
	@param	f	Source font handling structure.
	@return	Pointer to new copy on success, NULL on error.
*/
dk_fig_fonth_t *
dkfig_fnt_copy_fonth DK_P1(dk_fig_fonth_t *,f)
{
  dk_fig_fonth_t *back = NULL;
  
  if(f) {
    back = dk_new(dk_fig_fonth_t,1);
    if(back) {
      back->handling = f->handling;
      back->fontno   = f->fontno;
      back->fontsize = f->fontsize;
      back->ofontno  = f->ofontno;	/* bug fixed: was ... = f->fontno */
      back->oflags   = f->oflags;
    }
  }
  
  return back;
}



/**	Delete font handling structure, release memory.
	@param	f	Font handling structure to delete.
*/
void
dkfig_fnt_del_fonth DK_P1(dk_fig_fonth_t *,f)
{
  
  if(f) { 
    dk_delete(f);
  }
  
}




