package net.yura.grasshopper;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * An OutputStream that writes contents to a Logger upon each call to flush()
 */
class LoggingOutputStream extends ByteArrayOutputStream { 
 
    private String lineSeparator;
    private Logger logger;
    private Level level;
 
    /**
     * Constructor
     * @param logger Logger to write to
     * @param level Level at which to write the log message
     */
    public LoggingOutputStream(Level level) {
        super();
        this.logger = Logger.getLogger("");
        this.level = level;
        lineSeparator = System.getProperty("line.separator");
    }

    /**
     * upon flush() write the existing contents of the OutputStream
     * to the logger as a log record.
     * @throws java.io.IOException in case of error
     */
    public synchronized void flush() throws IOException {
        String record = this.toString();

        // only flush to logger if ends with a end of line char
        if (record.endsWith("\r\n") || record.endsWith("\n") || record.endsWith(lineSeparator)) {
            super.reset();

            if (record.endsWith("\r\n")) {
                record = record.substring(0, record.length()-2);
            }
            else if (record.endsWith("\n")) {
                record = record.substring(0, record.length()-1);
            }
            else {
                record = record.substring(0, record.length()-lineSeparator.length());
            }

            logger.log(level, record);
        }
    }
}
