package net.yura.grasshopper;

import java.io.FileDescriptor;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.logging.ErrorManager;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;

/**
 * Like the normal JUL ConsoleHandler except that it sends to System.out and System.err,
 * not just System.err like the JUL one.
 * This Handler does nothing on Android and so is not included in the shares source
 * @author Yura Mamyrin
 * @see java.util.logging.ConsoleHandler
 */
public class ConsoleHandler extends Handler {

    // we use a PrintStream as it does not throw IOExcption with using javaw (a Writer does)
    PrintStream out = new PrintStream(new FileOutputStream(FileDescriptor.out));
    PrintStream err = new PrintStream(new FileOutputStream(FileDescriptor.err));
    
    public ConsoleHandler() {
        setFormatter( new ReallySimpleFormatter() );
    }

    public synchronized void publish(LogRecord record) {
	if (!isLoggable(record)) {
	    return;
	}
	String msg;
	try {
 	    msg = getFormatter().format(record);
	} catch (Exception ex) {
	    // We don't want to throw an exception here, but we
	    // report the exception to any registered ErrorManager.
	    reportError(null, ex, ErrorManager.FORMAT_FAILURE);
	    return;
	}

	try {
            PrintStream ps = (record.getLevel().intValue() >= Level.WARNING.intValue() )?err:out;
	    ps.print(msg);
	} catch (Exception ex) {
	    // We don't want to throw an exception here, but we
	    // report the exception to any registered ErrorManager.
	    reportError(null, ex, ErrorManager.WRITE_FAILURE);
	}
    }

    public synchronized void flush() {
    }

    public synchronized void close() throws SecurityException {
        try {
            out.close();
            err.close();
        } catch (Exception ex) {
            // We don't want to throw an exception here, but we
            // report the exception to any registered ErrorManager.
            reportError(null, ex, ErrorManager.CLOSE_FAILURE);
        }
        out = null;
        err = null;
    }
}
