/*******************************************************************************
 * Copyright (c) 2004, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdktypes.h>
#include <gdk/gdkx.h>
#include <jni.h>

#define EPSILON 1e-10

struct _GdkWindowPaint
{
  GdkRegion *region;
  GdkPixmap *pixmap;
  gint x_offset;
  gint y_offset;
};
typedef struct _GdkWindowPaint GdkWindowPaint;

char* copyright = "Copyright (c) 2004 IBM Corporation";
GtkWidget* mainWidget;
GdkWindow* currentWindow;
int maxWidth, maxHeight;
#ifdef DEBUG
char msg[540000];
#endif 

//////////////////////////////////////////////////////
//
//////////////////////////////////////////////////////
void sendExposeSignal(GtkWidget* widget, GdkWindow* window){
	if(GTK_WIDGET_REALIZED(widget)){
		GdkEventExpose ee;
		ee.type=GDK_EXPOSE;
		ee.window = window;
		ee.send_event=TRUE;
		ee.area.x=0;ee.area.y=0;
		gdk_window_get_geometry(window, NULL,NULL,&ee.area.width,&ee.area.height,NULL);
		ee.region=gdk_region_rectangle(&ee.area);
		#ifdef DEBUG_EVENTS
		sprintf(msg, "%s\t>>>>>> SEND EXPOSE-EVENT:%s %x (%i,%i %ix%i)<<%x>>\r\n", msg,gtk_widget_get_name(widget), (unsigned int)window, ee.area.x, ee.area.y, ee.area.width, ee.area.height,  (ee.type == GDK_FOCUS_CHANGE || GTK_WIDGET_REALIZED(widget)));
		#endif
		gtk_widget_send_expose(widget,(GdkEvent*)&ee);
	}
}

//////////////////////////////////////////////////////
// Force expose on all widgets
//////////////////////////////////////////////////////
GtkSpinButton* currentSpinButton = NULL;
void signalExposeOnWidgets(GtkWidget*);
void signalExposeOnWidgetsCallBack(GtkWidget* widget, gpointer data){
	signalExposeOnWidgets(widget);
}
void signalExposeOnWidgets(GtkWidget* widget){
	sendExposeSignal(widget, currentWindow);
	// Bug 96228:
	if(strcmp("GtkSpinButton", gtk_widget_get_name(widget))==0){
		GtkSpinButton* spinButton = (GtkSpinButton*) widget;
		if(spinButton->panel == currentWindow){
			currentSpinButton = spinButton;
		}
	}
	
	// keep going through children
	// to find out the owner of the targetWindow
	if(GTK_IS_CONTAINER(widget)){
		GtkContainer* container = (GtkContainer*) widget;
		gtk_container_forall(container, signalExposeOnWidgetsCallBack, NULL);
	}
}

////////////////////////////////////////
// Adds and removes expose-event handlers
///////////////////////////////////////
#ifdef DEBUG_EVENTS
int eventAdd = 1;
int handlerIDCount=0;
gulong handlerID[500];
#endif
#ifdef DEBUG
void walkWidget(GtkWidget* , int );
void traverseChildrenCallback(GtkWidget* widget, gpointer data){
        int* levelp = (int*) data;
        walkWidget(widget, (*levelp)+1);
}
#endif
#ifdef DEBUG_EVENTS
void exposeCallback(GtkWidget* widget, gpointer data){
        GdkEventExpose* ee = (GdkEventExpose*) data;
        sprintf(msg, "%s \t\t <<<<<< RECEIVED EXPOSE-EVENT(%s:%x:mask=%x)(type=%x,window=%x,area=%i,%i %ix%i)\r\n", msg,gtk_widget_get_name(widget),(unsigned int)widget,gtk_widget_get_events(widget),ee->type,(unsigned int)ee->window,ee->area.x,ee->area.y,ee->area.width,ee->area.height);
}
#endif
#ifdef DEBUG
void walkWidget(GtkWidget* widget, int level){
        int lc;
        for(lc=0;lc<level;lc++)
                sprintf(msg,"%s****", msg);
        GdkWindow* window = widget->window;
        int wx,wy,ww,wh,wd;
        gdk_window_get_geometry(window, &wx,&wy,&ww,&wh,&wd);
        sprintf(msg, "%s(%s:%x:mask=%x) (window=%x)(db=%x) ((%i,%i) (%ix%i) [%i])\r\n", msg,gtk_widget_get_name(widget),(unsigned int)widget, gtk_widget_get_events(widget),(unsigned int) window,GTK_WIDGET_DOUBLE_BUFFERED(widget),wx,wy,ww,wh,wd);
        if(GTK_IS_CONTAINER(widget)){
                GtkContainer* container = (GtkContainer*) widget;
                gtk_container_forall(container, traverseChildrenCallback, &level);
        }
        #endif
        #ifdef DEBUG_EVENTS
        if(eventAdd==1){
                int hid = g_signal_connect_after(widget, "expose-event", G_CALLBACK(exposeCallback), NULL);
                sprintf(msg, "%s>> Adding signal %i - ID:%i\r\n", msg,handlerIDCount, hid);
                handlerID[handlerIDCount] = hid;
                handlerIDCount++;
        }else{
                sprintf(msg, "%s<< Removing signal %i - ID:%i\r\n", msg,handlerIDCount,(unsigned int)handlerID[handlerIDCount]);
                g_signal_handler_disconnect(widget, handlerID[handlerIDCount]);
                handlerIDCount++;
        }
        #endif
        #ifdef DEBUG
}
#endif
 
//////////////////////////////////////////
// 96228: Draw spinner buttons manually
//////////////////////////////////////////
gint spin_button_get_arrow_size (GtkSpinButton *spin_button) {
  gint size = pango_font_description_get_size (GTK_WIDGET (spin_button)->style->font_desc);
  gint arrow_size;

  arrow_size = MAX (PANGO_PIXELS (size), 6);

  return arrow_size - arrow_size % 2; /* force even */
}
gboolean spin_button_at_limit (GtkSpinButton *spin_button, GtkArrowType   arrow) {
  GtkArrowType effective_arrow;

  if (spin_button->wrap)
    return FALSE;

  if (spin_button->adjustment->step_increment > 0)
    effective_arrow = arrow;
  else
    effective_arrow = arrow == GTK_ARROW_UP ? GTK_ARROW_DOWN : GTK_ARROW_UP; 
  
  if (effective_arrow == GTK_ARROW_UP &&
      (spin_button->adjustment->upper - spin_button->adjustment->value <= EPSILON))
    return TRUE;
  
  if (effective_arrow == GTK_ARROW_DOWN &&
      (spin_button->adjustment->value - spin_button->adjustment->lower <= EPSILON))
    return TRUE;
  
  return FALSE;
}
void gtk_spin_button_draw_arrow (GtkSpinButton *spin_button, GtkArrowType   arrow_type) {
  GtkStateType state_type;
  GtkShadowType shadow_type;
  GtkWidget *widget;
  gint x;
  gint y;
  gint height;
  gint width;
  gint h, w;

  g_return_if_fail (GTK_IS_SPIN_BUTTON (spin_button));
  g_return_if_fail (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN);
  
  widget = GTK_WIDGET (spin_button);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      width = spin_button_get_arrow_size (spin_button) + 2 * widget->style->xthickness;
      if (arrow_type == GTK_ARROW_UP)
	{
	  x = 0;
	  y = 0;

	  height = widget->requisition.height / 2;
	}
      else
	{
	  x = 0;
	  y = widget->requisition.height / 2;

	  height = (widget->requisition.height + 1) / 2;
	}
      if (spin_button_at_limit (spin_button, arrow_type))
	{
	  shadow_type = GTK_SHADOW_OUT;
	  state_type = GTK_STATE_INSENSITIVE;
	}
      else
	{
	  if (spin_button->click_child == arrow_type)
	    {
	      state_type = GTK_STATE_ACTIVE;
	      shadow_type = GTK_SHADOW_IN;
	    }
	  else
	    {
	      if (spin_button->in_child == arrow_type &&
		  spin_button->click_child == 2)
		{
		  state_type = GTK_STATE_PRELIGHT;
		}
	      else
		{
		  state_type = GTK_WIDGET_STATE (widget);
		}
	      shadow_type = GTK_SHADOW_OUT;
	    }
	}
      gtk_paint_box (widget->style, spin_button->panel,
		     state_type, shadow_type,
		     NULL, widget,
		     (arrow_type == GTK_ARROW_UP)? "spinbutton_up" : "spinbutton_down",
		     x, y, width, height);
      height = widget->requisition.height;
      if (arrow_type == GTK_ARROW_DOWN)
	{
	  y = height / 2;
	  height = height - y - 2;
	}
      else
	{
	  y = 2;
	  height = height / 2 - 2;
	}
      width -= 3;
      if (widget && gtk_widget_get_direction (widget) == GTK_TEXT_DIR_RTL)
	x = 2;
      else
	x = 1;
      w = width / 2;
      w -= w % 2 - 1; /* force odd */
      h = (w + 1) / 2;
      x += (width - w) / 2;
      y += (height - h) / 2;
      height = h;
      width = w;
      gtk_paint_arrow (widget->style, spin_button->panel,
		       state_type, shadow_type, 
		       NULL, widget, "spinbutton",
		       arrow_type, TRUE, 
		       x, y, width, height);
    }
}

void drawSpinnerButtons(GtkSpinButton* spinButton){
	#ifdef DEBUG
		sprintf(msg, "%s>>>> Drawing Spinner buttons for GtkSpinButton[%x] \r\n", msg, (unsigned int) spinButton);
	#endif
	gtk_spin_button_draw_arrow(spinButton, GTK_ARROW_UP);
	gtk_spin_button_draw_arrow(spinButton, GTK_ARROW_DOWN);
}

 

//////////////////////////////////////////////////////
// Get the pixmap of the window
//////////////////////////////////////////////////////
GdkPixmap* getPixmapOfWindow(GdkWindow* window, int exposeAll){
	int wx,wy,ww,wh,wd;
	GdkRectangle grab_area;
	currentWindow = window;
	gdk_window_get_geometry(window, &wx,&wy,&ww,&wh,&wd);
	grab_area.x = 0;
    grab_area.y = 0;
    grab_area.width = ww;
    grab_area.height = wh;
    //sprintf(msg, "%s### Grabbing area[%x]: (%i,%i)(%ix%i)\r\n", msg,(unsigned int)window,grab_area.x,grab_area.y,grab_area.width,grab_area.height);
    gdk_window_begin_paint_region(window, gdk_region_rectangle(&grab_area));
    gdk_window_invalidate_region(window, gdk_region_rectangle(&grab_area), TRUE);
    currentSpinButton = NULL;
   	signalExposeOnWidgets(mainWidget);
   	
		// Bug:96228 There is a problem with capturing the up
		// and down arrows of the GtkSpinButton widget used by
		// the SWT spinner class in GTK versions 2.4 and above.
		// The problem has to do with the way double buffering 
		// works in GTK. GtkSpinButton is unique in that it has
		// a GdkWindow as a peer of its main GdkWindow, and 
		// because of which it does its own double buffering. In
		// GTK (>2.2) when double buffering ends, the buffer is 
		// being removed. This was not being done so in versions 
		// < 2.4. Opened GTK defect 313992.
		if(gtk_major_version>=2 && gtk_minor_version>2){
			if(currentSpinButton!=NULL){
				drawSpinnerButtons(currentSpinButton);
			}
		}
    	currentSpinButton = NULL;
    	
    gdk_window_process_updates(window, TRUE);
    GdkWindowObject *private = (GdkWindowObject *)window;
    GdkWindowPaint *wp  = private->paint_stack->data;
    GdkPixmap* pm = wp->pixmap;
    // Make a copy
    GdkPixmap* copy = gdk_pixmap_new(pm, ww, wh, -1);
    GdkGC* gc = gdk_gc_new(copy);
    gdk_draw_drawable(copy, gc, pm, 0,0,0,0,ww,wh);
    gdk_window_end_paint(window);
    g_object_unref(gc);
    
	return copy;
}


//////////////////////////////////////////////////////
// Determination of X11 Z Order section
//////////////////////////////////////////////////////
#ifdef DEBUG
void dumpWindowLists(GList* list, Window* windows, int numWindows){
	int lc=0;
	sprintf(msg,"%s---------------------\r\n",msg);
	sprintf(msg, "%sGdk Windows=", msg);
	for(lc=0;lc<g_list_length(list);lc++)
		sprintf(msg, "%s[%x]",msg,(unsigned int)g_list_nth_data(list, lc));
	sprintf(msg, "%s\r\nX11 Windows=",msg);
	for(lc=0;lc<numWindows;lc++)
		sprintf(msg, "%s[%x]",msg,(unsigned int)windows[lc]);
	sprintf(msg,"%s\r\n---------------------\r\n",msg);
}
#endif
//////////////////////////////////////////////////////
// Returns a list of GdkWindows, in the order in which their 
// X11 windows are stacked.
//////////////////////////////////////////////////////
GList* getChildrenByZOrder(GdkWindow* parentWindow){
	GList* sorted ;
	GList* rawList = gdk_window_get_children(parentWindow);

	int query = 0;
	Window root_return;
	Window parent_return;
	Window* children_return=NULL;
	unsigned int num_children_return=0;
	Display* parentWindowDisplay = GDK_WINDOW_XDISPLAY(parentWindow);
	Window parentWindowX11Window = GDK_WINDOW_XID(parentWindow);
	if(parentWindowDisplay!=NULL){
		query = XQueryTree(parentWindowDisplay, parentWindowX11Window, &root_return, &parent_return, &children_return, &num_children_return);
	}
	#ifdef DEBUG
	//sprintf(msg, "%s#### Children of (%x): [length=%i][query=%i][children_return=%x][num children=%i]\r\n",msg,(unsigned int) parentWindow,g_list_length(rawList),query,(unsigned int)children_return,num_children_return);
	#endif

	if(	g_list_length(rawList)<2 ||
		query == 0 || 
		query == BadWindow ||
		children_return==NULL ||
		num_children_return!=g_list_length(rawList)){
		// If anything is wrong or no need to sort, return the original
		sorted = rawList;
	}else{
		// There is now one-one correspondence between X11 Window and GdkWindow
		// There are now more than one Windows
		sorted = rawList;
		#ifdef DEBUG
		dumpWindowLists(sorted, children_return, num_children_return);
		#endif
		// Now find the neighbors
		int xC, gdkC;
		for(xC=0;xC<num_children_return;xC++){
			GdkWindow* correspondingGdkWindow = NULL;
			for(gdkC=0;gdkC<g_list_length(sorted);gdkC++){
				GdkWindow* child = g_list_nth_data(sorted, gdkC);
				#ifdef DEBUG
				//sprintf(msg, "%s\t\tGDK:<%x> --- X11:<%x>\r\n",msg, (unsigned int)GDK_WINDOW_XID(child), (unsigned int)children_return[xC]);
				#endif
				if(GDK_WINDOW_XID(child)==children_return[xC]){
					correspondingGdkWindow = child;
					break;
				}
			}
			if(correspondingGdkWindow==NULL){
				// something is wrong - we were expecting a GdkWindow and 
				// X11 window match, and it didnt take place - just return
				// the unsorted list
				#ifdef DEBUG
				sprintf(msg,"%s\t#### ERROR! Was not able to match X11(%x) and GDK windows - returning unsorted \r\n", msg, (unsigned int)children_return[xC]);
				#endif
				
				if(rawList!=NULL && G_IS_OBJECT(rawList))
					g_object_unref(rawList);
				if(sorted!=NULL && G_IS_OBJECT(sorted))
					g_object_unref(sorted);
				sorted = gdk_window_get_children(parentWindow);
				break;
			}else{
				// Corresponding GdkWindow has been found for the XWindow
				// Now remove it from the unsorted and add it to the sorted list
				#ifdef DEBUG
				sprintf(msg,"%s\t#### (X11=%x)(GDK=%x)\r\n", msg,(unsigned int)children_return[xC], (unsigned int)correspondingGdkWindow);
				#endif
				
				sorted = g_list_remove(sorted, correspondingGdkWindow);
				sorted = g_list_append(sorted, correspondingGdkWindow);
			}
		}
	}
	if(children_return!=NULL){
		XFree(children_return);
	}
	return sorted;
}

GdkPixmap* traverseWindow(GdkWindow* window, int level, int includeChildren){
	int lc = 0;
	int wx,wy,ww,wh,wd;
	int cx,cy,cw,ch,cd;

	//If window is NULL dont return any pixmap
	if(window==NULL){
		#ifdef DEBUG
		sprintf(msg, "%s!!! traverseWindow(): window=%x, level=%i, includeChildren=%i !!!\r\n", msg, (unsigned int)window, level, includeChildren);
		#endif
		return NULL;
	}

	gdk_window_get_geometry(window, &wx,&wy,&ww,&wh,&wd);
	GdkPixmap* myPM = NULL;
	GdkGC* myGC = NULL;
	if(wd!=0){
		#ifdef DEBUG
		for(lc=0;lc<level;lc++)
			sprintf(msg,"%s***",msg);
		sprintf(msg, "%sGetting pixmap for window=%x (%i,%i) (%ix%i) [%i]", msg,(unsigned int)window,wx,wy,ww,wh,wd);
		sprintf(msg, "%s\r\n", msg);
		#endif
		myPM = getPixmapOfWindow(window,1);
		myGC = gdk_gc_new(myPM);
	}
	
	if(includeChildren){
		GList* list = getChildrenByZOrder(window); // need to get children based on Z order of X11 
		for(lc=0;lc<g_list_length(list);lc++){
			GdkWindow* child = (GdkWindow*) g_list_nth_data(list, lc);
			if(child==NULL || !gdk_window_is_visible(child)){
				#ifdef DEBUG
				sprintf(msg, "%s!!!!Skipping NULL child [%i]!!!!", msg, lc);
				#endif
				continue;
			}
			GdkPixmap* childPM = traverseWindow(child, level+1, includeChildren);
			if(childPM!=NULL && myPM!=NULL && myGC!=NULL){
				gdk_window_get_geometry(child, &cx,&cy,&cw,&ch,&cd);
				gdk_draw_drawable(myPM, myGC, childPM, 0,0,cx,cy,cw,ch);
				g_object_unref(childPM);
			}
		}
		if(list!=NULL && G_IS_OBJECT(list))
			g_object_unref(list);
	}
	if(myGC!=NULL){
		g_object_unref(myGC);
	}
	return myPM;
}

GdkImage* getImageOfWidget(GtkWidget* widget, int includeChildren){
	int wx,wy,ww,wh,wd;
	mainWidget = widget;
	GdkWindow* window = widget->window;
	#ifdef DEBUG_EVENTS
	eventAdd=1;handlerIDCount=0;
	#endif
	#ifdef DEBUG
    msg[0]=0;
    walkWidget(mainWidget, 0);
	fprintf(stderr, "\r\n[[[WalkWidget 1]]]\r\n%s\r\n", msg);
    #endif
 	GdkPixmap* pm = traverseWindow(window, 0, includeChildren);
 	#ifdef DEBUG_EVENTS
 	eventAdd=0;handlerIDCount=0;
 	#endif
	#ifdef DEBUG
	fprintf(stderr, "\r\n[[[Traverse Window]]]\r\n%s\r\n", msg);
    msg[0]=0;
    walkWidget(mainWidget, 0);
	fprintf(stderr, "\r\n[[[WalkWidget 2]]]\r\n%s\r\n", msg);
    #endif
	GdkImage* image = NULL;
	if(pm!=NULL){
		gdk_window_get_geometry(window, &wx,&wy,&ww,&wh,&wd);
		if(ww>maxWidth){
			ww = maxWidth;
		}
		if(wh>maxHeight){
			wh = maxHeight;
		}
		image = gdk_drawable_get_image(pm, 0,0,ww,wh);
		g_object_unref(pm);
	}
	return image;
}

JNIEXPORT jintArray JNICALL Java_org_eclipse_ve_internal_swt_targetvm_unix_ImageCapture_getPixels
  (JNIEnv * env, jobject thisObj, jint winH, jint includeChildren, jint jMaxWidth, jint jMaxHeight, jint arg4){
	jintArray dataArray = NULL;
	jint *pixels;
	jboolean isCopy;
	int offset;
	int headerSize;
	int isDirectColor=1;
	
	maxWidth = (int) jMaxWidth;
	maxHeight = (int) jMaxHeight;
	#ifdef DEBUG
	fprintf(stderr, "MAX width = %i and MAX height = %i\r\n", maxWidth, maxHeight);
	#endif
	
	if(!GTK_IS_WIDGET(winH)){
		winH=(jint)0;
	}
	
	if(winH!=0){
		GtkWidget* widget = (GtkWidget*) winH;
		GdkImage* image = getImageOfWidget(widget, includeChildren);
		if(image!=NULL){
			#ifdef DEBUG
			fprintf(stderr, "############>> Widget[%x] = Image[%x](widthxheight=%ix%i)\r\n", (unsigned int)widget, (unsigned int) image, image->width, image->height);
			#endif
			GdkVisual* visual = gtk_widget_get_visual(widget);
			if(visual!=NULL) {
				isDirectColor = (visual->type==GDK_VISUAL_TRUE_COLOR || visual->type==GDK_VISUAL_DIRECT_COLOR);
				if(isDirectColor){
					// DIRECT RGB VALUES
					headerSize = 1+1+1+3; // width(1) + height(1) + type(1) + masks(3)
				}else{
					// INDEXED COLORS
					GdkColormap* cmap = gdk_image_get_colormap(image);
					headerSize = 1+1+1+1+(3*(cmap->size)); // width(1) + height(1) + type(1) + numColors(1) + colors(3*numColors)
				}
				int arrayLength = (headerSize+(image->width*image->height));
				dataArray = (*env)->NewIntArray(env, arrayLength);
				#ifdef DEBUG
				fprintf(stderr, "############>> new int array[%i] = %x\r\n", arrayLength, (unsigned int) dataArray);
				#endif
				pixels = (*env)->GetIntArrayElements(env, dataArray, &isCopy);
				#ifdef DEBUG
				fprintf(stderr, "############>> pixels=%x  isCopy=%x\r\n", (unsigned int) pixels, (unsigned int) isCopy);
				#endif
				pixels[0] = image->width;
				pixels[1] = image->height;
				if(isDirectColor){
					// DIRECT RGB VALUES
					pixels[2] = 1; // RGB VALUES FOLLOW
					pixels[3] = visual->red_mask; // RED mask
					pixels[4] = visual->green_mask; // GREEN mask
					pixels[5] = visual->blue_mask; // BLUE mask
				}else{
					// INDEXED COLORS - send the num of colors in colormap and the colormap colors
					pixels[2] = 1<<1; // INDEXED
					GdkColormap* cmap = gdk_image_get_colormap(image);
					pixels[3] = cmap->size; // Number of colors
					int pp=0;
					for(pp=0;pp<cmap->size;pp++){
						GdkColor color = cmap->colors[pp];
						// RGB class has a max limit of 255
						pixels[4+(color.pixel*3)+0] = ((color.red*255)/65535);
						pixels[4+(color.pixel*3)+1] = ((color.green*255)/65535);
						pixels[4+(color.pixel*3)+2] = ((color.blue*255)/65535);
						//fprintf(stderr, "#### %x,%x,%x\r\n", pixels[4+(color.pixel*3)+0], pixels[4+(color.pixel*3)+1], pixels[4+(color.pixel*3)+2]);
					}
				}
				offset = headerSize;
				int imgwidth, imgheight;
				for(imgheight=0;imgheight<image->height;imgheight++){
					for(imgwidth=0;imgwidth<image->width;imgwidth++){
						unsigned int pv = gdk_image_get_pixel(image, imgwidth, imgheight);
						pixels[offset] = pv;
						offset++;
					}
				}
				#ifdef DEBUG
				fprintf(stderr, "############>>  Releasing int array[%x] and GTK image[%x]\r\n", (unsigned int)pixels, (unsigned int)image);
				#endif
				g_object_unref(image);
				(*env)->ReleaseIntArrayElements(env, dataArray, pixels, 0);
			}
		}
	}
	#ifdef DEBUG
	fprintf(stderr, "############>> Finshed! return dataArray[%x]\r\n", (unsigned int)dataArray);
	#endif
	return dataArray;
}
