#ifndef _ECS_ENTMAIL_PCP_H_
#define _ECS_ENTMAIL_PCP_H_

/*============================================================================
 *  Prototypes des fonctions de base
 *   associées à la structure `ecs_entmail_t' décrivant une entité de maillage
 *  Ces fonctions concernent les entités de maillage PRINCIPALES
 *   par opposition avec les sous-entités de maillage
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilité
 *============================================================================*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C
 *----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_param_perio_glob.h"
#include "ecs_param_rc_glob.h"
#include "ecs_def.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_famille.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' publics  du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_entmail.h"


/*============================================================================
 *                       Prototypes de fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 * Fonction qui décompose les cellules d'une entité de maillage principale
 *  en ses faces
 *
 * Renvoie l'entité de maillage  obtenue par décomposition
 *----------------------------------------------------------------------------*/

ecs_entmail_t * ecs_entmail_pcp__decompose_cel
(
 ecs_entmail_t * entmail_cel          /* <-> Entite de maillage a décomposer  */
) ;


/*----------------------------------------------------------------------------
 *  Fonction réalisant la concaténation de deux entités de maillage principales
 *   Les 2 entités doivent être de même dimension
 *    c'est-a-dire 2 ECS_ENTMAIL_CEL, ou 2 ECS_ENTMAIL_FAC, etc.)
 *
 *  La 2nde entité de maillage `entmail_concat' est concaténée à la suite
 *   de l'entité de maillage réceptrice `this_entmail'
 *
 *  Le 3ème argument est un tableau sur les entités de maillage
 *   dont les éléments non NULL
 *   sont les entités de maillage pour lesquelles le champ "définition"
 *   sera modifie pour prendre en compte
 *   le décalage des références des éléments
 *   de l'entité de maillage à concaténer
 *   (décalage correspondant au nombre d'éléments de l'entité réceptrice)
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__concatene
(
 ecs_entmail_t *      this_entmail,
 ecs_entmail_t *      entmail_concat,
 ecs_entmail_t *const liste_entmail_connect[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction réalisant la fusion des éléments topologiquement identiques
 *   d'une entité de maillage principale donnée
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__fusionne
(
 ecs_entmail_t *      entmail,
 ecs_entmail_t *const liste_entmail_connect[]
) ;


/*----------------------------------------------------------------------------
 *  Fusion des sommets confondus et suppression des éléments dégénérés
 *
 *  On ne met pas à jour les champs de type attribut non héritable,
 *  de type ECS_CHAMP_STATUT_REF_ELT (ex : champ_fac_perio), ces champs
 *  pouvant être construits après l'appel de cette fonction.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__nettoie_descend
(
       ecs_entmail_t  *const *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *  Suppression des sommets ne participant pas à la connectivité
 *   et fusion des éléments surfaciques confondus éventuels
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__nettoie_nodal
(
       ecs_entmail_t  *const *const vect_entmail
) ;


/*----------------------------------------------------------------------------
 *  Correction si nécessaire de l'orientation des éléments en
 *   connectivité nodale.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__orient_nodal
(
       ecs_entmail_t  *const *const vect_entmail,   /*  -> Déf. maillage      */
       ecs_tab_int_t  *const        liste_cel_err,  /* <-  Liste cels. erreur
                                                     *     (optionnelle)      */
       ecs_tab_int_t  *const        liste_cel_cor,  /* <-  Liste cels. corr.
                                                     *     (optionnelle)      */
 const bool                         correc_orient   /*  -> Correction ou non  */
) ;


/*----------------------------------------------------------------------------
 * Fonction qui renvoie le nombre de faces de type "polygone"
 *  de l'entité de maillage de type "faces" donnée
 *  (faces définies par plus de 4 arêtes)
 *----------------------------------------------------------------------------*/

size_t  ecs_entmail_pcp__nbr_fac_poly
(
 ecs_entmail_t * entmail_fac
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui réalise le tri des éléments
 *   de l'entité de maillage principale donnée
 *   suivant leur type géométrique
 *  La fonction affiche le nombre d'éléments par type géométriques
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__trie_typ_geo
(
 ecs_entmail_t  * entmail,
 ECS_ENTMAIL_E    typ_entmail
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui crée la liste des faces intérieures à un maillage.
 *  On ne compte pas ici les faces périodiques parmi les faces intérieures,
 *  cette fonction étant destinée à filtrer le post-triatment, et les
 *  faces périodiques étant déjà affichables par ailleurs.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_entmail_pcp__liste_fac_int
(
 ecs_entmail_t  *const   vect_entmail[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui compte les faces intérieures et de bord et crée une liste
 *  correspondant à chacun de ces deux ensembles. On peut aussi renvoyer
 *  d'éventuelles listes de faces isolées ou de faces appartenant à plus de
 *  deux cellules (erreur de connectivité en cas de cellules partiellement
 *  superposées).
 *
 *  Dans chaque cas, si l'argument pointeur vers une liste donnée est
 *  à NULL, cette liste n'est pas crée.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__compte_typ_fac
(
 ecs_entmail_t  *const   vect_entmail[],
 ecs_tab_int_t  *const   liste_fac_erreur,
 ecs_tab_int_t  *const   liste_fac_interne,
 ecs_tab_int_t  *const   liste_fac_de_bord,
 ecs_tab_int_t  *const   liste_fac_isolee
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit la liste des cellules attachées à une liste
 *  de faces fournie en argument.
 *----------------------------------------------------------------------------*/

ecs_tab_int_t ecs_entmail_pcp__liste_cel_fac
(
       ecs_entmail_t  *const vect_entmail[],
 const ecs_tab_int_t         liste_fac
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui définit de nouvelles entités de maillage principales
 *   par extraction d'une partie des éléments
 *   d'une entité de maillage principale donnée
 *  Les éléments à extraire sont ceux qui ont un booléen à `true'
 *
 *  On construit automatiquement une filiation.
 *----------------------------------------------------------------------------*/

ecs_entmail_t * * ecs_entmail_pcp__extrait
(
       ecs_entmail_t    *const vect_entmail[],
       ECS_ENTMAIL_E           entmail_e,
 const ecs_tab_bool_t          bool_elt_select,
 const bool                    herite_attributs
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui sélectionne des éléments ayant une des couleurs de la liste
 *   des couleurs ou appartenant à un des groupes de la liste des groupes
 *  Seules sont concernées les entités de type `entmail_sel_e'
 *----------------------------------------------------------------------------*/

ecs_tab_bool_t  ecs_entmail_pcp__selectionne
(
       ecs_entmail_t     *vect_entmail[ECS_ENTMAIL_FIN],
       ECS_ENTMAIL_E      entmail_sel_e,
 const ecs_tab_int_t     *liste_filtre,
 const ecs_tab_int_t      liste_couleur,
 const ecs_tab_char_t     liste_groupe,
       bool               inv_selection
) ;


/*----------------------------------------------------------------------------
 * Fonction qui selectionne tous les elements ou ceux appartenant a une liste
 * Seules sont concernees les entites de type `entmail_sel_e'
 *----------------------------------------------------------------------------*/

ecs_tab_bool_t  ecs_entmail_pcp__selectionne_lst
(
       ecs_entmail_t     *vect_entmail[ECS_ENTMAIL_FIN],
       ECS_ENTMAIL_E      entmail_sel_e,
 const ecs_tab_int_t     *liste_filtre
) ;


/*----------------------------------------------------------------------------
 * Fonction qui construit la liste des éléments de l'entité principale
 *  sélectionnée portant des numéros de familles marqués dans le tableau
 *  indic_famille
 *----------------------------------------------------------------------------*/

ecs_tab_int_t  ecs_entmail_pcp__liste_ent_fam
(
       ecs_entmail_t        *const *const vect_entmail,
       ECS_ENTMAIL_E                      entmail_sel_e,
 const ecs_tab_int_t        *const        liste_filtre,
 const ecs_tab_bool_t       *const        indic_famille
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui calcule les coordonnées min et max du domaine
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__calc_coo_ext
(
 const ecs_entmail_t *const entmail_som
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui modifie les coordonnées du maillage
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__transf_coo
(
 const ecs_entmail_t  *const entmail_som,
 const double                matrice[3][4]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui recolle les faces non conformes
 *
 *  Les listes des faces nouvelles ou modifiées sont construites (et allouées)
 *  ici ; les structures liste_fac_new et liste_fac_mod correspondantes sont
 *  donc vides en entrée ; idem pour liste_fac_err qui indiquera les indices
 *  des faces pour lesquelles le découpage en sous-faces a échoué
 *
 *  On prend en entrée soit le nom d'un champ définissant une "visibilité"
 *  entre faces à recoller (par exemple une filiation), ou alors un
 *  tableau indicateur de faces sélectionnées.
 *  L'un de nom_champ_vis_vis et tab_bool_fac_select doit donc
 *  être à NULL, et l'autre non.
 *
 *  On renvoie 1 si l'on a effectivement appellé le recollement, 0 sinon
 *  (i.e. si la sélection de faces à recoller est vide ou le champ de type
 *  "visibilité" est introuvable.
 *----------------------------------------------------------------------------*/

ecs_int_t ecs_entmail_pcp__recolle
(
       ecs_entmail_t    *const *const vect_entmail,
 const char                    *const nom_champ_vis,
       ecs_tab_bool_t          *const tab_bool_fac_select,
       ecs_tab_int_t           *const liste_fac_new,
       ecs_tab_int_t           *const liste_fac_mod,
       ecs_tab_int_t           *const liste_fac_err,
 const ecs_param_rc_t                 param_rc
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui supprime les faces isolées.
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__compacte_faces
(
 ecs_entmail_t  *const *const vect_entmail,
 ecs_tab_int_t         *const liste_fac_isolees
) ;


/*----------------------------------------------------------------------------
 *  Fonction ne conservant d'une entité de maillage que le champ
 *  'filiation' (le seul nécessaire sur les faces d'un maillage destiné
 *  à une coupe après la sortie du maillage pour le post traitement des
 *  variables affectées aux faces)
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__epure_coupe
(
 ecs_entmail_t   *const entmail_coupe
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui vérifie la cohérence d'un maillage.
 *----------------------------------------------------------------------------*/

bool       ecs_entmail_pcp__verif
(
 ecs_entmail_t     *const *const vect_entmail,
 ecs_post_t               *const cas_post
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les familles
 *----------------------------------------------------------------------------*/

ecs_famille_t * * ecs_entmail_pcp__cree_famille
(
 ecs_entmail_t  *const vect_entmail[]
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui détruit le champ attribut déterminé par son nom
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__detruit_att_nom
(
       ecs_entmail_t  *const this_entmail,
 const char           *const nom_champ
) ;


/*----------------------------------------------------------------------------
 *  Fonction qui construit les attributs "groupe" et "couleur"
 *   à partir des familles
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__cree_attribut
(
 ecs_entmail_t *const this_entmail,
 ecs_famille_t *const famille
) ;


/*----------------------------------------------------------------------------
 * Fonction qui supprime les attributs "groupe" et "couleur"
 *----------------------------------------------------------------------------*/

void ecs_entmail_pcp__suppr_attribut
(
 ecs_entmail_t *const this_entmail
) ;


#endif /* _ECS_ENTMAIL_PCP_H_ */
