/*============================================================================
 *  Définitions des fonctions de base
 *   associées aux impressions dans un fichier
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2007 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================*
 *                                 Visibilité
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <stdarg.h>
#include <string.h>

#include <bft_file.h>
#include <bft_mem.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_def.h"


/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associé au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_fic.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' privés du  paquetage courant
 *----------------------------------------------------------------------------*/


/*============================================================================
 *  Constantes et Macros
 *============================================================================*/

/* Formats pour impressions de tableaux dans un fichier */

#define ECS_LOC_IMP_FMT_INT             "%12d"
#define ECS_LOC_IMP_FMT_REAL         "%#12.5E"
#define ECS_LOC_IMP_FMT_CHAR            "%12s"
#define ECS_LOC_IMP_FMT_SIZE_T          "%12u"
#define ECS_LOC_IMP_FMT_PTR             "%12p"
#define ECS_LOC_IMP_FMT_CONT    "............"
#define ECS_LOC_IMP_FMT_WIDTH               12

#define ECS_LOC_IMP_INDENT                   8

#define ECS_LOC_IMP_COL_VAL                  9
#define ECS_LOC_IMP_COL_IND                 51


/*============================================================================
 *                       Prototypes de fonctions privées
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *  Fonction d'impression du nom et de la valeur d'un pointeur
 *----------------------------------------------------------------------------*/

void ecs_fic__imprime_ptr
(
       bft_file_t  *const fic_imp,
 const int                profondeur_imp,
 const char        *const nom_ptr,
 const void        *const ptr
)
{

  /* Instructions */

  assert(fic_imp != NULL) ;


  bft_file_printf(fic_imp,
                  "%-*s%s  & ", ECS_LOC_IMP_INDENT * profondeur_imp, "",
                  nom_ptr) ;

  if (ptr == NULL)

    bft_file_printf(fic_imp, "%s\n", "NULL") ;

  else

    bft_file_printf(fic_imp, "%p\n", ptr) ;

}


/*----------------------------------------------------------------------------
 *  Fonction d'impression du nom et de la valeur d'une variable
 *----------------------------------------------------------------------------*/

void ecs_fic__imprime_val
(
       bft_file_t     *const fic_imp,
 const int                   profondeur_imp_nom,
 const char           *const nom,
       ecs_type_t            typ_e,
 const void           *const val
)
{

  /* Variables locales */

  int  nb_col_nom ;
  int  profondeur_imp_val ;


  /* Instructions */

  assert(fic_imp != NULL) ;


  if ((int)strlen(nom) < ECS_LOC_IMP_INDENT)
    nb_col_nom = 1 ;
  else
    nb_col_nom = 2 ;

  profondeur_imp_val
    = ECS_LOC_IMP_COL_VAL - profondeur_imp_nom - nb_col_nom - 1;

  bft_file_printf(fic_imp,
                  "%-*s%-*s%-*s",
                  ECS_LOC_IMP_INDENT * profondeur_imp_nom, "" ,
                  ECS_LOC_IMP_INDENT * nb_col_nom        , nom,
                  ECS_LOC_IMP_INDENT * profondeur_imp_val, ""  ) ;

  if (val == NULL)

    bft_file_printf(fic_imp, ECS_LOC_IMP_FMT_CHAR "\n", "NULL") ;

  else {

    switch(typ_e) {

    case ECS_TYPE_ecs_int_t:

      {
        const int val_imp = *(const int *)val ;
        bft_file_printf(fic_imp, ECS_LOC_IMP_FMT_INT "\n", val_imp) ;
      }

      break ;

    case ECS_TYPE_ecs_real_t:

      {
        const ecs_real_t val_imp = *(const ecs_real_t *)val ;
        bft_file_printf(fic_imp, ECS_LOC_IMP_FMT_REAL "\n", val_imp) ;
      }

      break ;

    case ECS_TYPE_char:

      {
        const char *val_imp = (const char *)val ;
        bft_file_printf(fic_imp, ECS_LOC_IMP_FMT_CHAR "\n", val_imp) ;
      }

      break ;

    case ECS_TYPE_size_t:

      {
        const size_t val_imp = *(const size_t *)val ;
        bft_file_printf(fic_imp, ECS_LOC_IMP_FMT_SIZE_T "\n", val_imp) ;
      }

      break ;

    case ECS_TYPE_void:

      bft_file_printf(fic_imp, ECS_LOC_IMP_FMT_PTR  "\n", val) ;

      break ;

    default:

      assert( typ_e == ECS_TYPE_ecs_int_t  ||
              typ_e == ECS_TYPE_ecs_real_t ||
              typ_e == ECS_TYPE_char       ||
              typ_e == ECS_TYPE_size_t     ||
              typ_e == ECS_TYPE_void         ) ;

    }

  }

  bft_file_flush(fic_imp) ;

}


/*============================================================================
 *                              Fonctions privées
 *============================================================================*/

