/*============================================================================
 *  Definitions des fonctions
 *   associees a la structure `ecs_maillage_t' decrivant un maillage
 *   et realisant les sorties pour post-traitement
 *============================================================================*/

/*
  This file is part of the Code_Saturne Preprocessor, element of the
  Code_Saturne CFD tool.

  Copyright (C) 1999-2009 EDF S.A., France

  contact: saturne-support@edf.fr

  The Code_Saturne Preprocessor is free software; you can redistribute it
  and/or modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2 of
  the License, or (at your option) any later version.

  The Code_Saturne Preprocessor is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with the Code_Saturne Preprocessor; if not, write to the
  Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor,
  Boston, MA  02110-1301  USA
*/


/*============================================================================
 *                                 Visibilite
 *============================================================================*/

#include "ecs_config.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' librairie standard C ou BFT
 *----------------------------------------------------------------------------*/

#include <assert.h>
#include <string.h>

#include <bft_mem.h>
#include <bft_printf.h>


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Utilitaire"
 *----------------------------------------------------------------------------*/

#include "ecs_chaine_glob.h"
#include "ecs_def.h"
#include "ecs_fic.h"
#include "ecs_tab_glob.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage global "Post-Traitement"
 *----------------------------------------------------------------------------*/

#include "ecs_post_ens.h"
#include "ecs_post.h"

#if defined(HAVE_CGNS)
#include "ecs_post_cgns.h"
#endif /* HAVE_CGNS */

#if defined(HAVE_MED)
#include "ecs_post_med.h"
#endif /* HAVE_MED */


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles des paquetages visibles
 *----------------------------------------------------------------------------*/

#include "ecs_entmail.h"
#include "ecs_entmail_pcp.h"
#include "ecs_entmail_post.h"
#include "ecs_famille.h"

#include "ecs_maillage.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' visibles du  paquetage courant
 *----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
 *  Fichier  `include' du  paquetage courant associe au fichier courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_post.h"


/*----------------------------------------------------------------------------
 *  Fichiers `include' prives   du  paquetage courant
 *----------------------------------------------------------------------------*/

#include "ecs_maillage_priv.h"



/*============================================================================
 *                              Fonctions privees
 *============================================================================*/


/*============================================================================
 *                             Fonctions publiques
 *============================================================================*/

/*----------------------------------------------------------------------------
 *    Fonction qui écrit le maillage sur fichier pour post-traitement
 *  Seuls les éléments de l'entité de rang `ent_post' sont écrits.
 *
 *    Les éléments de l'entité à écrire doivent être sous forme de
 *  connectivité nodale.
 *----------------------------------------------------------------------------*/

void
ecs_maillage_post__ecr(const char       *nom_maillage,
                       ecs_maillage_t   *maillage,
                       ECS_ENTMAIL_E     entmail_post,
                       ecs_post_type_t   type_post,
                       ecs_post_t       *cas_post)
{

  ecs_int_t        ient;
  bool             bool_ecrit_maillage;
  bool             bool_cree_famille;
  ecs_entmail_t  * entmail_inf;
  ecs_famille_t  * famille_inf;


  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/


  assert(entmail_post > ECS_ENTMAIL_SOM && entmail_post < ECS_ENTMAIL_FIN);


  /* On vérifie si l'on a besoin d'écrire le maillage */
  /*--------------------------------------------------*/

  if (ecs_maillage__ret_entmail_max(maillage) < entmail_post)
    return;


  bool_ecrit_maillage = false;

  if (   cas_post->post_ens == true
      && cas_post->opt_ens.ecr_type[type_post] == true)
    bool_ecrit_maillage = true;

#if defined(HAVE_CGNS)
  if (   cas_post->post_cgns == true
      && cas_post->opt_cgns.ecr_type[type_post] == true)
    bool_ecrit_maillage = true;
#endif

#if defined(HAVE_MED)
  if (   cas_post->post_med == true
      && cas_post->opt_med.ecr_type[type_post] == true)
    bool_ecrit_maillage = true;
#endif

  if (bool_ecrit_maillage == false)
    return;


  /* Construction des familles si non encore présentes */
  /*---------------------------------------------------*/

  bool_cree_famille = true;

  for (ient = ECS_ENTMAIL_FAC; ient < ECS_ENTMAIL_FIN; ient++) {
    if (maillage->famille[ient] != NULL)
      bool_cree_famille = false;
  }

  if (bool_cree_famille == true) {

    /* Construction des familles pour les elements */

    ecs_maillage__cree_famille(maillage,
                               ECS_ENTMAIL_FAC,
                               entmail_post);

  }


  /* Ajout du cas EnSight (si nécessaire) */
  /*--------------------------------------*/

  if (   cas_post->post_ens == true
      && cas_post->opt_ens.ecr_type[type_post] == true) {

    bft_printf(_("\n\n"
                 "EnSight output of mesh: %s\n"
                 "-----------------------\n"), nom_maillage);

    if (cas_post->cas_ens == NULL)
      cas_post->cas_ens
        = ecs_post_ens__cree_cas(cas_post->nom_cas,
                                 cas_post->opt_ens.single_part,
                                 cas_post->opt_ens.no_poly,
                                 cas_post->opt_ens.text,
                                 cas_post->opt_ens.big_endian);

  }


#if defined(HAVE_CGNS)

  /* Ajout du cas CGNS (si nécessaire) */
  /*-----------------------------------*/

  if (   cas_post->post_cgns == true
      && cas_post->opt_cgns.ecr_type[type_post] == true) {

    bft_printf(_("\n\n"
                 "CGNS file output of mesh: %s\n"
                 "-------------------------\n"), nom_maillage);


    if (cas_post->cas_cgns == NULL)
      cas_post->cas_cgns
        = ecs_post_cgns__cree_cas(cas_post->nom_cas,
                                  cas_post->opt_cgns.no_poly);

  }

#endif /* HAVE_CGNS */


#if defined(HAVE_MED)

  /* Ajout du cas MED (si nécessaire) */
  /*----------------------------------*/

  if (   cas_post->post_med == true
      && cas_post->opt_med.ecr_type[type_post] == true) {

    bft_printf(_("\n\n"
                 "MED file output of mesh: %s\n"
                 "------------------------\n"), nom_maillage);

    if (cas_post->cas_med == NULL)
      cas_post->cas_med = ecs_post_med__cree_cas(cas_post->nom_cas,
                                                 cas_post->opt_med.no_poly);

  }

#endif /* HAVE_MED */


  /* Appel a la fonction réalisant l'ecriture des élements */
  /*-------------------------------------------------------*/


  if (maillage->entmail[(ecs_int_t)entmail_post] != NULL) {

    int dim_ent_max;

    switch(entmail_post) {
    case ECS_ENTMAIL_FAC:
      dim_ent_max = 2;
      break;
    case ECS_ENTMAIL_CEL:
      dim_ent_max = 3;
      break;
    default:
      dim_ent_max = 0;
    }


    /* élements de dimension inférieure (pour cond. limites.) */

    if (entmail_post == ECS_ENTMAIL_CEL) {
      entmail_inf = maillage->entmail[(ecs_int_t)entmail_post - 1];
      famille_inf = maillage->famille[(ecs_int_t)entmail_post - 1];
    }
    else {
      entmail_inf = NULL;
      famille_inf = NULL;
    }


    /* Ensemble des éléments */

    ecs_entmail_post__ecr_elt(nom_maillage,
                              dim_ent_max,
                              maillage->entmail[(ecs_int_t)entmail_post],
                              entmail_inf,
                              maillage->entmail[ECS_ENTMAIL_SOM],
                              maillage->famille[(ecs_int_t)entmail_post],
                              famille_inf,
                              type_post,
                              cas_post);

  }


  /* Suppression des familles temporaires */
  /*--------------------------------------*/

  if (bool_cree_famille == true) {

    ecs_maillage__detruit_famille(maillage);

  }
}


/*----------------------------------------------------------------------------
 *  Écriture du maillage correspondant à une liste de faces sur fichier
 *  pour post-traitement.
 *
 *  Cette fonction crée une coupe correspondant à la liste de faces donnée
 *  (ce qui provoque automatiquement son post-traitement), puis la détruit.
 *  Le nom utilisé pour cette sortie ne sera donc plus disponible pour
 *  d'autres coupes.
 *
 *  Le maillage principal doit être en connectivité descendante.
 *----------------------------------------------------------------------------*/

void
ecs_maillage_post__ecr_fac_liste(const char           *nom_liste,
                                 ecs_maillage_t       *maillage,
                                 const ecs_tab_int_t   liste_fac,
                                 const bool            herite_attributs,
                                 ecs_post_type_t       type_post,
                                 ecs_post_t           *cas_post)
{
  ecs_maillage_t   *maillage_post = NULL;

  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  maillage_post = ecs_maillage__cree_coupe(nom_liste,
                                           maillage,
                                           liste_fac,
                                           0,
                                           NULL,
                                           herite_attributs,
                                           type_post,
                                           cas_post);

  maillage_post = ecs_maillage__detruit(maillage_post);
}


/*----------------------------------------------------------------------------
 *  Impression d'une variable en post-traitement valant 1 pour les entités
 *  enumérées dans cette liste, 0 pour les autres.
 *----------------------------------------------------------------------------*/

void
ecs_maillage_post__ecr_tab_liste(const char           *nom_var,
                                 ecs_maillage_t       *this_maillage,
                                 const ecs_tab_int_t   liste_entites,
                                 ECS_ENTMAIL_E         entmail_e,
                                 ecs_post_t           *cas_post)
{
  /*xxxxxxxxxxxxxxxxxxxxxxxxxxx Instructions xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx*/

  assert(cas_post != NULL);

  ecs_entmail_post__ecr_tab_liste(this_maillage->entmail[entmail_e],
                                  _(ECS_MAILLAGE_NOM_PRINCIPAL),
                                  nom_var,
                                  liste_entites,
                                  cas_post);

}

/*============================================================================*/
