<?php

/*
 * Notes:
 * 1. Deletion of directories will be done through the file system class, which inherits from the tree class
 */

/**
 * eFrontFileException class
 *
 * This class extends Exception class and is used to issue errors regarding files and filesystem
 * @author Venakis Periklis <pvenakis@efront.gr>
 * @version 1.0
 */
class EfrontFileException extends Exception
{
    //Note: values from 1 to 8 are upload errors
    const NO_ERROR                 = 0;
    const ILLEGAL_FILE_NAME        = 101;
    const FILE_NOT_EXIST           = 102;
    const ILLEGAL_PATH             = 103;
    const FILE_IN_BLACK_LIST       = 104;
    const FILE_NOT_IN_WHITE_LIST   = 105;
    const GENERAL_ERROR            = 106;
    const FILE_ALREADY_EXISTS      = 107;
    const DIRECTORY_ALREADY_EXISTS = 108;
    const FILE_DELETED             = 109;
    const ERROR_CREATE_ZIP         = 110;
    const ERROR_OPEN_ZIP           = 111;
    const UNKNOWN_COMPRESSION      = 112;
    const DIRECTORY_NOT_EXIST      = 113;
    const NOT_LESSON_FILE          = 114;
    const UNAUTHORIZED_ACTION      = 115;
    const UNKNOWN_ERROR            = 199;
    const DATABASE_ERROR           = 301;
}



/**
 * Interface for filesystem entities
 *
 * @since 3.5.0
 * @package eFront
 */
interface EfrontFileSystemEntity
{
    public function delete();
    public function copy($destination, $overwrite = true);
    public function move($destination, $overwrite = true);
    public function rename($newName);
    public function persist();
    public function compress($method = 'zip');
    public function uncompress();
}

/**
 * Class for files in Efront file system
 *
 * @since 3.5.0
 * @package efront
 */
class EfrontFile extends ArrayObject implements EfrontFileSystemEntity
{
    /**
     * An array of mime types
     *
     * @since 3.5.0
     * @var array
     * @access public
     * @static
     */
    public static $mimeTypes = array (
        'bmp'       =>   'image/bmp',
        'cgm'       =>   'image/cgm',
        'djv'       =>   'image/vnd.djvu',
        'djvu'      =>   'image/vnd.djvu',
        'gif'       =>   'image/gif',
        'ico'       =>   'image/x-icon',
        'ief'       =>   'image/ief',
        'jp2'       =>   'image/jp2',
        'jpe'       =>   'image/jpeg',
        'jpeg'      =>   'image/jpeg',
        'jpg'       =>   'image/jpeg',
        'mac'       =>   'image/x-macpaint',
        'pbm'       =>   'image/x-portable-bitmap',
        'pct'       =>   'image/pict',
        'pgm'       =>   'image/x-portable-graymap',
        'pic'       =>   'image/pict',
        'pict'      =>   'image/pict',
        'png'       =>   'image/png',
        'pnm'       =>   'image/x-portable-anymap',
        'pnt'       =>   'image/x-macpaint',
        'pntg'      =>   'image/x-macpaint',
        'ppm'       =>   'image/x-portable-pixmap',
        'qti'       =>   'image/x-quicktime',
        'qtif'      =>   'image/x-quicktime',
        'ras'       =>   'image/x-cmu-raster',
        'rgb'       =>   'image/x-rgb',
        'svg'       =>   'image/svg+xml',
        'tif'       =>   'image/tiff',
        'tiff'      =>   'image/tiff',
        'wbmp'      =>   'image/vnd.wap.wbmp',
        'xbm'       =>   'image/x-xbitmap',
        'xpm'       =>   'image/x-xpixmap',
        'xwd'       =>   'image/x-xwindowdump',
        'asc'       =>   'text/plain',
        'css'       =>   'text/css',
        'etx'       =>   'text/x-setext',
        'htm'       =>   'text/html',
        'html'      =>   'text/html',
        'ics'       =>   'text/calendar',
        'ifb'       =>   'text/calendar',
        'rtf'       =>   'text/rtf',
        'rtx'       =>   'text/richtext',
        'sgm'       =>   'text/sgml',
        'sgml'      =>   'text/sgml',
        'tsv'       =>   'text/tab-separated-values',
        'txt'       =>   'text/plain',
        'wml'       =>   'text/vnd.wap.wml',
        'wmls'      =>   'text/vnd.wap.wmlscript',
        'kar'       =>   'audio/midi',
        'm3u'       =>   'audio/x-mpegurl',
        'm4a'       =>   'audio/mp4a-latm',
        'm4b'       =>   'audio/mp4a-latm',
        'm4p'       =>   'audio/mp4a-latm',
        'mid'       =>   'audio/midi',
        'midi'      =>   'audio/midi',
        'mp2'       =>   'audio/mpeg',
        'mp3'       =>   'audio/mpeg',
        'mpga'      =>   'audio/mpeg',
        'ra'        =>   'audio/x-pn-realaudio',
        'ram'       =>   'audio/x-pn-realaudio',
        'snd'       =>   'audio/basic',
        'wav'       =>   'audio/x-wav',
        'aif'       =>   'audio/x-aiff',
        'aifc'      =>   'audio/x-aiff',
        'aiff'      =>   'audio/x-aiff',
        'au'        =>   'audio/basic',
        'avi'       =>   'video/x-msvideo',
        'mov'       =>   'video/quicktime',
        'movie'     =>   'video/x-sgi-movie',
        'mp4'       =>   'video/mp4',
        'mpe'       =>   'video/mpeg',
        'mpeg'      =>   'video/mpeg',
        'mpg'       =>   'video/mpeg',
        'm4u'       =>   'video/vnd.mpegurl',
        'm4v'       =>   'video/x-m4v',
        'dif'       =>   'video/x-dv',
        'dv'        =>   'video/x-dv',
        'mxu'       =>   'video/vnd.mpegurl',
        'qt'        =>   'video/quicktime',
        'iges'      =>   'model/iges',
        'igs'       =>   'model/iges',
        'mesh'      =>   'model/mesh',
        'msh'       =>   'model/mesh',
        'silo'      =>   'model/mesh',
        'vrml'      =>   'model/vrml',
        'wrl'       =>   'model/vrml',
        'xyz'       =>   'chemical/x-xyz',
        'pdb'       =>   'chemical/x-pdb',
        'ice'       =>   'x-conference/x-cooltalk',
        'ai'        =>   'application/postscript',
        'atom'      =>   'application/atom+xml',
        'bcpio'     =>   'application/x-bcpio',
        'bin'       =>   'application/octet-stream',
        'cdf'       =>   'application/x-netcdf',
        'class'     =>   'application/octet-stream',
        'cpio'      =>   'application/x-cpio',
        'cpt'       =>   'application/mac-compactpro',
        'csh'       =>   'application/x-csh',
        'dcr'       =>   'application/x-director',
        'dir'       =>   'application/x-director',
        'dll'       =>   'application/octet-stream',
        'dmg'       =>   'application/octet-stream',
        'dms'       =>   'application/octet-stream',
        'doc'       =>   'application/msword',
        'dtd'       =>   'application/xml-dtd',
        'dvi'       =>   'application/x-dvi',
        'dxr'       =>   'application/x-director',
        'eps'       =>   'application/postscript',
        'exe'       =>   'application/octet-stream',
        'ez'        =>   'application/andrew-inset',
        'gram'      =>   'application/srgs',
        'grxml'     =>   'application/srgs+xml',
        'gtar'      =>   'application/x-gtar',
        'hdf'       =>   'application/x-hdf',
        'hqx'       =>   'application/mac-binhex40',
        'jnlp'      =>   'application/x-java-jnlp-file',
        'js'        =>   'application/x-javascript',
        'latex'     =>   'application/x-latex',
        'lha'       =>   'application/octet-stream',
        'lzh'       =>   'application/octet-stream',
        'man'       =>   'application/x-troff-man',
        'mathml'    =>   'application/mathml+xml',
        'me'        =>   'application/x-troff-me',
        'mif'       =>   'application/vnd.mif',
        'ms'        =>   'application/x-troff-ms',
        'nc'        =>   'application/x-netcdf',
        'oda'       =>   'application/oda',
        'ogg'       =>   'application/ogg',
        'pdf'       =>   'application/pdf',
        'pgn'       =>   'application/x-chess-pgn',
        'ppt'       =>   'application/vnd.ms-powerpoint',
        'ps'        =>   'application/postscript',
        'rdf'       =>   'application/rdf+xml',
        'rm'        =>   'application/vnd.rn-realmedia',
        'roff'      =>   'application/x-troff',
        'sh'        =>   'application/x-sh',
        'shar'      =>   'application/x-shar',
        'sit'       =>   'application/x-stuffit',
        'skd'       =>   'application/x-koan',
        'skm'       =>   'application/x-koan',
        'skp'       =>   'application/x-koan',
        'skt'       =>   'application/x-koan',
        'smi'       =>   'application/smil',
        'smil'      =>   'application/smil',
        'so'        =>   'application/octet-stream',
        'spl'       =>   'application/x-futuresplash',
        'src'       =>   'application/x-wais-source',
        'sv4cpio'   =>   'application/x-sv4cpio',
        'sv4crc'    =>   'application/x-sv4crc',
        'swf'       =>   'application/x-shockwave-flash',
        't'         =>   'application/x-troff',
        'tar'       =>   'application/x-tar',
        'tcl'       =>   'application/x-tcl',
        'tex'       =>   'application/x-tex',
        'texi'      =>   'application/x-texinfo',
        'texinfo'   =>   'application/x-texinfo',
        'tr'        =>   'application/x-troff',
        'ustar'     =>   'application/x-ustar',
        'vcd'       =>   'application/x-cdlink',
        'vxml'      =>   'application/voicexml+xml',
        'wbmxl'     =>   'application/vnd.wap.wbxml',
        'wmlc'      =>   'application/vnd.wap.wmlc',
        'wmlsc'     =>   'application/vnd.wap.wmlscriptc',
        'xht'       =>   'application/xhtml+xml',
        'xhtml'     =>   'application/xhtml+xml',
        'xls'       =>   'application/vnd.ms-excel',
        'xml'       =>   'application/xml',
        'xsl'       =>   'application/xml',
        'xslt'      =>   'application/xslt+xml',
        'xul'       =>   'application/vnd.mozilla.xul+xml',
        'zip'       =>   'application/zip');
        
    /**
     * Class constructor
     *
     * The class constructor instantiates the object based on the $file parameter.
     * $file may be either:
     * - an array with file attributes
     * - a file id
     * - the full path to a physical file
     * - the full path to a file using its original file name
     * - The full path to a file, even if it doesn't have a corresponding database representation
     * <br/>Example:
     * <code>
     * $result = eF_getTableData("files", "*", "id=43");
     * $file = new EfrontFile($result[0]);                          //Instantiate object using array of values
     * $file = new EfrontFile(43);                                  //Instantiate object using id
     * $file = new EfrontFile('/var/www/test.txt');                 //Instantiate object using path and original name
     * $file = new EfrontFile('/var/www/32.txt.efront');            //Instantiate object using path and physical name
     * </code>
     *
     * @param mixed $file The file information, either an array, an id or a path string
     * @since 3.5.0
     * @access public
     */
    function  __construct($file) {
        if (is_array($file)) {                                    //Instantiate object based on the given array
            $fileArray = $file;
        } else {                                        
            if (eF_checkParameter($file, 'id')) {                //Instantiate object based on id
                $result = eF_getTableData("files", "*", "type = 'file' and id=".$file);
            } else {                                             //id-based instantiation failed; Check if the full path is specified
                $result = eF_getTableData("files", "*", "type = 'file' and file='$file'");
                if (sizeof($result) <= 0 || !is_file($result[0]['file'])) {                                //A file with the specified filename was not found. Check if the original file name was specified
                    $result = eF_getTableData("files", "*", "type = 'file' and original_name='".basename($file)."' and directory = '".EfrontFileSystem :: checkPath(dirname($file))."'");
                }
            }

            if (sizeof($result) > 0) {
                if (sizeof($result) > 1) {                            //if for some reason there is more than 1 database entries for the same file, keep only the latest (based on id)
                    for ($i = 0; $i < sizeof($result) - 1; $i++) {
                        eF_deleteTableData("files", "id=".$result[$i]['id']);
                        unlink($result[$i]['file']);
                    }
                    $fileArray = $result[$i];
                } else {
                    $fileArray = $result[0];
                }
            } else {
                if (is_file($file) && strpos($file, G_ROOTPATH) !== false) {                                                //Create object without database information
                    $fileArray = array('id'            => -1,                        //Set 'id' to -1, meaning this file has not a database representation
                                       'file'          => $file,
                                       'type'          => 'file',
                                       'physical_name' => basename($file),
                                       'original_name' => basename($file),
                                       'directory'     => dirname($file));
                } else if (strpos($file, G_ROOTPATH) === false) {
                    throw new EfrontFileException(_ILLEGALPATH.': '.$file, EfrontFileException :: ILLEGAL_PATH);
                } else {
                    throw new EfrontFileException(_FILEDOESNOTEXIST.': '.$file, EfrontFileException :: FILE_NOT_EXIST);
                }
            }
        }

        //Append extra useful (derived) information to the array: extension, size, mime type and whether it is a "renamed" file (a file that has its name changed for filesystem compatibility issues, i.e. Greek filenames in Windows file systems) 
        !isset($fileArray['extension'])                            ? $fileArray['extension'] = pathinfo($fileArray['original_name'], PATHINFO_EXTENSION) : null; 
        !isset($fileArray['size'])                                 ? $fileArray['size']      = round(filesize($fileArray['file'])/1024, 2)               : null;
        EfrontDirectory :: normalize($fileArray['original_name']) != EfrontDirectory :: normalize($fileArray['physical_name']) ? $fileArray['renamed'] = true : $fileArray['renamed'] = false;        //If the physical file name is different than the original name, it means that the file is renamed 
        
        if (!isset($fileArray['mime_type'])) {
            isset(EfrontFileSystem :: $mimeTypes[strtolower($fileArray['extension'])]) ? $fileArray['mime_type'] = EfrontFileSystem :: $mimeTypes[strtolower($fileArray['extension'])] : $fileArray['mime_type'] = 'application/'.$fileArray['extension'];
        }
        
        parent :: __construct($fileArray);                        //Create an ArrayObject from the given array
        
        if (!is_file($this['file'])) {                            //If the file does not actually exist, then delete it from database and issue exception
            eF_deleteTableData("files", "id=".$this['id']);
            throw new EfrontFileException(_FILEDOESNOTEXIST.': '.$file, EfrontFileException :: FILE_DELETED);
        } elseif ( strpos(EfrontDirectory :: normalize(dirname($this['file'])), EfrontDirectory :: normalize(G_ROOTPATH)) === false ) {
            throw new EfrontFileException(ILLEGAL_PATH.': '.dirname($this['file']), EfrontFileException :: FILE_DELETED);    //The file must be inside root path, otherwise it is illegal
        }
    }

    /**
     * Delete file
     *
     * This function deletes the file. It first unlinks (if it exists)
     * the physical file, and then deletes its entry from the database.
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(34);                          //Instantiate file
     * $file -> delete();                                   //Delete file
     * </code>
     *
     * @return boolean True if the file was deleted
     * @since 3.5.0
     * @access public
     */
    public function delete() {
        if (is_file($this['file']) && !unlink($this['file'])) {                       //If the file exists but could not be deleted, throw an exception. This way, even files that their equivalent physical file does not exist, may be deleted.
            throw new EfrontFileException(_CANNOTDELETEFILE, EfrontFileException :: GENERAL_ERROR);
        }
        if ($this['id'] != -1) {
            eF_deleteTableData("files", "id=".$this['id']);                   //Delete database representation of the file
        }
        return true;
    }

    /**
     * Copy file
     *
     * This function is used to copy the current file to a new
     * destination. If a file with the same name exists in the
     * destination and $overwrite is true, it will be overwritten
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(43);                                  //Instantiate file object
     * $file -> copy('/var/www/');                                  //Copy file to /var/www/
     * $file -> copy('/var/www/', true);                            //Copy file to /var/www/ and overwrite if it already exists
     * $dir = new EfrontDirectory('/var/www/');                     //Instantiate directory object
     * $file -> copy($dir);                                         //Copy file using EfrontDirectory object as destination
     * </code>
     * If the file being copied doesn't have a corresponding database representation, 
     * the new file won't have one either. Otherwise, a database entry will be created
     * for the new file (An EfrontFile object corresponds to a file without DB representation
     * when the id is 0)
     *
     * @param mixed $destination The destination directory, either a string or an EfrontDirectory object
     * @param boolean $overwrite If true, overwrite existing file with the same name
     * @return EfrontFile The copied file
     * @since 3.5.0
     * @access public
     */
    public function copy($destination, $overwrite = true) {
        if (!($destination instanceof EfrontDirectory)) {
            $destination = new EfrontDirectory($destination);
        }
        
        try {
            $file = new EfrontFile($destination['file'].'/'.$this['original_name']);                //Check if a file with the same *original* name already exists. If it does not, execution will jump to Catch block below
            if (!$overwrite) {                                                                      //If we are here, it means the file already exists. If $overwrite is false, then throw a FILE_ALREADY_EXISTS exception.  
                throw new Exception(_CANNOTCOPYFILE.': '._FILEALREADYEXISTS, EfrontFileException :: FILE_ALREADY_EXISTS);//Use plain Exception rather than EfrontFileException, since the latter is caught right from the following catch block
            } else {
                $file -> delete();
            }
        } catch (EfrontFileException $e) {}                                                     //This means the file does not exist

        if ($this['id'] == -1) {                                                                //An EfrontFile Object with id 0 is a file without DB entry. If so, just copy the file, without creating a DB entry for the new file 
            if (copy($this['file'], $destination['file'].'/'.$this['original_name'])) {
                $file = new EfrontFile($destination['file'].'/'.$this['original_name']);
                return $file;
            } else {
                throw new EfrontFileException(_CANNOTCOPYFILE, EfrontFileException :: UNKNOWN_ERROR);
            }
        } else {
            $fileid  = eF_insertTableData("files", array('id' => 0));                               //Insert an empty row to the files table, so you can get the new id
            $this['renamed'] ? $newName = $fileid.'.'.$this['extension'].'.'.POSTFIX : $newName = $this['original_name'];                                  //The new file name
            if (copy($this['file'], $destination['file'].$newName)) {
                $fields = array("file"          => $destination['file'].'/'.$newName,                   //Database entry for copied file
                                "physical_name" => $newName,
                                "original_name" => $this['original_name'],
                                "directory"     => $destination['file'],
                                "users_LOGIN"   => isset($_SESSION['s_login']) ? $_SESSION['s_login'] : $this['users_LOGIN'],
                                "timestamp"     => time(),
                                "type"          => 'file',
                                "description"   => $this['description'],
                                "groups_ID"     => $this['groups_ID'],
                                "access"        => $this['access']);
                
                eF_updateTableData("files", $fields, "id=$fileid");                                 //Update the empty table entry with specific data this time
    
                $file = new EfrontFile($destination['file'].'/'.$this['original_name']);
                return $file;
            } else {
                eF_deleteTableData("files", "id=$id");                                                //If copy failed, delete empty table entry
                throw new EfrontFileException(_CANNOTCOPYFILE, EfrontFileException :: UNKNOWN_ERROR);
            }
        }
    }

    /**
     * Move file
     *
     * This function is equivalent to copy(), except that it deletes the original
     * file after copying it.
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(43);                                  //Instantiate file object
     * $file -> move('/var/www/');                                  //Move file to /var/www/
     * $file -> move('/var/www/', true);                            //Move file to /var/www/ and overwrite if it already exists
     * $dir = new EfrontDirectory('/var/www/');                     //Instantiate directory object
     * $file -> move($dir);                                         //Move file using EfrontDirectory object as destination
     * </code>
     *
     * @param mixed $destination The destination directory, either a string or an EfrontDirectory object
     * @param boolean $overwrite If true, overwrite existing file with the same name
     * @return EfrontFile The copied file
     * @since 3.5.0
     * @access public
     * @see copy()
     */
    public function move($destination, $overwrite = true) {
        if ($file = $this -> copy($destination, $overwrite)) {
            $this -> delete();
        }
        return $file;
    }

    /**
     * Rename file
     *
     * This function is used to rename the file's original name
     * to the one specified
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(43);                                  //Instantiate file object
     * $file -> rename('new name');                                 //Rename file
     * </code>
     *
     * @param string $newName The new (original) file name
     * @return boolean True if everything is ok
     * @since 3.5.0
     * @access public
     */
    public function rename($newName) {
        if (eF_checkParameter($newName, 'filename')) {
            $newPath = dirname($this['file'])."/$newName";               //Concatenate new name with current path to get the new path
            if (is_file($newPath) || sizeof(eF_getTableData("files", "*", "file='$newPath'")) > 0) {        //Check if a file with the same name already exists
                throw new EfrontFileException(_CANNOTRENAMEFILE.': '.$newName, EfrontFileException :: FILE_ALREADY_EXISTS);
            }
            if (!$this['renamed']) {                                    //If the file is not 'renamed', i.e. the physical name is different than the original name, then the original name should change as well
                $result = rename($this['file'], $newPath);
                if (!$result) {
                    throw new EfrontFileException(_CANNOTRENAMEFILE.': '.$newName, EfrontFileException :: ILLEGAL_FILE_NAME);
                }
                $this['file']          = $newPath;                      //Update physical name and full path to the database
                $this['physical_name'] = $newName;
            }
            $this['original_name'] = $newName;                          //Update original name to the database
            
            if ($this['id'] != -1) {                                    //Only if a DB representation exists
                $this -> persist();
            }
            
            return true;
        } else {
            throw new EfrontFileException(_CANNOTRENAMEFILE.': '.$newName, EfrontFileException :: ILLEGAL_FILE_NAME);
        }
    }

    /**
     * Persist file values
     *
     * This function is used to persist any changed values
     * of the file.
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(43);                                  //Instantiate file object
     * $file -> file['description'] = 'New description';            //Change a file's property
     * $file -> persist();                                          //Persist changes
     * </code>
     *
     * @return boolean true if everything is ok
     * @since 3.5.0
     * @access public
     */
    public function persist() {
        $fields = array('file'          => $this['file'],
                        'type'          => $this['type'],
                        'physical_name' => $this['physical_name'],
                        'original_name' => $this['original_name'],
                        'directory'     => $this['directory'],
                        'description'   => $this['description'],
                        'groups_ID'     => $this['groups_ID'],
                        'access'        => $this['access'],
                        'lessons_ID'    => $this['lessons_ID'],
                        'shared'        => $this['shared']);
        return eF_updateTableData("files", $fields, "id=".$this['id']);
    }

    public function compress($method = 'zip') {

    }

    /**
     * Uncompress file
     *
     * This function is used to uncompress the current file.
     * If the file has a database representation, the uncompressed
     * files will also have one
     * <br/>Example:
     * <code>
     * $file = new EfrontFile('/var/www/test.zip');
     * $uncompressedFiles = $file -> uncompress();
     * </code> 
     * 
     * @return array An array of EfrontFile objects or file paths (depending on wheter a database representation exists)
     * @since 3.5.0
     * @access public
     */
    public function uncompress() {
        $zip = new ZipArchive;
        if ($zip -> open($this['file']) === true && $zip -> extractTo($this['directory'])) {            
            for ($i = 0; $i < $zip -> numFiles; $i++) {
                $zipFiles[] = EfrontDirectory :: normalize($this['directory']).'/'.$zip -> getNameIndex($i);
            }
            if ($this['id'] != -1) {
                $this['lessons_ID'] ? $options['lessons_ID'] = $this['lessons_ID'] : $options = array();
                $importedFiles = FileSystemTree :: importFiles($zipFiles, $options);
                return $importedFiles;
            } else {
                return $zipFiles;            
            }            
        } else {
            throw new EfrontFileException(_CANNOTOPENCOMPRESSEDFILE.': '.$this['file'], EfrontFileException :: ERROR_OPEN_ZIP);
        }
    }
    
    /**
     * Get the image for the file type
     * 
     * This function returns the url to an image representing the current
     * file type.
     * <br/>Example:
     * <code>
     * echo $file -> getTypeImage();            //Returns something like 'images/16x16/zip.png' if it's a zip file 
     * </code>
     * 
     * @return string The url to the image representing the file type
     * @since 3.5.0
     * @access public
     */
    public function getTypeImage() {
        if (is_file(G_IMAGESPATH.'file_types/'.$this['extension'].'.png')) {
            $image = 'images/file_types/'.$this['extension'].'.png';
        } else {
            $image = 'images/file_types/unknown.png';
        }        
        return $image;
    }

    /**
     * Share file
     * 
     * This function is used to make the current file available to the lesson's
     * students. It must belong to a lesson (that is, it must have a lesson id)
     * in order to do so.
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(43);
     * $file -> share();                            //The file is now visible to the shared files list
     * $file -> unshare();                          //The file was made hidden again
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    public function share() {
        if ($this['lessons_ID']) {
            $this['shared'] = true;
            $this -> persist();
        } else {
            throw new EfrontFileException(_NOTALESSONFILE.': '.$directory, EfrontFileException :: NOT_LESSON_FILE);
        }
    }
    
    /**
     * Unshare file
     * 
     * This function is used to make the current file unavailable to the lesson's
     * students. It must belong to a lesson (that is, it must have a lesson id)
     * in order to do so.
     * <br/>Example:
     * <code>
     * $file = new EfrontFile(43);
     * $file -> share();                            //The file is now visible to the shared files list
     * $file -> unshare();                          //The file was made hidden again
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    public function unshare() {
        if ($this['lessons_ID']) {
            $this['shared'] = false;
            $this -> persist();
        } else {
            throw new EfrontFileException(_NOTALESSONFILE.': '.$directory, EfrontFileException :: NOT_LESSON_FILE);
        }
    }
}

class EfrontFileUnitTest
{
    public static function test() {
        /**
         * __construct(): ok
         */
        //A file id
        //$file = new EfrontFile(1440); 
        //the full path to a physical file, with associated DB entry
        //$file = new EfrontFile('C:/Projects/efront/trunc/backups/1340.pdf.efront');
        //the full path to a file, with associated DB entry, using its original file name
        //$file = new EfrontFile('C:/Projects/efront/trunc/backups/eFront_price.pdf');
        //the full path to a physical file, without associated DB entry
        $file = new EfrontFile('C:/Projects/efront/trunc/backups/efront_backup_2008_04_23_03.20.30.tar.gz');
        //an array with file attributes
        //$file = new EfrontFile(array('file' => 'C:/Projects/efront/trunc/backups/efront_backup_2008_04_23_03.20.30.tar.gz'));
        
        /**
         * delete(): ok
         */
        //$file -> delete();

        /**
         * copy(): ok
         */
/*        
        $destination = new EfrontDirectory('C:/Projects/efront/trunc/backups/testdir');
        pr($destination);
        $newFile = $file -> copy($destination, false);
        pr($newFile);
*/
        /**
         * move(): ok
         */
/*
        $destination = new EfrontDirectory('C:/Projects/efront/trunc/backups/testdir');
        pr($destination);
        $newFile = $file -> move($destination, false);
        pr($newFile);
*/                
        /**
         * rename(): ok
         */
/*
        $newName = 'aaa.txt';
        $file -> rename($newName);
*/        
        /**
         * persist(): ok
         */
        //$file -> persist();
        
        /**
         * compress():
         */
        //$file -> compress($method = 'zip');
        
        /**
         * uncompress():
         */
        //$file -> uncompress();        
    }
}

/**
 * Class for directories in Efront file system
 *
 * @since 3.5.0
 * @package efront
 */
class EfrontDirectory extends ArrayObject implements EfrontFileSystemEntity
{

    /**
     * Class constructor
     *
     * The class constructor instantiates the object based on the $directory parameter.
     * $directory may be either:
     * - an array with directory attributes
     * - a directory id
     * - the full path to a physical directory
     * - the full path to a directory using its original directory name
     * <br/>Example:
     * <code>
     * $result = eF_getTableData("files", "*", "id=43");
     * $file = new EfrontDirectory($result[0]);                          //Instantiate object using array of values
     * $file = new EfrontDirectory(43);                                  //Instantiate object using id
     * $file = new EfrontDirectory('/var/www/test/');                    //Instantiate object using path and original name
     * $file = new EfrontDirectory('/var/www/32/');                      //Instantiate object using path and physical name
     * </code>
     *
     * @param mixed $directory The directory information, either an array, an id or a path string
     * @since 3.5.0
     * @access public
     */
    function  __construct($directory) {
        
        if (is_array($directory)) {
            $directoryArray = $directory;
        } else {
            if (eF_checkParameter($directory, 'id')) {
                $result = eF_getTableData("files", "*", "type = 'directory' and id=".$directory);
            } else {
                $result = eF_getTableData("files", "*", "type = 'directory' and file='$directory'");
                if (sizeof($result) <= 0 || !is_dir($result[0]['file'])) {                                //A directory with the specified filename was not found. Check if the original directory name was specified
                    $result = eF_getTableData("files", "*", "type = 'directory' and original_name='".basename($directory)."' and directory = '".EfrontFileSystem :: checkPath(dirname($directory))."'");
                }
            }

            if (sizeof($result) > 0) {
                if (sizeof($result) > 1) {                            //if for some reason there is more than 1 database entries for the same file, keep only the latest (based on id)
                    for ($i = 0; $i < sizeof($result) - 1; $i++) {
                        eF_deleteTableData("files", "id=".$result[$i]['id']);
                        unlink($result[$i]['file']);
                    }
                    $directoryArray = $result[$i];
                } else {
                    $directoryArray = $result[0];
                }
                
            } else {
                if (is_dir($directory) && strpos($directory, G_ROOTPATH) !== false) {                            //Create object without database information
                    $directoryArray = array('id'            => -1,                        //Set 'id' to -1, meaning this directory has not a database representation
                                            'file'          => rtrim($directory, '/').'/',
                                            'type'          => 'directory',
                                            'physical_name' => basename($directory),
                                            'original_name' => basename($directory),
                                            'directory'     => dirname($directory).'/');
                } else if (strpos($directory, G_ROOTPATH) === false) {
                    throw new EfrontFileException(_ILLEGALPATH.': '.$directory, EfrontFileException :: ILLEGAL_PATH);
                } else {
                    throw new EfrontFileException(_DIRECTORYDOESNOTEXIST.': '.$directory, EfrontFileException :: DIRECTORY_NOT_EXIST);
                }
            }
        }

        
        self :: normalize($directoryArray['original_name']) != self :: normalize($directoryArray['physical_name']) ? $directoryArray['renamed'] = true : $directoryArray['renamed'] = false;        //If the physical directory name is different than the original name, it means that the directory is renamed
        parent :: __construct($directoryArray);                        //Create an ArrayObject from the given array
        
        if (!is_dir($this['file'])) {
            eF_deleteTableData("files", "id=".$this['id']);
            throw new EfrontFileException(_DIRECTORYDOESNOTEXIST.': '.$this['file'], EfrontFileException :: FILE_DELETED);
        } elseif ( strpos(self :: normalize(dirname($this['file'])), self :: normalize(G_ROOTPATH)) === false ) {
            throw new EfrontFileException(ILLEGAL_PATH.': '.dirname($this['file']), EfrontFileException :: FILE_DELETED);    //The file must be inside root path, otherwise it is illegal
        } elseif (self :: normalize(dirname($this['file'])) !== $this['file']) {                                             //If the normalized path is different than the actual, make sure they match...
            $this['file'] = self :: normalize($this['file']);
            if ($this['id'] !== -1) {                                                                                        //...and store changed value, if a DB representation is set 
                $this -> persist();
            }
        }
    }

    /**
     * Normalize path
     *
     * This function is used to normalize a path string. It applies
     * realpath() to translate relevant paths, converts \ to / and trims
     * trailing /
     * <br/>Example:
     * <code>
     * $path = '..\..\..\test.php\';
     * echo EfrontDirectory :: normalize($path);    //Outputs c:/test.php
     * </code>
     * 
     * @param string $path The path to normalize
     * @return string The normalized path
     * @since 3.5.0
     * @access public
     */
    public static function normalize($path) {
        if (realpath($path)) {        
            return rtrim(str_replace("\\", "/", realpath($path)), "/");
        } else {
            return rtrim(str_replace("\\", "/", $path), "/");
        }
    }
        
    /**
     * Delete directory
     *
     * This function deletes the directory. The directory must be empty, otherwise
     * it will not be deleted. For deleting non-empty directories, see FileSystemTree
     * equivalent method
     * <br/>Example:
     * <code>
     * $directory = new EfrontDirectory(34);                     //Instantiate directory
     * $directory -> delete();                                   //Delete directory
     * </code>
     *
     * @return boolean True if the directory was deleted
     * @since 3.5.0
     * @access public
     */
    public function delete() {
        if (is_dir($this['file']) && !rmdir($this['file'])) {                       //If the directory exists but could not be deleted, throw an exception. This way, even directories that their equivalent physical value does not exist, may be deleted.
            throw new EfrontFileException(_CANNOTDELETEDIRECTORY, EfrontFileException :: GENERAL_ERROR);
        }
        eF_deleteTableData("files", "id=".$this['id']);                             //Delete database representation of the directory
        return true;
    }

    /**
     * Copy directory
     *
     * This function is used to copy the current directory to a new destination. The
     * directory contents are *not* copied. In order to copy contents as well, use
     * appropriate FileSystemTree method
     * <br/>Example:
     * <code>
     * $directory = new EfrontDirectory(43);                            //Instantiate directory object
     * $directory -> copy('/var/www/');                                 //Copy directory to /var/www/
     * $dir = new EfrontDirectory('/var/www/');                         //Instantiate directory object
     * $directory -> copy($dir);                                        //Copy directory using EfrontDirectory object as destination
     * </code>
     *
     * @param mixed $destination The destination directory, either a string or an EfrontDirectory object
     * @param boolean $overwrite There is not point in overwriting directories, so this parameter has no effect
     * @return EfrontDirectory The copied directory
     * @since 3.5.0
     * @access public
     */
    public function copy($destination, $overwrite = false) {        
        if (!($destination instanceof EfrontDirectory)) {
            $destination = new EfrontDirectory($destination);
        }

        try {
            $directory = new EfrontDirectory($destination['file'].'/'.$this['original_name']);                //Check if a directory with the same *original* name already exists. If it does not, execution will jump to Catch block below
            return $directory;
        } catch (EfrontFileException $e) {}                                                     //This means the directory does not exist

        if ($this['id'] == -1) {                                                                //An EfrontDirectory Object with id 0 is a file without DB entry. If so, just copy the file, without creating a DB entry for the new file 
            if (mkdir($destination['file'].'/'.$this['original_name'], 0755)) {
                $directory = new EfrontDirectory($destination['file'].'/'.$this['original_name']);
                return $directory;
            } else {
                throw new EfrontFileException(_CANNOTCOPYDIRECTORY, EfrontFileException :: UNKNOWN_ERROR);
            }
        } else {
            $fileid  = eF_insertTableData("files", array('id' => 0));                               //Insert an empty row to the files table, so you can get the new id
            $this['renamed'] ? $newName = $fileid : $newName = $this['original_name'];                                  //The new directory name
            if (mkdir($destination['file'].$newName, 0755)) {
                $fields = array("file"          => $destination['file'].'/'.$newName,                   //Database entry for copied directory
                                "physical_name" => $newName,
                                "original_name" => $this['original_name'],
                                "directory"     => $destination['file'],
                                "users_LOGIN"   => isset($_SESSION['s_login']) ? $_SESSION['s_login'] : $this['users_LOGIN'],
                                "timestamp"     => time(),
                                "type"          => 'directory',
                                "description"   => $this['description'],
                                "groups_ID"     => $this['groups_ID'],
                                "access"        => $this['access']);
                
                eF_updateTableData("files", $fields, "id=$fileid");                                 //Update the empty table entry with specific data this time
    
                $directory = new EfrontDirectory($destination['file'].'/'.$this['original_name']);
                return $directory;
            } else {
                eF_deleteTableData("files", "id=$id");                                                //If copy failed, delete empty table entry
                throw new EfrontFileException(_CANNOTCOPYDIRECTORY, EfrontFileException :: UNKNOWN_ERROR);
            }
        }        
    }

    /**
     * Move directory
     *
     * This function is equivalent to copy(), except that it deletes the original
     * directory after copying it. Directory contents are *not* moved
     * <br/>Example:
     * <code>
     * $directory = new EfrontDirectory(43);                             //Instantiate directory object
     * $directory -> move('/var/www/');                                  //Move directory to /var/www/
     * $dir = new EfrontDirectory('/var/www/');                          //Instantiate directory object
     * $directory -> move($dir);                                         //Move directory using EfrontDirectory object as destination
     * </code>
     *
     * @param mixed $destination The destination directory, either a string or an EfrontDirectory object
     * @return EfrontDirectory The copied directory
     * @since 3.5.0
     * @access public
     * @see copy()
     */
    public function move($destination, $overwrite = false) {
        if ($directory = $this -> copy($destination)) {
            try {
                $this -> delete();
            } catch (Exception $e) {                        //If the deletion could not be done, delete the newly created (empty) directory
                $directory -> delete();
                throw $e;
            }
        }
        return $directory;
    }

    /**
     * Rename directory
     *
     * This function is used to change the directory's original name
     * to the one specified. This function does not update directory's
     * contents to the new name. Use FileSystemTree renaming method instead.
     * <br/>Example:
     * <code>
     * $directory = new EfrontDirectory(43);                                  //Instantiate directory object
     * $directory -> rename('new name');                                      //Rename directory
     * </code>
     * If the directory doesn't have a database representation, only its physical name
     * is changed. Furthermore,if the directory is not "renamed", that is, its physical 
     * name matches its original name, then its physical name is changed also. In this case
     * however, normally all contents of the directory should be updated to include the new name.
     * In order to achieve this desired effect, you should use equivalent FileySystemTree function.
     *
     * @param string $newName The new (original) directory name
     * @return boolean True if everything is ok
     * @since 3.5.0
     * @access public
     */
    public function rename($newName) {
        if (eF_checkParameter($newName, 'filename')) {
            $newPath = dirname($this['file'])."/$newName";               //Concatenate new name with current path to get the new path
            if (is_file($newPath) || is_dir($newPath) || sizeof(eF_getTableData("files", "*", "file='$newPath'")) > 0) {        //Check if a directory with the same name already exists
                throw new EfrontFileException(_CANNOTRENAMEDIRECTORY.': '.$newName, EfrontFileException :: DIRECTORY_ALREADY_EXISTS);
            }
            if (!$this['renamed']) {                                    //If the directory is not 'renamed', i.e. the physical name is different than the original name, then the original name should change as well
                $result = rename($this['file'], $newPath);
                if (!$result) {                                         //The directory cannot be renamed
                    throw new EfrontFileException(_CANNOTRENAMEDIRECTORY.': '.$newName, EfrontFileException :: ILLEGAL_FILE_NAME);
                }
                $this['file']          = $newPath;                      //Update physical name and full path to the database
                $this['physical_name'] = $newName;
            }
            $this['original_name'] = $newName;                          //Update original name to the database
            
            if ($this['id'] != -1) {                                    //Only if a DB representation exists
                $this -> persist();
            }
            
            return true;
        } else {
            throw new EfrontFileException(_CANNOTRENAMEDIRECTORY.': '.$newName, EfrontFileException :: ILLEGAL_FILE_NAME);
        }
    }

    /**
     * Persist directory values
     *
     * This function is used to persist any changed values
     * of the directory.
     * <br/>Example:
     * <code>
     * $directory = new EfrontDirectory(43);                                  //Instantiate directory object
     * $directory -> file['description'] = 'New description';                 //Change a directory's property
     * $directory -> persist();                                               //Persist changes
     * </code>
     *
     * @return boolean true if everything is ok
     * @since 3.5.0
     * @access public
     */
    public function persist() {
        $fields = array('file'          => $this['file'],
                        'type'          => $this['type'],
                        'physical_name' => $this['physical_name'],
                        'original_name' => $this['original_name'],
                        'directory'     => $this['directory'],
                        'users_LOGIN'   => $this['users_LOGIN'],
                        'timestamp'     => $this['timestamp'],
                        'description'   => $this['description'],
                        'groups_ID'     => $this['groups_ID'],
                        'access'        => $this['access']);
        return eF_updateTableData("files", $fields, "id=".$this['id']);
    }

    /**
     * Create directory
     * 
     * This function is used to create a new directory along with its
     * database representation (as long as $addDB is not false)
     * <br/>Example:
     * <code>
     * EfrontDirectory :: createDirectory('/var/www/efront/www/content/lessons/32/new directory');
     * </code>
     *
     * @param string $fullPath The full path to the new directory
     * @param string $addDB Whether to create database represenation for the new directory, defaults to true 
     * @return boolean true if everything is ok
     * @since 3.5.0
     * @access public
     */
    public static function createDirectory($fullPath, $addDB = true) {
        $newName         = basename($fullPath);
        $parentDirectory = new EfrontDirectory(self :: normalize(dirname($fullPath)));
        $newPath         = self :: normalize($parentDirectory['file']).'/'.$newName;

        try {
            $directory = new EfrontDirectory($newPath);                //Check if a directory with the same *original* name already exists. If it does not, execution will jump to Catch block below
            throw new Exception(_COULDNOTCREATEDIRECTORY.': '._DIRECTORYALREADYEXISTS, EfrontFileException :: DIRECTORY_ALREADY_EXISTS);
        } catch (EfrontFileException $e) {}                                                     //This means the directory does not exist
        
        if (eF_checkParameter($newName, 'file') && strpos($newPath, EfrontDirectory :: normalize(G_ROOTPATH)) !== false && mkdir($newPath, 0755)) {
            if ($addDB) {                            //If the parent directory has a database representation, then create one for the new directory also 
                $fields = array("file"          => $newPath,
                                "physical_name" => $newName,
                                "original_name" => $newName,
                                "directory"     => $parentDirectory['file'],
                                "users_LOGIN"   => isset($_SESSION['s_login']) ? $_SESSION['s_login'] : '',
                                "timestamp"     => time(),
                                "type"          => 'directory');
                $result = eF_insertTableData("files", $fields);
            }
            if (!$result) {
                rmdir($newPath);                                    //If the database entry could not be created, delete the created directory
            }
            return $result;
        } else {
            return false;
        }
        
    }

    
    public function compress($method = 'zip') {

    }

    /**
     * Uncompress directory
     *
     * There is no point in uncompressing a directory
     * @return false
     * @since 3.5.0
     * @access public
     */
    public function uncompress() {
        return false;        
    }

    /**
     * Get the image for the directory
     * 
     * This function returns the url to an image representing the directory.
     * similar to EfrontFile :: getTypeImage().
     * <br/>Example:
     * <code>
     * echo $directory -> getTypeImage();           //Returns something like 'images/16x16/folder.png'  
     * </code>
     * 
     * @return string The url to the image representing the directory
     * @since 3.5.0
     * @access public
     */
    public function getTypeImage() {
        $image = 'images/file_types/folder.png';
        return $image;
    }
}

class EfrontDirectoryUnitTest
{
    public static function test() {
        /**
         * __construct(): ok
        */

        //A directory id
        $directory = new EfrontDirectory(1415); 
        //the full path to a physical directory, with associated DB entry
        //$directory = new EfrontDirectory('C:/Projects/efront/trunc/backups/1415');
        //the full path to a directory, with associated DB entry, using its original directory name
        //$directory = new EfrontDirectory('C:/Projects/efront/trunc/backups/testdir');
        //the full path to a physical directory, without associated DB entry
        //$directory = new EfrontDirectory('C:/Projects/efront/trunc/backups/testdiraa');
        //an array with directory attributes
        //$directory = new EfrontDirectory(array('file' => 'C:/Projects/efront/trunc/backups/testdiraa'));
        pr($directory);
        
        /**
         * delete(): ok
         */
        //$directory -> delete();

        /**
         * copy(): ok
         */
/*
        $destination = new EfrontDirectory('C:/Projects/efront/trunc/backups/testdir/');
        pr($destination);
        $directory -> copy($destination);
*/
                
        /**
         * move(): ok
         */
/*
        $destination = new EfrontDirectory('C:/Projects/efront/trunc/backups/testdir/');
        pr($destination);
        $directory -> move($destination);
*/        
        /**
         * rename(): ok
         */
/*
        $newName = 'testdir';
        $directory -> rename($newName);
*/
        /**
         * persist(): ok
         */
        //$directory -> persist();
        
        /**
         * createDirectory():
         */
        //EfrontDirectory :: createDirectory();
        
        /**
         * compress():
         */
        //$directory -> compress($method = 'zip');
        
        /**
         * uncompress():
         */
        //$directory -> uncompress();
    }
}

/**
 * File system tree
 *
 * This class represents the file system tree, with directories being
 * branches or leafs and files being only leafs
 * @since 3.5.0
 * @author Venakis Periklis <pvenakis@efront.gr>
 */
class FileSystemTree extends EfrontTree
{
    /**
     * The tree's root directory
     *
     * @var string
     * @since 3.5.0
     * @access protected
     */
    protected $dir = '';

    /* Initialize tree
     *
     * This function is used to initialize the file system tree
     * <br/>Example:
     * <code>
     * $fileSystemTree = new EfrontFileSystemTree();
     * </code>
     *
     * @param string $dir The root directory for the filesystem tree
     * @since 3.5.0
     * @access public
     */
    function __construct($dir = G_ROOTPATH) {
        if ($dir instanceof EfrontDirectory) {
            $dir = $dir['file'];
        }
        if (!is_dir($dir)) {
            throw new EfrontFileException(_DIRECTORYDOESNOTEXIST.': '.$dir, EfrontFileException :: DIRECTORY_NOT_EXIST);
        }
        
        $this -> dir = new EfrontDirectory($dir);
        $this -> reset();
    }

    /**
     * Insert node to the tree
     *
     * This function is not used by this EfrontTree implementation,
     * so it always returns false
     *
     * @param EfrontFileSystemEntity $node
     * @param int $parentNode
     * @param int $previousNode
     * @return boolean Always false
     * @since 3.5.0
     * @access public
     */
    public function insertNode($node, $parentNode = false, $previousNode = false) {
        return false;
    }

    /**
     * Remove node from tree
     *
     * This function is not used by this EfrontTree implementation,
     * so it always returns false
     *
     * @param EfrontFileSystemEntity $node
     * @return boolean Always false
     * @since 3.5.0
     * @access public
     */
    public function removeNode($node) {
        return false;
    }

    /**
     * Delete directory
     * 
     * This function is used in order to delete the specified directory.
     * The directory must be part of the filesystem tree. Its contents are
     * deleted recursively.
     * <br/>Example:
     * <code>
     * $filesystem = new FileSystemTree('/path/to/dir/');
     * $filesystem -> deleteDirectory('/path/to/dir/deletethis/');
     * </code>
     *
     * @param mixed $directory The directory to delete, either an id, a path or an EfrontDirectory object
     * @since 3.5.0
     * @access public
     */
    public function deleteDirectory($directory) {
        if (eF_checkParameter($directory, 'id')) {                    //If $directory is an id, instantiate EfrontDirectory object
            $directory = new EfrontDirectory($directory);
        }
        if ($directory instanceof EfrontDirectory) {                  //If $directory is an EfrontDirectory object, get its path
            $directory = $directory['file'];
        }
        
        $node = new RecursiveArrayIterator($this -> seekNode($directory));
        
        $iterator = new EfrontFileOnlyFilterIterator(new RecursiveIteratorIterator($node, RecursiveIteratorIterator :: SELF_FIRST));
        foreach ($iterator as $key => $value) {
            $value -> delete();
        }
        $iterator = new EfrontDirectoryOnlyFilterIterator(new RecursiveIteratorIterator($node, RecursiveIteratorIterator :: SELF_FIRST));
        foreach ($iterator as $key => $value) {
            $directories[] = $value;
        }

        //Directories must be put in reverse order before we can delete them, otherwise they won't be empty and an exception will be issued
        if (isset($directories)) {
            array_reverse($directories);
            foreach ($directories as $key => $value) {
                $value -> delete();
            }
        }
        //Delete the current directory as well
        $directory = new EfrontDirectory($directory);
        $directory -> delete();
                
        $this -> reset();
        
    }
    
    /**
     * Copy directory
     * 
     * This function is used to recursively copy a directory
     * <br/>Example:
     * <code>
     * $sourceDir = new EfrontDirectory(34);
     * $targetDir = new EfrontDirectory(54);
     * $tree = new FileSystemTree();
     * $tree -> copyDirectory($sourceDir, $targetDir);
     * </code>
     *
     * @param mixed $directory The source directory, can be a string, an id or an EfrontDirectory object
     * @param mixed $destination The destination directory, can be a string, an id or an EfrontDirectory object
     * @param boolean $overwrite Whether to overwrite destination files
     * @since 3.5.0
     * @access public
     */
    public function copyDirectory($directory, $destination, $overwrite = true) {
        if (!($destination instanceof EfrontDirectory)) {
            $destination = new EfrontDirectory($destination);
        }        

        if (!($directory instanceof EfrontDirectory)) {
            $directory = new EfrontDirectory($directory);
        }                
        
        $node = new RecursiveArrayIterator($this -> seekNode($directory['file']));
        
        //First, copy the root directory
        $directory -> copy($destination, $overwrite);
        //Then, create directory structure
        $iterator = new EfrontDirectoryOnlyFilterIterator(new RecursiveIteratorIterator($node, RecursiveIteratorIterator :: SELF_FIRST));
        foreach ($iterator as $key => $value) {
            $dest = dirname(str_replace(EfrontDirectory :: normalize(dirname($directory['file'])), EfrontDirectory :: normalize($destination['file']), $value['file']));
            $value -> copy($dest, $overwrite);
        }        
        //Now copy files to the right locations
        $iterator = new EfrontFileOnlyFilterIterator(new RecursiveIteratorIterator($node, RecursiveIteratorIterator :: SELF_FIRST));
        foreach ($iterator as $key => $value) {
            $dest = dirname(str_replace(EfrontDirectory :: normalize(dirname($directory['file'])), EfrontDirectory :: normalize($destination['file']), $value['file']));
            try {
                $value -> copy($dest, $overwrite);
            } catch (Exception $e) {
                if ($e -> getCode() != EfrontFileException :: FILE_ALREADY_EXISTS) {            //Move on to the next file, if the exception is due to same file existing 
                    throw($e);
                } 
            }
        }
        
        $this -> reset();        
    }
    
    /**
     * Move directory
     * 
     * This function is used to recursively move a directory. 
     * <br/>Example:
     * <code>
     * $sourceDir = new EfrontDirectory(34);
     * $targetDir = new EfrontDirectory(54);
     * $tree = new FileSystemTree();
     * $tree -> moveDirectory($sourceDir, $targetDir);
     * </code>
     *
     * @param mixed $directory The source directory, can be a string, an id or an EfrontDirectory object
     * @param mixed $destination The destination directory, can be a string, an id or an EfrontDirectory object
     * @param boolean $overwrite Whether to overwrite destination files
     * @since 3.5.0
     * @access public
     */
    public function moveDirectory($directory, $destination, $overwrite = true) {
        $this -> copy($directory, $destination);
        $this -> deleteDirectory($directory);
    }
    
    /**
     * Rename directory
     * 
     * This function is used to rename the current directory.
     * It makes sure that all its contents' database descriptions
     * are updated accordingly
     * <br/>Example:
     * <code> 
     * $directory = new EfrontDirectory(34);
     * $tree = new FileSystemTree();
     * $tree -> renameDirectory($sourceDir, 'newName');
     * </code>
     * 
     * @param mixed $directory The directory to rename, can be a string, an id or an EfrontDirectory object
     * @param string $newName The new directory name
     * @since 3.5.0
     * @access public
     */
    public function renameDirectory($directory, $newName) {
        if (!($directory instanceof EfrontDirectory)) {
            $directory = new EfrontDirectory($directory);
        }                
        
        $node    = new RecursiveArrayIterator($this -> seekNode($directory['file']));
        $oldPath = $directory['file'];
        $directory -> rename($newName);
        
        //If the directory is renamed, then only its original name is changed (stored in the database), so we do not need to update its contents
        if (!$directory['renamed']) {
            $iterator = new EfrontNodeFilterIterator(new RecursiveIteratorIterator($node, RecursiveIteratorIterator :: SELF_FIRST));
            foreach ($iterator as $key => $value) {
                if ($value['id'] != -1) {                        //Only contents with database representations need to change
                    $value['file']      = str_replace($oldPath, $directory['file'], $value['file']);
                    $value['directory'] = str_replace($oldPath, $directory['file'], $value['directory']);
                    $value -> persist();
                }
            }
        }
    }  
    
    /**
     * Reset filesystem tree
     *
     * This function is used to reset (or initially set) the filesystem
     * tree to its original state. The function is normally called by the
     * constructor.
     * <br/>Example:
     * <code>
     * $tree -> reset();
     * </code>
     *
     * @since 3.5.0
     * @access public
     */
    public function reset() {
        $result = eF_getTableData("files", "*");
        foreach ($result as $file) {
            $files[rtrim($file['file'], '/')] = $file;
        }
        
        $it = new EfrontREFilterIterator(new RecursiveIteratorIterator(new RecursiveDirectoryIterator($this -> dir['file']), RecursiveIteratorIterator :: SELF_FIRST), array('/.svn/'), false);        
        foreach ($it as $node => $value) {           
            $current         = EfrontDirectory :: normalize($node);            
            //Instantiate file/directory object. We are using an approach that doesn't require any database queries
            if (isset($files[$current])) {
                $files[$current]['type'] == 'file' ? $nodes[$current] = new EfrontFile($files[$current]) : $nodes[$current] = new EfrontDirectory($files[$current]);
            } else {
                $fileArray = array('id'            => -1,                        //Set 'id' to -1, meaning this file/directory has not a database representation
                                   'file'          => $current,
                                   'type'          => ($value -> isFile()) ? 'file' : 'directory',
                                   'physical_name' => basename($current),
                                   'original_name' => basename($current),
                                   'directory'     => dirname($current),
                                   'timestamp'     => filemtime($current));
                $fileArray['type'] == 'file' ? $nodes[$current] = new EfrontFile($fileArray) : $nodes[$current] = new EfrontDirectory($fileArray);
            }
            $nodes[$current]['directory']     = EfrontDirectory :: normalize($nodes[$current]['directory']);
            //$nodes[$current]['physical_name'] = EfrontDirectory :: normalize($nodes[$current]['physical_name']);
            //$nodes[$current]['original_name'] = EfrontDirectory :: normalize($nodes[$current]['original_name']);
            $nodes[$current]['file']          = EfrontDirectory :: normalize($nodes[$current]['file']);

        }

        $parentNode = EfrontDirectory :: normalize($this -> dir['file']);
        $rejected   = array();
        $tree       = $nodes;
        $count      = 0;                                                                          //$count is used to prevent infinite loops
        while (sizeof($tree) > 0 && $count++ < 1000) {                                       //We will merge all branches under the main tree branch, the 0 node, so its size will become 1
            foreach ($nodes as $key => $value) {
                if ($value['directory'] == $parentNode || in_array($value['directory'], array_keys($nodes))) {        //If the unit parent (directory) is in the $nodes array keys - which are the unit ids- or it is 0, then it is  valid
                    $parentNodes[$value['directory']][]      = $value;               //Find which nodes have children and assign them to $parentNodes
                    $tree[$value['directory']][$value['file']] = array();              //We create the "slots" where the node's children will be inserted. This way, the ordering will not be lost
                } else {
                    $rejected = $rejected + array($value['file'] => $value);                   //Append units with invalid parents to $rejected list
                    unset($nodes[$key]);                                                     //Remove the invalid unit from the units array, as well as from the parentUnits, in case a n entry for it was created earlier
                    unset($parentNodes[$value['directory']]);
                }
            }

            if (isset($parentNodes)) {                                                       //If the unit was rejected, there won't be a $parentNodes array
                $leafNodes = array_diff(array_keys($nodes), array_keys($parentNodes));       //Now, it's easy to see which nodes are leaf nodes, just by subtracting $parentNodes from the whole set
                foreach ($leafNodes as $leaf) {
                    $parent_id = $nodes[$leaf]['directory'];                         //Get the leaf's parent
                    $tree[$parent_id][$leaf] = $tree[$leaf];                                 //Append the leaf to its parent's tree branch
                    unset($tree[$leaf]);                                                     //Remove the leaf from the main tree branch
                    unset($nodes[$leaf]);                                                    //Remove the leaf from the nodes set
                }
                unset($parentNodes);                                                         //Reset $parentNodes; new ones will be calculated at the next loop
            }
        }
        
        if (sizeof($tree) > 0 && !isset($tree[$this -> dir['file']])) {                                         //This is a special case, where only one node exists in the tree
            $tree = array($tree);
        } 

        if (sizeof($rejected) > 0) {                                            //Append rejected nodes to the end of the tree array, updating their parent/previous information
            foreach ($rejected as $key => $value) {
                //eF_updateTableData("directions", array("parent_direction_ID" => 0), "id=".$key);
                //$value['parent_direction_ID'] = 0;
                //$tree[0][] = $value;
            }
        }
        
        if (sizeof($tree) > 0) {
            $this -> tree = new RecursiveArrayIterator($tree[$this -> dir['file']]);
        } else {
            $this -> tree = new RecursiveArrayIterator(array());
        }
    }
    
    /**
     * Get the upload form
     * 
     * This function is responsible for creating the "upload file" 
     * form, as well as the equivalent HTML code. 
     *
     * @param HTML_QuickForm $form The form to populate 
     * @return string The HTML code of the form
     * @since 3.5.0
     * @access protected
     */
    protected function getUploadForm(& $form) {
        $form -> registerRule('checkParameter', 'callback', 'eF_checkParameter');                   //Register this rule for checking user input with our function, eF_checkParameter
        $form -> addElement('file', 'file_upload', null, 'class = "inputText"');
        $form -> addElement('hidden', 'upload_current_directory', null, 'id = "upload_current_directory" class = "inputText"');
        $form -> addElement('submit', 'submit_upload_file', _UPLOAD, 'class = "flatButton"');
        $form -> setMaxFileSize($this -> getUploadMaxSize() * 1024);            //getUploadMaxSize returns size in KB
        
        $renderer =& new HTML_QuickForm_Renderer_ArraySmarty($smarty);
        $form -> accept($renderer);
        $formArray = $renderer -> toArray();

        $formString = '
                    '.$formArray['javascript'].'
                    <form '.$formArray['attributes'].'>
                    '.$formArray['hidden'].'
                    <table width = "100%">
                        <tr><td class = "labelCell">'._UPLOADFILE.':&nbsp;</td>
                            <td class = "elementCell">'.$formArray['file_upload']['html'].'</td></tr>
                        <tr><td></td>
                            <td class = "elementCell">
                                '.$formArray['submit_upload_file']['html'].'
                            </td></tr>
                    </table>
                    </form>';
        
        return $formString;
    }

    /**
     * Get the create directory form
     * 
     * This function is responsible for creating the "create directory" 
     * form, as well as the equivalent HTML code. 
     *
     * @param HTML_QuickForm $form The form to populate 
     * @return string The HTML code of the form
     * @since 3.5.0
     * @access protected
     */
    protected function getCreateDirectoryForm(& $form) {
        $form -> registerRule('checkParameter', 'callback', 'eF_checkParameter');                   //Register this rule for checking user input with our function, eF_checkParameter
        $form -> addElement('text', 'create_directory', null, 'class = "inputText"');
        $form -> addElement('hidden', 'current_directory', null, 'id = "current_directory" class = "inputText"');
        $form -> addElement('submit', 'submit_create_directory', _CREATE, 'class = "flatButton"');
        
        $renderer =& new HTML_QuickForm_Renderer_ArraySmarty($smarty);
        $form -> accept($renderer);
        $formArray = $renderer -> toArray();

        $formString = '
                    '.$formArray['javascript'].'
                    <form '.$formArray['attributes'].'>
                    '.$formArray['hidden'].'
                    <table width = "100%">
                        <tr><td class = "labelCell">'._FOLDERNAME.':&nbsp;</td>
                            <td class = "elementCell">'.$formArray['create_directory']['html'].'</td></tr>
                        <tr><td></td>
                            <td class = "elementCell">
                                '.$formArray['submit_create_directory']['html'].'
                            </td></tr>
                    </table>
                    </form>';
        
        return $formString;
    }
        
    /**
     * Create HTML representation of file system tree
     *
     * This function creates the file manager HTML code. It also handles any AJAX calls,
     * composes and prints upload and create directory forms, as well as makes sure the 
     * correct folder contents are displayed.
     * <code>
     * $basedir    = G_LESSONSPATH.'test/';     
     * $filesystem = new FileSystemTree($basedir);                                  //Set the base directory that the file manager displayes
     * $url        = 'administrator.php?ctg=control_panel&op=file_manager';         //Set the url where file manager resides
     * echo $filesystem -> toHTML($url);                                            //Display file manager
     * </code>
     * 
     * @param string $url The url where the file manager resides
     * @param string $currentDirectory The directory to use as base directory
     * @param array $ajaxOptions AJAX-specific options: sort, order, limit, offset, filter  
     * @param array $options Options for the file manager: db_files_only, root_name, lessons_ID
     * @return string The HTML representation of the file system
     * @since 3.5.0
     * @access public
     */
    public function toHTML($url, $currentDirectory = '', $ajaxOptions, $options) {        
        if (!$treeId) {
            $treeId = 'dhtml_filesystem_tree';
        }

        $uploadForm         = new HTML_QuickForm("upload_file_form", "post", $url, "", "id = create_folder_form", true);
        $uploadFormString   = $this -> getUploadForm($uploadForm);
        if ($uploadForm -> isSubmitted() && $uploadForm -> validate()) {
            $curDir = EfrontDirectory :: normalize($uploadForm -> exportValue('upload_current_directory'));
            if (strpos($curDir, $this -> dir['file']) !== false) {               
                $uploadedFile = $this -> uploadFile('file_upload', $curDir);                
                if ($options['lessons_ID'] && eF_checkParameter($options['lessons_ID'], 'id')) {
                    $uploadedFile['lessons_ID'] = $options['lessons_ID'];
                    $uploadedFile -> persist();
                }
                if ($options['permissions'] && eF_checkParameter($options['permissions'], 'id')) {
                    $uploadedFile['access'] = $options['permissions'];
                    $uploadedFile -> persist();
                }
                $this -> reset();
            } 
        }
        $createFolderForm   = new HTML_QuickForm("create_folder_form", "post", $url, "", null, true);
        $createFolderString = $this -> getCreateDirectoryForm($createFolderForm);
        if ($createFolderForm -> isSubmitted() && $createFolderForm -> validate()) {
            $newDir = basename(EfrontDirectory :: normalize($createFolderForm -> exportValue('create_directory')));
            if ($createFolderForm -> exportValue('current_directory')) {
                $curDir = EfrontDirectory :: normalize($createFolderForm -> exportValue('current_directory'));
            } else {
                $curDir = $this -> dir['file'];
            }
            if (strpos($curDir, $this -> dir['file']) !== false) {               
                EfrontDirectory :: createDirectory($curDir.'/'.$newDir);
                $this -> reset();
            } 
        }
        
        if ($currentDirectory) {
            $currentDir      = $this -> seekNode($currentDirectory);
        } else {
            $currentDir      = $this -> tree;
        }        
        
        $iterator   = new EfrontDirectoryOnlyFilterIterator(new RecursiveIteratorIterator($this -> tree, RecursiveIteratorIterator :: SELF_FIRST));
        if ($options['db_files_only']) {                                    //Filter out directories without database representation
            $iterator = new EfrontDBOnlyFilterIterator($iterator);
        }
        $iterator   -> rewind();
        $current    = $iterator -> current();
        $depth      = $iterator -> getDepth();
        $treeString = '';
        $count      = 0;                                //Counts the total number of nodes, used to signify whether the tree has content
        while ($iterator -> valid()) {
            $iterator -> next();
            $fullName = $current['original_name'];                                                    //Full name will be needed for the link title, which is never truncated
            $dirName  = $current['original_name'];
            if (mb_strlen($current['original_name']) > 20) {
                $dirName = mb_substr($current['original_name'], 0, 20).'...&nbsp;&nbsp;&nbsp;';
            }

            $treeString  .= '
                <li style = "white-space:nowrap;" class = "" id = "'.$current['file'].'" noDrag = "true" noRename = "true" noDelete = "true">
                    <a class = "treeLink" href = "javascript:void(0)" onclick = "$(\''.$treeId.'\').select(\'a\').each(function (s) {s.removeClassName(\'drag_tree_current\')});this.addClassName(\'drag_tree_current\');eF_js_sortTable(false, \''.urlencode($current['file']).'\')" title = "'.$fullName.'">&nbsp;'.$dirName.'</a>
                    <a href = "javascript:void(0)" style = "display:none"><img src = "images/16x16/edit.png" alt = "'._EDIT.'" title = "'._EDIT.'" border = "0" style = "vertical-align:middle"></a>
                    <a href = "javascript:void(0)"><img src = "images/16x16/delete.png" alt = "'._DELETE.'" title = "'._DELETE.'" border = "0" onclick = "deleteFolder(this, \''.$current['id'].'\')" style = "vertical-align:middle"></a>';
            
            $iterator -> getDepth() > $depth ? $treeString .= '<ul>' : $treeString .= '</li>';
            for ($i = $depth; $i > $iterator -> getDepth(); $i--) {
                $treeString .= '</ul></li>';
            }
            $current = $iterator -> current();
            $depth   = $iterator -> getDepth();
            $count++;
        }
        
        $directoriesCode .= '
            <script>
                var expandTree = false;                     //Set expandTree to true in any position, to make the tree be expanded by default  
                function updateExpandCollapseLink(status) {
                    status ? $("expand_collapse_link").update("'._COLLAPSEALL.'") : $("expand_collapse_link").update("'._EXPANDALL.'");
                }
            </script>
            <table style = "width:100%">
                <tr><td class = "topTitle">'._DIRECTORIES.'</td></tr>
                <tr><td>
                        <div style = "display:none" id = "expand_collapse_div" '.(isset($expand) ? 'expand = "'.$expand.'"' : null).'>
                            <b><a id = "expand_collapse_link" href = "javascript:void(0)" onclick = "treeObj.setTreeId(\''.$treeId.'\');if (treeObj.status) {treeObj.collapseAll();this.innerHTML = \''._EXPANDALL.'\';} else {treeObj.expandAll();this.innerHTML = \''._COLLAPSEALL.'\';}">'._COLLAPSEALL.'</a></b><br/>
                        </div>
                    </td></tr>
                <tr><td>
                    <ul id = "'.$treeId.'" class = "dhtmlgoodies_tree" selectedNode = "'.$currentDirectory.'">
                        <li style = "white-space:nowrap;" class = "" id = "'.$this -> dir['file'].'"  noDrag = "true" noRename = "true" noDelete = "true">
                            <a class = "treeLink drag_tree_current" href = "javascript:void(0)" onclick = "$(\''.$treeId.'\').select(\'a\').each(function (s) {s.removeClassName(\'drag_tree_current\')});this.addClassName(\'drag_tree_current\');eF_js_sortTable(false, \'\')" title = "'.($options['root_name'] ? $options['root_name'] : $this -> dir['original_name']).'">&nbsp;'.($options['root_name'] ? $options['root_name'] : $this -> dir['original_name']).'</a>&nbsp;
                            <ul>'.$treeString.'</ul></li>
                    </ul>
                </td></tr>
            </table>';
        
        $files     = array();
        
        $iterator  = new EfrontFileOnlyFilterIterator(new ArrayIterator($currentDir, RecursiveIteratorIterator :: SELF_FIRST));    //Plain ArrayIterator so that it iterates only on the current's folder files
        if ($options['db_files_only']) {                                    //Filter out directories without database representation
            $iterator = new EfrontDBOnlyFilterIterator($iterator);
        }
        foreach ($iterator as $key => $value) {                    //We convert iterator to a complete array if files, so we can apply sorting, filtering etc more easily
            $fileArrays[]  = (array)$iterator -> current();         //Array representation of file objects, on which we can apply sorting, filtering, etc
        }

        isset($ajaxOptions['order']) && $ajaxOptions['order'] == 'asc' ? $ajaxOptions['order'] = 'asc' : $ajaxOptions['order'] = 'desc';
        !isset($ajaxOptions['sort'])   ? $ajaxOptions['sort']   = 'name' : null;
        !isset($ajaxOptions['limit'])  ? $ajaxOptions['limit']  = 20     : null;
        !isset($ajaxOptions['offset']) ? $ajaxOptions['offset'] = 0      : null;
        !isset($ajaxOptions['filter']) ? $ajaxOptions['filter'] = ''     : null;

        $size       = sizeof($fileArrays);
        $fileArrays = eF_multiSort($fileArrays, $ajaxOptions['sort'], $ajaxOptions['order']);
        $ajaxOptions['filter'] ? $fileArrays = eF_filterData($fileArrays, $ajaxOptions['filter']) : null;
        $fileArrays = array_slice($fileArrays, $ajaxOptions['offset'], $ajaxOptions['limit']);

        $filesCode = '
                        <table class = "sortedTable" style = "width:100%" size = "'.$size.'" sortBy = "1" id = "filesTable" useAjax = "1" rowsPerPage = "20" other = "'.urlencode($currentDirectory).'" url = "'.$url.'&">
                            <tr><td class = "topTitle" name = "extension">'._TYPE.'</td>
                                <td class = "topTitle" name = "original_name" id = "original_name">'._FILENAME.'</td>
                                <td class = "topTitle" name = "size">'._SIZE.'</td>
                                <td class = "topTitle" name = "timestamp">'._LASTMODIFIED.'</td>
                                <td class = "topTitle centerAlign">'._OPERATIONS.'</td>
                                <td class = "topTitle centerAlign">'._SELECT.'</td></tr>';
        foreach ($fileArrays as $key => $value) {
            $value       = new EfrontFile($value);                        //Restore file representation, so we can use its methods
            $toolsString = '';
            if ($value['extension'] == 'zip') {
                $toolsString .= '<a href = "javascript:void(0)"><img src = "images/16x16/box.png" alt = "'._UNCOMPRESS.'" title = "'._UNCOMPRESS.'" border = "0" onclick = "uncompressFile(this, \''.$value['id'].'\')"  /></a>&nbsp;';
            }
            if ($value['id'] != -1) {
                $toolsString .= '<a href = "'.$url.'&download='.$value['id'].'"><img src = "images/16x16/import2.png" alt = "'._DOWNLOADFILE.'" title = "'._DOWNLOADFILE.'" border = "0"/></a>';
                if ($value['lessons_ID']) {
                    $toolsString .= $value['shared'] ? '<a href = "javascript:void(0)"><img src = "images/16x16/folder_refresh.png" alt = "'._UNSHARE.'" title = "'._UNSHARE.'" onclick = "unshareFile(this, \''.$value['id'].'\')" border = "0"/></a>&nbsp;' : '<a href = "javascript:void(0)"><img src = "images/16x16/folder_forbidden.png" alt = "'._SHARE.'" title = "'._SHARE.'" onclick = "shareFile(this, \''.$value['id'].'\')" border = "0"/></a>&nbsp;';
                }
                //$toolsString .= '<a href = "'.$url.'&file_info='.$value['id'].'"><img src = "images/16x16/about.png" alt = "'._FILEINFORMATION.'" title = "'._FILEINFORMATION.'" border = "0"/></a>';
            }
            $filesCode .= '
                            <tr class = "defaultRowHeight '.(fmod($i++, 2) ? 'oddRowColor' : 'evenRowColor').'">
                                <td><span style = "display:none">'.$value['extension'].'</span><img src = "'.$value -> getTypeImage().'" alt = "'.$value['mime_type'].'" title = "'.$value['mime_type'].'"/></td>
                                <td><a href = "'.$url.'&view='.$value['id'].'" onclick = "eF_js_showDivPopup(\''._PREVIEW.'\', 2, \'preview_table\')" target = "PREVIEW_FRAME">'.$value['original_name'].'</a></td>
                                <td>'.$value['size'].' '._KB.'</td>
                                <td>'.formatTimestamp($value['timestamp']).'</td>
                                <td class = "centerAlign">
                                    '.$toolsString.'
                                    <a href = "javascript:void(0)"><img src = "images/16x16/delete.png" alt = "'._DELETE.'" title = "'._DELETE.'" onclick = "if (confirm(\''._IRREVERSIBLEACTIONAREYOUSURE.'\')) {deleteFile(this, \''.$value['id'].'\')}" border = "0"/></a>
                                </td>
                                <td class = "centerAlign"><input type = "checkbox" id = "'.$value['id'].'" value = "'.$value['id'].'" /></td></tr>';
        }
        if ($size) {
            $filesCode .= '
                        </table>';
            $massOperationsCode = '
                        <div class = "horizontalSeparatorAbove">
                            <span  style = "vertical-align:middle">'._WITHSELECTEDFILES.':</span> 
                            <a href = "javascript:void(0)"><img src = "images/16x16/delete.png" title = "'._DELETESELECTED.'" alt = "'._DELETESELECTED.'" border = "0" style = "vertical-align:middle" onclick = "if (confirm(\''._IRREVERSIBLEACTIONAREYOUSURE.'\')) deleteSelected()"></a>
                            <a href = "javascript:void(0)"><img src = "images/16x16/folder_forbidden.png" title = "'._UNSHARESELECTED.'" alt = "'._UNSHARESELECTED.'" border = "0" style = "vertical-align:middle" onclick = "unshareSelected()"></a>
                            <a href = "javascript:void(0)"><img src = "images/16x16/folder_refresh.png" title = "'._SHARESELECTED.'" alt = "'._SHARESELECTED.'" border = "0" style = "vertical-align:middle" onclick = "shareSelected()"></a>
                        </div>';
        } else {
            $filesCode .= '
                            <tr class = "oddRowColor defaultRowHeight"><td colspan = "100%" class = "emptyCategory centerAlign">'._NODATAFOUND.'</td></tr>
                        </table>';            
        }
         
        $str = '
            <table>
                <tr><td style = "border-right:1px solid black">
                        <img src = "images/16x16/add2.png" alt = "'._UPLOADFILE.'" title = "'._UPLOADFILE.'" style = "vertical-align:middle">&nbsp;
                        <a href = "javascript:void(0)" onclick = "$(\''.$treeId.'\').select(\'a\').each(function (s) {if (s.hasClassName(\'drag_tree_current\')) {$(\'upload_current_directory\').value = s.up().id;} });eF_js_showDivPopup(\''._UPLOADFILE.'\', 0, \'upload_file_table\')" style = "vertical-align:middle">'._UPLOADFILE.'</a>&nbsp;
                    </td><td>
                        &nbsp;<img src = "images/16x16/folder_add.png" alt = "'._CREATEFOLDER.'" title = "'._CREATEFOLDER.'" style = "vertical-align:middle">&nbsp;
                        <a href = "javascript:void(0)" onclick = "$(\''.$treeId.'\').select(\'a\').each(function (s) {if (s.hasClassName(\'drag_tree_current\')) {$(\'current_directory\').value = s.up().id;} });eF_js_showDivPopup(\''._CREATEFOLDER.'\', 0, \'create_directory_table\')" style = "vertical-align:middle">'._CREATEFOLDER.'</a>
                    <td>
                    </td></tr>
            </table>
            <br/>
            <table style = "width:100%">
                <tr><td id = "'.$treeId.'_treeCell" style = "vertical-align:top;width:150px;" class = "verticalSeparator">
                        '.$directoriesCode.'
                    </td>
                    <td id = "'.$treeId.'_filesCell" style = "vertical-align:top;">
<!--ajax:filesTable-->
                        '.$filesCode.'
<!--/ajax:filesTable-->
                        '.$massOperationsCode.'                 
                    </td></tr>
            </table>
            <script>
                function deleteSelected() {
                    $("filesTable").select("input[type=checkbox]").each(function (s) {
                                                                            if (s.checked && s.id) {
                                                                                s.up().previous().select("img").each (function (p) {if (p.src.match(/delete/)) {deleteFile(p, s.value);}});
                                                                            }
                                                                        });
                }
                function shareSelected() {
                    $("filesTable").select("input[type=checkbox]").each(function (s) {
                                                                            if (s.checked && s.id) {
                                                                                s.up().previous().select("img").each (function (p) {if (p.src.match(/forbidden/)) {shareFile(p, s.value);}});
                                                                            }
                                                                        });
                }
                function unshareSelected() {
                    $("filesTable").select("input[type=checkbox]").each(function (s) {
                                                                            if (s.checked && s.id) {
                                                                                s.up().previous().select("img").each (function (p) {if (p.src.match(/folder_refresh/)) {unshareFile(p, s.value);}});
                                                                            }
                                                                        });
                }                               
                function deleteFile(img, fileId) {
                    Element.extend(img);
                    img.src = "images/others/progress1.gif";
                    url     = "'.$url.'&delete="+fileId;
                    new Ajax.Request(url, {
                        method:\'get\',
                        asynchronous:true,
                        onFailure: function (transport) {
                            img.src   = "images/16x16/delete.png";
                            failImage = new Element("img", {src:"images/16x16/delete2.png", title:transport.statusText}).hide();
                            img.up().up().insert(failImage);
                            new Effect.Appear(failImage.identify());
                            window.setTimeout(\'Effect.Fade("\'+failImage.identify()+\'")\', 10000);
                        },
                        onSuccess: function (transport) {
                            new Effect.Fade(img.up().up().up());
                            }
                        });             
                }
                function shareFile(img, fileId) {
                    Element.extend(img);
                    img.src = "images/others/progress1.gif";
                    url     = "'.$url.'&share="+fileId;
                    new Ajax.Request(url, {
                        method:\'get\',
                        asynchronous:true,
                        onFailure: function (transport) {
                            img.src   = "images/16x16/folder_forbidden.png";
                            failImage = new Element("img", {src:"images/16x16/delete2.png", title:transport.statusText}).hide();
                            img.up().up().insert(failImage);
                            new Effect.Appear(failImage.identify());
                            window.setTimeout(\'Effect.Fade("\'+failImage.identify()+\'")\', 10000);
                        },
                        onSuccess: function (transport) {
                            img.hide();
                            img.src   = "images/16x16/folder_refresh.png";
                            new Effect.Appear(img);
                            }
                        });             
                }               
                function unshareFile(img, fileId) {
                    Element.extend(img);
                    img.src = "images/others/progress1.gif";
                    url     = "'.$url.'&unshare="+fileId;
                    new Ajax.Request(url, {
                        method:\'get\',
                        asynchronous:true,
                        onFailure: function (transport) {
                            img.src   = "images/16x16/folder_refresh.png";
                            failImage = new Element("img", {src:"images/16x16/delete2.png", title:transport.statusText}).hide();
                            img.up().up().insert(failImage);
                            new Effect.Appear(failImage.identify());
                            window.setTimeout(\'Effect.Fade("\'+failImage.identify()+\'")\', 10000);
                        },
                        onSuccess: function (transport) {
                            img.hide();
                            img.src   = "images/16x16/folder_forbidden.png";
                            new Effect.Appear(img);
                            }
                        });             
                }
                function uncompressFile(img, fileId) {
                    Element.extend(img);
                    img.src = "images/others/progress1.gif";
                    url     = "'.$url.'&uncompress="+fileId;
                    new Ajax.Request(url, {
                        method:\'get\',
                        asynchronous:true,
                        onFailure: function (transport) {
                            img.src   = "images/16x16/box.png";
                            failImage = new Element("img", {src:"images/16x16/delete2.png", title:transport.statusText}).hide();
                            img.up().up().insert(failImage);
                            new Effect.Appear(failImage.identify());
                            window.setTimeout(\'Effect.Fade("\'+failImage.identify()+\'")\', 10000);
                        },
                        onSuccess: function (transport) {
                            img.hide();
                            img.src   = "images/16x16/box.png";
                            new Effect.Appear(img);
                            eF_js_sortTable($("original_name").down(), \''.urlencode($current['file']).'\');
                            }
                        });         
                }
                function deleteFolder(img, directoryId) {
                    Element.extend(img);
                    img.src = "images/others/progress1.gif";
                    url     = "'.$url.'&delete_folder="+directoryId;
                    new Ajax.Request(url, {
                        method:\'get\',
                        asynchronous:true,
                        onFailure: function (transport) {
                            img.src   = "images/16x16/delete.png";
                            failImage = new Element("img", {src:"images/16x16/delete2.png", title:transport.statusText}).hide().setStyle({verticalAlign:"middle"});
                            img.up().up().insert(failImage);
                            new Effect.Appear(failImage.identify());
                            window.setTimeout(\'Effect.Fade("\'+failImage.identify()+\'")\', 10000);
                        },
                        onSuccess: function (transport) {
                            new Effect.Fade(img.up());
                            img.up().up().remove();
                            eF_js_sortTable($("original_name").down(), \'\');
                            }
                        });             
                }
            </script>
            <div id = "upload_file_table"      style = "display:none;text-align:center;width:100%">'.$uploadFormString.'</div>
            <div id = "create_directory_table" style = "display:none;text-align:center;width:100%">'.$createFolderString.'</div>
            <div id = "preview_table" style = "display:none">
                <iframe name = "PREVIEW_FRAME" style = "border-width:0px;width:100%;height:100%;padding:0px 0px 0px 0px"></iframe>
            </div>';

        
        return $str;
    }

    /**
     * Handle AJAX actions
     * 
     * This function is used to perform the necessary ajax actions,
     * that may be fired by the file manager
     * <br/>Example:
     * <code>
     * $basedir    = $currentLesson -> getDirectory();
     * $filesystem = new FileSystemTree($basedir);
     * $filesystem -> handleAjaxActions();
     * </code>
     *
     * @param EfrontUser $currentUser The current user
     * @since 3.5.0
     * @access public
     */
    public function handleAjaxActions($currentUser) {
        if (isset($_GET['delete']) && (eF_checkParameter($_GET['delete'], 'id') || strpos($_GET['delete'], $this -> dir['file']) !== false)) {
            try {
                $file = new EfrontFile($_GET['delete']);
                if ($file['users_LOGIN'] == $currentUser -> user['login']) {
                    $file -> delete();
                } else {
                    throw new EfrontFileException(_YOUDONTHAVEPERMISSION.': '.$file['file'], EfrontFileException :: UNAUTHORIZED_ACTION);
                }
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;
        } else if (isset($_GET['share']) && (eF_checkParameter($_GET['share'], 'id') || strpos($_GET['share'], $this -> dir['file']) !== false)) {
            try {
                $file = new EfrontFile($_GET['share']);
                if ($file['users_LOGIN'] == $currentUser -> user['login']) {
                    $file -> share();
                } else {
                    throw new EfrontFileException(_YOUDONTHAVEPERMISSION.': '.$file['file'], EfrontFileException :: UNAUTHORIZED_ACTION);
                }
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;
        } else if (isset($_GET['unshare']) && (eF_checkParameter($_GET['unshare'], 'id') || strpos($_GET['unshare'], $this -> dir['file']) !== false)) {
            try {
                $file = new EfrontFile($_GET['unshare']);
                if ($file['users_LOGIN'] == $currentUser -> user['login']) {
                    $file -> unshare();
                } else {
                    throw new EfrontFileException(_YOUDONTHAVEPERMISSION.': '.$file['file'], EfrontFileException :: UNAUTHORIZED_ACTION);
                }
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;
        } else if (isset($_GET['uncompress']) && (eF_checkParameter($_GET['uncompress'], 'id') || strpos($_GET['uncompress'], $this -> dir['file']) !== false)) {
            try {
                $file = new EfrontFile($_GET['uncompress']);
                $file -> uncompress();
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;
        } elseif (isset($_GET['delete_folder']) && (eF_checkParameter($_GET['delete_folder'], 'id') || strpos($_GET['delete_folder'], $this -> dir['file']) !== false)) {
            try {
                $directory = new EfrontDirectory($_GET['delete_folder']);
                if ($directory['users_LOGIN'] == $currentUser -> user['login']) {
                    $this -> deleteDirectory($directory);
                } else {
                    throw new EfrontFileException(_YOUDONTHAVEPERMISSION.': '.$directory['file'], EfrontFileException :: UNAUTHORIZED_ACTION);
                }
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;   
        } elseif (isset($_GET['download']) && (eF_checkParameter($_GET['download'], 'id') || strpos($_GET['download'], $this -> dir['file']) !== false)) {
            try {
                $file = new EfrontFile($_GET['download']);
                if ($file['users_LOGIN'] == $currentUser -> user['login'] || $file['access'] != 0) {
                    header("content-type:".$file['mime_type']);
                    header('content-disposition: attachment; filename= "'.urlencode($file['original_name']).'"');
                    readfile($file['file']);
                } else {
                    throw new EfrontFileException(_YOUDONTHAVEPERMISSION.': '.$file['file'], EfrontFileException :: UNAUTHORIZED_ACTION);
                }
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;   
        } elseif (isset($_GET['view']) && (eF_checkParameter($_GET['view'], 'id') || strpos($_GET['view'], $this -> dir['file']) !== false)) {
            try {
                $file = new EfrontFile($_GET['view']);
                if ($file['users_LOGIN'] == $currentUser -> user['login'] || $file['access'] != 0) {
                    header("content-type:".$file['mime_type']);
                    header('content-disposition: inline; filename= "'.urlencode($file['original_name']).'"');
                    readfile($file['file']);
                } else {
                    throw new EfrontFileException(_YOUDONTHAVEPERMISSION.': '.$file['file'], EfrontFileException :: UNAUTHORIZED_ACTION);
                }
            } catch (Exception $e) {
                header("HTTP/1.0 500 ".$e -> getMessage().' ('.$e -> getCode().')');
            }            
            exit;   
        }                      
    }
    
    /**
     * Handle uploaded file
     *
     * This function is used to handle an uploaded file. Given the name of the form field
     * that was used to upload the file, as well as the destination directory, the function
     * creates the corresponding database entry and moves the file to the designated position,
     * using the appropriate name.
     * <br/>Example:
     * <code>
     * $directory = new eF_Directory('some_dir');                    //the directory to upload the file to.
     * try {
     *   $uploadedFile = EfrontFileSystem :: uploadFile('file_upload', $directory -> getFullPath());
     * } catch (EfrontFileException $e) {
     *   echo $e -> getMessage();
     * }
     * </code>
     *
     * @param string $fieldName The form file field name
     * @param mixed $destinationDirectory The destination for the uploaded file, either a string or an EfrontDirectory object
     * @param string $offset If the field name is on the form file[x] (array-like), then specifying specific offset (x) allows for handling of it
     * @return object An object of eF_File class, corresponding to the newly uploaded file.
     * @access public
     * @since 3.0
     * @static
     */
    public function uploadFile($fieldName, $destinationDirectory, $offset = false) {
        if (!($destinationDirectory instanceof EfrontDirectory)) {
            $destinationDirectory = new EfrontDirectory($destinationDirectory);
        }
        if (strpos($destinationDirectory['file'], $this -> dir['file']) === false) {
            throw new EfrontFileException(_ILLEGALPATH.': '.$destinationDirectory['file'], EfrontFileException :: ILLEGAL_PATH);
        } else {
            if ($offset !== false) {
                $error    = $_FILES[$fieldName]['error'][$offset];
                $size     = $_FILES[$fieldName]['size'][$offset];
                $name     = $_FILES[$fieldName]['name'][$offset];
                $tmp_name = $_FILES[$fieldName]['tmp_name'][$offset];
            } else {
                $error    = $_FILES[$fieldName]['error'];
                $size     = $_FILES[$fieldName]['size'];
                $name     = $_FILES[$fieldName]['name'];
                $tmp_name = $_FILES[$fieldName]['tmp_name'];
            }

            if ($error) {
                switch ($error) {
                    case UPLOAD_ERR_INI_SIZE :
                        throw new EfrontFileException(_THEFILE." "._MUSTBESMALLERTHAN." ".ini_get('upload_max_filesize'), UPLOAD_ERR_INI_SIZE );
                        break;
                    case UPLOAD_ERR_FORM_SIZE :
                        throw new EfrontFileException(_THEFILE." "._MUSTBESMALLERTHAN." ".sprintf("%.0f", $_POST['MAX_FILE_SIZE']/1024)." "._KILOBYTES, UPLOAD_ERR_FORM_SIZE);
                        break;
                    case UPLOAD_ERR_PARTIAL :
                        throw new EfrontFileException(_FILEWASPARTIALLYUPLOADED, UPLOAD_ERR_PARTIAL);
                        break;
                    case UPLOAD_ERR_NO_FILE :
                        throw new EfrontFileException(_NOFILEUPLOADED, UPLOAD_ERR_NO_FILE);
                        break;
                    case UPLOAD_ERR_NO_TMP_DIR :
                        throw new EfrontFileException(_NOTMPDIR, UPLOAD_ERR_NO_TMP_DIR);
                        break;
                    case UPLOAD_ERR_CANT_WRITE :
                        throw new EfrontFileException(_UPLOADCANTWRITE, UPLOAD_ERR_CANT_WRITE);
                        break;
                    case UPLOAD_ERR_EXTENSION :
                        throw new EfrontFileException(_UPLOADERREXTENSION, UPLOAD_ERR_EXTENSION);
                        break;
                    default:
                        throw new EfrontFileException(_ERRORUPLOADINGFILE, EfrontFileException :: UNKNOWN_ERROR);
                        break;
                }
            } elseif ($size == 0) {
                throw new EfrontFileException(_FILEDOESNOTEXIST, EfrontFileException :: FILE_NOT_EXIST);
            } elseif (!EfrontFileSystem :: checkFile($name)) {
                throw new EfrontFileException(_ILLEGALFILENAME, EfrontFileException :: ILLEGAL_FILE_NAME);
            } else {
                $id = eF_insertTableData("files", array('file' => 'temp'));                        //Insert bogus entry
                strtolower(mb_detect_encoding($name)) != 'ascii' ? $newName = $id : $newName = $name;
                move_uploaded_file($tmp_name, $destinationDirectory['file'].'/'.$newName);
                $fields = array('file'          => $destinationDirectory['file'].'/'.$newName,
                                'type'          => 'file',
                                'physical_name' => $newName,
                                'original_name' => $name,
                                'directory'     => $destinationDirectory['file'],
                                'users_LOGIN'   => $_SESSION['s_login'],
                                'timestamp'     => time());
                eF_updateTableData("files", $fields, "id=$id");
                return new EfrontFile($id);
            }

        } 
    }
    
    /**
     * Get maximum upload size
     *
     * This function is used to calculate the maximum alloweded upload
     * file size (in KB). The size is the smallest among the following:
     * - The 'memory_limit' PHP ini setting
     * - The 'upload_max_filesize' PHP ini setting
     * - The 'post_max_size' PHP ini setting
     * - The maximum file size configuration setting
     * <br/>Example:
     * <code>
     * echo FileSystemTree :: getUploadMaxSize();   //returns something like 10000, which is 10000KB 
     * </code>
     *
     * @return int The maximum file size, in Kilobytes
     * @see FileSystemTree :: uploadFile()
     * @since 3.0
     * @access public
     * @static
     */
    public static function getUploadMaxSize() {
        preg_match('/(\d+)/', ini_get('memory_limit'), $memory_limit);
        preg_match('/(\d+)/', ini_get('upload_max_filesize'), $upload_max_filesize);
        preg_match('/(\d+)/', ini_get('post_max_size'), $post_max_size);

        $max_upload = min($memory_limit[1] * 1024, $upload_max_filesize[1] * 1024, $post_max_size[1] * 1024, $GLOBALS['configuration']['max_file_size']);

        return $max_upload;
    }    
    
    /**
     * Get specific file types
     *
     * This function can be used to return extensions and mime types of specific file
     * classes, such as images or media.
     * <br/>Example:
     * <code>
     * $imageMimeTypes = FileSystemTree :: getFileTypes('image');   //$imageMimeTypes now contains the arrays 'jpg' => 'image/png', 'png' => 'image/png', etc
     * </code>
     *
     * @param mixed $type The file classes: 'image', 'media', 'java'. If none is specified, then all file types available are returned
     * @return array The file types in extension => mime type pairs
     * @see EfrontFile :: $mimeTypes
     * @since 3.0
     * @access public
     * @static
     */
    public static function getFileTypes($type = false) {
        $fileTypes = array();
        foreach (self :: $mimeTypes as $key => $filetype) {
            switch ($type) {
                case 'image':
                    if (strpos($filetype, 'image/') === 0) {
                        $fileTypes[$key] = $filetype;
                    }
                    break;
                case 'media':
                    if (strpos($filetype, 'audio/') === 0 || strpos($filetype, 'video/') === 0 || $key == 'swf') {
                        $fileTypes[$key] = $filetype;
                    }
                    break;
                case 'java':
                    if ($key == 'class') {
                        $fileTypes[$key] = $filetype;
                    }
                    break;
                default:
                    $fileTypes[$key] = $filetype;
                    break;
            }
        }
        return $fileTypes;
    }
       
    /**
     * Import files to filesystem
     *
     * This function imports the specified files (in $list array) to the filesystem,
     * by creating a corresponding database representation. The $list
     * array should contain full paths to the files. The function returns an array
     * of the same size and contents as $list , but this time the file ids being the keys
     * <br/>Example:
     * <code>
     * $list = array('/var/www/text.txt', '/var/www/user.txt');
     * $newList = EfrontFileSystem :: importFiles($list);
     * </code>
     *
     * @param array $list The files list
     * @param array $options extra options to set for the files, such as whether they are part of a lesson, or the proper permissions
     * @return array An array with the new file ids
     * @access public
     * @since 3.0
     * @static
     */
    public static function importFiles($list, $options = array()) {
        if (!is_array($list)) {
            $list = array($list);
        }

        $allFiles = eF_getTableDataFlat("files", "file");                       //Get all files, so that if a file already exists, a duplicate entry in the database won't be created
        for ($i = 0; $i < sizeof($list); $i++) {
            if (!in_array($list[$i], $allFiles['file']) && strpos(dirname($list[$i]), G_ROOTPATH) !== false) {
                $fields = array('file'          => $list[$i],
                                'physical_name' => basename($list[$i]),
                                'original_name' => basename($list[$i]),
                                'directory'     => EfrontDirectory :: normalize(dirname($list[$i])),
                                'type'          => is_dir($list[$i]) ? 'directory' : 'file',
                                'users_LOGIN'   => isset($_SESSION['s_login']) ? $_SESSION['s_login'] : '',
                                'timestamp'     => time());
                isset($options['lessons_ID']) ? $fields['lessons_ID'] = $options['lessons_ID'] : null;
                isset($options['access'])     ? $fields['access']     = $options['access']     : null;
                
                $fileId = eF_insertTableData("files", $fields);
                if ($fileId) {
                    $newList[$fileId] = $list[$i];
/*                    
                    if ($rename) {
                        $newName = $fields['directory'].$fileId;
                    }
                    if (rename($list[$i], $newName)) {
                        eF_updateTableData("files", array("file" => $newName, "physical_name" => basename($newName)), "id=$fileId");
                    }
*/
                }
            }
        }

        return $newList;
    }

    
    
}

class FileSystemTreeUnitTest
{
    public static function test() {
        /**
         * __construct(): ok
        */
        $dir = '';
        $tree = new FileSystemTree(G_LESSONSPATH.'54');
        
        /**
         * deleteDirectory(): ok
         */
        //$tree -> deleteDirectory(G_LESSONSPATH.'54/test folder');
        
        /**
         * copyDirectory(): ok
         */
        //$tree -> copyDirectory(G_LESSONSPATH.'54/test folder', G_LESSONSPATH.'54/aaaaa', false);
        
        //moveDirectory
        $tree -> renameDirectory(G_LESSONSPATH.'54/test folder', 'new name');
    }
}

/**
 * Return directories only
 *
 */
class EfrontDirectoryOnlyFilterIterator extends FilterIterator
{
    /**
     * Accept method
     *
     * The accept method returns true only if the current element
     * is a directory
     *
     * @return boolean True if the current element is a directory
     * @since 3.5.0
     * @access public
     */
    function accept() {
        return is_dir($this -> key());
    }
}
/**
 * Return files only
 *
 */
class EfrontFileOnlyFilterIterator extends FilterIterator
{
    /**
     * Accept method
     *
     * The accept method returns true only if the current element
     * is a file
     *
     * @return boolean True if the current element is a file
     * @since 3.5.0
     * @access public
     */
    function accept() {
        return is_file($this -> key());
    }
}

class EfrontDBOnlyFilterIterator extends FilterIterator
{
    /**
     * Accept method
     *
     * The accept method returns true only if the current element
     * has a db representation (equivalent to having an id different than -1)
     *
     * @return boolean True if the current element has a DB representation
     * @since 3.5.0
     * @access public
     */
    function accept() {
        if ($this -> current() -> offsetGet('id') != -1) {
            return true;
        }
    }
}
/**
 * Filter files based on a Regular Expression
 *
 */
class EfrontREFilterIterator extends FilterIterator
{
    /**
     * The Regular Expression to use to filter files
     *
     * @var string
     * @since 3.5.0
     * @access public
     */
    public $re;

    /**
     * Whether to include the filtered files (true) or exclude them (false)
     *
     * @var boolean
     * @since 3.5.0
     * @acess public
     */
    public $mode;

    /**
     * Class constructor
     *
     * The class constructor calls the FilterItearator constructor and assigns
     * the $re and $mode parameters
     *
     * @param ArrayIterator $it The iterator
     * @param string $re The regular expression to use
     * @param boolean $mode Whether to include or exclude the filtered files from the data set
     */
    function __construct($it, $re, $mode = true) {
        parent :: __construct($it);
        is_array($re) ? $this -> re = $re : $this -> re = array($re);
        $this -> mode = $mode;
    }

    /**
     * Accept method
     *
     * The accept method filters in or out (based on $mode value) the files
     * from the data set
     *
     * @return boolean True if the current element is a file
     * @since 3.5.0
     * @access public
     */
    function accept() {
        $result = array();
        foreach ($this -> re as $regExp) {
            $this -> mode ? $result[] = preg_match($regExp, $this -> key()) : $result[] = !preg_match($regExp, $this -> key());
        }
        return array_sum($result);
    }
}






?>