<?php

/**
 *
 */
abstract class EfrontHcdUser
{
    /**
     * The employee login, defined at constructor.
     *
     */
    public $login;


    /**
     * Create new hcd-user: employee, supervisor, administrator
     *
     * This function is used to create a new hcd - user in the system
     * The user is created based on a a properties array, in which
     * the user login must be present, otherwise
     * an EfrontUserException is thrown. Apart from these, all the other
     * user elements are optional, and defaults will be used if they are left
     * blank.
     * Once the database representation is created, the constructor tries to create the
     * user directories, G_UPLOADPATH.'login/module_hcd/' .
     * The function instantiates the user based on its type.
     * <br/>Example:
     * <code>
     * $properties = array('login' => 'jdoe', 'name' => 'john', 'surname' => 'doe', 'email' => 'jdoe@example.com');
     * $employee   = EfrontHcdUser :: createUser($properties);
     * </code>
     *
     * @param array $employeeProperties The new user properties
     * @return EfrontHcdUser The new employee
     * @since 3.5.0
     * @access public
     */
    public static function createUser($employeeProperties) {

        // TODO: put the rest of the controls for the validity of the input data here
        if (!isset($employeeProperties['users_login']) || !eF_checkParameter($employeeProperties['users_login'], 'login')) {
            throw new EfrontUserException(_INVALIDLOGIN.': '.$employeeProperties['users_login'], EfrontUserException :: INVALID_LOGIN);
        }

        if (eF_insertTableData("module_hcd_employees", $employeeProperties)) {

            $login = $employeeProperties['users_login'];

            global $currentUser;

            /*** Event logging ***/
            // Register new and hired on events into the event log
            eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['NEW'],
                                                          "users_login"   => $login,
                                                          "author"        => $currentUser -> login,
                                                          "specification" => _INSERTEDINTOTHESYSTEM,
                                                          "timestamp"     => time()));

            $hired_date = $employeeProperties['hired_on'];
            if ( $hired_date  )  {
                if (strpos($hired_date, "/")) {
                   $result = explode("/",$hired_date);
                } else {
                   $result = explode("-",$hired_date);
                }

                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['HIRED'],
                                                              "users_login"   => $login,
                                                              "author"        => $currentUser -> login,
                                                              "specification" => _HIRED,
                                                              "timestamp"     => mktime(0,0,0,$result[1],$result[0],$result[2])));
            }

            $left_date = $employeeProperties['left_on'];
            if ( $left_date  )  {
                if (strpos($left_date, "/")) {
                   $result = explode("/",$left_date);
                } else {
                   $result = explode("-",$left_date);
                }

                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['LEFT'],
                                                              "users_login"   => $login,
                                                              "author"        => $currentUser -> login,
                                                              "specification" => _HASLEFT,
                                                              "timestamp"     => mktime(0,0,0,$result[1],$result[0],$result[2])));
            }

            $wage = $employeeProperties['wage'];
            if ( $wage ) {
                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['WAGE_CHANGE'],
                                                              "users_login"   => $login,
                                                              "author"        => $currentUser -> login,
                                                              "specification" => _EMPLOYEESNEWWAGEIS . " " .$wage,
                                                              "timestamp"     => time()));
            }

            // Create folders for uploading user files
            if (!is_dir(G_UPLOADPATH.$login.'/module_hcd/')) {
                  mkdir(G_UPLOADPATH.$login.'/module_hcd/',0755);
            }
            if (!is_dir(G_UPLOADPATH.$login.'/module_hcd/private/')) {
                  mkdir(G_UPLOADPATH.$login.'/module_hcd/private/',0755);
            }
            if (!is_dir(G_UPLOADPATH.$login.'/module_hcd/public/')) {
                  mkdir(G_UPLOADPATH.$login.'/module_hcd/public/',0755);
            }
            if (!is_dir(G_UPLOADPATH.'/supervisors/')) {
                  mkdir(G_UPLOADPATH.'/supervisors/',0755);
            }
            if (!is_dir(G_UPLOADPATH.'/supervisors/'.$login)) {
                  mkdir(G_UPLOADPATH.'/supervisors/'.$login,0755);
            }

        } else {
            throw new EfrontUserException(_COULDNOTINSERTUSER.': '.$userProperties['login'], EfrontUserException :: DATABASE_ERROR);
        }

        return $true;
    }


    /**
     * Delete the hcd-user: employee, supervisor, administrator
     *
     * This function is used to create a new hcd - user in the system
     * The user is created based on a a properties array, in which
     * the user login must be present, otherwise
     * an EfrontUserException is thrown. Apart from these, all the other
     * user elements are optional, and defaults will be used if they are left
     * blank.
     * Once the database representation is created, the constructor tries to create the
     * user directories, G_UPLOADPATH.'login/module_hcd/' .
     * The function instantiates the user based on its type.
     * <br/>Example:
     * <code>
     * $properties = array('login' => 'jdoe', 'name' => 'john', 'surname' => 'doe', 'email' => 'jdoe@example.com');
     * $employee   = EfrontHcdUser :: createUser($properties);
     * </code>
     *
     * @param array $employeeProperties The new user properties
     * @return EfrontHcdUser The new employee
     * @since 3.5.0
     * @access public
     */
    public function delete() {

        eF_deleteTableData("module_hcd_employees", "users_login='".$this -> login."'");
        eF_deleteTableData("module_hcd_employee_has_skill", "users_login='".$this -> login."'");
        eF_deleteTableData("module_hcd_employee_has_job_description", "users_login='".$this -> login."'");
        eF_deleteTableData("module_hcd_employee_works_at_branch", "users_login='".$this -> login."'");
        eF_deleteTableData("module_hcd_events", "users_login='".$this -> login."'");
        // Register user's firing into the event log

        eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['FIRED'],
                                                      "users_login"   => $this -> login,
                                                      "specification" => _FIRED,
                                                      "timestamp"     => time()));
    }

}


class EfrontEmployee extends EfrontHcdUser
{


    /**
     * The employee array.
     *
     * @since 3.5.0
     * @var array
     * @access public
     */
    public $employee = array();


    /**
     * The employee jobs array.
     * Each record is of the form $jobs = array ("job_description_ID" => array("branch_ID", "branch",  "job_description", "role"));    // sososos: mipws evala too much?
     * @since 3.5.0
     * @var array
     * @access public
     */
    public $jobs = false;

    /**
     * The user skills array.
     *
     * @since 3.5.0
     * @var array
     * @access public
     */
    public $skills = false;

    /**
     * The evaluations regarding the user
     *
     * @since 3.5.0
     * @var array
     * @access public
     */
    public $evaluations = false;

    /**
     * Variable which denotes whether the user is an employee.
     * This information derives from the existence of a record in
     * table module_hcd_employees
     *
     * @since 3.5.0
     * @var array
     * @access public
     */
    public $isEmployee = true;

    /**
     * Get employee type
     *
     * @return string The employee type
     * @since 3.5.0
     * @access public
     */
    public function getType() {
        return _EMPLOYEE;
    }

    /**
     * Check whether this employee is a supervisor
     *
     * @return string The employee type
     * @since 3.5.0
     * @access public
     */
    public function isSupervisor() {
        return false;
    }


   /**
     * Get the employee's jobs list
     *
     * This function is used to get a list of all jobs with the employees jobs in the form job_description_id array("branch_ID", "branch", "job_description", "role").
     * The list is returned using the objects cache (unless this is the first access to the method).
     * <br/>Example:
     * <code>
     * $jobList    = $employee -> getJobs();                         //Returns an array with pairs [job_description_ID] => [array("branch_ID", "branch", "job_description_ID", "job_description", "role")]
     * </code>
     * If $returnObjects is specified, then each skill in the skills array will
     * contain an additional field holding information on the user's skill status
     *
     * @return array An array with pairs [job_description_ID] => [array("branch_ID", "branch", "job_description_ID", "job_description", "role")]
     * @since 3.5.0
     * @access public
     */
    public function getJobs() {
        if (! $this -> jobs) {
            $jobs = eF_getTableData("module_hcd_employees LEFT OUTER JOIN module_hcd_employee_has_job_description ON module_hcd_employees.users_login = module_hcd_employee_has_job_description.users_login LEFT OUTER JOIN module_hcd_job_description ON module_hcd_job_description.job_description_ID = module_hcd_employee_has_job_description.job_description_ID LEFT OUTER JOIN module_hcd_employee_works_at_branch ON module_hcd_employee_works_at_branch.users_login = module_hcd_employees.users_login AND module_hcd_employee_works_at_branch.branch_ID = module_hcd_job_description.branch_ID LEFT OUTER JOIN module_hcd_branch ON module_hcd_job_description.branch_ID = module_hcd_branch.branch_ID", "module_hcd_job_description.job_description_ID, module_hcd_job_description.branch_ID, module_hcd_branch.name, module_hcd_branch.father_branch_ID, module_hcd_job_description.description, module_hcd_employee_works_at_branch.supervisor", "module_hcd_employees.users_login='".$this -> login."' AND module_hcd_job_description.job_description_ID != '0'");

            $size = sizeof($jobs);

            for ($i = 0 ; $i < $size; $i++) {
                $this->jobs[$jobs[$i]['job_description_ID']] = $jobs[$i];
 // The following is superfluous: TODO: remove when the tpl file is adjusted
 //               unset($this->jobs[$jobs[$i]['job_description_ID']]['job_description_ID']);
            }

        }
        return $this -> jobs;
    }


    /**
     * Checks if the supervisor is assigned to a branch. The branches
     * an employee is assigned to are those, in which this employee has job descriptions.
     * Careful: For supervisors, the assigned and the supervised branches might
     * not be the same (due to the fact, that all subbranches of a branch supervised with a job description
     * are supervised as well without any job descriptions been assigned there)
     */
    public function isAssignedToBranch($branchID) {
        if (!$this -> jobs) {
            $this -> getJobs();
        }

        // Better way?
        foreach  ($this -> getJobs() as $jobID => $job) {
            if ($job['branch_ID'] == $branchID) {
                return $jobID;
            }
        }
        return false;

    }

    /**
     * Assign a job to the employee
     *
     * This function can be used to assign a job to the current employee.
     * <br/>Example:
     * <code>
     * $user = EfrontUserFactory :: factory('jdoe');
     * $employee = EfrontHcdUserFactory :: factory('jdoe');
     * $employee -> addJob($user,23,1,1,"Lead Programmer");  //Add job 23 ("Lead Programmer") at branch 1 and set this employee's role as SUPERVISOR
     * $employee -> addJob($user,12,2,0);                    // Add job 2 (no description given) at branch 2 and set this employee's role as EMPLOYEE
     * </code>
     *
     * @param $jobID the ID of the new job to be assigned
     * @param $branchID the branch where this job placements belongs to
     * @param boolean $position the employee's role for this job: 0-Employee, 1-Supervisor
     * @param EfrontUser $userAspect, the user aspect of the employee needed to find his lessons
     * @return an Item of class EfrontHcdUser. This is done, because the role might change the class where the employee belongs.
     * @since 3.5.0
     * @access public
     */
    public function addJob($userAspect, $jobID, $branchID, $position, $job_description = false) {

        if ($branchID == "0" || ($position != "0" && $position != "1") || ($jobID == "" || $jobID == "0")) {
           throw new EfrontUserException(_NOBRANCHSELECTED . " ", EfrontUserException :: WRONG_INPUT_TYPE);
        }

        if (!$this -> jobs) {
            $this -> getJobs();
        }

        // Check if this job description is already assigned
        if (!in_array($jobID, array_keys($this -> jobs))) {

            // The employee must be promoted to a Supervisor -> a new EfrontSupervisor instance must be returned
            if ($position == "1") {

                // Create a new supervisor instance, with no jobs assigned
                $supervisor = new EfrontSupervisor($this -> employee, array());
                $supervisor -> skills = $this -> skills;
                $supervisor = $supervisor -> addJob($userAspect, $jobID, $branchID, $position, $job_description);    // the jobs will be taken by the addJob

                return $supervisor;

            } else {
                // Write to the database the new branch assignment: employee to branch (if such an assignment is not already true)

                if (! $this->isAssignedToBranch($branchID)) {
                    eF_insertTableData("module_hcd_employee_works_at_branch", array('users_login' => $this -> login, 'supervisor' => $position, 'assigned' => '1', 'branch_ID' => $branchID));
                }

                // Write to database the new job assignment: employee to job description
                $insert_job = array('users_login' => $this -> login, 'job_description_ID' => $jobID);

                if($ok = eF_insertTableData("module_hcd_employee_has_job_description", $insert_job)) {
                    $message      = _OPERATIONCOMPLETEDSUCCESFULLY;
                    $message_type = 'success';

                    $bname = eF_getTableData("module_hcd_branch", "name, father_branch_ID","branch_ID = '".$branchID."'");

                    if (!$job_description) {
                        $jdescription = eF_getTableData("module_hcd_job_description", "description", "job_description_ID= '".$jobID."'");
                        $job_description = $jdescription[0]['description'];
                    }
                    $this -> jobs[$jobID] = array("job_description_ID" => $jobID, "branch_ID" => $branchID, "name" => $bname[0]['name'],  "father_branch_ID" => $bname[0]['father_branch_ID'], "description" => $job_description, "supervisor" => $position);

                    // Register job release into the event log
                    eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['JOB'],
                                                                  "users_login"   => $this -> login,
                                                                  "author"        => $_SESSION['s_login'],
                                                                  "specification" => _EMPLOYEEWASASSIGNEDJOB .": " .$job_description. " ". _ATBRANCH . " " .$bname[0]['name'],
                                                                  "timestamp"     => time()));

                    // Can be optimized in cost of programmability interface
                    $lessons_to_add = eF_getTableData("module_hcd_lesson_to_job_description", "lessons_ID", "job_description_ID = '". $jobID ."'");
                    $size = sizeof($lessons_to_add);

                    if ($size) {
                        $currentLessonIds = array_keys($userAspect -> getLessons());

                        $at_least_one_new_lesson = 0;
                        $sql_query = "INSERT INTO users_to_lessons (users_LOGIN, lessons_ID, active, from_timestamp, user_type, positions, done_content, issued_certificate, comments) VALUES ";
                        for ($k = 0; $k<$size; $k++) {
                            if ( !in_array($lessons_to_add[$k]['lessons_ID'], $currentLessonIds) ) {
                                $at_least_one_new_lesson = 1;
                                $sql_query .= "('". $this -> login . "','" . $lessons_to_add[$k]['lessons_ID'] . "', '1', '".time()."', 'student','','','','')";
                                if ($k != $size-1) {
                                    $sql_query = $sql_query . ",";
                                }

                                //$userAspect -> addLessons($lessons_to_add[$k]['lessons_ID']);
                            }

                        }

                        if ($at_least_one_new_lesson) {
                            $ok = eF_execute($sql_query);
                        }
                    }
//echo "meta to query<br>";


                } else {
                   throw new EfrontUserException(_PLACEMENTCOULDNOTBEASSIGNED . " ", EfrontUserException :: DATABASE_ERROR);
                }

                return $this;
            }
        } else {
           throw new EfrontUserException(_JOBALREADYASSIGNED . " ", EfrontUserException :: WRONG_INPUT_TYPE);
        }
    }


    /**
     * Remove a job to the employee
     *
     * This function can be used to remove a job to the current employee.
     * <br/>Example:
     * <code>
     * $user = EfrontHcdUserFactory :: factory('jdoe');
     * $user -> removeJob(12);                     // Removejob 12
     * </code>
     *
     * @param $jobID the ID of the job to be removed
     * @return an Item of class EfrontHcdUser. This is done, because the when the last supervisor job is removed then the employee changes class.
     * @since 3.5.0
     * @access public
     */
    public function removeJob($jobID) {

        if ($jobID == "" || $jobID == "0") {
           throw new EfrontUserException("", EfrontUserException :: WRONG_INPUT_TYPE);
        }

        if (!$this -> jobs) {
            $this -> getJobs();
        }

        // If this job description is actually assigned
        if (in_array($jobID, array_keys($this -> jobs))) {

            // We need to check whether other placements are also assigned to the employee at that branch
            $branchID = $this -> jobs[$jobID]['branch_ID'];
            $branch_name = $this -> jobs[$jobID]['name'];
            $job_description = $this -> jobs[$jobID]['description'];

            // Delete this job from cache
            unset($this -> jobs[$jobID]);

            // If no other job exists in that branch
            if (sizeof($this->jobs) == 0 || !($another_job_at_branch = $this -> isAssignedToBranch($branchID))) {
                eF_deleteTableData("module_hcd_employee_works_at_branch", "users_login='".$this -> login."' AND branch_ID ='". $branchID ."'" );
            }

            eF_deleteTableData("module_hcd_employee_has_job_description", "users_login='".$this -> login."' AND job_description_ID ='".$jobID."'");

            $message      = _EMPLOYEERELEASEDFROMJOB;
            $message_type = 'success';

            // Register job release into the event log
            eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['JOB'],
                                                          "users_login"   => $this -> login,
                                                          "author"        => $_SESSION['s_login'],
                                                          "specification" => _EMPLOYEEWASRELEASEDFROMDJOB .": " .$job_description. " ". _ATBRANCH . " " .$branch_name,
                                                          "timestamp"     => time()));

        } else {
           throw new EfrontUserException(_JOBNOTASSIGNED . " ", EfrontUserException :: WRONG_INPUT_TYPE);
        }

        return $this;
    }


    /**
     * Update the data of this employee
     *
     * <br/>Example:
     * <code>
     * $properties = array('login' => 'jdoe', 'name' => 'john', 'surname' => 'doe', 'email' => 'jdoe@example.com');
     * $job        = array("branch_ID" => "2", "job_description" => "Programmer", "role" => "Supervisor");    // or "role" => "Employee"
     * $employee   = EfrontHcdUser :: createUser($properties, $job, "2");
     * $new_properties = array('name' => 'john', 'surname' => 'doe', 'email' => 'jdoe@example.com');
     * $new_job        = array("branch_ID" => "3", "job_description" => "Manager", "role" => "Employee");    // or "role" => "Supervisor"
     * $employee = $employee -> updateEmployeeData($new_properties, $new_job, "5");     // the self-assignment is needed in case the employee changes type from employee<->supervisor
     *
     * </code>
     *
     * @param array $employeeProperties The employee new properties
     * @param array $employeeJob The new job of the new employee
     * @return EfrontHcdUser The new employee
     * @since 3.5.0
     * @access public
     */
    public function updateEmployeeData($newEmployeeProperties) {

        eF_updateTableData("module_hcd_employees", $newEmployeeProperties, "users_login='". $this -> login ."'");

        global $currentUser;
        // Check for changes in hired date/wage (raise or decrease) and register corresponding events into the log
        if ($this -> employee['hired_on'] != $newEmployeeProperties['hired_on']) {

            // Only one hiring date can be valid - delete the previous existing one
            if ($this -> employee['hired_on'] && $this -> employee['hired_on'] != "") {
                eF_deleteTableData("module_hcd_events", "event_code = '".$GLOBALS['MODULE_HCD_EVENTS']['HIRED']."' AND users_login = '".$this -> login."'");
            }


            if (($hired_date = $newEmployeeProperties['hired_on']) != "" )  {
                if (strpos($hired_date, "/")) {
                   $result = explode("/",$hired_date);
                } else {
                   $result = explode("-",$hired_date);
                }
                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['HIRED'],
                                                              "users_login"   => $this -> login,
                                                              "author"        => $currentUser -> login,
                                                              "specification" => _HIRED,
                                                              "timestamp"     => mktime(0,0,0,$result[1],$result[0],$result[2])));
            }
        }


        if ($this -> employee['left_on'] != $newEmployeeProperties['left_on'] ) {
            if ($this -> employee['left_on'] && $this -> employee['left_on'] != "") {
                eF_deleteTableData("module_hcd_events", "event_code = '".$GLOBALS['MODULE_HCD_EVENTS']['LEFT']."' AND users_login = '".$this -> login."'");
            }

            if (  ($left_date = $newEmployeeProperties['left_on'])  != "" )  {

                if (strpos($left_date, "/")) {
                   $result = explode("/",$left_date);
                } else {
                   $result = explode("-",$left_date);
                }
                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['LEFT'],
                                                              "users_login"   => $this -> login,
                                                              "author"        => $currentUser -> login,
                                                              "specification" => _HASLEFT,
                                                              "timestamp"     => mktime(0,0,0,$result[1],$result[0],$result[2])));
            }
        }

        // Wage changes
        if ($this -> employee['wage'] != $newEmployeeProperties['wage']) {
            $init_wage = (integer)$this -> employee['wage'];
            $final_wage = (integer)$newEmployeeProperties['wage'];

            if (!$init_wage || $init_wage == 0)
            {
                $spec = _EMPLOYEESNEWWAGEIS . " " .$final_wage;
            } else if ( $init_wage  < $final_wage )  {
                $diff = ($final_wage-$init_wage);
                $raise_percent = round((float)$diff/(float)$init_wage*100,2);
                $spec = _RAISEOF ." ". $diff . " (" . $raise_percent . "%)." . _EMPLOYEESNEWWAGEIS . " " .$final_wage;
            } else if ( $init_wage  > $final_wage ) {
                $diff = ($init_wage-$final_wage);
                $raise_percent = round((float)$diff/(float)$init_wage*100,2);
                $spec = _DECREASEOF ." ". $diff . " (" . $raise_percent . "%)." . _EMPLOYEESNEWWAGEIS . " " .$final_wage;
            }

            eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['WAGE_CHANGE'],
                                                          "users_login"   => $this -> login,
                                                          "author"        => $currentUser -> login,
                                                          "specification" => $spec,
                                                          "timestamp"     => time()));

        }

        return $this;
    }

    /**
     * Get the employees skills list
     *
     * This function is used to get a list of ids with the employees lessons.
     * The list is returned using the objects cache (unless $returnObjects is true).
     * <br/>Example:
     * <code>
     * $skillsList    = $user -> getSkills();                         //Returns an array with pairs [skills id] => [employee specification for this skill]
     * $skillsObjects = $user -> getSkills(true);                     //Returns an array of skill objects - TODO when skills become objects
     * </code>
     * If $returnObjects is specified, then each skill in the skills array will
     * contain an additional field holding information on the user's skill status
     *
     * @param boolean $returnObjects Whether to return skill objects
     * @return array An array of [skill id] => [user type] pairs, or an array of skill objects
     * @since 3.5.0
     * @access public
     */
    public function getSkills() {
        if (! $this -> skills ) {
            $result = eF_getTableData("module_hcd_employee_has_skill JOIN module_hcd_skills ON module_hcd_skills.skill_ID = module_hcd_employee_has_skill.skill_ID", "module_hcd_employee_has_skill.skill_ID, module_hcd_employee_has_skill.specification, module_hcd_skills.description", "users_LOGIN = '".$this -> login."'");
            foreach ($result as $skill) {
                $id = $skill['skill_ID'];
                $this -> skills[$id] = $skill;
            }

        }
        return $this -> skills;
    }


    /**
     * Assign skills to user.
     *
     * This function can be used to assign a skill to the current employee. If $skillDescriptions
     * is specified, then the skill is assigned to the employee with this description. Otherwise, '' is used
     * <br/>Example:
     * <code>
     * $user = EfrontHcdUserFactory :: factory('jdoe');
     * $user -> addSkills(23);                         //Add a single skill with id 23
     * $user -> addSkills(23, 'Three years of work experience');            //Add a signle lesson with id 23 and set the skill description to 'Three years of work experience'
     * $user -> addSkills(array(23,24,25));            //Add multiple skills using an array
     * $user -> addSkills(array(23,24,25), array('Three years of work experience', 'Fair', 'Excellent sql knowledge'));            //Add multiple skills using an array for lesson ids and another for corresponding descriptions
     * $user -> addSkills(23, '2 years delivery', true);    // Complement the description of skill id 23 with string '2 years delivery'
     * </code>
     *
     * @param mixed $lessonIds Either a single lesson id, or an array if ids
     * @param mixed $userTypes The corresponding user types for the specified lessons
     * @param boolean $complement if complement = true, then if the skill exists the new description will complement the oldone
     * @return int The array of lesson ids.
     * @since 3.5.0
     * @access public
     * @todo auto_projects
     */
    public function addSkills($skillIds, $skillDescriptions, $complement = false) {

        if (!$this -> skills) {
            $this -> getSkills();
        }

        if (!is_array($skillIds)) {
            $skillIds = array($skillIds);
        }
        if (!is_array($skillDescriptions)) {
            $skillDescriptions = array($skillDescriptions);
        }

        global $currentUser;
        $currentSkillIds = array_keys($this -> skills);

        foreach ($skillIds as $key => $skillId) {
            if (eF_checkParameter($skillId, 'id')) {
                isset($skillDescriptions[$key])  ? $skillDescription = $skillDescriptions[$key] : $skillDescription = "";

                // Check if the skill is already assigned, if so, complement the existing specification
                if (isset($this -> skills[$skillId])) {
                    if ($complement) {
                        eF_updateTableData("module_hcd_employee_has_skill", array("specification" => $this -> skills[$skillId]['specification'] . ", ". $skillDescription, "author_login" => $currentUser -> login), "users_LOGIN = '".$this -> login."' AND skill_ID=".$skillId);
                        $this -> skills[$skillId]['specification'] = $this -> skills[$skillId]['specification'] . ", ". $skillDescription;
                    } else {
                        eF_updateTableData("module_hcd_employee_has_skill", array("specification" => $skillDescription,"author_login" => $currentUser -> login), "users_LOGIN = '".$this -> login."' AND skill_ID=".$skillId);
                        $this -> skills[$skillId]['specification'] = $skillDescription;
                    }
                    // Register skill specifications change into the event log
                    eF_insertTableData("module_hcd_events", array("event_code"    => 5,
                                                                  "users_login"   => $this -> login,
                                                                  "author"        => $currentUser -> login,
                                                                  "specification" => _EMPLOYEESSKILLSPECIFICATIONSFORSKILL. ' "' .$this -> skills[$skillId]['description']. '" ' ._WERECHANGEDTO ." ".$this -> skills[$skillId]['specification'],
                                                                  "timestamp"     => time()));
                } else {
                    $result = eF_getTableData("module_hcd_skills", "description", "skill_ID = '".$skillId."'");

                    if ($ok = eF_insertTableData("module_hcd_employee_has_skill", array("users_LOGIN" => $this -> login, "skill_ID" => $skillId, "specification" => $skillDescription, "author_login" => $currentUser -> login))) {


                        // Register skill assignment into the event log
                        eF_insertTableData("module_hcd_events", array("event_code"    => 5,
                                                                      "users_login"   => $this -> login,
                                                                      "author"        => $currentUser -> login,
                                                                      "specification" => _EMPLOYEEWASASSIGNEDSKILL .' "' . $result[0]['description']. '" ' . (($skillDescription != "") ? '('.$skillDescription.')':''),
                                                                      "timestamp"     => time()));

                        $this -> skills[$skillId]['specification'] = $skillDescription;
                    } else {
                        throw new EfrontUserException(_OPERATIONWASNOTCOMPLETEDSUCCESFULLY.': '.$ok, EfrontUserException :: DATABASE_ERROR);
                    }
                }
            }
        }

        return $this -> skills;
    }

    /**
     * Remove skills from employee.
     *
     * This function can be used to remove a skill from the current employee.
     * <br/>Example:
     * <code>
     * $employee = EfrontHcdUserFactory :: factory('jdoe');
     * $employee -> removeSkills(23);                          //Remove a signle skill with id 23
     * $employee -> removeSkills(array(23,24,25));             //Remove multiple skills using an array
     * </code>
     *
     * @param int $skillIds Either a single skill id, or an array if ids
     * @return int The array of skill ids.
     * @since 3.5.0
     * @access public
     */
    public function removeSkills($skillIds) {
        if (!$this -> skills) {
            $this -> getSkills();
        }

        if (!is_array($skillIds)) {
            $skillIds = array($skillIds);
        }
        foreach ($skillIds as $key => $skillId) {
            if (!eF_checkParameter($skillId, 'id')) {
                unset($skillIds[$key]);                                        //Remove illegal vaues from skills array.
            }
        }

        eF_deleteTableData("module_hcd_employee_has_skill", "users_login = '".$this -> login."' and skill_ID in (".implode(",", $skillIds).")");    //delete skills from list

        global $currentUser;

        foreach ($this -> skills as $key => $skillId) {
            if (in_array($skillId['skill_ID'], $skillIds)) {
                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['SKILL'],
                                                          "users_login"   => $this -> login,
                                                          "author"        => $currentUser -> login,
                                                          "specification" => _SKILL ." " .$skillId['description']. " ". _WASRELEASEDFROMEMPLOYEE,
                                                          "timestamp"     => time()));

                unset($this -> skills[$key]);                                        //Remove skills from cache array.
            }
        }

        return $this -> skills;
    }



    /**
     * Update employee skills.
     *
     * This function can be used to update the specification for a skill of the current employee.
     * <br/>Example:
     * <code>
     * $employee = EfrontHcdUserFactory :: factory('jdoe');
     * $employee -> updateSkills(23, "Better java knowledge than before");                          //Update skill with id 23
     * </code>
     *
     * @param int $skillId a single skill id, $newSkillDescription the new description
     * @return int The array of skill ids.
     * @since 3.5.0
     * @access public
     */
    public function updateSkills($skillId, $newSkillDescription) {
        // Check if the new parameters are valid
        if (!eF_checkParameter($skillId, 'id')) {
            throw new EfrontUserException(_WRONGPARAMETERTYPE.': '._SKILLID, EfrontUserException :: WRONG_INPUT_TYPE);
        }

        if (!$this -> skills) {
            $this -> getSkills();
        }

        // Check if this skill already exists; otherwise, it cannot be updated
        if (!in_array($skillId, array_keys($this -> skills))) {
            throw new EfrontUserException(_WRONGPARAMETERTYPE.': '._SKILLID, EfrontUserException :: WRONG_INPUT_TYPE);
        }

        global $currentUser;
        // Update only if the specification is different
        if ($this -> skills[$skillId] != $newSkillDescription) {
            $fields_update = array("users_LOGIN"    =>  $this -> login,
                                   "specification"  =>  $newSkillDescription,
                                   "author_login" => $currentUser -> login,
                                   "skill_ID"       =>  $skillId);
            eF_updateTableData("module_hcd_employee_has_skill", $fields_update, "users_LOGIN = '".$this -> login."' AND skill_ID=".$skillId);

            // Update the cache (which has surely been defined here) as well
            $this -> skills[$skillId]['specification'] = $newSkillDescription;

            // Register skill specifications change into the event log
            eF_insertTableData("module_hcd_events", array("event_code"    => 5,
                                                          "users_login"   => $this -> login,
                                                          "author"        => $currentUser -> login,
                                                          "specification" => _EMPLOYEESSKILLSPECIFICATIONSFORSKILL. ' "' .$this -> skills[$skillId]['description']. '" ' ._WERECHANGEDTO ." ".$this -> skills[$skillId]['specification'],
                                                          "timestamp"     => time()));

        }

        return $this -> skills;
    }

    /**
     * Get the employees evaluations list
     *
     * This function is used to get the list of evaluations written FOR this employees.
     * The list is returned using the objects cache.
     * <br/>Example:
     * <code>
     * $evaluationsList    = $user -> getEvaluations();          //Returns an array with of evaluations
     * </code>
     *
     * @return array An array of [eventID] => [author, specification, timestamp] pairs
     * @since 3.5.0
     * @access public
     */
    public function getEvaluations() {
        if (sizeof($this -> evaluations) == 0) {
            $result = eF_getTableData("users JOIN module_hcd_events ON login = author","login, name, surname,module_hcd_events.*","module_hcd_events.users_login = '".$this -> login."' && event_code >= 10","timestamp","","login");

            foreach ($result as $evaluation) {
                $evID = $evaluation['event_ID'];
                $this -> evaluations[$evID] = array('event_ID' => $evID, 'author' => $evaluation['author'], 'author_name' => $evaluation['name'],'author_surname' => $evaluation['surname'],'specification' => $evaluation['specification'], 'timestamp' => $evaluation['timestamp']);
            }
        }
        return $this -> evaluations;

    }

    /**
     * Evaluate a user wit an evaluation.
     *
     * This function can be used to assign an evaluation to the employee of this object.
     * The author is set as the currentEmployee using the system
     * <br/>Example:
     * <code>
     * $user -> addEvaluation("Lazy");                         //Add a single evaluation: lazy
     * $user -> addEvaluation(array("Higher","Faster","Stronger"); //Add three evaluations: higher, faster, stronger
     * </code>
     *
     * @param mixed $evaluation Either a single evaluation, or an array of evaluations
     * @return int The array of evaluations.
     * @since 3.5.0
     * @access public
     * @todo auto_projects
     */
    public function addEvaluation($evaluation) {
        if (sizeof($this -> evaluations) == 0) {
            $this -> getEvaluations();
        }
        if (!is_array($evaluation)) {
            $evaluation = array($evaluation);
        }

        global $currentUser;
        foreach ($evaluation as $temp_eval) {
            $eval_item = array('specification'  => $temp_eval,
                               'event_code'     => 10,
                               'users_login'    => $this -> login,
                               'author'         => $currentUser -> login,
                               'timestamp'      => time());
            if ($ok = eF_insertTableData("module_hcd_events", $eval_item)) {
                $this -> evaluations[$ok] = array('event_ID' => $ok, 'author' => $eval_item['author'], 'author_name' => $currentUser -> user['name'],'author_surname' => $currentUser -> user['surname'], 'specification' => $eval_item['specification'], 'timestamp' => $eval_item['timestamp']);
            }
            else {
                throw new EfrontUserException(_EVALUATIONCOULDNOTBECREATED.': '.$ok, EfrontUserException :: DATABASE_ERROR);
            }

        }

        return $this -> evaluations;
    }

    /**
     * Remove an evaluation from an employee according to its Id
     * This function can be used to remove a evaluation from the current employee.
     * <br/>Example:
     * <code>
     * $employee -> removeEvaluations(23);                          //Remove a signle evaluation with id 23
     * $employee -> removeEvaluations(array(23,24,25));             //Remove multiple evaluations using an array
     * </code>
     *
     * @param int $evaluationIds Either a single evaluation id, or an array if ids
     * @return int The array of evaluation ids.
     * @since 3.5.0
     * @access public
     */
    public function removeEvaluations($evaluationIds) {

        if (!is_array($evaluationIds)) {
            $evaluationIds = array($evaluationIds);
        }

        // If the evaluations cache has not been defined yet
        if (sizeof($this -> evaluations) == 0) {
            $this -> getEvaluations();
        }

        global $currentUser;
        foreach ($evaluationIds as $key => $evaluationId) {
            if ($currentUser -> getType() != 'administrator' && ($this -> evaluations[$evaluationId] && $this -> evaluations[$evaluationId]['author'] != $currentUser -> login)) {
                throw new EfrontUserException(_YOUCANNOTDELETESOMEELSESEVALUATION, EfrontUserException :: WRONG_INPUT_TYPE);
            }

            if (!eF_checkParameter($evaluationId, 'id')) {
                unset($evaluationIds[$key]);                                        //Remove illegal vaues from evaluations array.
            }
        }

        if (eF_deleteTableData("module_hcd_events", "users_login = '".$this -> login."' and event_ID in (".implode(",", $evaluationIds).")")) {    //delete evaluations from list

            foreach ($evaluationIds as $evaluationId) {
                unset($this -> evaluations[$evaluationId]);                                  //Remove evaluations from cache array.
            }
        }

        return $this -> evaluations;
    }


    /**
     * Update an employee evaluation.
     *
     * This function can be used to update the specification for a evaluation of the current employee.
     * <br/>Example:
     * <code>
     * $employee -> updateEvaluations(23, "Not so lazy any more");                          //Update a single evaluation with id 23
     * </code>
     *
     * @param int $evaluationId a single evaluation id, $newEvaluationDescription the new description
     * @return int The array of evaluation ids.
     * @since 3.5.0
     * @access public
     */
    public function updateEvaluations($evaluationId, $newEvaluationDescription) {
        // Check if the new parameters are valid
        if (!eF_checkParameter($evaluationId, 'id')) {
            throw new EfrontUserException(_WRONGPARAMETERTYPE.': '._EVALUATIONID, EfrontUserException :: WRONG_INPUT_TYPE);
        }

        if (sizeof($this -> evaluations) == 0) {
            $this -> getEvaluations();
        }

        // Check if this evaluation already exists; otherwise, it cannot be updated
        if (!in_array($evaluationId, array_keys($this -> evaluations))) {
            throw new EfrontUserException(_WRONGPARAMETERTYPE.': '._EVALUATIONID, EfrontUserException :: WRONG_INPUT_TYPE);
        } else {
            global $currentUser;
            // Check if the currentEmployee has permission to edit this evaluation: he is either the administrator or the author
            if ($currentUser -> getType() != 'administrator' && $this -> evaluations[$evaluationId]['author'] != $currentUser -> login) {
                throw new EfrontUserException(_YOUCANNOTEDITSOMEELSESEVALUATION, EfrontUserException :: WRONG_INPUT_TYPE);
            }

        }

        // Update only if the specification is different
        if ($this -> evaluations[$evaluationId] != $newEvaluationDescription) {
            // We also update the author field, in case the administrator is the one editing the evaluation
            $fields_update = array("author"         =>  $currentUser -> login,
                                   "specification"  =>  $newEvaluationDescription,
                                   "timestamp"      =>  time());
            eF_updateTableData("module_hcd_events", $fields_update, "users_login = '".$this -> login."' AND event_ID=".$evaluationId);

            // Update the cache (which has surely been defined here) as well
            $this -> evaluations[$evaluationId]['specification'] = $newEvaluationDescription;
            $this -> evaluations[$evaluationId]['timestamp'] = time();
            $this -> evaluations[$evaluationId]['author'] = $currentUser -> login;
            $this -> evaluations[$evaluationId]['name'] = $currentUser -> users['name'];
            $this -> evaluations[$evaluationId]['surname'] = $currentUser -> users['surname'];
        }

        return $this -> evaluations;
    }

    /**
     *
     * Export employee form to PDF
     * <br/>Example:
     * <code>
     * $employee_array = eF_getTableData("module_hcd_employee", "*", "users_login = 'jdoe'");
     * $user = EfrontHcdUserFactory :: factory($employee_array);            //Use factory to instantiate user object with login 'jdoe'
     * $user = new EfrontEmployee('jdoe')                  //Instantiate administrator user object with login 'jdoe'
     * $user -> printToPDF($eFrontUser, $skill_categories, $courses, $lessons, $averages, "logo.jpg");
     *
     * @param $skill_categories, $courses, $lessons, $all_average, $logo
     * @param
     * @since 3.5.0
     * @access public
     **/
    public function printToPdf($eFrontUser, $evaluations, $skill_categories, $courses, $lessons, $all_average, $logo) {
        $pdf = new TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true);

        $pdf->SetCreator(PDF_CREATOR);
        $pdf->SetAuthor(PDF_AUTHOR);
        $pdf->SetTitle($doc_title);
        $pdf->SetSubject($doc_subject);
        $pdf->SetKeywords($doc_keywords);

        //set margins
        $pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);

        //set auto page breaks
        $pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
        $pdf->SetHeaderMargin(PDF_MARGIN_HEADER);
        $pdf->SetFooterMargin(PDF_MARGIN_FOOTER);
        $pdf->setImageScale(PDF_IMAGE_SCALE_RATIO); //set image scale factor

        $pdf->setHeaderFont(Array('FreeSerif', 'I', 11));
        $pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
//        $pdf->setHeaderData('','','', _EMPLOYEEFORM.":".$this -> login);
//        $pdf->setLanguageArray($l);     //set language items

        //initialize document
        $pdf->AliasNbPages();

        $pdf->AddPage();

        if ($logo) {
            $pdf->Image($logo,16,16,15,15);
        }

        $x = 45;
        $y = 19;
        $pdf->SetXY($x,$y);
        $pdf->SetFont("FreeSerif","B",16); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(140, 10, _EMPLOYEEFORM . ": " . $eFrontUser -> user['name'] ." ". $eFrontUser -> user['surname'], 0, 1, L, 0);
        $pdf->Cell(180, 2, " ", B, 1, L, 0);

$pdf->SetXY(15,40);
        try {
            $avatar_file = new eF_File($eFrontUser -> user['avatar']);
        }catch(Exception $e) {
            $avatar_file = new eF_File(G_IMAGESPATH . "avatars/system_avatars/unknown_small.gif");
        }
        $avatar_fn = $avatar_file -> createImage();
        if ($avatar_file) {
            $pdf->Image($avatar_fn,26,45,20,20);
            unlink($avatar_fn);
        }
$x = 60;
$y = 40;
$z = 25;

$pdf->SetXY($x,$y);
        $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 5, _NAME, 0, 0, L, 0);
        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        $pdf->Cell(170, 5, $eFrontUser -> user['surname'] ." ". $eFrontUser -> user['name'], 0, 1, L, 0);

if ($this -> employee['birthday']) {
        $y +=5; $pdf->SetXY($x,$y);$z -= 5;
        $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 5, _BIRTHDAY,   0, 0, L, 0);
        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        $pdf->Cell(170, 5, $this -> employee['birthday'], 0, 1, L, 0);

}

if ($this -> employee['address']) {
        $y +=5; $pdf->SetXY($x,$y);$z -= 5;
        $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 5, _ADDRESS,   0, 0, L, 0);
        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        $pdf->Cell(170, 5, $this -> employee['address'], 0, 1, L, 0);
}
if ($this -> employee['city']) {
        $y +=5; $pdf->SetXY($x,$y);$z -= 5;
        $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 5, _CITY,   0, 0, L, 0);
        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        $pdf->Cell(170, 5, $this -> employee['city'], 0, 1, L, 0);
}
if ($this -> employee['hired_on']) {
        $y +=5; $pdf->SetXY($x,$y);$z -= 5;
        $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 5, _HIREDON,   0, 0, L, 0);
        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        $pdf->Cell(170, 5, $this -> employee['hired_on'], 0, 1, L, 0);
}

if ($this -> employee['left_on']) {
        $y +=5; $pdf->SetXY($x,$y);$z -= 5;
        $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 5, _LEFTON,   0, 0, L, 0);
        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        $pdf->Cell(170, 5, $this -> employee['left_on'], 0, 1, L, 0);
}

$pdf->SetXY($x,$y+$z);

        $pdf->Cell(70, 5, " ", 0, 1, L, 0);

        // Jobs
        $placements = $this -> getJobs();

        $pdf->SetFont("FreeSerif","B",16); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 10, _PLACEMENTS , 0, 1, L, 0);

        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        foreach ($placements as $placement) {
            if ($placement['supervisor']) {
                $supervisor = "("._SUPERVISOR.")";
            } else {
                $supervisor = "";
            }
            $pdf->Cell(270, 5, $placement['name'] . ": " . $placement['description']. " " . $supervisor, 0, 1, L, 0);

        }
        $pdf->Cell(70, 5, " ", 0, 1, L, 0);

        // Evaluations
        $pdf->SetFont("FreeSerif","B",16); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 10, _EVALUATIONS , 0, 1, L, 0);

        $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
        foreach ($evaluations as $evaluation) {
            $tmp_value = str_replace("#filter:timestamp-"," ",$evaluation['timestamp']);
            $tmp_endValue = str_replace("#", " ", $tmp_value);
            $pdf->SetTextColor(0,0,255);
            $pdf->Cell(270, 5, strftime('%d-%m-%Y', $tmp_endValue).": ". $evaluation['specification'] . " [" . $evaluation['surname'] . " " . $evaluation['name'] . "]" , 0, 1, L, 0);
        }
        $pdf->Cell(70, 5, " ", 0, 1, L, 0);

        // Evaluations
        $pdf->SetFont("FreeSerif","B",16); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 10, _SKILLS , 0, 1, L, 0);

        // Skill categories
        foreach ($skill_categories as $id => $skill_cat) {
            $notfound = 1;
            $pdf->SetFont("FreeSerif","B",12); $pdf->SetTextColor(0,0,0);
            $pdf->Cell(70, 5, $skill_cat['description'], 0, 1, L, 0);

            foreach ($skill_cat['skills'] as $id => $skill ) {

                $notfound  = 0;

                $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,0);
                $pdf->Cell(70, 5, $skill['description'] . ": ", 0, 0, L, 0);

                $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,255);
                $pdf->Cell(170, 5, $skill['specification'] . " (" . $skill['surname'] . " " . $skill['name'] . ")", 0, 1, L, 0);
            }

            if ($notfound) {
                $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,0);
                $pdf->Cell(70, 5, _NOSKILLSASSIGNED, 0, 1, L, 0);
            }
            $pdf->Cell(170, 5, " ",   0, 1, L, 0);
        }

        $pdf->SetFont("FreeSerif","B",16); $pdf->SetTextColor(0,0,0);
        $pdf->Cell(70, 12, _TRAININGCAP , 0, 1, L, 0);

        foreach ($courses as $course) {
            $pdf->SetFont("FreeSerif","B",14); $pdf->SetTextColor(0,0,0);
            $pdf->Cell(70, 10, $course['name'], 0, 1, L, 0);
            foreach ($course['lessons'] as $lesson) {
                $pdf->SetFont("FreeSerif","B",11); $pdf->SetTextColor(0,0,0);
                if ($lesson['completed']) {
                    $tmp_value = str_replace("#filter:timestamp-"," ",$lesson['to_timestamp']);
                    $tmp_endValue = str_replace("#", " ", $tmp_value);
                    $to_timestamp =  strftime('%d-%m-%Y', $tmp_endValue);
                    $score  = ceil($lesson['score']) . "%";
                } else {
                    $to_timestamp = "-";
                    $score  = "";
                }
                $pdf->Cell(70, 8, $lesson['name'] . " " . $score, 0, 1, L, 0);
                $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,0);
                $pdf->Cell(70, 5, _COMPLETED . ": " . $to_timestamp, 0, 1, L, 0);

                foreach ($lesson['tests'] as $test) {
                    $pdf->Cell(50, 5, $test['name'] . ":", 0, 0, L, 0);
                    if ($test['score'] > 60) {
                        $pdf->SetFillColor(0,255,0);
                    } else {
                        $pdf->SetFillColor(255,0,0);
                    }
                    $pdf->Cell(10, 5, ceil($test['score']) . "%", 0, 1, L, 1);
                }

                if ($lesson['tests_count'] > 0) {
                    if ($lesson['tests_average'] > 60) {
                        $pdf->SetFillColor(0,255,0);
                    } else {
                        $pdf->SetFillColor(255,0,0);
                    }
                    $pdf->Cell(50, 5, _AVERAGESCORE . ":", 0, 0, L, 0);
                    $pdf->Cell(10, 5, ceil($lesson['tests_average']) . "%", 0, 1, L, 1);
                }
            $pdf->Cell(70, 5, " ", 0, 1, L, 0);
            }

        }

        foreach ($lessons as $lesson) {
            $pdf->SetFont("FreeSerif","B",14); $pdf->SetTextColor(0,0,0);
            if ($lesson['completed']) {
                $tmp_value = str_replace("#filter:timestamp-"," ",$lesson['to_timestamp']);
                $tmp_endValue = str_replace("#", " ", $tmp_value);
                $to_timestamp =  strftime('%d-%m-%Y', $tmp_endValue);
                $score  = ceil($lesson['score']) . "%";
            } else {
                $to_timestamp = "-";
                $score  = "";
            }
            $pdf->Cell(70, 10, $lesson['name'] . " " . $score, 0, 1, L, 0);
            $pdf->SetFont("FreeSerif","",10); $pdf->SetTextColor(0,0,0);
            $pdf->Cell(70, 5, _COMPLETED . ": " . $to_timestamp, 0, 1, L, 0);

            foreach ($lesson['tests'] as $test) {
                $pdf->Cell(50, 5, $test['name'] . ":", 0, 0, L, 0);
                if ($test['score'] > 60) {
                    $pdf->SetFillColor(0,255,0);
                } else {
                    $pdf->SetFillColor(255,0,0);
                }
                $pdf->Cell(10, 5, ceil($test['score']) . "%", 0, 1, L, 1);
            }

            if ($lesson['tests_count'] > 0) {
                if ($lesson['tests_average'] > 60) {
                    $pdf->SetFillColor(0,255,0);
                } else {
                    $pdf->SetFillColor(255,0,0);
                }
                $pdf->Cell(50, 5, _AVERAGESCORE . ":", 0, 0, L, 0);
                $pdf->Cell(10, 5, ceil($lesson['tests_average']) . "%", 0, 1, L, 1);
            }
            $pdf->Cell(70, 5, " ", 0, 1, L, 0);
        }

        foreach ($all_average as $average) {
            $pdf->SetFont("FreeSerif","B",11); $pdf->SetTextColor(0,0,0);
            $pdf->Cell(80, 5, $average['title'] . ":", 0, 0, L, 0);
            if ($average['avg'] > 60) {
                $pdf->SetFillColor(0,255,0);
            } else {
                $pdf->SetFillColor(255,0,0);
            }
            $pdf->Cell(10, 5, $average['avg'] . "%", 0, 1, L, 1);
        }
        $pdf->Output();
    }


    /**
     * Instantiate class
     *
     * This function instantiates a new EfrontEmployee object based on the given
     * employee array. Either the EfrontUserFactory may be used, or directly the
     * EfrontX class.
     * <br/>Example:
     * <code>
     * $employee_array = eF_getTableData("module_hcd_employee", "*", "users_login = 'jdoe'");
     * $user = EfrontHcdUserFactory :: factory($employee_array);            //Use factory to instantiate user object with login 'jdoe'
     * $user = new EfrontAdministrator('jdoe')                  //Instantiate administrator user object with login 'jdoe'
     * </code>
     *
     * @param string $login The user login
     * @param string $password An enrypted password to check for the user
     * @since 3.5.0
     * @access public
     */
    function __construct($employee) {

        $this -> employee  = $employee;
        if ($this -> employee['users_login']) {
            $this -> login = $this -> employee['users_login'];
        } else if ($this -> employee['login']) {
            $this -> login = $this -> employee['login'];
        } else {
            throw new EfrontUserException(_NOBRANCHSELECTED . " ", EfrontUserException :: WRONG_INPUT_TYPE);
        }
    }

}


class EfrontSupervisor extends EfrontEmployee
{
    /**
     * The branches that the supervisor supervises.
     *
     * @since 3.5.0
     * @var array
     * @access public
     */
     public $supervisesBranches = array();

    /**
     * Get employee type -- this function overrides the function of the parent class
     *
     * @return string The employee type
     * @since 3.5.0
     * @access public
     */
    public function getType() {
        return _SUPERVISOR;
    }

    /**
     * Check whether this employee is a supervisor
     *
     * @return boolean: is employee a supervisor?
     * @since 3.5.0
     * @access public
     */
    public function isSupervisor() {
        return true;
    }

    /**
     * Check whether this supervisor supervises employee with login = $login
     *
     * <br/>Example:
     * <code>
     * $supervisor = new EfrontSupervisor($employee, $supervisesBranches);
     * if ($supervisor -> supervisesEmployee('jdoe')) {...}     //If the employee with the login 'jdoe' works at any branch supervised by $supervisor then true
     * </code>
     * @return string The employee type
     * @since 3.5.0
     * @access public
     */
    public function supervisesEmployee($login) {
        // Find all branches where employee $login works
        $works_at_branches = eF_getTableDataFlat("module_hcd_employees LEFT OUTER JOIN module_hcd_employee_works_at_branch ON module_hcd_employee_works_at_branch.users_login = module_hcd_employees.users_login", "branch_ID", "module_hcd_employee_works_at_branch.users_login = '".$login."'");

        // If the employee works at no branch yet (unassigned) or works at one of the branches supervised by this supervisor, return true
        if (empty($works_at_branches)) {
            return true;
        } else {
            $works_for_my_branches = array_intersect($works_at_branches['branch_ID'], $this -> supervisesBranches);
            if (!empty($works_for_my_branches)) {
                return true;
            } else {
                return false;
            }
        }
    }


    /**
     * Check whether this supervisor supervises the branch with branch_ID = $branch_ID
     *
     * <br/>Example:
     * <code>
     * $supervisor = new EfrontSupervisor($employee, $supervisesBranches);
     * if ($supervisor -> supervisesBranch(21)) {...}     //If the $supervisor supervises the branch with id 21
     * </code>
     * @param The branch id
     * @return boolean
     * @since 3.5.0
     * @access public
     */
    public function supervisesBranch($branch_ID) {

        // If the employee works at no branch yet (unassigned) or works at one of the branches supervised by this supervisor, return true
        if (!empty($this -> supervisesBranches) && in_array($branch_ID, $this -> supervisesBranches)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Get branches that are supervised by this supervisor
     *
     * <br/>Example:
     * <code>
     * $supervisor = new EfrontSupervisor($employee, $supervisesBranches);
     * if ($supervisor -> supervisesBranch(21)) {...}     //If the $supervisor supervises the branch with id 21
     * </code>
     *
     * @return array with supervised branches in the form [branch_ID,name,father_branch_ID]
     * @since 3.5.0
     * @access public
     */
    public function getSupervisedBranches() {
        return eF_getTableData("module_hcd_branch", "branch_ID, name, father_branch_ID","branch_ID IN (" . implode(',',$this -> supervisesBranches) . ")");
    }

    /**
     * Assign a job to the employee of type Supervisor: different logic than the assignment to a regular employee
     *
     * This function can be used to assign a job to the current employee.
     * <br/>Example:
     * <code>
     * $user = EfrontUserFactory :: factory('jdoe');
     * $employee = EfrontHcdUserFactory :: factory('jdoe');
     * $employee -> addJob($user,23,1,1,"Lead Programmer");  //Add job 23 ("Lead Programmer") at branch 1 and set this employee's role as SUPERVISOR
     * $employee -> addJob($user,12,2,0);                    // Add job 2 (no description given) at branch 2 and set this employee's role as EMPLOYEE
     * </code>
     *
     * @param $jobID the ID of the new job to be assigned
     * @param $branchID the branch where this job placements belongs to
     * @param boolean $position the employee's role for this job: 0-Employee, 1-Supervisor
     * @param EfrontUser $userAspect, the user aspect of the employee needed to find his lessons
     * @return an Item of class EfrontHcdUser. This is done, because the role might change the class where the employee belongs.
     * @since 3.5.0
     * @access public
     */
    public function addJob($userAspect, $jobID, $branchID, $position, $job_description = false) {
        if ($branchID == "0" || ($position != "0" && $position != "1") || ($jobID == "" || $jobID == "0")) {
           throw new EfrontUserException(_NOBRANCHSELECTED . " ", EfrontUserException :: WRONG_INPUT_TYPE);
        }

        if (!$this -> jobs) {
            $this -> getJobs();
        }

        if (!in_array($jobID, array_keys($this -> jobs))) {

            // If the supervisor is already assigned to the branch and already supervises it, then we have nothing to change
            if (! ($this -> isAssignedToBranch($branchID) && in_array($branchID, $this -> supervisesBranches) )) {

                if (in_array($branchID, $this -> supervisesBranches)) {
                    // The supervisor did already supervise this branch (as it being a subbranch o a supervising branch) but did not hold an actual position ther
                    eF_updateTableData("module_hcd_employee_works_at_branch", array("assigned" => '1') , "users_login = '". $this -> login ."' AND branch_ID = '". $branchID ."'");

                    // Keep the object cache consistent
                    /*
                    $job_at_branch = $this -> isAssignedToBranch($branchID);
                    $this -> jobs[$job_at_branch]["supervisor"] = 1;*/

                    $promotion = 0;
                } else {
                    // The supervisor did not supervise this branch before, this is a promotion

                    // If the supervisor was assigned to this branch but as an employee and not as a supervisor
                    if ($job_at_branch = $this -> isAssignedToBranch($branchID)) {

                        eF_updateTableData("module_hcd_employee_works_at_branch", array("supervisor" => $position) , "users_login = '". $this -> login ."' AND branch_ID = '". $branchID ."'");
                        $this -> jobs[$job_at_branch]["supervisor"] = 1;
                    } else {

                        // The supervisor didn't supervise the branch, nor was he assigned a job placement to it.
                        eF_insertTableData("module_hcd_employee_works_at_branch", array('users_login' => $this -> login, 'supervisor' => $position, 'assigned' => '1', 'branch_ID' => $branchID));
                    }
                    if ($position == 1) {
                        $promotion = 1;
                    }
                }

                // We need to propagate supervision rights to the subbranches of the assigned branch
                if ($promotion == "1") {
                    // The next two calls, create an array of all branches that need to be supervised by the supervisor
                    $all_branches = eF_getTableData("module_hcd_branch", "branch_ID, father_branch_ID","");
                    $branches_to_supervise = eF_subBranches($branchID, $all_branches);

                    // Create the SQL strings to manage the module_hcd_employee_works_at_branch table:
                    // insert where the supervisor is not assigned yet an entry with 'supervisor'=1 and assigned='0'
                    // update where the supervisor is assigned all entries with assigned='0'
                    if (!empty($branches_to_supervise)) {
                        $string_to_insert = "";
                        $branches_to_update = "";
                        $size = sizeof($branches_to_supervise);

                        for ($i = 0 ; $i < $size; $i++) {
                            // If the supervisor already supervises the subbranch then nothing needs to happen, otherwise...
                            if (!in_array($branches_to_supervise[$i], $this -> supervisesBranches )) {

                                // If he is already assigned to the branch, but did not supervise it, then add this branch to the $branches_to_update record
                                if ($job_at_branch = $this -> isAssignedToBranch($branches_to_supervise[$i])) {

                                    ($branches_to_update == "")? ($branches_to_update = $branches_to_supervise[$i]) : ($branches_to_update .= ", " . $branches_to_supervise[$i]);
                                    $this -> jobs[$job_at_branch]["supervisor"] = 1;
                                } else {
                                    // Otherwise the supervisor did not supervise or was assigned to that branch => insert a new record to module_hcd_employee_works_at_branch
                                    if ($string_to_insert != "") {
                                        $string_to_insert .= ", (";
                                    }
                                    $string_to_insert .=  "'" . $this -> login  . "', '1', '0', '" . $branches_to_supervise[$i] . "')";

                                }

                                $this -> supervisesBranches[$branches_to_supervise[$i]] = $branches_to_supervise[$i];
                            }

                        }

                        /* We insert multiple values and therefore we cannot use insertTableData */
                        if ($string_to_insert != "") {
                            eF_execute("INSERT INTO module_hcd_employee_works_at_branch VALUES (" . $string_to_insert);
                        }

                        if ($branches_to_update != "") {
                            eF_updateTableData("module_hcd_employee_works_at_branch", array("supervisor" => '1'), "users_login = '". $this -> login ."' AND branch_ID IN (" . $branches_to_update . ")");
                        }
                    }
                }
            } else {
                $position = 1;
            }

            // Update the jobs cache
            $bname = eF_getTableData("module_hcd_branch", "name, father_branch_ID","branch_ID = '".$branchID."'");
            if (!$job_description) {
                $jdescription = eF_getTableData("module_hcd_job_description", "description", "job_description_ID= '".$jobID."'");
                $job_description = $jdescription[0]['description'];
            }
            $this -> jobs[$jobID] = array("job_description_ID" => $jobID, "branch_ID" => $branchID, "name" => $bname[0]['name'],  "father_branch_ID" => $bname[0]['father_branch_ID'], "description" => $job_description, "supervisor" => $position);

            // Write to database the new job assignment: employee to job description
            $insert_job = array('users_login' => $this -> login, 'job_description_ID' => $jobID);
            if($ok = eF_insertTableData("module_hcd_employee_has_job_description", $insert_job)) {
                $message      = _OPERATIONCOMPLETEDSUCCESFULLY;
                $message_type = 'success';
                // Register job release into the event log
                eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['JOB'],
                                                              "users_login"   => $this -> login,
                                                              "author"        => $_SESSION['s_login'],
                                                              "specification" => _EMPLOYEEWASASSIGNEDJOB .": " .$job_description. " ". _ATBRANCH . " " .$bname[0]['name'],
                                                              "timestamp"     => time()));

                // Can be optimized in cost of programmability interface
                $lessons_to_add = eF_getTableData("module_hcd_lesson_to_job_description", "lessons_ID", "job_description_ID = '". $jobID ."'");
                $size = sizeof($lessons_to_add);

                if ($size) {
                    $currentLessonIds = array_keys($userAspect -> getLessons());

                    $at_least_one_new_lesson = 0;
                    $sql_query = "INSERT INTO users_to_lessons (users_LOGIN, lessons_ID, active, from_timestamp, user_type, positions, done_content, issued_certificate, comments) VALUES ";
                    for ($k = 0; $k<$size; $k++) {
                        if ( !in_array($lessons_to_add[$k]['lessons_ID'], $currentLessonIds) ) {
                            $at_least_one_new_lesson = 1;
                            $sql_query .= "('". $this -> login . "','" . $lessons_to_add[$k]['lessons_ID'] . "', '1', '".time()."', 'student','','','','')";
                            if ($k != $size-1) {
                                $sql_query = $sql_query . ",";
                            }

                            //$userAspect -> addLessons($lessons_to_add[$k]['lessons_ID']);
                        }

                    }

                    if ($at_least_one_new_lesson) {
                        $ok = eF_execute($sql_query);
                    }
                }

            } else {
               throw new EfrontUserException(_PLACEMENTCOULDNOTBEASSIGNED . " ", EfrontUserException :: DATABASE_ERROR);
            }

            return $this;

        } else {
           throw new EfrontUserException(_JOBALREADYASSIGNED . " ", EfrontUserException :: WRONG_INPUT_TYPE);
        }
    }


    /**
     * Remove a job from the supervisor
     *
     * This function can be used to remove a job to the current employee.
     * <br/>Example:
     * <code>
     * $user = EfrontHcdUserFactory :: factory('jdoe');
     * $user -> addJob(12,2,0);                    // Add job 12 at branch 2
     * $user -> removeJob(12);                     // Removejob 12
     * </code>
     *
     * @param $jobID the ID of the job to be removed
     * @return an Item of class EfrontHcdUser. This is done, because the when the last supervisor job is removed then the employee changes class.
     * @since 3.5.0
     * @access public
     */
    public function removeJob($jobID) {

        if ($jobID == "" || $jobID == "0") {
           throw new EfrontUserException("", EfrontUserException :: WRONG_INPUT_TYPE);
        }

        if (!$this -> jobs) {
            $this -> getJobs();
        }

        if ($this -> jobs[$jobID]['supervisor'] == 0) {
            parent::removeJob($jobID);
            return $this;
        } else {
            $job_to_delete = $this -> jobs[$jobID];
            unset($this -> jobs[$jobID]);

            // If no other job exists in that branch, then delete the record from module_hcd_employee_works_at_branch
            $another_job_at_branch = 1;
            if (sizeof($this->jobs) == 0 || !($another_job_at_branch = $this -> isAssignedToBranch($job_to_delete['branch_ID']))) {


                // if no other job exists, then the supervisor might only remain his role is he is supervising
                // a branch higher in the branch hierarchy. This means that we need to check if he is supervising the father branch
                if (!in_array($job_to_delete['father_branch_ID'], $this -> supervisesBranches)) {

                    eF_deleteTableData("module_hcd_employee_works_at_branch", "users_login='".$this -> login."' AND branch_ID ='". $job_to_delete['branch_ID'] ."'" );

                    // If the control flow reaches here then: the supervisor does not have another placement in this branch nor does he
                    // supervise the father branch. This means that his supervisor rights must be revoked for all branches between this
                    // one and the next highest branches in the hiearchy where he has jobs assigned.

                    /* If we reach this point then all privileges to the branches below need to be recalled */
                    $all_branches = eF_getTableData("module_hcd_branch", "branch_ID, father_branch_ID", "");

                    $branches_to_recall_supervision = eF_subBranches($job_to_delete['branch_ID'], $all_branches);
                    array_push($branches_to_recall_supervision, $job_to_delete['branch_ID']);
                    $strictly_supervises_branches = eF_getTableDataFlat("module_hcd_employee_works_at_branch", "branch_ID", "supervisor = '1' and assigned = '1' and users_login = '".$this -> login."'");


                    // Get all branches that are strictly supervised by the supervisor i.e. not supervised due to the supervision of father branches
                    $strictly_supervises_branches = array();
                    foreach ($this -> jobs as $job) {
                        if ($job['supervisor'] == '1') {
                            $strictly_supervises_branches[] = $job['branch_ID'];
                        }
                    }

                    $i = 0;
                    $branches_to_keep = array();
                    foreach($strictly_supervises_branches as $strict_branch) {
                        if($strict_branch != $job_to_delete['branch_ID'] && eF_isSubbranch($strict_branch, $job_to_delete['branch_ID'], $all_branches)) {
                            $results = eF_subBranches($strict_branch, $all_branches);
                            $branches_to_keep[$i++] = $strict_branch;
                            foreach ($results as $another_branch_to_keep) {
                                $branches_to_keep[$i++] = $another_branch_to_keep;
                            }
                        }
                    }

                    $branches_to_recall_supervision_list = implode("','", $branches_to_recall_supervision);

                    $branches_to_keep_list = implode("','", $branches_to_keep);

                    // FINALLY: delete all subbranches that exist in the branches_to_recall_supervision list and that do not exist in the branches_to_keep_list list
                    $string_to_insert = "DELETE FROM module_hcd_employee_works_at_branch WHERE supervisor='1' AND assigned = '0' AND users_login = '".$this -> login."'";
                    if (!empty($branches_to_recall_supervision)) {
                        $string_to_insert = $string_to_insert . " AND branch_ID IN ('" . $branches_to_recall_supervision_list . "')";
                    }
                    if (!empty($branches_to_keep)) {
                        $string_to_insert = $string_to_insert . " AND branch_ID NOT IN ('" . $branches_to_keep_list . "')";
                    }


                    // Recall supervision rights
                    foreach ($branches_to_recall_supervision as $recall_branches) {
                        unset($this -> supervisesBranches[$recall_branches]);
                    }


                    //echo  $string_to_insert. "<br>";
                    /* We delete multiple values and therefore we cannot use deleteTableData */
                    // At least one record is always deleted - no security needed
                    eF_execute($string_to_insert);

                } else {
                    // No need to change supervision rights, since the father branch remains supervised, just set the assigned flag to this branch equal to zero
                    eF_updateTableData("module_hcd_employee_works_at_branch", array("assigned" => "0") , "users_login='".$this -> login."' AND branch_ID ='". $job_to_delete['branch_ID']  ."' AND supervisor = '1' AND assigned = '1'" );
                }
            }


            eF_deleteTableData("module_hcd_employee_has_job_description", "users_login='".$this -> login."' AND job_description_ID ='".$jobID."'");

            $message      = _EMPLOYEERELEASEDFROMJOB;
            $message_type = 'success';

            // Register job release into the event log
            eF_insertTableData("module_hcd_events", array("event_code"    => $GLOBALS['MODULE_HCD_EVENTS']['JOB'],
                                                          "users_login"   => $this -> login,
                                                          "author"        => $_SESSION['s_login'],
                                                          "specification" => _EMPLOYEEWASRELEASEDFROMDJOB .": " .$job_to_delete['description']. " ". _ATBRANCH . " " .$job_to_delete['name'],
                                                          "timestamp"     => time()));

        }

        if (empty($this -> supervisesBranches)) {
            $employee = new EfrontEmployee($this -> employee);
            $employee -> jobs = $this -> jobs;
            $employee -> skills = $this -> skills;
            return $employee;
        } else {
            return $this;
        }
    }


    /**
     * Instantiate class
     *
     * This function instantiates a new EfrontSupervisor object based on the given
     * employee array and an array of the branches to be supervised. Either the EfrontUserFactory may be used, or directly the
     * EfrontX class.
     * <br/>Example:
     * <code>
     * $employee_array = eF_getTableData("module_hcd_employee", "*", "users_login = 'jdoe'");
     * $user = EfrontUserFactory :: factory($employee_array);            //Use factory to instantiate user object with login 'jdoe'
     * $user = new EfrontSupervisor()                  //Instantiate administrator user object with login 'jdoe'
     * </code>
     *
     * @param string $login The user login
     * @param string $password An enrypted password to check for the user
     * @since 3.5.0
     * @access public
     */
    function __construct($employee, $sbranches) {
        parent::__construct($employee);

        foreach($sbranches as $sbranch) {
            $this -> supervisesBranches[$sbranch] = $sbranch;
        }
    }

}



class EfrontHcdAdministrator extends EfrontEmployee
{

    /**
     * Get employee type
     *
     * @return string The employee type
     * @since 3.5.0
     * @access public
     */
    public function getType() {
        return "administrator";
    }


    /**
     * Get branches that are supervised by the administrator: ALL BRANCHES
     *
     * <br/>Example:
     * <code>
     * $supervisor = new EfrontSupervisor($employee, $supervisesBranches);
     * if ($supervisor -> supervisesBranch(21)) {...}     //If the $supervisor supervises the branch with id 21
     * </code>
     *
     * @return array with supervised branches in the form [branch_ID,name,father_branch_ID]
     * @since 3.5.0
     * @access public
     */
    public function getSupervisedBranches() {
        return eF_getTableData("module_hcd_branch", "branch_ID, name, father_branch_ID", "");
    }


    /**
     * Instantiate class
     *
     */
    function __construct($employee) {
        parent::__construct($employee);
    }

}


/**
 *
 */
class EfrontEmployeeFactory
{
    /**
     * Construct user object
     *
     * This function is used to construct a user object, based on the user type.
     * Specifically, it creates a EfrontSupervisor or an EfrontEmployee.
     * $user is a login name and the function queries the database for an employee with that name
     * <br/>Example :
     * <code>
     * $user = EfrontEmployeeFactory :: factory('jdoe');            //Use factory function to instantiate user object with login 'jdoe'
     * $userData = eF_getTableData("users", "*", "login='jdoe'");
     * $user = EfrontEmployeeFactory :: factory($userData[0]);      //Use factory function to instantiate user object using prepared data
     * </code>
     *
     * @param mixed $user A user login or an array holding user data
     * @param string $password An optional password to check against
     * @return EfrontUser an object of a class extending EfrontUser
     * @since 3.5.0
     * @access public
     * @static
     */
    public static function factory($employee, $password = false) {

        // The following code overlaps with EfrontUserFactory: needed here for Object independence - should it be removed?
        if (eF_checkParameter($employee, 'login')) {
            $user = eF_getTableData("users", "*", "login='".$employee."'");

            if (sizeof($user) == 0) {
                throw new EfrontUserException(_USERDOESNOTEXIST.': '.$employee, EfrontUserException :: USER_NOT_EXISTS);
            } else if ($password != false && $password != $employee[0]['password']) {
                throw new EfrontUserException(_INVALIDPASSWORDFORUSER.': '.$user[0]['login'], EfrontUserException :: INVALID_PASSWORD);
            }
            $user = $user[0];
        }

        /*** Get the employee's data ***/
        $employee_array = eF_getTableData("module_hcd_employees", "*", "users_login='".$employee."'");


        if (sizeof($employee_array) == 0) {
            $employee = array ("users_login" => $employee);
            // Create a new
            eF_insertTableData("module_hcd_employees", $employee);
        } else {
            $employee = $employee_array[0];
        }


        if ($user['user_type'] != "administrator") {
            /*** Get the branches this employee supervises and depending on the result create a EfrontSupervisor or a EfrontEmployee instance ***/
            $supervisor_at_branches = eF_getTableDataFlat("module_hcd_employee_works_at_branch", "branch_ID","users_login = '" .$employee['users_login']. "' AND supervisor='1'");
            if (!empty($supervisor_at_branches)) {
                $factory = new EfrontSupervisor($employee, $supervisor_at_branches['branch_ID']);
            } else {
                $factory = new EfrontEmployee($employee);
            }
        } else {
            $factory = new EfrontHcdAdministrator($employee);
        }

        return $factory;
    }
}



?>