<?php
/**
* Efront System Classes file
*
* @package efront
* @version 3.5.0
*/

define("_INCOMPATIBLEVERSIONS", "Incompatible versions");
define("_BACKUPVERSION", "Backup version");
define("_CURRENTVERSION", "Current Version");

class EfrontException extends Exception
{
	
	public function toHTML() {
		
	}
}

class EfrontSystemException extends Exception
{
	const INCOMPATIBLE_VERSIONS = 10; 
}

/**
 * System class
 * 
 * This class incorporates system-wise static functions
 *
 * @since 3.5.0
 * @package eFront
 */
class EfrontSystem 
{
	/**
	 * Backup system
	 * 
	 * This function is used to backup the system. There are 2 types of backup, database only and full.
	 * In the first case (the default), only the database is backed up, while in the second case, files 
	 * are backed up as well.
	 * <br/>Example:
	 * <code>
	 * $backupFile = EfrontSystem :: backup('13_3_2007');			//Backup database only 
	 * </code>
	 *
	 * @param string $backupName The name of the backup 
	 * @param int $backupType Can be either 0 or 1, where 0 siginifies database only backup and 1 is for including backup files as well
	 * @return eF_File The compressed file of the backup
	 * @since 3.5.0
	 * @access public
	 */
	public static function backup($backupName, $backupType = 0) {
		try {														//Delete old backup directories that may be left
			$directory = new eF_Directory(G_BACKUPPATH.'db_backup');
			$directory -> delete();
		} catch (Exception $e) {}
		$directory = EfrontFileSystem :: makeDirectory('db_backup', G_BACKUPPATH);
		$tables    = $GLOBALS['db'] -> GetCol("show tables");                                              //Get the database tables
		
		foreach ($tables as $table) {
			$data[$table] = eF_getTableData($table);
			if ($table == 'files') {
				foreach ($data['files'] as &$value) {										//Remove system-specific information
					$value['file'] 		= str_replace(G_ROOTPATH, '', $value['file']);
					$value['directory'] = str_replace(G_ROOTPATH, '', $value['directory']);
				}
			}
			file_put_contents($directory -> getFullPath().$table, serialize($data[$table]));
			$result       = eF_ExecuteNew("show create table $table");
			$temp         = $result -> GetAll();
			$definition[] = "drop table ".$temp[0]['Table'];
			$definition[] = $temp[0]['Create Table'];
		}
		file_put_contents($directory -> getFullPath().'sql.txt', implode(";\n", $definition));
		file_put_contents($directory -> getFullPath().'/version.txt', G_VERSION_NUM);

		$compressedFile = $directory -> compress('tar.gz');
		$compressedFile -> rename($backupName.'.tar.gz');
		$directory -> delete();
		if ($backupType == 1) {
			$compressedFile -> addDirectory(G_LESSONSPATH);
			$compressedFile -> addDirectory(G_UPLOADPATH);
			$compressedFile -> addDirectory(G_AVATARSPATH);
		}
		
		return $compressedFile;
	}
	
	/**
	 * Restore system
	 * 
	 * This function is used to restore a backup previously taken
	 * <br/>Example:
	 * <code>
	 * </code> 
	 *
	 * @param unknown_type $restoreFile
	 */
	public static function restore($restoreFile) {
		if (!($restoreFile instanceof eF_File)) {
			$restoreFile = new eF_compressedFile($restoreFile);
		}
//error_reporting(E_ALL);$GLOBALS['db']->debug=true;
		$backupFiles = eF_getTableData("files", "*", "directory='".G_BACKUPPATH."'");		//This guards the current backup files from the restore
		if (is_dir(G_BACKUPPATH.'temp')) {
			$tempDir = new eF_Directory(G_BACKUPPATH.'temp');
			$tempDir -> deleteFS();
		}
		mkdir(G_BACKUPPATH.'temp', 0755);

		$restoreFile = $restoreFile -> copy(G_BACKUPPATH.'temp/');
		$restoreFile -> uncompressFS();
		$tempDir 	   = new eF_Directory(G_BACKUPPATH.'temp/');
		$dirContents   = $tempDir -> getContentsFs();
		foreach ($dirContents as $value) {
			if (strpos($value, '/version.txt') !== false) {
				$backupVersion = file_get_contents($value);
			}
		}

		if (version_compare($backupVersion, G_VERSION_NUM) != 0) {
			throw new Exception (_INCOMPATIBLEVERSIONS.'<br/> '._BACKUPVERSION.':'.$backupVersion.' / '._CURRENTVERSION.': '.G_VERSION_NUM, EfrontSystemException::INCOMPATIBLE_VERSIONS);
		}
		$newdir = new eF_Directory(G_BACKUPPATH.'temp/db_backup/');
		
		$sql = file_get_contents($newdir -> getFullPath().'/sql.txt');
		$sql = explode(";", $sql);

		foreach ($sql as $query) {
			eF_executeNew($query);
		}
		$contents = $newdir -> getContentsFS();

		foreach ($contents as $file) {
			if (strpos($file, 'sql.txt') === false && strpos($file, 'version.txt') === false) {			//These 2 files do not contain serialized database information
				$data = unserialize(file_get_contents($file));
				if (basename($file) == 'files') {
					for ($i = 0; $i < sizeof($data); $i++) {
						$data[$i]['directory'] = G_ROOTPATH.$data[$i]['directory'];
						$data[$i]['file'] 	   = G_ROOTPATH.$data[$i]['file'];
					}
				}
				eF_insertTableDataMultiple(basename($file), $data);
			}
		}

		if (is_dir(G_BACKUPPATH.'temp/upload')) {
			try {
				$uploadDir = new eF_Directory(G_UPLOADPATH);
				$uploadDir -> deleteFS();
			} catch (EfrontFileException $e) {}
			$uploadDirBackup = new eF_Directory(G_BACKUPPATH.'temp/upload');
			$uploadDirBackup -> copyFS(G_ROOTPATH, false, true);
		}
		if (is_dir(G_BACKUPPATH.'temp/lessons')) {
			try {
				$lessonsDir = new eF_Directory(G_LESSONSPATH);
				$lessonsDir -> deleteFS();
			} catch (EfrontFileException $e) {}
			$lessonsDirBackup = new eF_Directory(G_BACKUPPATH.'temp/lessons');
			$lessonsDirBackup -> copyFS(G_ROOTPATH.'www/content/', false, true);
		}
		if (is_dir(G_BACKUPPATH.'temp/avatars')) {
			try {
				$avatarsDir = new eF_Directory(G_AVATARSPATH);
				$avatarsDir -> deleteFS();
			} catch (EfrontFileException $e) {}
			$avatarsDirBackup = new eF_Directory(G_BACKUPPATH.'temp/avatars');
			$avatarsDirBackup -> copyFS(G_ROOTPATH.'www/images/', false, true);
		}

		foreach ($backupFiles as $file) {
			$result = eF_getTableData("files", "count(*)", "id=".$file['id']);
			if ($result[0]['count(*)'] == 0) {
				eF_insertTableData("files", $file);
			}
		}
		if (is_dir(G_BACKUPPATH.'temp')) {
			$tempDir = new eF_Directory(G_BACKUPPATH.'temp');
			$tempDir -> deleteFS();
		}
			
		return true;
	}
	 
	
}


?>