<?php

/**
* @package efront
*/

/**
 * Format timestamp
 * 
 * This function is used to convert the given timestamp into 
 * human-readable format. The order is derived from the 'date_format' 
 * configuration option. 
 * <br/>Example:
 * <code>
 * $timestamp = time();
 * $dateString = formatTimestamp($timestamp);					//Returns something like 20 May 2008
 * $dateString = formatTimestamp($timestamp, 'time');			//Returns something like 11:50:23, 20 May 2008
 * </code>
 * The function is the equivalent of the "formatTimestamp" smarty filter
 *
 * @param int $timestamp The timestamp to convert
 * @param string $mode The mode to use. currently only 'time' is available
 * @return string The date in human-readable format
 */
function formatTimestamp($timestamp, $mode = false) {
    switch ($GLOBALS['configuration']['date_format']) {
        case "YYYY/MM/DD": $format = '%Y %b %d'; break;
        case "MM/DD/YYYY": $format = '%b %d %Y'; break;
        case "DD/MM/YYYY": default: $format = '%d %b %Y'; break;
    }
    
    switch ($mode) {
        case 'time': $format .= ', %H:%M:%S'; break;
        default: break;
    }
    $dateString = iconv(_CHARSET, 'UTF-8', strftime($format, $timestamp));
    
    return $dateString;
}
/**
*
*/
function eF_multiSort($array, $sort_by, $sort_order = 'asc') {
    if (!in_array($sort_by, array_keys(current($array)))) {
        return $array;
    }

    $keys        = array_keys($array);
    $sort_values = array();
    foreach ($array as $value) {
        $sort_values[] = $value[$sort_by];
    }
    if (is_numeric($sort_values[0])) {    //If the column consists of numeric data, we want the default sorting to be descending, so we reverse the parameter
            $sort_order == 'asc' ? $sort_order = SORT_DESC : $sort_order = SORT_ASC;
    } else {
            $sort_order == 'asc' ? $sort_order = SORT_ASC : $sort_order = SORT_DESC;
    }

    array_multisort($sort_values, $sort_order, $keys);
    foreach ($keys as $key) {
        $temp[] = $array[$key];
    }

    return $temp;
}

/**
* Get calendar events
*
* This function is used to retrieve calendar events for the current lesson.
* If a timestamp is specified, then only events for the day that this timestamp
* belongs to are returned. If timestamp is omitted, all events for the current
* lesson are returned
*
* @param int $type 0,1 or 2 corresponding Organization profile, Current lesson, All lessons calendar events
* @param int $timestamp The timestamp
* @return array the list of events on that day
* @version 1.0
*/
function eF_getCalendar($timestamp = false, $type = 1) {

    //$lessons_ID = $_SESSION['s_lessons_ID'];

    // The type is going to define the lessons array - for backwards compatibility
    // if no type is defined then events regarding all lessons are to be returned
    if ($type == 0) {
        $lessons = array("0" => "0");
    } else if ($type == 2) {
        $lessons = array("0" => $_SESSION['s_lessons_ID']);
    } else {
        $login = $_SESSION['s_login'];
        $lessons = array();

        $tmp = eF_getTableData("users_to_lessons", "lessons_ID", "users_LOGIN='".$login."'");
        for ($i = 0; $i < sizeof($tmp); $i++)
        {
            $lessons[] = $tmp[$i]['lessons_ID'];
        }
    }
    $l = implode(",", $lessons);

    if (!$timestamp || !eF_checkParameter($timestamp, 'timestamp')) {

        if (MODULE_HCD_INTERFACE) {
            $result = eF_getTableData("calendar c LEFT OUTER JOIN lessons l ON c.lessons_ID = l.ID", "c.id, c.timestamp, c.data, l.name", "c.lessons_ID in (".$l.") AND c.active=1", "timestamp");
        } else {
            $result = eF_getTableData("calendar c LEFT OUTER JOIN lessons l ON c.lessons_ID = l.ID", "c.id, c.timestamp, c.data, l.name", "c.lessons_ID in (".$l.") AND c.active=1", "timestamp");

//            $result = eF_getTableData("calendar c, lessons l", "c.id, c.timestamp, c.data, l.name", "c.lessons_ID in (".$l.") AND c.active=1 AND c.lessons_ID = l.ID", "timestamp");
        }
    } else {

        $timestamp_info = getdate($timestamp);
        $timestamp_from = mktime(0, 0, 0, $timestamp_info['mon'], $timestamp_info['mday'], $timestamp_info['year']);     //today first sec
        $timestamp_to   = mktime(23, 23, 59, $timestamp_info['mon'], $timestamp_info['mday'], $timestamp_info['year']);  //today last sec
        if (MODULE_HCD_INTERFACE) {
            $result = eF_getTableData("calendar c LEFT OUTER JOIN lessons l ON c.lessons_ID = l.ID", "c.id, c.timestamp, c.data, l.name", "c.lessons_ID in (".$l.") AND c.active=1 AND timestamp >= ".($timestamp_from)." AND timestamp <= ".($timestamp_to), "timestamp");
        } else {
            $result = eF_getTableData("calendar c LEFT OUTER JOIN lessons l ON c.lessons_ID = l.ID", "c.id, c.timestamp, c.data, l.name", "c.lessons_ID in (".$l.") AND c.active=1 AND timestamp >= ".($timestamp_from)." AND timestamp <= ".($timestamp_to), "timestamp");

//            $result = eF_getTableData("calendar c, lessons l", "c.id, c.timestamp, c.data, l.name", "c.lessons_ID in (".$l.") AND c.active=1 AND c.lessons_ID = l.ID AND timestamp >= ".($timestamp_from)." AND timestamp <= ".($timestamp_to), "timestamp");
        }
    }

    foreach ($result as $event) {
        $events[$event['timestamp']]['data'][] = $event['data'];
        $events[$event['timestamp']]['id'][]   = $event['id'];
        $events[$event['timestamp']]['lesson'][]   = $event['name'];
    }

    return $events;
}

/**
* Get status for lessons in course
*
* This function checks for the given course and the given student, which lessons
* he has the right to attend based on the course rules
* <br/>Example:<br/>
* <code>
* $status = eF_getCourseRulesPassed(5, array(2, 3, 4), 'student');
* //returns:
* Array
* (
*     [2] => 1
*     [3] => 0
*     [4] => 1
* )
* </code>
*
* @param int $course_id The course id
* @param mixed $lessons Either a single lesson id, or an array of lesson ids
* @param string $users_LOGIN The user to check access for
* @return array An array where keys are lesson ids and values are 0 or 1, depending on whether he may have access to this lesson
* @version 1.0
* @author Venakis Periklis
* @date 2007/26/09
*/
function eF_getCourseRulesPassed($course_id, $lessons, $users_LOGIN) {
    if (!is_array($lessons)) {                                       //Convert to array, if a single value is given
        $lessons = array($lessons);
    }

    //$user_lessons = eF_getTableDataFlat("users_to_lessons", "lessons_ID, completed", "users_LOGIN='".$users_LOGIN."'");
    $user_lessons = eF_getTableDataFlat("users_to_lessons, lessons", "lessons_ID, completed, lessons.name", "lessons.id=users_to_lessons.lessons_ID and users_LOGIN='".$users_LOGIN."'");
    $lesson_names = array_combine($user_lessons['lessons_ID'], $user_lessons['name']);
    $user_lessons = array_combine($user_lessons['lessons_ID'], $user_lessons['completed']);
    $course       = eF_getTableData("courses", "rules, lessons", "id=$course_id");
    foreach ($lessons as $lesson_id) {
        $course_lessons = unserialize($course[0]['lessons']);
        $course_rules   = unserialize($course[0]['rules']);

        if (in_array($lesson_id, $course_lessons) && isset($course_rules[$lesson_id])) {
            $rule = $course_rules[$lesson_id];

            $rule_string = '';
            foreach ($rule['lesson'] as $key => $lesson) {
                $condition = '';
                $condition_string = '';
                if (isset($rule['condition'][$key + 1])) {
                    if ($rule['condition'][$key + 1] == 'and') {                                        //Construct the bit sequence, for example 1 & 0 & 0 | 1
                        $condition        = '&';
                        $condition_string = _AND;
                    } else if ($rule['condition'][$key + 1] == 'or') {
                        $condition        = '|';
                        $condition_string = _OR;
                    }
                }
                $rule_string .= $user_lessons[$lesson].$condition;
                $lesson_permitted[$lesson_id]['conditions'] .= $lesson_names[$lesson]." ".$condition_string." ";
            }

            eval("\$lesson_permitted[\$lesson_id]['selectable'] = $rule_string;");                                    //Evaluate bit sequence

        } else {
            $lesson_permitted[$lesson_id]['selectable'] = 1;
        }
    }

    return $lesson_permitted;
}

/**
* Prints a lessons list box
*
* This function prints a list box options containing all the lessons of the system, of the specified language. If no
* language is specified, the current language is used. If $category is specified, then all lesson that belong to the
* specified category are printed.
* <br/>Example:
* <code>
* print eF_printSelectLessons('greek');                            //Prints a list box with all the greek lessons
* print eF_printSelectLessons('', 'Category 1');                   //Prints a list box with all the lessons that belong to "Category 1"
* print eF_printSelectLessons();                                   //Prints a list box with all the lessons of the current language (specified in session)
* </code>
*
* @param string $language The lessons language
* @param id $category The lessons category id
* @param string $selected The selected option
* @return string The HTML code of the list
* @version 1.0
* @todo separate lessons per category
*/
function eF_printSelectLessons($language = false, $category = false, $selected = false)
{
    $lessons = eF_getLessons($language, $category);                         //Get the list of users

    $lessons_str = '';
    for ($i = 0; $i < sizeof($lessons); $i++) {
        ($selected && $lessons[$i]['id'] == $selected) ? $selected_str = 'selected' : $selected_str = '';
        $lessons_str .= '
            <option value = "'.$lessons[$i]['id'].'" '.$selected_str.'>'.$lessons[$i]['name'].'</option>';
    }

    return $lessons_str;
}


/**
* Gets system lessons
*
* This function returns an array containing all the system lessons of the specified language, or the current language.
* If $category is specified, then only lessons that belong to this category are returned
*
* @param string $language The lessons language
* @param id $category The lessons category id
* @return string The HTML code of the list
* @version 1.0
* @todo separate lessons per category
*/
function eF_getLessons($language = false, $category = false)
{
    if (!$language || !in_array($language, eF_getLanguages())) {
        $language = $_SESSION['s_language'];
    }

    if (!$category) {
        $lessons = eF_getTableData("lessons", "*", "languages_NAME='$language'");
    } else {
        $lessons = eF_getTableData("lessons, directions", "*", "languages_NAME='$language' AND directions_ID=directions.id");
    }

    return $lessons;
}


/**
* Prints a users list box
*
* This function prints a list box options containing all the users of the specified lesson and type, or everyone
* if no arguments are passed.
* <br/>Example:
* <code>
* print eF_printSelectUsers('demo_lesson', 'student');           //Prints a list box with all the students enrolled in 'demo_lesson'
* print eF_printSelectUsers('', 'administrator');                //Prints a list box with all the users of type 'admnistrator'
* print eF_printSelectUsers();                                   //Prints a list box with all the system users
* </code>
* If $onlyactive parameter is set, then only active users are returned.
*
* @param int $lessons_ID The lesson id
* @param string $user_type The user type
* @param bool $onlyactive If specified, only active users are returned
* @return string The HTML code of the list
* @version 0.1
* @todo if an argument is specified, separate user types
*/
function eF_printSelectUsers($lessons_ID = '', $user_type = '', $onlyactive = false)
{
    $users = eF_getUsers($lessons_ID, $user_type, $onlyactive);                         //Get the list of users

    $users_str = '';
    foreach ($users['login'] as $value) {
        $users_str .= '
            <option value = "'.$value.'">'.$value.'</option>';
    }

    return $users_str;
}

/**
* Gets an array with the specified users
*
* This function returns an array containing all the users of the specified lesson and type, or everyone
* if no arguments are passed.
* If $onlyactive parameter is set, then only active users are returned.
*
* @param int $lessons_ID The lesson id
* @param string $user_type The user type
* @param bool $onlyactive If specified, only active users are returned
* @return array The users array
* @version 0.1
* @todo if an argument is specified, separate user types

*
* changes for version 3  25/5/2007
* NOW user_type is specified in users_to_lessons table or if it is NULL there, in users table
*
*/
/*backup for a while
function eF_getUsers($lessons_ID = '', $user_type = '', $onlyactive = false)
{
    $onlyactive ? $active_sql = ' AND users.active = 1' : $active_sql = '';

    if ($lessons_ID != '') {
        if ($user_type != '') {
            $users = eF_getTableDataFlat("users_to_lessons, users", "users.login", "users_to_lessons.users_LOGIN=users.login AND users.user_type = '$user_type' AND users_to_lessons.lessons_ID=$lessons_ID$active_sql");         //Get all users enrolled in a lesson, of type $user_type
        } else {
            $users = eF_getTableDataFlat("users_to_lessons, users", "users.login", "users_to_lessons.users_LOGIN=users.login AND users_to_lessons.lessons_ID=$lessons_ID$active_sql");                                            //Get all users enrolled in a lesson
        }
    } else {
        if ($user_type != '') {
            $users = eF_getTableDataFlat("users", "login", "user_type = '$user_type'$active_sql");          //Get all users of the specified type.
        } else {
            if ($onlyactive) {
                $users = eF_getTableDataFlat("users", "login", "active = 1");                    //Get all active users.
            } else {
                $users = eF_getTableDataFlat("users", "login");                                  //Get all users.
            }
        }
    }

    return $users;
}*/
function eF_getUsers($lessons_ID = '', $user_type = '', $onlyactive = false)
{
    $onlyactive ? $active_sql = ' AND users.active = 1' : $active_sql = '';
    //for a spesific basic_user_type we need also subtypes
    if ($user_type != '') {
        $temp[] = "'".$user_type."'";
        $user_types = eF_getTableData("user_types", "user_type", "basic_user_type='$user_type'");
        for($k=0; $k < sizeof($user_types); $k++){
            $temp[] = "'".$user_types[$k]['user_type']."'";

        }
        $types = implode(",", $temp);

    }
    if ($lessons_ID != '') {
        if ($user_type != '') {
            $users1 = eF_getTableDataFlat("users_to_lessons, users", "users.login", "users_to_lessons.users_LOGIN=users.login AND users_to_lessons.user_type IN ($types) AND users_to_lessons.lessons_ID=$lessons_ID$active_sql"); //user_type from users_to_lessons
            $users2 = eF_getTableDataFlat("users_to_lessons, users", "users.login", "users_to_lessons.users_LOGIN=users.login AND users.user_type IN ($types) AND users_to_lessons.user_type IS NULL AND users_to_lessons.lessons_ID=$lessons_ID$active_sql"); //user_type from users
            $users = array_merge($users1, $users2);
            //pr($users);
        } else {
            $users = eF_getTableDataFlat("users_to_lessons, users", "users.login", "users_to_lessons.users_LOGIN=users.login AND users_to_lessons.lessons_ID=$lessons_ID$active_sql");                                            //Get all users enrolled in a lesson
        }
    } else {
        if ($user_type != '') {
            if ($user_type != "administrator") { //administrator user_type only from users table
                $users1 = eF_getTableDataFlat("users,users_to_lessons", "login", "users_to_lessons.users_LOGIN=users.login AND users_to_lessons.user_type IN ($types) $active_sql"); //user_type from users_to_lessons
                $users2 = eF_getTableDataFlat("users,users_to_lessons", "login", "users_to_lessons.users_LOGIN=users.login AND users_to_lessons.user_type IS NULL AND users.user_type IN ($types) $active_sql"); //user_type from users
                $users['login'] = array_merge($users1['login'], $users2['login']);
            } else {
                $users = eF_getTableDataFlat("users", "login", "user_type IN ($types) $active_sql");
            }
        } else {
            if ($onlyactive) {
                $users = eF_getTableDataFlat("users", "login", "active = 1");                    //Get all active users.
            } else {
                $users = eF_getTableDataFlat("users", "login");                                  //Get all users.
            }
        }
    }

    return $users;
}
/**
* Prints dates list boxes
*
* This function prints 3 list boxes (drop-down menu) used to select dates. There are 3 list boxes,
* one for each of day, month and year. The function accepts many arguments: $prefix is a string that
* is prefixed on the select name, so that the list boxes are named $prefixDay, $prefixMonth and $prefixYear
* respectively. $timestamp is the current date. The $selected_timestamp corresponds to the default date displayed
* on the list boxes and if ommited defaults to now. Finally, printDay, printMonth and printYear are boolean
* variables that set whether the corresponding element will be visible. They default to true.
* <br/>Example:
* <code>
* print eF_selectDate('selected_', time(), time()-86400);
* </code>
* The function returns the equivalent HTML code.
*
* @param string $prefix The prefix of the select boxes names
* @param int $timestamp The current timestamp
* @param int $selected_timestamp The timestamp corresponding to the date displayed by default on the list boxes
* @param bool $printDay If set to true, the Day list box is displayed
* @param bool $printMonth If set to true, the Month list box is displayed
* @param bool $printYear If set to true, the Year list box is displayed
* @return string The HTML code corresponding to the list boxes
* @version 1.0
* @todo The variable $timestamp is useless, we can calculate current time inside the function
* @todo maybe $MONTH global variable should be replaced...
* @todo print
*/
function eF_selectDate($prefix, $timestamp, $selected_timestamp = "", $printDay = true, $printMonth = true, $printYear = true)
{
    global $MONTH;

    if ($selected_timestamp == "") {
        $selected_timestamp = time();
    }

    $year        = strftime("%Y", $timestamp);
    $selectday   = strftime("%d", $selected_timestamp);
    $selectmonth = strftime("%m", $selected_timestamp);
    $selectyear  = strftime("%Y", $selected_timestamp);

    if ($printDay) {
        $str = '
            <select name = "'.$prefix.'Day">';
        for ($i = 1; $i <= 31; $i++) {
            $str .= "
                <option value = '$i' ".($i == $selectday ? 'selected' : '').">$i</option>";                       //Display as selected the day corresponding to $selected_timestamp
        }
        $str .= '
            </select>';
    }

    if ($printMonth) {
        $str .= '
            <select name = "'.$prefix.'Month">';
        for ($i = 1; $i <= 12; $i++) {
            $str .= "
                <option value = '$i' ".($i == $selectmonth ? 'selected' : '').">".$MONTH[sprintf("%02d",$i)]."</option>";      //Display as selected the month corresponding to $selected_timestamp
        }
        $str .= '
            </select>';
    }

    if ($printYear) {
        $str .= '
            <select name = "'.$prefix.'Year">';
        for ($i = $year - 2; $i < $year + 5; $i++) {
            $str .= "
                <option value = '$i' ".($i == $selectyear ? 'selected' : '').">$i</option>";                       //Display as selected the year corresponding to $selected_timestamp
        }
        $str .= '
            </select>';
    }

    return $str;
}


/**
* Prints time list boxes
*
* This function prints 2 list boxes (drop-down menu) used to select time. There are 2 list boxes,
* one for each of hour and minutes. The function accepts 5 arguments: $prefix is a string that
* is prefixed on the select name, so that the list boxes are named $prefixHour and $prefixMinute
* respectively. $timestamp is the current date. The $selected_timestamp corresponds to the default
* time displayed on the list boxes and if ommited defaults to now. Finally, printHour and printMinute
* are boolean variables that set whether the corresponding element will be visible and default to true.
* <br/>Example:
* <code>
* print eF_selectTime('selected_', time(), time()-3600);
* </code>
* The function returns the equivalent HTML code.
*
* @param string $prefix The prefix of the select boxes names
* @param int $timestamp The current timestamp
* @param int $selected_timestamp The timestamp corresponding to the time displayed by default on the list boxes
* @param bool $printHour If set to true, the Hour list box is displayed
* @param bool $printMinute If set to true, the Minute list box is displayed
* @return string The HTML code corresponding to the list boxes
* @version 1.0
* @todo The variable $timestamp is useless, we can calculate current time inside the function
* @todo print
*/
function eF_selectTime($prefix, $timestamp, $selected_timestamp = "", $printHour = true, $printMinute = true)
{
    if ($selected_timestamp == "") {
        //$selected_timestamp = mktime(0, 0, 0, strftime("%M", time()), strftime("%d", time()), strftime("%Y", time()));          //Now timestamp... weird way, so changed
        $selected_timestamp = time();
    }

    $selecthour   = strftime("%H", $selected_timestamp);
    $selectminute = strftime("%M", $selected_timestamp);

    if ($printHour) {
        $str = '
            <select name="'.$prefix.'Hour">';
        for ($i = 0; $i <= 23; $i++) {
            $str .= "<option value = '$i' ".($i == $selecthour ? 'selected' : '').">".sprintf("%02d",$i)."</option>";                       //Display as selected the hour corresponding to $selected_timestamp
        }
        $str .= '
            </select>';
    }

    if ($printMinute) {
        $str .= '
            <select name="'.$prefix.'Minute">';
        for ($i = 0; $i <= 59; $i++) {
            $str .= "<option value = '$i' ".($i == $selectminute ? 'selected' : '').">".sprintf("%02d",$i)."</option>";                       //Display as selected the minute corresponding to $selected_timestamp
        }
        $str .= '
            </select>';
    }

    return $str;
}

/**
* Converts a timestamp to human-readable date format
*
* This function takes a timestamp as an argument. It returns a string containing a textual representation
* of the date corresponding to this timestamp. If no input argument is given, it defaults to current timestamp.
* <br/>Example:
* <code>
* print eF_TimestampToDate(time()-86400);
* //prints: May 12 2005
* </code>
*
* @param int $timestamp The timestamp to be converted
* @return string The date corresponding to the given timestamp
* @version 1.0
* @todo Must change the conversion, so that both a better way is employed and internationalization issues are fixed
*/
function eF_timestampToDate($timestamp = "")
{
    global $MONTH;

    if ($timestamp == '') {
        $timestamp = time();
    }

    $month_num = strftime("%m", $timestamp);
    return strftime("%d ", $timestamp).$MONTH[$month_num].strftime(" %Y", $timestamp);                  //This has to change... we may use date() function
}


/**
* Converts a timestamp to human-readable date and time format
*
* This function takes a timestamp as an argument. It returns a string containing a textual representation
* of the date and time corresponding to this timestamp. If no input argument is given, it defaults to current timestamp.
* <br/>Example:
* <code>
* print eF_TimestampToTime(time()-3600);
* //prints: May 12 2005, 14:51
* </code>
*
* @param int $timestamp The timestamp to be converted
* @param bool $onlytime If set, then only the cirrent time is displayed
* @return string The date and time corresponding to the given timestamp
* @todo get rid of global
* @version 1.2
* Changes from version 1.1 to 1.2:
* - Removed new conversion method and reverted to old one
* Changes from version 1.0 to 1.1:
* - Added $onlytime parameter
* - Changed conversion method to use date() function
*/
function eF_timestampToTime($timestamp = false, $onlytime = false)
{
    global $MONTH;

    if ($timestamp === false) {
        $timestamp = time();
    }

    $month_num = strftime("%m", $timestamp);
    if ($onlytime) {
        $date_str = date("H:i:s", $timestamp);
    } else {
        $date_str = date("d ".$MONTH[$month_num]." Y, H:i", $timestamp);
    }
    return strftime("%d ", $timestamp).$MONTH[$month_num].strftime(" %Y", $timestamp).",  ".strftime(" %H:%M", $timestamp); //This should change, as well as eF_timeStampToDate() above
    //return $date_str;
}

/**
* Converts a greek string so that it doesn't includes accented or capital characters
*
* This function replaces all the accented greek characters in the input string to their equivalent not-accented.
* It also converts text to lower case. This function is used mainly for automatic test correction
* <br/>Example:
* <code>
* print eF_noTonos('????????');
* //prints: ????????
* </code>
*
* @param string $str The string to be converted
* @return string The string without accented or capital characters
* @version 1.0
*/

function eF_noTonos($str)
{
    $str = mb_strtolower($str);
    $str = str_replace("?", "?", $str);
    $str = str_replace("?", "?", $str);
    $str = str_replace("?", "?", $str);
    $str = str_replace("?", "?", $str);
    $str = str_replace("?", "?", $str);
    $str = str_replace("?", "?", $str);
    $str = str_replace("?", "?", $str);
    return $str;
}

/**
* Calculates the percentage representation of a number
*
* This function accepts a number and returns its equivalent percentage, with two decimal zeros leading
* <br/>Example:
* <code>
* print eF_percent(0.35);
* //prints: 35%
* </code>
*
* @param int $value The number to be converted
* @return string The percentage as a string
* @version 1.0
*/
function eF_percent($value)
{
    if($value == 0 ){
        return sprintf("%.0f%%", $value * 100);
    } else {
        return sprintf("%.2f%%", $value * 100);
    }
}


/**
* Presents a comma separated list of units related to a lesson
*
* This function accepts a results table, of the same structure as returned by the eF_getTableData() function and
* optionally a $key string. It searches through the table for $key, and appends results in the list.
* <br/>Example:
* <code>
* $content       = eF_getTableData("content", "*", "lessons_ID=3");
* $content_list  = eF_makeListIDs($content);
* echo $content_list;
* //prints: 2952,2956,2958,2960,2961,2962,2963,2979,2980,2981,2982,2985,3023,3022,3020
* </code>
*
* @param array $data The table to search for results
* @param string $key The key to search for
* @return string The comma-separated list of results
* @version 1.0
* @deprecated. Should use getTableDataFlat() with implode() instead
*/
function eF_makeListIDs($data, $key = "id")
{
    if ($data) {
        for ($i = 0; $i < sizeof($data); $i++) {
            $IDs[] = $data[$i][$key];
        }
        $list = implode(",", $IDs);
        return $list;
    } else {
        return false;
    }
}


/**
* As eF_makeListIDs for string elements
*
* This function accepts a results table, of the same structure as returned by the eF_getTableData() function and
* optionally a $key string. It searches through the table for $key, and appends results in the list.
* <br/>Example:
* <code>
* $students = ef_getProfessorStudents($_SESSION['s_login']);
* $students_list  = eF_makeListStrings($students,"login");
* echo $students_list;
* //prints: 'makriria','student','peris'
* </code>
*
* @param array $data The table to search for results
* @param string $key The key to search for
* @return string The comma-separated list of results
* @version 1.0
*/
function eF_makeListStrings($data, $key = "login")
{
    if ($data) {
        for ($i = 0; $i < sizeof($data); $i++) {
            $strings[] = $data[$i][$key];
        }
        $list = "'".implode("','", $strings)."'";
        return $list;
    } else {
        return false;
    }
}

/**
* Checks if the given user and password pair is valid
*
* The function is used to check if a certain username/password pair is valid, and to return this user's
* type. If no arguments are given, equivalent Session variables are considered.
* <br/>Example:
* <code>
* //The following code may be put at a page's header, indicating that only a valid user of type 'professor' may access it
* if (eF_checkUser($_SESSION['s_login'], $_SESSION['s_password']) != "professor") {
*     header("location:index.php");
*     exit;
* }
* </code>
*
* @param string $login The user name
* @param string $password The user password
* @return mixed The user type, one of 'professor', 'student' and 'administrator' or false if the combination is not valid.
* @version 1.0
* @todo Remove the global variable...
* - Added $lessons_ID parameter
* -version 2.6 From now on user_type depends also from lesson
*/

function eF_checkUser($login = false, $password = false, $lessons_ID = false)
{
    global $message;
    global $configuration;

    if (!$login){                                                                                  //?? ??? ???????? ????????, ???? ?? ??? ?? SESSION
        $login = $_SESSION['s_login'];
        if (!$password) {
            $password = $_SESSION['s_password'];
        }
    }
    if(!$lessons_ID){
        if (isset($_SESSION['s_lessons_ID']))
            $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    $res = eF_getTableData("users", "password, user_type, active", "login='$login'");
    if (!$res) {
        $message = _YOUCANNOTLOGINAS." ".$login;
        return false;
    } elseif ($res[0]['password'] == 'ldap') {
//        if (eF_checkUserLdap($login, str_rot13($password))) {                                       //rot13 applies a very simple encryption/decryption to the password, since it is stored as clear text (not hashed) to the session
            $user_type = eF_getUserBasicType($login, $lessons_ID);
            return $user_type;
//        } else {
//            $message = _WRONGPASSWORDTRYAGAIN;
//            return false;
//        }
    } elseif($res[0]['password'] != $password) {
        $message = _WRONGPASSWORDTRYAGAIN;
        return false;
    } elseif($res[0]['active'] != 1) {
        if ($configuration['activation'] == 0 && $configuration['mail_activation'] == 1){
            $message = _YOUWILLRECEIVEMAILFORACCOUNTACTIVATION;
        }else{
            $message = _NOTACTIVATEDACCOUNTTRYAGAINLATER;
        }
        return false;
    } else {
        $user_type = eF_getUserBasicType($login, $lessons_ID);
        return $user_type;
    }
}

/**
* Checks whether the specified unit and lesson are eligible for this user
*
* This function accepts a trinity of arguments: the user login, a lesson id and a content id.
* It then checks if the specified user has enrolled to the designated lesson and if the specified
* unit is available. If not, suitable error messages appear.
* <br/>Example:
* <code>
* //Checks if the logged-in user may access unit with id 102 from lesson with id 5
* eF_checkAccess($_SESSION['s_login'], 5, 102);
* </code>
*
* @param string $login The user name
* @param int $lessons_ID The lesson id
* @param int $content_ID The unit id
* @version 1.0
* @todo Very useful function, but needs also one to return, not just check and print.
* @DEPRECATED
*/
function eF_checkAccess($login, $lessons_ID, $content_ID)
{
    if ($lessons_ID) {
        $res = eF_getTableData("lessons",  "id",  "id=$lessons_ID AND active=1");
        if ($res[0]['id'] != $lessons_ID) {
            $message .= _LESSONDOESNOTEXISTS."!<BR>";
        }
    }
    if ($login AND $lessons_ID) {
        $res = eF_getTableData("users_to_lessons", "users_LOGIN", "users_LOGIN='$login' AND lessons_ID=$lessons_ID AND active=1");
        if ($res[0]['users_LOGIN'] != $login) {
            $message .= _YOUHAVENOTREGISTERTHISLESSON."!<BR>";
        }
    }
    if ($content_ID AND $lessons_ID) {
        $res = eF_getTableData("content", "lessons_ID", "id=$content_ID");
        if($res[0]['lessons_ID'] != $lessons_ID) {
            $message .= _UNITBELONGSTOOTHERLESSON."!<BR>";
        }
    }
    if (isset($message)) {
        eF_printMessage($message);
        exit;
    }
}


/**
* Clears any timed-out users
*
* This function checks if there are any long inactive sessions. If it finds any, it automatically logs out
* the corresponding users. The inactive sessions are checked against the G_SESSION_TIMEOUT configuration variable.
*
* @param int $lessons_ID The lesson id
* @param int $content_ID The unit id
* @version 2.0
* @todo Very useful function, but needs also one to return, not just check.
* Changes from 1.0 to 2.0:
* -Rewritten to be AdoDB compatible
*/
function eF_clearSessions()
{
    $users = eF_getTableDataFlat("logs", "distinct users_LOGIN");
    foreach ($users['users_LOGIN'] as $login) {
        $result = eF_getTableDataFlat("logs", "timestamp, action", "users_LOGIN='$login'", "timestamp desc limit 1");
        if ($result['timestamp'][0]< (time() - G_SESSION_TIMEOUT) && $result['action'][0] != 'logout') {
            $user = EfrontUserFactory :: factory($login);
            $user -> logout();
        }
    }
}

/**
* Delete removed tests from content
*
* This function checks if there are any tests in the content that are absent from the tests table.
* If so, it means that these tests where deleted, so they must be removed from the content table also.
*
* @version 1.0
* @Deprecated: Was causing problems, when a user reloaded the content page while not finishing the test addition...
*/
function eF_clearBadTests()
{
    $content = eF_getTableData("content, tests", "content.id AS id", "ctg_type='tests' AND tests.content_ID=content.id");
    $list    = eF_makeListIDs($content);
    if ($list) {
        eF_deleteTableData("content", "ctg_type='tests' AND id NOT IN ($list)");
    }
}





/**
* Get the forum id corresponding to this lesson.
*
* For each lesson there is a corresponding forum section. This function returns the forum id
* for the designated lesson
*
* @param int $lessons_ID the lesson id
* @return int the forum id
* @version 1.0
*/
function EfrontLessonToForum($lessons_ID)
{
    $forum = eF_getTableData("forum_forums", "forum_id", "lessons_ID=$lessons_ID");
    return $forum[0]['forum_id'];
}

/**
* Processes data string before insertion to database
*
* This function takes the data string of a unit and processes it.
*
* @param string $str The unit content
* @param int $content_ID the unit id
* @return string The processed data.
* @version 1.0
* @todo Too much complicated, can't tell exactly what it's doing, needs revisiting
* @todo get rid of global
*/
function eF_postProcess($str, $content_ID = false)
{
    global $message;

    if (mb_substr($str, 0, 3) == "<P>" AND mb_substr($str, -4) == "</P>") {
        $str = mb_substr($str, 3, -4);
    }

    $str = str_replace('</P>', '', str_replace("<P>", "<br/><br/>", $str));

    $temp_str = $str;                                                                               // ?????????? Copy - Paste ??? ?????
    $i = 0;

    while ($temp = strrev(strrchr(strrev(strstr($temp_str, "file:///")), '"'))) {
        $image_files[$i] = mb_substr($temp, 8, -2);
        $temp_str = mb_substr(strstr($temp_str, "file:///"), 8 + mb_strlen($image_files[$i]));
        $i++;
    }

    if (isset($image_files) && sizeof($image_files) > 0) {
        $files   = urlencode(implode(";", $image_files));
        $message = '<SCRIPT LANGUAGE = "JavaScript">popUp(\'/add_files.php?lessons_ID='.$_SESSION['s_lessons_ID'].'&content_ID='.$content_ID.'&files='.$files.'\', 700, 700);</script>';
        for ($i = 0; $i < sizeof($image_files); $i++) {
            $name           = mb_substr(strrchr($image_files[$i], "\\"), 1);
            $image_urls[$i] = G_LESSONSLINK.$_SESSION['s_lessons_ID']."/".$name;
        }

        $str = str_replace($image_files, $image_urls, $str);
        $str = str_replace("file:///", "", $str);
        $str = str_replace("<v:imagedata", "<img", $str);
        $str = str_replace("</v:imagedata", "</img", $str);
        $str = str_replace("<SCRIPT language=JavaScript>", "", $str);
        $str = str_replace("<SCRIPT language=JavaScript title=WebHelpNavBar>", "", $str);
        $str = str_replace("</SCRIPT>", "", $str);
        $str = mb_ereg_replace("<!--([^-]*([^-]|-([^-]|-[^>])))*//-->", "", $str);
    }

    return $str;
}


/**
* Checks if a user is eligible to access a unit, based on rules
*
* This function checks of the currently logged-in student is allowed to view a unit's content,
* based on access rules.
*
* @param int $content_ID the d of the current unit
* @return bool if the student is allowed to access the unit.
* @version 1.0
* @deprecated
*/
function eF_checkRulesOld($content_ID)
{
    $message = "";
    $parents = eF_getParents($content_ID);
    $list    = implode(",", $parents['id']);
    $rules   = eF_getTableData("rules", "rule_type, rule_content_ID, rule_option", "content_ID IN ($list) AND (users_LOGIN='*' OR users_LOGIN='".$_SESSION['s_login']."')");
    $allow   = true;
    for($i = 0; $i < sizeof($rules); $i++) {
        if ($rules[$i]['rule_type'] == "always") {
            $allow    = false;
            $message .= _YOUHAVEBEENEXCLUDEDBYPROFESSOR.'<BR>';
        } elseif($rules[$i]['rule_type'] == "hasnot_seen") {
            $res = eF_getTableData("logs", "id", "users_LOGIN='".$_SESSION['s_login']."' AND comments=".$rules[$i]['rule_content_ID']);
            if (!$res) {
                $content  = eF_getTableData("content", "name, ctg_type", "content.id=".$rules[$i]['rule_content_ID']);
                $allow    = false;
                $message .= _MUSTFIRSTREADUNIT.' <a href="student.php?ctg='.$content[0]['ctg_type'].'&content_ID='.$rules[$i]['rule_content_ID'].'">'.$content[0]['name'].'</a><BR>';
            }
        } elseif($rules[$i]['rule_type'] == "hasnot_passed") {
            $res = eF_getTableData("tests, done_tests", "done_tests.id", "users_LOGIN='".$_SESSION['s_login']."' AND tests.content_ID=".$rules[$i]['rule_content_ID']." AND done_tests.tests_ID=tests.id AND score>".$rules[$i]['rule_option']);
            if(!$res) {
                $content  = eF_getTableData("content", "name", "content.id=".$rules[$i]['rule_content_ID']);
                $allow    = false;
                $message .= _MUSTFIRSTTAKEATLEAST.' '.($rules[$i]['rule_option'] * 100).' % '._ATTEST.' <a href="student.php?ctg=tests?&content_ID='.$rules[$i]['rule_content_ID'].'">'.$content[0]['name'].'</a><BR>';
            }
        }
    }

    if ($allow === true) {
        return ($allow);
    } else {
        return ($message);
    }
}

/**
* Checks if a user is eligible to access a unit, based on rules
*
* This function checks of the currently logged-in student is allowed to view a unit's content,
* based on access rules.
*
* @param int $content_ID the d of the current unit
* @return mixed true if the student is allowed to access the unit, or an explanatory message otherwise
* @version 2.0
* @Changes from version 1.0 to 2.0 (2007/09/29 - venakis):
* - Rewritten so that it supports the new way of checking whether a unit is seen (through the explicit student action)
*/
function eF_checkRules($content_ID)
{
$seen_content = eF_getSeenContent($_SESSION['s_login'], $_SESSION['s_lessons_ID']);

    $message = "";
    $parents = eF_getParents($content_ID);
    $list    = implode(",", $parents['id']);
    $rules   = eF_getTableData("rules", "rule_type, rule_content_ID, rule_option", "content_ID IN ($list) AND (users_LOGIN='*' OR users_LOGIN='".$_SESSION['s_login']."')");
    $allow   = true;

    for ($i = 0; $i < sizeof($rules); $i++) {
        if ($rules[$i]['rule_type'] == "always") {
            $allow    = false;
            $message .= _YOUHAVEBEENEXCLUDEDBYPROFESSOR.'<BR>';
        } elseif ($rules[$i]['rule_type'] == "hasnot_seen") {
            if (!in_array($rules[$i]['rule_content_ID'], $seen_content)) {
                $content  = eF_getTableData("content", "name", "id = ".$rules[$i]['rule_content_ID']);
                $allow    = false;
                $message .= _MUSTFIRSTREADUNIT.' <a href="student.php?ctg=content&view_unit='.$rules[$i]['rule_content_ID'].'">'.$content[0]['name'].'</a><br/>';
            }
        } elseif ($rules[$i]['rule_type'] == "hasnot_passed") {
            $res = eF_getTableData("tests, done_tests", "done_tests.id", "users_LOGIN='".$_SESSION['s_login']."' AND tests.content_ID=".$rules[$i]['rule_content_ID']." AND done_tests.tests_ID=tests.id AND score > ".$rules[$i]['rule_option']);
            if (!$res) {
                $content  = eF_getTableData("content", "name", "content.id=".$rules[$i]['rule_content_ID']);
                $allow    = false;
                $message .= _MUSTFIRSTTAKEATLEAST.' '.($rules[$i]['rule_option'] * 100).' % '._ATTEST.' <a href="student.php?ctg=tests&view_unit='.$rules[$i]['rule_content_ID'].'">'.$content[0]['name'].'</a><br/>';
            }
        }
    }

    if ($allow === true) {
        return ($allow);
    } else {
        return ($message);
    }
}


/**
* Prints a count-down timer
*
* This function takes only one argument, which defines the timer's timeout, in seconds.
* It then creates the equivalent HTML code that displays a little count-down timer, using javascript.
*
* @param int $seconds_total The timer time-out, in seconds
* @return string The HTML cod of the timer.
* @version 1.0
* @deprecated
*/
function eF_timer($seconds_total)
{
    $hours   = floor($seconds_total / 3600);
    $minutes = floor($seconds_total / 60) - $hours * 60;
    $seconds = $seconds_total % 60;

    $script = '
        <input type = "text" size = "8" name = "time_left"/>
            <script>
            <!--
                var hours='.$hours.'
                var minutes='.$minutes.'
                var seconds='.$seconds.'
                var min=new String(3)
                var sec=new String(3)

                function display()
                {
                if( hours==0 && minutes==0 && seconds==0) {
                    document.form_test.submit_test_done.value = "true";
                    document.form_test.submit();
                    alert("'._YOURTIMEISUP.'!");
                }
                else {
                    if(seconds>=1)
                        seconds-=1
                    else {
                        if(seconds==0)
                            seconds=59
                        if(minutes>=1)
                            minutes-=1
                        else {
                            if(minutes==0)
                                minutes=59
                            if(hours>=1)
                                hours-=1
                             else
                                hours=0
                        }
                    }

                    min=minutes.toString()
                    if(min.length==1)
                        min="0"+min
                    sec=seconds.toString()
                    if(sec.length==1)
                        sec="0"+sec
                    document.form_test.time_left.value=hours+":"+min+":"+sec
                    setTimeout("display()",1000)
                }
            }

            display()
        -->
        </script>';

    return $script;
}


/**
* Checks if a user is logged in already
*
* This function checks the given login name and if that user is found logged in,
* returns true; otherwise it returns false.
*
* @param string $login The user login name
* @return bool If the user is logged in
* @version 1.1
* Changes from version 1.0 to 1.1:
* - Got rid of forum
*/
function eF_userAlreadyIn($login)
{
    $result = eF_getTableData('logs', 'action', "users_LOGIN='$login' ", "timestamp desc limit 1");

    if (sizeof($result) != 0 && $result[0]['action'] != 'logout') {                                             //0 means that the user may have been just registered, so there are no ogs for him in the system
        $login_ip = eF_getTableData('logs', 'session_ip', "users_LOGIN='$login' and action = 'login'", "timestamp desc limit 1");

        if (eF_encodeIP($_SERVER['REMOTE_ADDR']) == $login_ip[0]['session_ip'] || $login_ip[0]['session_ip'] == '')  {      //If the IP of the user is the same with the one he logged in, or there is no ip recorded, log out the user.
            $user = EfrontUserFactory :: factory($login);
            $user -> logout();
            return false;
        } else {
            return true;
        }
    } else {
        return false;
    }
}


/**
* Displays a list of the languages available
*
* This function searches in the languages directory on the disk for language files
* and returns an array with the ones found.
*
* @param $translations If true, it returns an array holding both languages and language names translated in the corresponding language
* @param $only_active If true, it returns only active languages. ADDED in version 3
* @return array The languages available
* @version 1.2
* Changes from 1.1 to 1.2:
* - Added $translations parameter
* Changes from 1.0 to 1.1:
* - Modified so that an array is returned instead of HTML code
* - Changed name from eF_selectLanguage to this
* - Removed parameter
*
*/
function eF_getLanguages($translations = false, $only_active = 1)
{
    global $path;

    if ($handle = opendir($path.'language')) {
        $i = 0;
        $content = '';
        while (false !== ($file = readdir($handle))) {
            if ($file != "." && $file != "..") {
                if (preg_match("/^lang\-(.+)\.php/", $file, $matches)) {
                    $languageslist[$i] = $matches[1];
                    $i++;
                }
            }
        }
        closedir($handle);
    }

    if ($translations) {
        foreach ($languageslist as $language) {
            if (defined("_".mb_strtoupper($language))) {
                eval("\$lang_translation[\$language] = _".mb_strtoupper($language).";");
            } else {
                $lang_translation[$language] = $language;
            }
        }
        $languageslist = $lang_translation;
        if ($only_active) {
            $result = eF_getTableDataFlat("languages","name","active=1");
            foreach ($languageslist as $lang) {
                if (!in_array($lang, $result['name'])) {
                    unset($languageslist[$lang]);
                }
            }
        }
        return $languageslist;
    } else {
        if ($only_active) {
            $temp = eF_getTableDataFlat("languages","name","active=1");
            $active_langs = $temp["name"];
            $languageslist = array_intersect($languageslist,$active_langs);
        }
        return $languageslist;
    }

}




/**
* Returns the system colors
*
* This function returns an array with the default system colors, per category. These
* categories are:
* - control
* - lessons
* - theory
* - examples
* - exercises
* - tests
* - users
* - forum
* - personal
* - statistics
* - nocontent
* - inactive
* - content
* - current_content
* - emails
* - chat
* Each array element has a key equal to one of these categories, and is valued with an HTML color
*
* @return array The system colors
* @version 1.0
*/
function eF_getDefaultColors()
{
    $result = eF_getTableData("configuration", "value", "name = 'colors'");
    $colors = unserialize($result[0]['value']);
    return $colors;
}

/**
* Gets the system default language
*
* This function reads the configuration table in the database and returns the default language
*
* @return string The default language
* @version 1.0
*/
function eF_getDefaultLanguage()
{
    $data = eF_getTableData("configuration", "value", "name='default_language'");
    return $data[0]['value'];
}

/**
* Returns the administrator's email address
*
* This function reads the configuration table in the database and returns the administrator's email
*
* @return string The administrator's email
* @version 1.0
* @DEPRECATED. Use $GLOBALS['configuration']['system_email'] instead
*/
function eF_getAdminEmail()
{
    return $GLOBALS['configuration']['system_email'];
}


/**
* Returns whether the footer will be visible
*
* This function reads the configuration table in the database and returns true if the footer
* is to be visible
*
* @return bool Whether the footer will be visible
* @version 1.0
*/
function eF_getShowFooter()
{
     return $GLOBALS['configuration']['show_footer'];
}


/**
* Searches for a user
*
* This function accepts a string and retrieves users matching to it. The search is performed on
* user logins and names. Optionally, there may be specified the user type, or a lesson to which
* he has enrolled. It returns an array with the results.
* <br/>Example:
* <code>
* $users = eF_findUsers('pe', 'professor');     //returns all professors starting with 'pe'
* </code>
*
* @param string $search The search string
* @param string $user_type The user type
* @param int lessons_ID The lesson id
* @return array The search results
* @version 1.0
*/
function eF_findUsers($search = false, $user_type = "", $lessons_ID = "")
{
    $users = false;
    if ($search && eF_checkParameter($search, 'login')) {
        $username_search = "%".$search."%";
        $str_extra = "";
        if ($user_type != "" && eF_checkParameter($user_type, 'string')) {
            $str_extra .= "user_type='".$user_type."' AND ";
        }
        if ($lessons_ID != "" && eF_checkParameter($lessons_ID, 'id')) {
            $str_extra .= "users_to_lessons.lessons_ID='".$lessons_ID."' AND users_to_lessons.users_LOGIN=users.login AND ";
        }
        $res = eF_getTableData("users,users_to_lessons","DISTINCT users.login",$str_extra." (users.login LIKE '".str_replace("\'","''",$username_search)."' OR users.name LIKE '".str_replace("\'","''",$username_search)."' OR users.surname LIKE '".str_replace("\'","''",$username_search)."')");
        if ($res) {
            for ($i = 0;$i < sizeof($res); $i++) {
                $users[$i] = $res[$i]['login'];
            }
        }
    }
    return $users;
}





/**
* Encodes an IP to its hexadecimal equivalent
*
* This function takes an IP representation and converts it to its hexadecimal equivalent
* <br/> Example:
* <code>
* eF_encodeIP('127.0.0.1');         //Outputs: 7f000001
* </code>
*
* @param string $dotquad_ip The string representing the IP
* @return string The hexadecimal representation of the IP
* @version 1.0
*/
function eF_encodeIP($dotquad_ip) {
    $ip_sep = explode('.',  $dotquad_ip);
    return sprintf('%02x%02x%02x%02x',  $ip_sep[0],  $ip_sep[1],  $ip_sep[2],  $ip_sep[3]);
}

/**
* Decodes an IP from hexadecimal to its equivalent human-readable format
*
* This function takes a hexadecimal IP representation and converts it to its
* equivalent human readable format.
* <br/> Example:
* <code>
* eF_decodeIP('7f000001');         //Outputs: 127.0.0.1
* </code>
*
* @param string $hex_ip The hexadecimal representation of the IP
* @return string The human readable representation of the IP
* @version 1.0
*/
function eF_decodeIP($hex_ip) {

    if (!$hex_ip) {
        return '';
    }

    $dotquad_ip = hexdec(mb_substr($hex_ip,0,2)).'.'.
                  hexdec(mb_substr($hex_ip,2,2)).'.'.
                  hexdec(mb_substr($hex_ip,4,2)).'.'.
                  hexdec(mb_substr($hex_ip,6,2));

    return $dotquad_ip;
}

/**
* Checks if a client may access the system;
*
* This function checks the client IP against the stored values. If there exist values and the client IP is
* not included, the system denies access to it.
*
* @return bool true if the client may access the system
* @version 1.0
*/
function eF_checkIP()
{
    $client_ip = $_SERVER['REMOTE_ADDR'];

    $allowedIPs = $GLOBALS['configuration']['ip_white_list'];                                              //Read the allowed IPs
    if (!$allowedIPs || !$client_ip) {                                                    //If the database doesn't
        $ok1 = true;
    } else {
        $client_ip_parts = explode('.', $client_ip);
        $allowed_ips     = explode(",", preg_replace("/\s+/", "", $allowedIPs));                                    //explode ips into an array, after stripping off any whitespace
        $ok1 = false;
        foreach($allowed_ips as $ip) {
            $ip_parts = explode('.', $ip);
            $count = 0;
            $temp  = true;
            while ($temp && $count++ < 4) {
                if ($client_ip_parts[$count] != $ip_parts[$count] && $ip_parts[$count] != '*') {
                    $temp = false;
                }
            }
            $ok1 = $ok1 | $temp;
        }
    }

    $disAllowedIPs = $GLOBALS['configuration']['ip_black_list'];                                              //Read the allowed IPs
    if (!$disAllowedIPs || !$client_ip) {                                                    //If the database doesn't
        $ok2 = false;
    } else {
        $client_ip_parts = explode('.', $client_ip);
        $allowed_ips     = explode(",", preg_replace("/\s+/", "", $disAllowedIPs));                                    //explode ips into an array, after stripping off any whitespace
        $ok2 = false;
        foreach($allowed_ips as $ip) {
            $ip_parts = explode('.', $ip);
            $count = 0;
            $temp  = true;
            while ($temp && $count++ < 4) {
                if ($client_ip_parts[$count] != $ip_parts[$count] && $ip_parts[$count] != '*') {
                    $temp = false;
                }
            }
            $ok2 = $ok2 | $temp;
        }
    }


    return $ok1 & !$ok2;                                                                                                    //For the user to be able to login, he must either be in the first group or the second group
}

/**
* Converts string to smilies
*
* This function parses a string and replaces text smilies occurences with the
* equivalent icons
*
* @param string $str The string to parse
* @return string The string returned
* @version 0.8
*/
function eF_convertTextToSmilies($str) {

    $img_str     = '<image src = "images/smilies/icon_';
    $text_array    = array(':)', ':-)',
                           ':(', ':-(',
                           ';)', ';-)',
                           ':oops:',
                           ':D', ':-D',
                           ':o', ':-o',
                           '8O', '8-O',
                           ':?', ':-?',
                           '8)', '8-)',
                           ':lol:',
                           ':x', ':-x',
                           ':P', ':-P',
                           ':cry:',
                           ':evil:',
                           ':twisted:',
                           ':roll:');
    $smilies_array = array($img_str.'smile.gif" />',     $img_str.'smile.gif" />',
                           $img_str.'sad.gif" />',       $img_str.'sad.gif" />',
                           $img_str.'wink.gif" />',      $img_str.'wink.gif" />',
                           $img_str.'redface.gif" />',
                           $img_str.'biggrin.gif" />',   $img_str.'biggrin.gif" />',
                           $img_str.'surprised.gif" />', $img_str.'surprised.gif" />',
                           $img_str.'eek.gif" />',       $img_str.'eek.gif" />',
                           $img_str.'confused.gif" />',  $img_str.'confused.gif" />',
                           $img_str.'cool.gif" />',      $img_str.'cool.gif" />',
                           $img_str.'lol.gif" />',
                           $img_str.'mad.gif" />',       $img_str.'mad.gif" />',
                           $img_str.'razz.gif" />',      $img_str.'razz.gif" />',
                           $img_str.'cry.gif" />',
                           $img_str.'evil.gif" />',
                           $img_str.'twisted.gif" />',
                           $img_str.'rolleyes.gif" />');

    $smilied_text = str_replace($text_array, $smilies_array, $str);

    return $smilied_text;
}

/**
* Checks if the designated user is ldap registered
*
* The input are a login / password pair, based on which the system tries to authenticate the user to
* the designated in the configuration LDAP server.
*
* @param string $login The user name
* @param string $password The user password
* @return mixed The user type, one of 'professor', 'student' and 'administrator' or false if the combination is not valid.
* @version 0.1
* @todo implementation
* @todo Remove the global variable...
*/
function eF_checkUserLdap($login, $password)
{
    $basedn   = $GLOBALS['configuration']['ldap_basedn'];
    $ldap_uid = $GLOBALS['configuration']['ldap_uid'];

    $ds = eF_ldapConnect();
    $sr = ldap_search($ds, $basedn, $ldap_uid.'='.$login);

    if (ldap_count_entries($ds, $sr) != 1) {
        return false;                                       //User either does not exist or more than 1 users found
    }
    $dn = ldap_get_dn($ds, ldap_first_entry($ds, $sr));

    $b = ldap_bind($ds, $dn, $password);
    if (!$b) {
        return 0;                                       //login / password values don't match
    }

    return true;

}

/**
* Get values for specified ldap attributes
*
* This function accepts a search filter and an array of attributes,
* and returns the equivalent values from the ldap server
*
* @param string $filter The search filter
* @param array $attributes The LDAP attributes
* @return array The array of attribute values
* @version 1.0
*/
function eF_getLdapValues($filter, $attributes)
{
    $basedn   = $GLOBALS['configuration']['ldap_basedn'];          //The base DN is needed to perform searches

    $ds = eF_ldapConnect();
    $sr = ldap_search($ds, $basedn, $filter, $attributes);

    $result = ldap_get_entries($ds, $sr);

    return $result;
}

/**
* Connect to LDAP server
*
* This function uses configuration values toattempt a connect to the LDAP server
*
* @return resource The LDAP link identifier
* @version 1.0
*/
function eF_ldapConnect() {
    $server   = eF_getTableData("configuration", "value", "name='ldap_server'");
    $port     = eF_getTableData("configuration", "value", "name='ldap_port'");
    $binddn   = eF_getTableData("configuration", "value", "name='ldap_binddn'");
    $bind_pwd = eF_getTableData("configuration", "value", "name='ldap_password'");
    $protocol = eF_getTableData("configuration", "value", "name='ldap_protocol'");

    $ds = ldap_connect($GLOBALS['configuration']['ldap_server'], $GLOBALS['configuration']['ldap_port']);
    ldap_set_option($ds, LDAP_OPT_PROTOCOL_VERSION, $GLOBALS['configuration']['ldap_protocol']);
    ldap_set_option($ds, LDAP_OPT_TIMELIMIT, 10);

    $b  = ldap_bind($ds, $GLOBALS['configuration']['ldap_binddn'], $GLOBALS['configuration']['ldap_password']);

    return $ds;
}

/**
* Check a parameter against a type
*
* This function accepts a parameter and a type. It then checks the parameter against a regular expression corresponding
* to the type specified. If the regular expression is met, then the parameter is returned. Otherwise, false is returned
* Supported types are:<br>
* - string: Only characters, [A-Za-a]
* - uint: Only positive numbers or zero, [0-9]
* - id: Alias for uint
* - login: Valid login names are made of alphanumeric characters and @, no spaces
* - email: Valid email address
* - filename: Valid filenames must not include special characters, such as /,\,..
* - hex: Hexadecimal number
* - alnum: Alphanumeric characters, [A-Za-z0-9]
* - alnum_with_spaces: Alphanumeric characters, but spaces are valid as well, [A-Za-z0-9\s]
* - ldap_attribute: Valid ldap attribute names
* - text: A string with plain characters, digits, and symbols, but not quotes or other special characters (like $, / etc)
*
* <br>Example:
* <code>
* $param = 'Hello world!';
* if (eF_checkParameter($param, 'string')) {
*     echo "Parameter is String";
* }
*
* $param = '123';
* if (eF_checkParameter($param, 'unit')) {
*     echo "Parameter is Unsigned integer";
* }
*
* </code>
* But be careful:
* <code>
* $param = '0';
* if (eF_checkParameter($param, 'unit')) {                      //Wrong way! This will not evalute to true, since eF_checkParameter will return $param, which is 0.
*     echo "Parameter is Unsigned integer";
* }
*
* if (eF_checkParameter($param, 'unit') !== false) {             //Correct way, since we make sure that the value returned is actually false.
*     echo "Parameter is Unsigned integer";
* }
* </code>
*
* @param mixed $param The parameter to check
* @param string $type The parameter type
* @return mixed The parameter, if it is of the specified type, or false otherwise
* @version 1.0.1
* Changes from 1.0 to 1.1:
* - Modified email declaration, so it can detect emails that have a dot (.) in the first part (before the '@').
*/
function eF_checkParameter($parameter, $type, $correct = false)
{
    switch ($type) {
        case 'string':
            if (!preg_match("/^[A-Za-z]{1,100}$/", $parameter)) {
                return false;
            }
            break;

        case 'uint':
        case 'id':
            if (!preg_match("/^[0-9]{1,100}$/", $parameter)) {                              //Caution: If 0 is met, then it will return 0 and not false! so, it must checked against false to make sure
                return false;
            }
            break;

        case 'login':
            if (!preg_match("/^_*\w+(\w*[._@-]*\w*)*$/", $parameter)) {                      //This means: begins with 0 or more '_', followed by at least 1 word character, followed by any combination of _,-,@ and word characters.
                return false;
            }
            break;

        case 'email':
            if (!preg_match("/^([a-zA-Z0-9_\.\-])+\@(([a-zA-Z0-9\-])+\.)+([a-zA-Z0-9]{2,4})+$/", $parameter)) {  //This means: begins with 0 or more '_' or '-', followed by at least 1 word character, followed by any combination of '_', '-', '.' and word characters, then '@', then the same as before, then the '.' and then 1 ore more characters.
                return false;
            }
            break;

        case 'filename':
            if (preg_match("/^.*((\.\.)|(\/)|(\\\)).*$/", $parameter)) {                      //File name must not contain .. or slashes of any kind
                return false;
            }
            break;

        case 'directory':
            if (preg_match("/^.*((\.\.)|(\\\)).*$/", $parameter)) {                      //Directory is the same as filename, except that it may contain forward slashes
                return false;
            }
            break;

        case 'hex':
            if (!preg_match("/^[0-9a-fA-F]{1,100}$/", $parameter)) {
                return false;
            }
            break;

        case 'timestamp':
            if (!preg_match("/^[0-9]{10}$/", $parameter)) {
                return false;
            }
            break;

        case 'date':
            if (!preg_match("/^[0-3]?[0-9]\-[0-1]?[0-9]\-[0-9]{4}$/", $parameter)) {
                return false;
            }
            break;

        case 'alnum':
            if (!preg_match("/^[A-Za-z0-9]{1,100}$/", $parameter)) {
                return false;
            }
        break;

        case 'ldap_attribute':
            if (!preg_match("/^[A-Za-z0-9:;\-_]{1,100}$/", $parameter)) {                     //An ldap attribute may be of the form: cn:lang-el;
                return false;
            }
        break;

       case 'alnum_with_spaces':
            if (!preg_match("/^[A-Za-z0-9_\s]{1,100}$/", $parameter)) {
                return false;
            }
       break;

       case 'alnum_general':
            if (!preg_match("/^[\.,_\-A-Za-z0-9\s]{1,100}$/", $parameter)) {
                return false;
            }
       break;

       case 'text':
            if (preg_match("/^.*[$\/\'\"]+.*$/", $parameter)) {
                return false;
            }
       break;

       case 'path':
            if (preg_match("/^.*[$\'\"]+.*$/", $parameter)) {
                return false;
            }
       break;

       default:
            break;
    }
    return $parameter;

}

/**
*
*   Checks if the corresponding file is a valid language file
*   @version 0.5
*
*/
function ef_checkLanguageFile($filename){
    $isfirst = true;
    $handle = @fopen($filename, "r");
    if ($handle) {
        while (!feof($handle)) {
            $line =fgets($handle);
            $line = trim($line);
            if (strlen($line)>0){
                if ($isfirst){  //first line
                    if ( strlen($line) > 4 && substr($line,0,5) == "<?php" ){
                        $isfirst = false;
                    }
                    else{
                        return false;
                    }
                }
                else{   //lines other than first
                    if (strlen($line) >6){
                        if (substr($line,0,6) == "define"){
                            if (substr($line,-1,1) != ";"){
                                return false;
                            }
                        }
                        else{
                            return false;
                        }
                    }
                    elseif ($line == "?>")
                        return true;
                    else{
                        return false;
                    }
                }
            }
        }
        fclose($handle);
        return true;
    }
    else
        return false;

}

/**
* Undocumented, not used
*/
function eF_errorHandler($errno, $errstr, $errfile, $errline, $errcontext)
{
    $error_level = ini_get('error_reporting');

    switch ($errno) {
        case E_USER_ERROR:
            if (($error_level & $errno) == $errno) {                //???? ? ??????, ?????? ??? ??????? ?? ?? error level, ?? ???????? ?? ???????????? ? ???.
                eF_printMessage("FATAL: ".$errstr);
            }
            exit(1);
            break;
        case E_USER_WARNING:
            if (($error_level & $errno) == $errno) {
                eF_printMessage($errstr);
            }
            break;
        case E_USER_NOTICE:
            if (($error_level & $errno) == $errno) {
                eF_printMessage("NOTICE: ".$errstr);
            }
            break;
        case E_WARNING:
            if (($error_level & $errno) == $errno) {
                echo "Warning: $errstr<br />\n";
            }
        case E_NOTICE:
            if (($error_level & $errno) == $errno) {
                echo "Notice (error number $errno): $errstr in file <b>$errfile</b>, line <b>$errline</b><br />\n";
            }
            break;
        default:
            if (($error_level & $errno) == $errno) {
                echo "Unkown error type: [$errno] $errstr<br />\n";
            }
            break;
    }
}


/**
* Alerts admininstrator for any hacking attempts
*
* This function inserts to the log file information regarding actions that may be taken as
* hacking attempts. The administrator may be contacted via email, if it is set so in the
* configuration table.
*
* @param string $login The user that made the attempt
* @param string $timestamp The attempt time
* @param string $message The message related to the action
* @version 1.0
*/
function eF_alertAdmin($login, $timestamp, $message) {
    $fields_insert = array('users_LOGIN' => $login,
                           'timestamp'   => $timestamp,
                           'action'      => 'hack',
                           'comments'    => $message);
    eF_insertTableData("logs", $fields_insert);

    $result = eF_getTableData("configuration", "value", "name='hacking_alert'");
    if (sizeof($result) > 0 && $result[0]['value'] == 'yes') {
        eF_mail(eF_getAdminEmail(), _HACKINGATTEMPT, $message);
    }
}




/**
* Convert backward slashes to forward slashes
*
* Thi function converts all the backslashes contained in a string to forward slashes.
* This is used to ensure cross-platform compatibility of the code (which, in general, uses
* forward slashes, '/').
*
* @param string $str The string to convert
* @return string The converted string
* @version 1.0
*/
function eF_revertSlashes($str)
{
    $str = str_replace('\\', '/', $str);
    return $str;
}

/**
* Register a user with the system
*
* This function takes an array containing user information and stores it to the users
* database table. It also accepts two boolean parameters, corresponding to system configuration
* options. It returns a string with a success or failure message.
*
* @param array $values The user information array
* @param bool $is_automatic_activation Specifies whether new users are activated automatically.
* @param bool $is_ldap_user Specifies whether the user is ldap-based
* @return string An error or success message
* @version 1.2
* @todo Check for permissions in folder
* @todo return other than true
* Changes from version 1.1 to 1.2:
* - Added if clauses and delete user on failure.
* Changes from version 1.0 to 1.1:
* - Got rid of forum
*/
function eF_registerUser($values, $is_automatic_activation, $is_ldap_user = false, $type="student")
{
    //$db -> debug = true;
    foreach ($values as $key => $value) {
        $fields_insert[$key] = $value;                                                                      //The values that will be inserted in the database
    }
    if ($type == ''){
        $fields_insert['user_type'] = "student";
    }else{
        $fields_insert['user_type'] = $type;
    }
    if($fields_insert['login'] == ''){
        $fields_insert['login'] = 'student';
    }

    if ($fields_insert['password'] == '') {
        $fields_insert['password']  = md5($fields_insert['login'].G_MD5KEY);
    } else {
        $fields_insert['password']  = md5($fields_insert['password'].G_MD5KEY);
    }

    if($fields_insert['languages_NAME'] == ''){
        $fields_insert['languages_NAME'] = $_SESSION['s_language'];
    }

    if ($is_ldap_user) {
        $fields_insert['password'] = 'ldap';                //For an LDAP user, the password always is 'ldap', so we can tell by looking at the password field that this is an LDAP user that authenticates to the LDAP server
    }

    $fields_insert['timestamp'] = time();
    if (!isset($fields_insert['active'])) {     //If the active field is not already set in the values, derive it
        if (!$is_automatic_activation) {      //Does the system permit automatic activation of new users?
            $fields_insert['pending'] = 1;
            $fields_insert['active']  = 0;
        } else {
            $fields_insert['pending'] = 0;
            $fields_insert['active']  = 1;
        }
    }
    //$fields_insert['valid_until_timestamp'] = time() + 365 * 86400;                                         //1 year

    //default avatar for new user
    $target_filename = time().'_prefix_unknown_small.png';
    copy (G_ROOTPATH.'www/images/avatars/system_avatars/unknown_small.png', G_ROOTPATH.'www/images/avatars/'.$target_filename);
    $fields_insert['avatar'] = $target_filename;

    if (eF_insertTableData("users", $fields_insert)) {                                                       //Create the new user by inserting a new entry in the users table
        if (is_dir(G_ROOTPATH.'upload/'.$values['login']) || mkdir(G_ROOTPATH.'upload/'.$values['login'])) {
            mkdir(G_ROOTPATH.'upload/'.$values['login'].'/message_attachments/');
            mkdir(G_ROOTPATH.'upload/'.$values['login'].'/message_attachments/Incoming');
            mkdir(G_ROOTPATH.'upload/'.$values['login'].'/message_attachments/Sent');
            mkdir(G_ROOTPATH.'upload/'.$values['login'].'/message_attachments/Drafts');
            mkdir(G_ROOTPATH.'upload/'.$values['login'].'/projects');
            $fields_insert = array('name'        => 'Incoming',
                                   'users_LOGIN' => $values['login'],
                                   'parent_id'   => 0);
            eF_insertTableData("f_folders", $fields_insert);


            $fields_insert = array('name'        => 'Sent',
                                   'users_LOGIN' => $values['login'],
                                   'parent_id'   => 0);
            eF_insertTableData("f_folders", $fields_insert);


            $fields_insert = array('name'        => 'Drafts',
                                   'users_LOGIN' => $values['login'],
                                   'parent_id'   => 0);
            eF_insertTableData("f_folders", $fields_insert);

            return true;
        } else {
            eF_deleteTableData("users", "login='".$values['login']."'");            //Delete inserted user in case of failure
            return false;
        }
    } else {
        return false;
    }

}

/**
* Delete user
*
* This function deletes the designated user
* @param string $login The user to delete
* @return bool true if the user was deleted
* @todo Ask if to delete forum data
* @version 1.0
*/
function eF_deleteUser($login) {
    if (eF_checkParameter($login, 'login')) {
        $user = eF_getTableData("users", "user_type, avatar", "login='".$login."'");

        eF_deleteTableData("users", "login='".$login."'");
        eF_deleteTableData("comments", "users_LOGIN='".$login."'");
        eF_deleteTableData("logs", "users_LOGIN='".$login."'");
        eF_deleteTableData("rules", "users_LOGIN='".$login."'");
        eF_deleteTableData("users_to_lessons", "users_LOGIN='".$login."'");
        eF_deleteTableData("scorm_data", "users_LOGIN='".$login."'");

        $result          = eF_getTableData("done_tests", "id", "users_LOGIN='".$login."'");
        $done_tests_list = eF_makeListIDs($result);
        if ($done_tests_list) {
            eF_deleteTableData("done_questions", "done_tests_ID IN ($done_tests_list)");
        }
        eF_deleteTableData("done_tests", "users_LOGIN='".$login."'");

        eF_deleteTableData("f_folders", "users_LOGIN='".$login."'");
        eF_deleteTableData("f_messages", "users_LOGIN='".$login."'");
        eF_deleteTableData("f_personal_messages", "users_LOGIN='".$login."'");
        eF_deleteTableData("chatmessages", "users_LOGIN='".$login."'");

        if ($user[0]['user_type'] != 'administrator') {                    //Delete chat and some other data, only if the users is not an administrator, otherwise you risk loosing much useful data
            eF_deleteTableData("groups", "users_LOGIN='".$login."'");
            eF_deleteTableData("chatrooms", "users_LOGIN='".$login."'");
        }

        if (is_dir(G_UPLOADPATH.$login)) {                            //Delete personal messages folder from system directory
            eF_deleteFolder(G_UPLOADPATH.$login);
        }

        if (is_file(G_AVATARSPATH.$user[0]['avatar'])) {                    //Delete user avatar
            unlink(G_AVATARSPATH.$user[0]['avatar']);
        }

        return true;
    } else {
        return false;
    }
}

/**
* Form for registering a new user's lessons
*
* This function displays a form with the lessons eligible for the new user to select.
* It reads the available directions and lessons and presents the ones in the same language
* as the user.
*
* @param string $login The new user
* @param string $language The new user language
* @param string $message A message that will be propagated on with a hidden tag. Normally the result of eF_registerUser.
* @return string The HTML code of the form with the lessons
* @version 1.0
* @deprecated
*/
function eF_registerLessons($login, $language, $message)
{
    $all_directions = eF_getTableData("directions", "id,name", "active=1");                             //Get the available directions

    $lessons_data = '
    <form name = "register_lessons_form" action = "'.$_SERVER['PHP_SELF'].'" method = "post">
    <input type = "hidden" name = "login" value = "'.$login.'" />
    <table border = "0" width = "100%" cellspacing = "10">
        <tr>';

    $counter = -1;
    for ($i = 0; $i < sizeof($all_directions); $i++) {

        $directions_ID   = $all_directions[$i]['id'];
        $directions_name = $all_directions[$i]['name'];

        $all_lessons = eF_getTableData("lessons", "name,id,info,price", "lessons.directions_ID=$directions_ID AND active=1 AND languages_NAME='".$language."'");
        if (sizeof($all_lessons) > 0 ) {
            $counter++;
            $lessons_data .= '
            <td valign = "top" width = "50%">
                <table>
                    <tr><td colspan = "2"><b>'.$directions_name.'</b></td></tr>';
            for ($k = 0; $k < sizeof($all_lessons); $k++) {
                $font_class = "";
                $checked    = "";
                $lessons_data .= '
                    <tr><td width = "1%">
                            <input type = "checkbox" name = "lessons['.$all_lessons[$k]['id'].']" value = "'.$all_lessons[$k]['name'].'" '.$checked.'/></td>
                        <td class = "'.$font_class.'">'.$all_lessons[$k]['name'].'</td></tr>
                    <tr><td></td><td>';
                if ($all_lessons[$k]['price'] == 0) {
                    $lessons_data .= '
                        <input type = "hidden" name = "lessons_free['.$all_lessons[$k]['id'].']" value = "free"><b>'._FREE.'</b><br/>';
                }
                $lessons_data .= $all_lessons[$k]['info'].'</td></tr>';
            }
            $lessons_data .= "
                </table>
            </td>";
        }
        if ($counter % 2 == 1) {
            $lessons_data .= "
        </tr>
        <tr>";
        }
    }

    if ($counter == -1) {
        $lessons_data .='<td>'._NOLESSONSDEFINEDCONTACTADMIN.'</td></tr>
        <tr><td align = "center" colspan = "2">
            <input type = "submit" value = "'._CONTINUE.'" onclick = "index.php"/>
        </td></tr>';
    } else {
        $lessons_data .='</tr>
        <tr><td align = "center" colspan = "2">
            <input type = "submit" name = "p_submit_lessons" value = "'._LESSONSREGISTRATION.'"/>
        </td></tr>';

    }

    $lessons_data .= '
    </table>
    <input type = "hidden" name = "p_previous_message" value = "'.$message.'" />
    </form>';

    return $lessons_data;
}



/**
* Delete a personal message
*
* This function is used to delete a message, including any attachments it may have
*
* @param int $msg_id The message id
* @return bool True if the deletion was succesful
* @version 0.1
*/
function eF_deletePersonalMessage($msg_id) {
    if (eF_checkParameter($msg_id, 'id')) {
        $res = eF_getTableData("f_personal_messages", "users_LOGIN, attachments, f_folders_ID", "id=".$msg_id);

        if ($_SESSION['s_login'] == $res[0]['users_LOGIN'] || $_SESSION['s_type'] == 'administrator') {
            eF_deleteTableData("f_personal_messages", "id=".$msg_id);

            if ($res[0]['attachments'] != '') {
                $attached_file = new eF_File($res[0]['attachments']);
                $attached_file -> delete();
            }

            return true;
        } else {
            $message = 'You cannot delete this message';
            return $message;
        }
    } else {
        $message = _INVALIDID;
        return $message;
    }
}

/**
* Move a personal message
*
* This function is used to move a message, including any attachments it may have
*
* @param int $msg_id The message id
* @param int $target_folder_id the target folder id
* @return bool True if the move was succesful
* @version 0.1
*/
function eF_movePersonalMessage($msg_id, $target_folder_id) {
    if (eF_checkParameter($msg_id, 'id') && eF_checkParameter($target_folder_id, 'id')) {
        $res = eF_getTableData("f_personal_messages", "users_LOGIN, attachments, f_folders_ID", "id=".$msg_id);
        $folder_name = eF_getTableData("f_folders", "name", "users_LOGIN='".$_SESSION['s_login']."' and id=".$target_folder_id);
        if ($_SESSION['s_login'] == $res[0]['users_LOGIN'] || $_SESSION['s_type'] == 'administrator') {
            eF_updateTableData("f_personal_messages", array("f_folders_ID" => $target_folder_id), "id=".$msg_id);
            if ($res[0]['attachments'] && sizeof($folder_name) > 0) {
                $attachments = unserialize($res[0]['attachments']);
                foreach($attachments as $attach) {
                    rename($attach, G_UPLOADPATH.$_SESSION['s_login'].'/message_attachments/'.$folder_name[0]['name'].'/'.basename($attach));
                    $new_attachments[] = G_UPLOADPATH.$_SESSION['s_login'].'/message_attachments/'.$folder_name[0]['name'].'/'.basename($attach);
                }
                eF_updateTableData("f_personal_messages", array("attachments" => serialize($new_attachments)), "id=".$msg_id);
            }
            return true;
        } else {
            $message = 'You cannot move this message';
            return $message;
        }
    } else {
        $message = _INVALIDID;
        return $message;
    }
}

/**
* Print a select box
*
* This function prints select options based on the $values array.
* If $labels is set, then it is used as the option label.
*
* @param array $values The option values
* @param array $labels The option labels
* @return string The html code of the select options
* @version 0.1
*/
function eF_printSelect($values, $labels = false) {
    $str = '';
    if ($labels === false || sizeof($labels) != sizeof($values)) {
        $labels = $values;
    }

    for($i = 0; $i < sizeof($values); $i++) {
        $str .= '
            <option value = "'.$values[$i].'">'.$labels[$i].'</option>';
    }
    return $str;
}

/**
* Get recent forum messages
*
* This function reads recent forum messages from the forum category with the same name as the current lesson.
* It returns an array with the forum title and information about the message, which can be used to access it.
* If a lesson is not specified, the current (session) is considered.
* <br/>Example:
* <code>
* $messages = eF_getForumMessages($lessons_ID);
* print_r($messages);
* //Returns:
*Array
*(
*    [0] => Array
*        (
*            [title] => Some message
*            [id] => 53
*            [topic_id] => 27
*            [users_LOGIN] => periklis
*            [timestamp] => 1125765345
*        )
*
*    [1] => Array
*        (
*            [title] => Another message
*            [id] => 54
*            [topic_id] => 27
*            [users_LOGIN] => periklis
*            [timestamp] => 1125752345
*        )
*
*    [2] => Array
*        (
*            [title] => This is a large mess...
*            [id] => 55
*            [topic_id] => 27
*            [users_LOGIN] => admin
*            [timestamp] => 1125751543
*        )
*
*)
* </code>
*
* @param int $lessons_ID The lesson id
* @param int $limit The results limit
* @return array The messages array
* @version 1.1
* Changes from 1.0 to 1.1 (15/11/2005):
* - Added $limit parameter
*/
function eF_getForumMessages($lessons_ID = false, $limit = false)
{
    if (!$lessons_ID) {
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }

    if ($limit && eF_checkParameter($limit, 'uint')) {
        $limit_str = ' limit '.$limit;
    } else {
        $limit_str = '';
    }

    $messages_array = array();
    if ($lessons_ID) {
        $messages_array = eF_getTableData("f_messages m, f_topics t, f_categories c", "m.title, m.id, t.id as topic_id, m.users_LOGIN, m.timestamp", "c.lessons_ID=".$lessons_ID." AND t.f_categories_ID=c.id AND m.f_topics_ID=t.id", "m.timestamp desc".$limit_str);
    }
    return $messages_array;
}

/**
* Get unread personal messages
*
* This function gets the unread personal messages from the user's inbox. It returns an array with
* the message title and information about the message, which can be used to access it. If $login
* is not set, then messages for the current (session) user are returned.
* <br/>Example:
* <code>
* $messages = eF_getPrivateMessages();
* print_r($messages);
* //Returns:
*Array
*(
*    [0] => Array
*        (
*            [title] => Hello!
*            [id] => 26
*        )
*
*    [1] => Array
*        (
*            [title] => It's me again!
*            [id] => 24
*        )
*
*    [2] => Array
*        (
*            [title] => This is a large mess...
*            [id] => 23
*        )
*
*)
* </code>
*
* @param string $login The user login name
* @param int $limit The results limit
* @return array The messages array
* @version 1.1
* Changes from 1.0 to 1.1 (15/11/2005):
* - Added $limit parameter
*/
function eF_getPersonalMessages($login = false, $limit = false) {

    if (!$login) {
        $login = $_SESSION['s_login'];
    } elseif (sizeof(eF_getTableData("users", "login", "login=$login")) == 0) {                             //This user does not exist
        return array();
    }

    if ($limit && eF_checkParameter($limit, 'uint')) {
        $limit_str = ' limit '.$limit;
    } else {
        $limit_str = '';
    }

    $messages_array = eF_getTableData("f_personal_messages pm, f_folders", "pm.title, pm.id, pm.timestamp, pm.sender", "pm.users_LOGIN='".$login."' and f_folders_ID=f_folders.id and f_folders.name='Incoming' and viewed='no'", "pm.timestamp desc".$limit_str);         //Get unseen messages in Incoming folder

    return $messages_array;
}

/**
* Get announcements
*
* This function gets the lesson announcements (news). It returns an array holding the announcement title, id
* and timestamp.
* From v3 you can add news which display from a specific time in the future...makriria(2007/07/17)
* <br/>Example:
* <code>
* $news = eF_getNews();
* print_r($news);
* //Returns:
*Array
*(
*    [0] => Array
*        (
*            [title] => announcement 1
*            [id] => 3
*            [timestamp] => 1125751731
*            [users_LOGIN] => admin
*        )
*
*    [1] => Array
*        (
*            [title] => Important announcem...
*            [id] => 5
*            [timestamp] => 1125751012
*            [users_LOGIN] => peris
*        )
*)
* </code>
*
* @param int $lessons_ID The lesson id
* @return array The news array
* @version 1.0
*/
function eF_getNews($lessons_ID = false) {

    if (!$lessons_ID) {
        isset($_SESSION['s_lessons_ID']) ? $lessons_ID = $_SESSION['s_lessons_ID'] : $lessons_ID = 0;
    }

    if ($_SESSION['s_type'] == "student"){                                  // students see only previous news
        $news_array = eF_getTableData("news", "*", "(lessons_ID=$lessons_ID OR lessons_ID=0) AND timestamp<=".time(), "lessons_ID, timestamp desc"); //"
    } elseif ($_SESSION['s_type'] == "administrator") {                     // administrators see only news with lessons_Id=0
        $news_array = eF_getTableData("news", "*", "lessons_ID=0", "timestamp desc");
    } else {
        $news_array = eF_getTableData("news", "*", "lessons_ID=$lessons_ID OR lessons_ID=0", "lessons_ID, timestamp desc");
    }

    return $news_array;
}

/**
* Get system announcements
*
* This function gets the system announcements. It returns an array holding the announcement title, id
* and timestamp.
* It works the same way as eF_getNews above, but it returns the data, too
* @return array The news array
* @version 0.5
*/

function eF_getSystemAnnouncements() {

    $announcements_array = eF_getTableData("system_announcements", "id, title, data, timestamp, users_LOGIN", "", "timestamp desc");

    return $announcements_array;
}

/**
* Get comments
*
* This function gets the lesson comments. It returns an array holding the name of the lesson where the comment was put,
* the comment id, the comment itself (which is put as a title on the lesson name link), and finally the timestamp and the
* user that posted it. IF a lesson id is not specified, then comments for the current lesson are returned.If a login is
* specified, then only comments that the specified user has posted are returned. If a content id is specified, then only
* comments of this unit are displayed.
* <br/>Example:
* <code>
* $comments = eF_getComments();
* print_r($comments);
* //Returns:
*Array
*(
*    [0] => Array
*        (
*            [id] => 3
*            [data] => This is a comment
*            [users_LOGIN] => admin
*            [timestamp] => 1125751731
*            [content_name] => unit 1.2
*            [content_id] => 145
*            [content_type] => theory
*        )
*)
* </code>
*
* @param int $lessons_ID The lesson id
* @param string $login The user login
* @param int $content_ID The unit id to return its comments
* @param int $limit The results limit
* @return array The comments array
* @version 1.1
* @Changes from 1.0 to 1.1: 15/11/2005
* - Added $limit parameter
*/
function eF_getComments($lessons_ID = false, $login = false, $content_ID = false, $limit = false) {

    if (!$lessons_ID || !eF_checkParameter($lessons_ID, 'id')) {
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }

    if ($login && eF_checkParameter($login, 'login')) {
        $login_str = " AND comments.users_LOGIN='$login'";
    } else {
        $login_str = '';
    }

    if ($content_ID && eF_checkParameter($content_ID, 'id')) {
        $content_ID_str = ' AND content.id='.$content_ID;
    } else {
        $content_ID_str = '';
    }

    if ($limit && eF_checkParameter($limit, 'uint')) {
        $limit_str = ' limit '.$limit;
    } else {
        $limit_str = '';
    }

    $comments_array = eF_getTableData("comments, content", "comments.id AS id, comments.data AS data, comments.users_LOGIN AS users_LOGIN, comments.timestamp AS timestamp, content.name AS content_name, content.id AS content_ID, content.ctg_type AS content_type", "content.lessons_ID=$lessons_ID AND comments.content_ID=content.id AND content.active=1 AND comments.active=1".$login_str.$content_ID_str, "comments.timestamp DESC".$limit_str);

    return $comments_array;


}

/**
* default $basic_user_type = false.. it returns lessons for all user types.. if $basic_user_type specified, it returns lessons with coreesponding role and subroles
*/
function eF_getUserLessons($login = false, $new_content = false, $basic_user_type = false)
{
    if (!$login) {
        $login = $_SESSION['s_login'];
    }

    $lessons_array = array();
    $lessons       = eF_getTableData("lessons l, users_to_lessons ul, directions d, users u", "distinct l.name as lesson_name, l.id as lesson_id, d.name as direction_name, l.info as lesson_info", "l.directions_ID=d.id and ul.lessons_ID=l.id and ul.users_LOGIN='$login' and u.languages_NAME = l.languages_NAME and ul.active=1 and u.active=1 and l.active=1", "d.name, l.name");
    for ($i = 0; $i < sizeof($lessons); $i ++) {
//        if ($new_content) {
//            sizeof(eF_getNewContent($login, $lessons[$i]['lesson_id'], 'content'));
//        }
        //$lessons_array[$lessons[$i]['direction_name']][] = array('id' => $lessons[$i]['lesson_id'], 'name' => $lessons[$i]['lesson_name'], 'content' => getLessonContentUnits($lessons[$i]['lesson_id']));
  /*      $lessons_array[$lessons[$i]['direction_name']][] = array('id' => $lessons[$i]['lesson_id'], 'name' => $lessons[$i]['lesson_name'], 'content' => getLessonContentUnits($lessons[$i]['lesson_id']), 'new_content' => sizeof(eF_getNewContent($login, $lessons[$i]['lesson_id'], 'content')));
    */
    $res_per  = eF_getTableData("periods", "id", "lessons_ID=".$lessons[$i]['lesson_id']);
    $user_type =    eF_getUserBasicType($login, $lessons[$i]['lesson_id']);
            if(!$basic_user_type){ // if false... write all user types
                if (sizeof($res_per) == 0){
                    $lessons_array[$lessons[$i]['direction_name']][] = array('id' => $lessons[$i]['lesson_id'], 'name' => $lessons[$i]['lesson_name'],'user_type' => $user_type, 'content' => getLessonContentUnits($lessons[$i]['lesson_id']), 'new_content' => sizeof(eF_getNewContent($login, $lessons[$i]['lesson_id'], 'content')), 'info' => $lessons[$i]['lesson_info']);
                } else {
                    $lessons_array[$lessons[$i]['direction_name']][] = array('id' => $lessons[$i]['lesson_id'], 'name' => $lessons[$i]['lesson_name'],'user_type' => $user_type , 'content' => getLessonContentUnits($lessons[$i]['lesson_id']), 'new_content' => sizeof(eF_getNewContent($login, $lessons[$i]['lesson_id'], 'content', true)), 'info' => $lessons[$i]['lesson_info']);
                }
            }else{
                if ($basic_user_type == $user_type){
                    if (sizeof($res_per) == 0){
                        $lessons_array[$lessons[$i]['direction_name']][] = array('id' => $lessons[$i]['lesson_id'], 'name' => $lessons[$i]['lesson_name'],'user_type' => $user_type, 'content' => getLessonContentUnits($lessons[$i]['lesson_id']), 'new_content' => sizeof(eF_getNewContent($login, $lessons[$i]['lesson_id'], 'content')), 'info' => $lessons[$i]['lesson_info']);
                    } else {
                        $lessons_array[$lessons[$i]['direction_name']][] = array('id' => $lessons[$i]['lesson_id'], 'name' => $lessons[$i]['lesson_name'],'user_type' => $user_type , 'content' => getLessonContentUnits($lessons[$i]['lesson_id']), 'new_content' => sizeof(eF_getNewContent($login, $lessons[$i]['lesson_id'], 'content', true)), 'info' => $lessons[$i]['lesson_info']);
                    }
                }
            }

    }

    return $lessons_array;

}


/**
* Get modules
*
* This function returns an array with all modules for the specific type of user
*
*/
function eF_getModules()
{
    $modules     = eF_getTableData("modules", "*", "active=1");//$modules[1] = $modules[0];

    $user_module = array('administrator' => array(), 'professor' => array(), 'student' => array());
    foreach ($modules as $module) {                                         //Copy the module to an array with keys "student", "professor", "administrator"
        $permissions = explode(",", $module['permissions']);
        foreach ($permissions as $user_type) {
            $user_module[$user_type][] = $module;//array('name' => $module['name'], 'title' => $module['title'], 'position' => $module['position']);
        }
    }

    $modules = array();
    $s_type = $_SESSION['s_type'];

    foreach ($user_module[$s_type] as $value) {
        if ($value['position'] == 'left') {
            $modules[$value['name']] = array('title' => $value['title'], 'link' => $s_type .'.php?ctg='.$value['name'], 'image' => 'component_green');
        }
    }

    return $modules;

}

/**
* Get navigation menu
*
* This function returns an array with all the elements and links of the user navigation menu. The array
* is in the form $menu[category][type] => array('title' => 'link'). There are 2 types of categories: 'lesson',
* which refers to lesson specific options, and 'general', which refers to general options.
* For example, some of the entries of the student menu are shown below:
* <code>
*Array
*(
*    [lesson][lessons] => Array
*        (
*            [title] => Lessons
*            [link] => student.php?ctg=lessons&op=lessons_list
*        )
*
*    [lesson][theory] => Array
*        (
*            [title] => Theory
*            [link] => student.php?ctg=theory
*        )
*
*    [lesson][examples] => Array
*        (
*            [title] => Examples
*            [link] => student.php?ctg=examples
*        )
*    [general][personal] => Array
*        (
*            [title] => Options
*            [link] => student.php?ctg=personal
*        )
* </code>
*
* @return array The navigation menu
* @version 0.5
*/

function eF_getMenu()
{

    $menu = array();

    switch($_SESSION['s_type']) {
        case 'administrator':
            $menu['general']['control_panel'] = array('title' => _CONTROLCENTER, 'link' => 'administrator.php?ctg=control_panel', 'image' => 'home');
            /** MODULE HCD: Present HCD link **/
            if (MODULE_HCD_INTERFACE) {
               $menu['general']['module_hcd']          = array('title' => _ORGANIZATION,          'link' => 'administrator.php?ctg=module_hcd',   'image' => 'factory');
            }
            $menu['general']['users']         = array('title' => _USERS,         'link' => 'administrator.php?ctg=users',         'image' => 'user1');
            $menu['general']['lessons']       = array('title' => _LESSONS,       'link' => 'administrator.php?ctg=lessons',       'image' => 'lessons');
            $menu['general']['directions']    = array('title' => _DIRECTIONS,    'link' => 'administrator.php?ctg=directions',    'image' => 'kdf');
            $menu['general']['courses']       = array('title' => _COURSES,       'link' => 'administrator.php?ctg=courses',       'image' => 'books');
            //$menu['general']['user_types']    = array('title' => _ROLES,         'link' => 'administrator.php?ctg=user_types',    'image' => 'users_family');
            $menu['general']['user_groups']   = array('title' => _GROUPS,        'link' => 'administrator.php?ctg=user_groups',    'image' => 'users3');
            if ($_SESSION['s_role_options']['statistics'] == 1){
                $menu['general']['statistics'] = array('title' => _STATISTICS,'link' => 'administrator.php?ctg=statistics',       'image' => 'chart');
            }
            $menu['general']['cms']           = array('title' => _CMS,           'link' => 'administrator.php?ctg=cms',           'image' => 'document_text');
            $menu['general']['forum']         = array('title' => _FORUM,         'link' => 'forum/forum_index.php',               'image' => 'messages');
            $menu['general']['messages']      = array('title' => _MESSAGES,      'link' => 'forum/messages_index.php',            'image' => 'mail2');
            $menu['general']['emails']        = array('title' => _EMAILS,        'link' => 'administrator.php?ctg=emails',        'image' => 'mail');
            $menu['general']['chat']          = array('title' => _CHAT,          'link' => 'chat/chat_index.php',                 'image' => 'user1_message');

            foreach ($user_module['administrator'] as $value) {
                if ($value['position'] == 'left') {
                    $menu['general'][$value['name']] = array('title' => $value['title'], 'link' => 'administrator.php?ctg='.$value['name'], 'image' => 'component_green');
                }
            }
            $menu['general']['logout']        = array('title' => _LOGOUT,        'link' => 'index.php?logout=true',               'image' => 'exit');

        break;

        case 'professor' :
            if ($_SESSION['s_lessons_ID'] != false) {
                $menu['lesson']['control_panel'] = array('title' => _MAINPAGE, 'link' => 'professor.php?ctg=control_panel', 'image' => 'home', 'id' => 'lesson_main_a');

                $menu['lesson']['content']       = array('title' => _CONTENTMANAGEMENT, 'link' => 'professor.php?ctg=content',       'image' => 'tests', 'id' => 'content_a');
                //$menu['lesson']['scheduling']    = array('title' => _SCHEDULING,        'link' => 'professor.php?ctg=scheduling',    'image' => 'date-time', 'id' => 'scheduling_a');
                if (isset($_SESSION['s_lesson_options']['projects'])) {
                    $menu['lesson']['projects'] = array('title' => _PROJECTS, 'link' => 'professor.php?ctg=projects', 'image' => 'exercises', 'id' => 'exercises_a');
                }
                if (isset($_SESSION['s_lesson_options']['tests'])) {
                    $menu['lesson']['tests'] = array('title' => _TESTS, 'link' => 'professor.php?ctg=tests', 'image' => 'document_edit', 'id' => 'tests_a');
                }
                if (isset($_SESSION['s_lesson_options']['rules'])) {
                    $menu['lesson']['rules'] = array('title' => _ACCESSRULES, 'link' => 'professor.php?ctg=rules', 'image' => 'recycle', 'id' => 'rules_a');
                }

                if (isset($_SESSION['s_lesson_options']['glossary'])) {
                    $menu['lesson']['glossary'] = array('title' => _GLOSSARY, 'link' => 'professor.php?ctg=glossary', 'image' => 'book_open2', 'id' => 'glossary_a');
                }
                if (isset($_SESSION['s_lesson_options']['forum'])) {
                    $forums_id = eF_getTableData("f_forums", "id", "lessons_ID=".$_SESSION['s_lessons_ID']);
                    if (sizeof($forums_id) > 0) {
                        $menu['lesson']['forum'] = array('title' => _FORUM, 'link' => 'forum/forum_index.php?forum='.$forums_id[0]['id'],'image' => 'messages', 'id' => 'forum_a');
                    } else {
                        $menu['lesson']['forum'] = array('title' => _FORUM, 'link' => 'forum/forum_index.php','image' => 'messages', 'id' => 'forum_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['chat']))  {
                    $chat_lessons_ID = eF_getTableData('chatrooms', 'id,active', 'lessons_ID='.$_SESSION['s_lessons_ID']);
                    if (sizeof($chat_lessons_ID) > 0 && $chat_lessons_ID[0]['active'] !=0) {
                        $menu['lesson']['chat'] = array('title' => _CHAT, 'link' => 'chat/chat_index.php?chatrooms_ID='.$chat_lessons_ID[0]['id'], 'image' => 'user1_message', 'id' => 'chat_a');
                    } else {
                        $menu['lesson']['chat'] = array('title' => _CHAT, 'link' => 'chat/chat_index.php', 'image' => 'user1_message', 'id' => 'chat_a');
                    }
                }
                if ($_SESSION['s_lesson_options']['survey'] && $_SESSION['s_role_options']['survey'] == 1) {
                    $menu['lesson']['survey']        = array('title' => _SURVEY,         'link' => 'professor.php?ctg=survey',       'image' => 'form_green', 'id' => 'survey_a');
                }
                /*if (isset($_SESSION['s_lesson_options']['calendar'])) {
                    $menu['lesson']['calendar']      = array('title' => _CALENDAR,       'link' => 'professor.php?ctg=calendar',     'image' => 'calendar', 'id' => 'calendar_a');
                }*/
                $menu['lesson']['file_manager']      = array('title' => _FILEMANAGER, 'link' => 'professor.php?ctg=content&op=file_manager',     'image' => 'folder_view', 'id' => 'file_manager_a');
                $menu['lesson']['settings']      = array('title' => _LESSONSETTINGS, 'link' => 'professor.php?ctg=settings',     'image' => 'gear', 'id' => 'settings_a');

                foreach ($user_module['professor'] as $value) {
                    if ($value['position'] == 'left' && ($module['mandatory'] != 'false' || isset($_SESSION['s_lesson_options'][$module['name']]))) {
                        $menu['lesson'][$value['name']] = array('title' => $value['title'], 'link' => 'professor.php?ctg='.$value['name'], 'image' => 'component_green', 'id' => 'module');
                    }
                }
            }

            $menu['general']['lessons']  = array('title' => _LESSONS,  'link' => 'professor.php?ctg=lessons',  'image' => 'lessons');
            if (MODULE_HCD_INTERFACE) {
               $menu['general']['module_hcd'] = array('title' => _ORGANIZATION,   'link' => 'professorpage.php?view=organization',   'image' => 'factory', 'target' => '_top');
            }

            if ($_SESSION['s_role_options']['statistics'] == 1) {
                $menu['general']['statistics'] = array('title' => _STATISTICS, 'link' => 'professor.php?ctg=statistics', 'image' => 'chart');
            }
            else{
                $menu['general']['statistics'] = array('title' => _STATISTICS, 'link' => 'professor.php?ctg=statistics', 'image' => 'chart_gray');
            }
            $menu['general']['calendar'] = array('title' => _CALENDAR,       'link' => 'professor.php?ctg=calendar',     'image' => 'calendar');
            $menu['general']['messages'] = array('title' => _MESSAGES, 'link' => 'forum/messages_index.php',   'image' => 'mail2');
            $menu['general']['personal'] = array('title' => _SETTINGS, 'link' => 'professor.php?ctg=personal', 'image' => 'gears');
            $menu['general']['logout']   = array('title' => _LOGOUT,   'link' => 'index.php?logout=true',      'image' => 'exit');
        break;

        case 'student':
            if ($_SESSION['s_lessons_ID'] != false) {
                $menu['lesson']['control_panel'] = array('title' => _MAINPAGE, 'link' => 'student.php?ctg=control_panel', 'image' => 'home', 'target' => "mainframe", 'id' => 'lesson_main_a');

                if (isset($_SESSION['s_lesson_options']['theory'])) {
                    $num = eF_numCurrentContent('theory');
                    if ($num > 0) {
                        $menu['lesson']['theory'] = array('title' => _THEORY, 'link' => 'student.php?ctg=content&type=theory', 'num' => $num, 'image' => 'book_blue', 'target' => "mainframe", 'id' => 'theory_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['examples'])) {
                    $num = eF_numCurrentContent('examples');
                    if ($num > 0) {
                        $menu['lesson']['examples'] = array('title' => _EXAMPLES, 'link' => 'student.php?ctg=content&type=examples', 'num' => $num, 'image' => 'lightbulb_on', 'target' => "mainframe", 'id' => 'examples_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['projects'])) {
                    $num = eF_numCurrentContent('projects');
                    if ($num > 0) {
                        $menu['lesson']['projects'] = array('title' => _PROJECTS, 'link' => 'student.php?ctg=projects', 'num' => $num, 'image' => 'exercises', 'target' => "mainframe", 'id' => 'exercises_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['tests'])) {
                    $num = eF_numCurrentContent('tests');
                    if ($num > 0) {
                        $menu['lesson']['tests'] = array('title' => _TESTS, 'link' => 'student.php?ctg=content&type=tests', 'num' => $num, 'image' => 'tests', 'target' => "mainframe", 'id' => 'tests_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['forum'])) {
                    $forums_id = eF_getTableData("f_forums", "id", "lessons_ID=".$_SESSION['s_lessons_ID']);
                    if (sizeof($forums_id) > 0) {
                        $menu['lesson']['forum'] = array('title' => _FORUM, 'link' => 'forum/forum_index.php?forum='.$forums_id[0]['id'], 'image' => 'messages', 'target' => "mainframe", 'id' => 'forum_a');
                    } else {
                        $menu['lesson']['forum'] = array('title' => _FORUM, 'link' => 'forum/forum_index.php', 'image' => 'messages', 'target' => "mainframe", 'id' => 'forum_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['chat']))  {
                    $chat_lessons_ID = eF_getTableData('chatrooms', 'id,active', 'lessons_ID='.$_SESSION['s_lessons_ID']);
                    if (sizeof($chat_lessons_ID) > 0 && $chat_lessons_ID[0]['active'] !=0) {
                        $menu['lesson']['chat'] = array('title' => _CHAT, 'link' => 'chat/chat_index.php?chatrooms_ID='.$chat_lessons_ID[0]['id'], 'image' => 'user1_message', 'target' => "mainframe", 'id' => 'chat_a');
                    } else {
                        $menu['lesson']['chat'] = array('title' => _CHAT, 'link' => 'chat/chat_index.php', 'image' => 'user1_message', 'target' => "mainframe", 'id' => 'chat_a');
                    }
                }
                if (isset($_SESSION['s_lesson_options']['glossary']) && $_SESSION['s_role_options']['read_content'] == 1) {
                        $menu['lesson']['glossary'] = array('title' => _GLOSSARY, 'link' => 'student.php?ctg=glossary', 'image' => 'book_open2', 'target' => "mainframe", 'id' => 'glossary_a');
                    }
                /*if (isset($_SESSION['s_lesson_options']['calendar'])) {
                        $menu['lesson']['calendar'] = array('title' => _CALENDAR, 'link' => 'student.php?ctg=calendar', 'image' => 'calendar', 'target' => "mainframe", 'id' => 'calendar_a');
                    }*/

                foreach ($user_module['student'] as $value) {
                    if ($value['position'] == 'left' && ($module['mandatory'] != 'false' || isset($_SESSION['s_lesson_options'][$module['name']]))) {
                        $menu['lesson'][$value['name']] = array('title' => $value['title'], 'link' => 'student.php?ctg='.$value['name'], 'image' => 'component_green', 'target' => "mainframe", 'id' => 'modules_i');
                    }
                }
            }

            $menu['general']['lessons']  = array('title' => _LESSONS,  'link' => 'student.php?ctg=lessons',  'image' => 'lessons', 'target' => "mainframe");
            if (MODULE_HCD_INTERFACE) {
               $menu['general']['module_hcd'] = array('title' => _ORGANIZATION,   'link' => 'studentpage.php?view=organization',   'image' => 'factory', 'target' => '_top');
            }

            if ($_SESSION['s_lessons_ID'] != false) {
                $menu['general']['statistics'] = array('title' => _STATISTICS, 'link' => 'student.php?ctg=statistics', 'image' => 'chart', 'target' => "mainframe");
            }
            $menu['general']['calendar'] = array('title' => _CALENDAR, 'link' => 'student.php?ctg=calendar', 'image' => 'calendar', 'target' => "mainframe");
            $menu['general']['messages'] = array('title' => _MESSAGES,   'link' => 'forum/messages_index.php', 'image' => 'mail2', 'target' => "mainframe");
            $menu['general']['personal'] = array('title' => _SETTINGS,   'link' => 'student.php?ctg=personal', 'image' => 'gears', 'target' => "mainframe");
            $menu['general']['logout']   = array('title' => _LOGOUT,     'link' => '/index.php?logout=true',    'image' => 'exit',  'target' => "mainframe");

        break;
    }

    return $menu;
}

/**
* Delete content unit
*
* This function is used to delete the designated unit. The algorithm is as follows:
* 0. Call V the unit to be deleted, Next(U) the next unit of an arbitrary unit U and Prev(U) the previous one.
* 1. Get the tree levels
* 2. Get the prev(V) and Next(V), and set U = V
* 3. While the Next(U) has a lower level than V (which means it is it's child), add Next(U) to the array holding units that will be deleted,
*    and set U = Next(U) . This way, we are trying to find V's next unit that is not a child of it (since its children will be deleted also).
*    After deletion, the unit found will be put after Prev(V).
* 4. If a next unit is found, then set the update array, so that, after deletion, the unit next to V in the same level has Prev equal to Prev(V).
*
* @param int $id The id of the unit to delete
* @param array $tree The content tree
* @return array The new tree, without the deleted nodes.
* @version 1.0
*
* @todo when deleting a copied unit, copy its content to its copies
*/
function eF_deleteUnit($id, $tree = false) {

    if ($tree === false) {
        $tree = eF_getContentTree($nouse, $_SESSION['s_lessons_ID'], 0, false, false);
    }

    $result_unit_to_delete = eF_getTableData("content", "id, data, lessons_ID", "id=$id");
    if (sizeof($result_unit_to_delete) == 0) {                          //If the designated unit does not exist (which would be the case if it is already deleted), just return
        return $tree;
    }

    if (sizeof($tree) > 1) {                                                                //If there is only one node in the tree, there is no need to search for its previous and next units
        $levels    = eF_getLevels($tree);
        $prev_unit = eF_getPreviousContent($id, false, false);
        $next_unit = eF_getNextContent($id, false, false);
        $count     = 0;
        while ($levels[$next_unit['id']] >= $levels[$id] + 1 && $count < sizeof($levels) && $next_unit) {       //Search for the next node, at the same or higher level
            $next_unit = eF_getNextContent($next_unit['id'], false, false);
            $count++;
        }
        if ($next_unit) {
            $update_fields = array('previous_content_ID' => $prev_unit['id']);              //If there was a unit after the one being deleted (at the same or higher level), then update it to point at the unit's previous node
        }
    }

    $delete_ids   = eF_getChildren($id);                                                    //These are the nodes that will be deleted
    $delete_ids[] = $id;                                                                    //Add the selected unit to the list

    $users_units = eF_getTableData("users_to_lessons", "*", "user_type = 'student' and lessons_ID=".$result_unit_to_delete[0]['lessons_ID']);
    for ($i = 0; $i < sizeof($users_units); $i++) {
        $done_units      = unserialize($users_units[$i]['done_content']);
        $different_units = array_diff($done_units, $delete_ids);

        if (is_array($done_units) && sizeof($different_units) != sizeof($done_units)) {

            $fields_update['done_content'] = sizeof($different_units) > 0 ? serialize(array_combine($different_units, $different_units)) : '';
            eF_updateTableData("users_to_lessons", $fields_update, "users_LOGIN='".$users_units[$i]['users_LOGIN']."' and lessons_ID=".$result_unit_to_delete[0]['lessons_ID']);
        }
        if (in_array($users_units[$i]['current_unit'], $delete_ids)) {
            eF_updateTableData("users_to_lessons", array('current_unit' => 0), "users_LOGIN='".$users_units[$i]['users_LOGIN']."' and lessons_ID=".$result_unit_to_delete[0]['lessons_ID']);
        }
    }

    foreach ($delete_ids as $id) {
        $result_copied = eF_getTableData("content", "id", "data='<:link:>".$result_unit_to_delete[0]['id']."'");            //When deleting a unit, check for units that are linked to this unit. If any is found, then copy the unit's content to the copied units, thus destroying the link.
        for ($i = 0; $i < sizeof($result_copied); $i++) {
            eF_updateTableData("content", array('data' => $result_copied[$i]['data']), "id=".$result_copied[$i]['id']);
        }

        eF_deleteTableData("content", "id=$id");                                                                            //Delete the unit
        eF_deleteTableData("current_content", "content_ID=$id");                                                            //Delete the unit from the scheduled content
        eF_deleteTableData("comments", "content_ID=$id");                                                                   //Delete any comments in the unit
        eF_deleteTableData("logs", "comments='$id'");                                                                       //Delete any logs regarding the unit
        eF_deleteTableData("questions", "content_ID=$id");                                                                  //Delete any questions assigned to the unit
        eF_deleteTableData("rules", "rule_content_ID=$id");                                                                 //Delete any rules assigned to the unit, or...
        eF_deleteTableData("rules", "content_ID=$id");                                                                      //...regarding the unit
        eF_deleteTableData("search_keywords", "table_name='content' AND foreign_ID=$id");                                   //Delete unit keywords from the search table
        eF_deleteTableData("scorm_data", "content_ID=$id");                                                              //Delete any scorm data associated with this unit

        $result_test = eF_getTableData("tests", "id", "content_ID=$id");
        eF_deleteTableData("tests", "content_ID=$id");                                                                      //Delete any tests underneath the unit

        for ($i = 0; $i < sizeof($result_test); $i++) {                                                                     //For each one of the deleted test
            $tests_ID = $result_test[$i]['id'];
            if ($tests_ID) {
                eF_deleteTableData("tests_to_questions", "tests_ID=$tests_ID");                                             //Delete tests_to_questions entries
                $result_done_test = eF_getTableData("done_tests", "id", "tests_ID=$tests_ID");
                $done_tests_list  = eF_makeListIDs($result_done_test);
                eF_deleteTableData("logs", "action='tests' AND comments='$id'");                                            //Delete log information regarding the test
                if ($done_tests_list) {
                    eF_deleteTableData("done_tests", "id IN ($done_tests_list)");                                           //Delete done tests
                    eF_deleteTableData("done_questions", "done_tests_ID IN ($done_tests_list)");                            //Delete done questions
                }
            }
        }
    }

    if (isset($update_fields)) {
        eF_updateTableData("content", $update_fields, "id=".$next_unit['id']);              //Apply the order update after the deletion, since we need the correct tree order to perform the deletion
    }

    $tree = eF_getContentTree($nouse, $_SESSION['s_lessons_ID'], 0, false, false);          //Return the new tree
    return $tree;
}

/**
* Inserts a content unit
*
* This function is used to insert a unit to the content table. It makes sure that the unit is placed at the
* right place in the content tree, correcting the order as necessary. It also inserts the unit to the search table
*
* @param array $fields_insert The values that will be inserted to the database, except for previous_content_ID
* @param array $tree The content tree
* @return int The id of the newly inserted unit
* @version 1.0
*/
function eF_insertUnit($fields_insert, $tree) {

    $parent_content_ID = $fields_insert['parent_content_ID'];
    $levels            = eF_getLevels($tree);

    if ($parent_content_ID == 0) {
        if (sizeof($tree) > 0) {
            $next_unit = eF_getNextContent($tree[0]['id'], false, false);       //Get the next unit of the first one.
            if (!$next_unit) {
                $previous_content_ID = $tree[0]['id'];                          //If there is no next unit, this means that the last unit is the first (and only) one.
            }
        } else {
            $next_unit           = false;                                       //There are no other units in the tree; this is the first one.
            $previous_content_ID = 0;
        }
        $levels[0] = -1;
    } else {
        $next_unit = eF_getNextContent($parent_content_ID, false, false);
        if (!$next_unit) {
            $previous_content_ID = $parent_content_ID;                          //If there is no next unit, this means that the last unit is the parent.
        }
    }

    if ($next_unit) {
        $prev_unit = eF_getPreviousContent($next_unit['id'], false, false);
        $count     = 0;
        while ($levels[$next_unit['id']] >= $levels[$parent_content_ID] + 1 && $count < sizeof($levels) && $next_unit) {
            $prev_unit = $next_unit;
            $next_unit = eF_getNextContent($next_unit['id'], false, false);
            $count++;
        }
        $previous_content_ID = $prev_unit['id'];
    }

    $fields_insert['previous_content_ID'] = $previous_content_ID;
    $content_ID = eF_insertTableData("content", $fields_insert);

    if ($next_unit) {
        eF_updateTableData("content", array('previous_content_ID' => $content_ID), "id=".$next_unit['id']);
    }

    EfrontSearch :: insertText($fields_insert['name'], $content_ID, "content", "title");

    return $content_ID;

}


/**
* Prints a list with tests that the student hasn't yet done
*
* This function prints a list with the tests the student hasn't done yet.
* If the $lessons_ID is specified, then only tests for this lesson are displayed.
*
* @param string $login The user login name
* @param int $lessons_ID The lessons id
* @return array An array with the unsolved tests information
* @version 1.0
* @todo example in documentation
*/
function eF_getUnsolvedTests($usrname, $lessons_ID = false)
{
    $unsolved_tests_info = array();

    $lessons_ID ? $sql = " AND c.lessons_ID=$lessons_ID" : $sql = "";

    $result = eF_getTableDataFlat("done_tests d, tests t, content c", "d.tests_ID", "d.tests_ID=t.id and t.content_ID=c.id $sql");
    sizeof($result) == 0 ? $tmp = -1 : $tmp = implode(",", $result['tests_ID']);


    $periods = eF_getTableData("periods", "count(*)", "lessons_ID=".$_SESSION['s_lessons_ID']);
    if ($periods[0]['count(*)'] == 0) {
        $result = eF_getTableData("tests t, content c", "distinct c.id,c.name", "t.content_ID=c.id $sql and t.id not IN ($tmp) AND c.active=1");
    } else {
        $result = eF_getTableData("tests t, content c, current_content cc", "distinct c.id,c.name", "t.content_ID=c.id AND cc.content_ID=c.id $sql and t.id not IN ($tmp) AND c.active=1");
    }

    //$result = eF_getTableData("tests t, content c, current_content cc", "distinct c.id,c.name", "t.content_ID=c.id AND cc.content_ID=c.id $sql and t.id not IN ($tmp) AND c.active=1");

    foreach ($result as $value) {
        if (eF_isDoneContent($value['id'])) {
            $unsolved_tests_info[] = array('content_ID' => $value['id'],
                                           'name'       => $value['name']);
        }
    }

    return $unsolved_tests_info;
}

/**
* Prints a list with tests that the student has done
*
* This function prints a list with the tests that the student has done.
* If the $lessons_ID is specified, then only tests for this lesson are displayed.
*
* @param string $login The user login name
* @param int $lessons_ID The lessons id
* @return array An array with the unsolved tests information
* @version 1.0
* @todo example in documentation
*/
function eF_getSolvedTests($login, $lessons_ID = 0, $type = "student")
{
    $solved_tests_info = array();
    $lessons_ID ? $sql = " AND c.lessons_ID=$lessons_ID" : $sql = "";

    $result = eF_getTableData("done_tests d, tests t, content c", "c.id,c.name,d.score", "d.users_LOGIN='$login' and d.tests_ID=t.id and t.content_ID=c.id $sql");

    foreach ($result as $value) {
        $solved_tests_info[] = array('score'      => round($value['score'] * 100, 2),
                                     'content_ID' => $value['id'],
                                     'name'       => $value['name'],
                                     'login'      => $login);
    }

    return $solved_tests_info;
}



/**
* Converts a timestamp interval to time interval
*
* This function is used to convert the interval specified into a human - readable format.
* <br/> Example:
* <code>
* $timestamp_from = mktime(10, 34, 27, 10, 7, 2005);
* $timestamp_to = mktime(11, 47, 4, 10, 7, 2005);
* $interval = $timestamp_to - $timestamp_from;
* print_r(eF_convertIntervalToTime($interval));
* </code>
* Returns:
* <code>
*Array
*(
*    [hours] => 1
*    [minutes] => 12
*    [seconds] => 37
*)
* </code>
*/
function eF_convertIntervalToTime($interval)
{
    $seconds = $interval % 60;
    $minutes = (($interval - $seconds) / 60) % 60;
    $hours   = ($interval - $seconds - ($minutes * 60)) / 3600;

    //return "$interval $hours:$minutes:$seconds";
    return array('hours' => $hours, 'minutes' => $minutes, 'seconds' => $seconds);
}

/**
* Converts a timestamp interval to time interval
*
* This function is used to convert the interval specified into a human - readable format.
* <br/> Example:
* <code>
* $timestamp_from = mktime(10, 34, 27, 10, 7, 2005);
* $timestamp_to = mktime(11, 47, 4, 10, 7, 2005);
* $interval = $timestamp_to - $timestamp_from;
* print_r(eF_convertIntervalToTimeFull($interval));
* </code>
* Returns:
* <code>
*Array
*(
*    [weeks] => 0
*    [days] => 0
*    [hours] => 1
*    [minutes] => 12
*    [seconds] => 37
*)
* </code>
*/
function eF_convertIntervalToTimeFull($interval)
{
    $seconds = $interval % 60;
    $minutes = (($interval - $seconds) / 60) % 60;
    $hours   = ($interval - $seconds - ($minutes * 60)) / 3600 % 24;
    $days    = floor(($interval - $seconds - ($minutes * 60) - ($hours*24))/ 86400 % 7);
    $weeks   = floor(($interval - $seconds - ($minutes * 60) - 7*($hours*24)) / 604800);
    //return "$interval $string:$weeks:$days:$hours:$minutes:$seconds";
    $string = null;
    if($weeks == 1)         $string .= ' '.$weeks.' '._WEEKs.',';
    elseif($weeks > 1)      $string .= ' '.$weeks.' '._WEEKS.',';
    if($days == 1)          $string .= ' '.$days.' '._DAY.',';
    elseif($days > 1)       $string .= ' '.$days.' '._DAYS.',';
    if($hours == 1)         $string .= ' '.$hours.' '._HOUR.',';
    elseif($hours > 1)      $string .= ' '.$hours.' '._HOURS.',';
    if($minutes == 1)       $string .= ' '.$minutes.' '._MINUTE.',';
    elseif($minutes > 1)    $string .= ' '.$minutes.' '._MINUTES.',';
    if($seconds > 0)        $string .= ' '.$seconds.' '._SECONDS;

    return array('string' => $string,'weeks' => $weeks, 'days' => $days, 'hours' => $hours, 'minutes' => $minutes, 'seconds' => $seconds);
}

/**
* Processes a message string
*
* This function gets a string as an argument, that comes from either a message editor (forum or personal), or the chat text box.
* It then formats its according to some modifiers that maybe present: [b][/b] for bold, [i][/i] for italics, [u][/u] for underlined
* and [link=<target>][/link] for a link reference. It also converts text smilies to their corresponding gif icons.
*
* @param string $msg The message to process
* @return string The processed message
* @version 1.0 (20/11/2005)
*/
function eF_messageProcess($msg)
{
    $search  = array("/(.*)\n(.*)/U",
                     "/\[b\](.*)\[\/b\]/U",
                     "/\[i\](.*)\[\/i\]/U",
                     "/\[u\](.*)\[\/u\]/U",
                     "/\040/U",
                     "/\[link=(.*)\](.*)\[\/link\]/U",
                     "/\[quote\](.*)\[\/quote\]/");
    $replace = array("\$1<br>\$2",
                     "<b>\$1</b>",
                     "<i>\$1</i>",
                     "<u>\$1</u>",
                     "&nbsp;",
                     "<a href = \"javascript:void(0)\" onclick = \"popUp(\'\$1\', 600, 400, 1)\">\$2</a>",
                     "<table><tr><td style = \"background-color:lightblue\">\$1</td></tr></table>");

    $new_msg = preg_replace($search, $replace, $msg);
    $new_msg = eF_convertTextToSmilies($new_msg);

    return $new_msg;
}

/**
* Revert process of a message string
*
* This function gets a string as an argument and applies the reverse process of the previous function.
*
* @param string $msg The message to process
* @return string The processed message
* @version 1.0 (25/11/2005)
* @see eF_messageProcess()
*/
function eF_revertMessageProcess($msg)
{
    $search  = array("/<b>(.*)<\/b>/U",
                     "/<i>(.*)<\/i>/U",
                     "/<u>(.*)<\/u>/U",
                     "/<a href = 'javascript:void\(0\)' onclick = 'popUp\(\"(.*)\", 600, 400, 1\)'>(.*)<\/a>/U",
                     "/&nbsp;/U",
                     "/<table><tr><td style = 'background-color:lightblue'>(.*)<\/td><\/tr><\/table>/U",
                     "/(.*)<br>(.*)/U");
    $replace = array("[b]\$1[/b]",
                     "[i]\$1[/i]",
                     "[u]\$1[/u]",
                     "[link=\$1]\$2[/link]",
                     " ",
                     "[quote]\$1[/quote]",
                     "\$1\n\$2");

    $new_msg = preg_replace($search, $replace, $msg);
    $new_msg = eF_convertTextToSmilies($new_msg);

    return $new_msg;
}

/**
* 18_1_2007 add case for periods...makriria.. now, in lessons_list op, it appears only current content numbers
*/
function getLessonContentUnits($lessons_ID = false, $nonempty = false) {
    if (!$lessons_ID) {
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    ($nonempty) ? $nonempty = " AND content.data != '' " : $nonempty = "";

    $time = time();
    $offset = eF_getOffset();
    $result = eF_getTableData("periods", "id", "lessons_ID=".$lessons_ID);
    if (sizeof($result) > 0) {
        $units = eF_getTableData("current_content,periods,content", "count(*)", "ctg_type != 'tests' AND periods.lessons_ID = ".$lessons_ID." AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND periods.to_timestamp+$offset>$time AND content.id=current_content.content_ID AND content.active=1".$nonempty);

        $tests = eF_getTableData("current_content,periods,content", "count(*)", "ctg_type = 'tests' AND periods.lessons_ID = ".$lessons_ID." AND current_content.periods_ID=periods.id AND periods.from_timestamp+$offset<=$time AND periods.to_timestamp+$offset>$time AND content.id=current_content.content_ID AND content.active=1");

    } else {
        $units = eF_getTableData("content", "count(*)", "ctg_type != 'tests' and lessons_ID = $lessons_ID and active = 1".$nonempty);
        $tests = eF_getTableData("content", "count(*)", "ctg_type = 'tests' and lessons_ID = $lessons_ID and active = 1");
    }


    return array('units' => $units[0]['count(*)'], 'tests' => $tests[0]['count(*)']);
}



/**
* Get number of unread messages
*
* This function returns the number of a user's unread messages.
* If $login is not set, the current user is assumed.
*
* @param string $login The user to check messages for
* @return int The number of unread messages
* @version 1.0 (9/12/2005)
*/
function eF_getUnreadMessagesNumber($login = false) {
    if (!$login) {
        $login = $_SESSION['s_login'];
    }

    $messages = eF_getTableData("f_personal_messages pm, f_folders", "count(*)", "pm.users_LOGIN='".$login."' and viewed='no' and f_folders_ID=f_folders.id and f_folders.name='Incoming'");

    return $messages[0]['count(*)'];
}

/**
* Recalculate test score
*
* ? ????????? ???? ??????? ??? ?????? ?? id ??? ???????? (??? ???? ???????) ??? ?????????? ?? ???? ??? ???? ??? ??? ?????????????
* @version 2.1 (6/12/2006 - makriria)
*/

function eF_test_recalculate($question_id) {

$done_tests_changed = eF_getTableData("done_tests as dt,tests_to_questions as tq","*","dt.tests_ID=tq.tests_ID AND tq.questions_ID=".$question_id);

for($l = 0; $l<sizeof($done_tests_changed); $l++) {   // gia kathe test poy exei eggrafi sto done_tests kai periexei afti tin erotisi
    $tests_ID = $done_tests_changed[$l]['tests_ID'];
    $done_tests_ID = $done_tests_changed[$l]['id'];

    $res0 = eF_getTableData("tests_to_questions","SUM(weight) AS all_weight","tests_ID=$tests_ID");
    $all_weight     = $res0[0]['all_weight'];

    $score          = 0;
    $text_questions = 0;


    $questions_ids = eF_getTableData("questions, tests_to_questions", "questions.id", "tests_to_questions.tests_ID=$tests_ID AND tests_to_questions.questions_ID=questions.id","questions.content_ID, questions.id ASC");

    $given_answers = eF_getTableData("done_questions,tests_to_questions","answer,done_questions.questions_ID","done_questions.questions_ID=tests_to_questions.questions_ID AND done_questions.done_tests_ID=".$done_tests_ID." AND tests_ID=".$tests_ID);

    $q =array();
    for($k = 0; $k<sizeof($given_answers); $k++) {   // fernei tis dosmenes apantiseis sti morfi poy einai otan ginetai kai submit to test
        $q[$k] = array($given_answers[$k]['questions_ID'] => $given_answers[$k]['answer']);
    }

    foreach ($q as $key0 => $value0) {    //gia kathe erotisi aftoy toy test
        foreach ($value0 as $questions_ID => $s_answer){

            if ($question['type'] != "raw_text") {
                $answer = unserialize($s_answer);
            }else{
                $answer = $s_answer;
            }

            $res      = eF_getTableData("questions","*","id=".$questions_ID);
            $question = $res[0];
            $res1     = eF_getTableData("tests_to_questions","weight","questions_ID=$questions_ID AND tests_ID=$tests_ID");
            $q_weight = $res1[0]['weight'];


            if ($question['type'] == "raw_text") {
                $text_questions++;
                unset($q_score);
            } elseif ($question['type'] == "multiple_one") {
                $right_answer = unserialize($question['answer']);
                $right_answer = $right_answer[0] - 1;

                if ($answer == $right_answer) {
                    $q_score = 1;
                } else {
                    $q_score = 0;
                }

            } elseif ($question['type'] == "multiple_many") {
                $right_answer = unserialize($question['answer']);
                sort($right_answer, SORT_NUMERIC);
                for ($i = 0; $i < sizeof($right_answer); $i++) {
                    $right_answer[$i]--;
                }
                $result       = array_intersect($right_answer,$answer);
                $more_answers = sizeof($answer)-sizeof($right_answer);
                $more_answers = ($more_answers > 0) ? $more_answers : 0;
                $q_score      = (sizeof($result) - $more_answers) / sizeof($right_answer);
                $q_score      = ($q_score > 0) ? $q_score : 0;
            }elseif ($question['type'] == "match") {
                $result = 0;

                $right_answer = unserialize($question['answer']);

                for($j=0; $j<sizeof($right_answer); $j++) {
                    if ($answer[$j] == $right_answer[$j]){
                            $result++;
                    }

                }
                $q_score = $result / sizeof($answer);
            } elseif ($question['type'] == "empty_spaces") {
                $right_answer = unserialize($question['answer']);
                $result       = 0;
                for ($i = 0; $i < sizeof($answer); $i++) {
                    $temp1 = eF_noTonos($right_answer[$i]);
                    $temp2 = eF_noTonos($answer[$i]);
                    if (strcasecmp($temp1,$temp2) == 0) {
                        $answer[$i] = $right_answer[$i];
                        $result++;
                    }
                }
                $q_score = $result / sizeof($answer);
            }
            $score += ($q_weight * $q_score) / $all_weight;

            unset($fields);

            $fields['score']         = $q_score;

            eF_updateTableData("done_questions", $fields,"done_tests_ID=".$done_tests_ID." AND questions_ID=".$questions_ID);
}
}   //ta 2 foreach
            unset($q);
            unset($fields);
            $fields['score'] = $score;
            eF_updateTableData("done_tests", $fields, "id=$done_tests_ID");

unset($score);

}
}
function eF_structureTableUsers($table, $change_field) {   //$table opos epistrefetai apo mia eggrafi toy user_profile

if($table['db_type'] == "text") {
    $db_type_str = "varchar(255)";       // needs modification in SQL server
}elseif($table['db_type'] == "number") {
    $db_type_str = "int(255)";
}

if($table['mandatory'] == 1) {
    $allow_null_str = "NOT NULL";
}else{
    $allow_null_str = "";
}

if($table['default_value'] != "" && $table['type']!= "checkbox")  {
    $default_str = "default '".$table['default_value']."'";
}elseif($table['type'] == "checkbox"){
    $table['default_value'] == "true" ? $default_str = "default '1'" : $default_str = "default '0'";
}

switch ($table['type']) {
    case 'textbox':

    break;

    case 'checkbox':
        $db_type_str = "tinyint(1)";   // needs modification in SQL server
    break;
  /*  case 'selectbox':
        $possible_values = unserialize($table['options']);

        $db_type_str = "enum( ";                 // needs modification in SQL server

        foreach($possible_values as $pos_value) {
            $db_type_str .= "'".$pos_value."',";
        }
        $db_type_str = mb_substr($db_type_str, 0, sizeof($db_type_str)-2);
        $db_type_str .= ")";   */
    case 'selectbox':
        $possible_values = unserialize($table['options']);

        $db_type_str = "varchar(255)";                 // needs modification in SQL server

//echo $db_type_str; exit;
    break;
    default:
    break;

}
    if($change_field == "") {
        eF_alterTableStructureAdd("users", $table['name'], $db_type_str, $default_str, $allow_null_str);
    }elseif($change_field != ""){
        eF_alterTableStructureChange("users", $change_field, $db_type_str, $default_str, $allow_null_str);
    }
}
//afti tha paei sto db kapote
function eF_alterTableStructureAdd($table, $column, $db_type_str, $default_str, $allow_null_str) {

    eF_execute("alter table `".$table."` add `". $column."` ".$db_type_str." ".$allow_null_str." ".$default_str);
}

function eF_alterTableStructureChange($table, $column, $db_type_str, $default_str, $allow_null_str) {

    eF_execute("alter table `".$table."` change `". $column."` `". $column."` ".$db_type_str." ".$allow_null_str." ".$default_str);
}

/**
* Get glossary words
*
* This function is used to return an array of words, divided per initial letter, given an array of
* glossary entries.
*
* @return array letters: An array of words, divided in subarrays per letter
* @version 0.9
* @todo
*/
function eF_getAllGlossaryWords($words) {

    $initials = array();
    foreach($words as $key => $value) {
        $letter = mb_substr($value['name'],0,1);
        if($letter=="?" || $letter=="?")
            $letter="?";
        if($letter=="?" || $letter=="?")
            $letter="?";
        if($letter=="?" || $letter=="?")
            $letter="?";
        if($letter=="?" || $letter=="?" || $letter=="?" || $letter=="?" || $letter=="?")
            $letter="?";
        if($letter=="?" || $letter=="?")
            $letter="?";
        if($letter=="?" || $letter=="?" || $letter=="?" || $letter=="?" || $letter=="?")
            $letter="?";
        $letter = mb_strtoupper($letter);
        //echo "LETTER: ".$letter." ASCII: ".ord($letter)."<br/>";
        if (preg_match("/[0-9]/", $letter)) {
            $initials["0-9"][$letter][] = $words[$key];
        } else if (!preg_match("/\w/", $letter)) {
            $initials["Symbols"][$letter][] = $words[$key];
        } else {
            $initials[$letter][] = $words[$key];
        }//$initials[$letter]=1;
    }
    $setNum = isset($initials["0-9"]);
    $setSym = isset($initials["Symbols"]);
    if( $setNum || $setSym )
    {
        $tempNum = $initials["0-9"];
        $tempSym = $initials["Symbols"];
        unset($initials["0-9"]);
        unset($initials["Symbols"]);
        ksort($initials);
        if($setNum)
            $initials["0-9"] = $tempNum;
        if($setSym)
            $initials["Symbols"] = $tempSym;
    }
    else
        ksort($initials);

    //print_r($initials);

    return $initials;
}
function  eF_checkUserType($username){
    $user_type = eF_getTableData("users","user_type","login=\"".$username."\"");
    return $user_type;
}



/**
* Returns the current lesson name
*
* This function returns the name of the current lesson
*
* @return string The lesson name
* @version 1.0
* @Date 2007/03/09
*/
function eF_getLessonName() {
    if (isset($_SESSION['s_lessons_ID'])) {
        $lesson = eF_getTableData("lessons", "name", "id=".$_SESSION['s_lessons_ID']);
        return $lesson[0]['name'];
    } else {
        return false;
    }
}

/**
* Returns questions of a test (for unsolved test)in correct order
*
* This function returns the questions of a test in correct order
* @param id $tests_ID The test id
* @return array with the questions of test
* in new format there is previous_question_ID field. If there is one row with previous_question_ID=0, then there is defined order of questions. Otherwise there is no order defined and the array returned in the old way.
* @version 1.0
* @Date 2007/03/16
*/
function eF_getTestQuestions($tests_ID, $shuffle = false){

  if ($shuffle == true){
     $questions_order = eF_getTableData("questions, tests_to_questions", "questions.*,tests_to_questions.weight AS weight", "tests_to_questions.tests_ID=$tests_ID AND tests_to_questions.questions_ID=questions.id","questions.content_ID, questions.id ASC");
     shuffle($questions_order);
 } else {

    $res = eF_getTableData("questions,tests_to_questions","questions.*,tests_to_questions.weight AS weight,questions_ID","tests_to_questions.tests_ID=$tests_ID AND tests_to_questions.questions_ID=questions.id AND previous_question_ID=0");
    if (sizeof($res) == 1) {  //new format of tests_to_questions
        $previous_ques_ID = $res[0]['questions_ID'];
        $questions_order[] = $res[0];
        while (sizeof($res)!=0) {
            $res = eF_getTableData("questions,tests_to_questions","questions.*,tests_to_questions.weight AS weight,questions_ID","tests_to_questions.tests_ID=$tests_ID AND tests_to_questions.questions_ID=questions.id AND previous_question_ID=".$previous_ques_ID);
            if (sizeof($res)!=0 ){
                $previous_ques_ID = $res[0]['questions_ID'];
                $questions_order[] = $res[0];

            }
        }
    } else { //old format of tests_to_questions
        $questions_order = eF_getTableData("questions, tests_to_questions", "questions.*,tests_to_questions.weight AS weight", "tests_to_questions.tests_ID=$tests_ID AND tests_to_questions.questions_ID=questions.id","questions.content_ID, questions.id ASC");
    }
  }
    return $questions_order;
}

/**
* Returns questions of a test (for solved test)in correct order
*
* This function returns the questions of a test in correct order
* @param id $tests_ID The test id
* @param string $login  The login of person who solve the test
* @return array with the questions of test
* in new format there is previous_question_ID field. If there is one row with previous_question_ID=0, then there is defined order of questions. Otherwise there is no order defined and the array returned in the old way.
* @version 1.0
* @Date 2007/03/16
*/
function eF_getTestQuestionsForSolved($login,$tests_ID, $shuffle = false){
 if($shuffle == true){

    $temp = eF_getTableData("users_to_done_tests","questions_order","users_LOGIN='$login' AND tests_ID=$tests_ID");
    $temp = unserialize($temp[0][questions_order]);
    for($k=0;$k < sizeof($temp); $k++) {
        $temp2 = eF_getTableData("questions, tests_to_questions, done_questions, done_tests", "DISTINCT questions.*", "done_tests.users_LOGIN='$login' AND done_questions.questions_ID=questions.id AND done_questions.done_tests_ID=done_tests.id AND tests_to_questions.tests_ID = done_tests.tests_ID AND tests_to_questions.questions_ID = questions.id AND done_tests.tests_ID=$tests_ID AND questions.id=".$temp[$k]);
        $questions_order[$k] = $temp2[0];
    }
 } else{
    $res = eF_getTableData("questions, tests_to_questions, done_questions, done_tests", "DISTINCT questions.*,tests_to_questions.questions_ID", "done_tests.users_LOGIN='$login' AND done_questions.questions_ID=questions.id AND done_questions.done_tests_ID=done_tests.id AND done_tests.tests_ID=$tests_ID AND tests_to_questions.tests_ID = done_tests.tests_ID AND tests_to_questions.questions_ID = questions.id AND tests_to_questions.previous_question_ID=0");
    if (sizeof($res) == 1) {  //new format of tests_to_questions
        $previous_ques_ID = $res[0]['questions_ID'];
        $questions_order[] = $res[0];
        while (sizeof($res)!=0) {
            $res = eF_getTableData("questions, tests_to_questions, done_questions, done_tests", "DISTINCT questions.*,tests_to_questions.questions_ID", "done_tests.users_LOGIN='$login' AND done_questions.questions_ID=questions.id AND done_questions.done_tests_ID=done_tests.id AND done_tests.tests_ID=$tests_ID AND tests_to_questions.tests_ID = done_tests.tests_ID AND tests_to_questions.questions_ID = questions.id AND previous_question_ID=".$previous_ques_ID);
            if (sizeof($res)!=0 ){
                $previous_ques_ID = $res[0]['questions_ID'];
                $questions_order[] = $res[0];

            }
        }
    } else { //old format of tests_to_questions
    $questions_order = eF_getTableData("questions, tests_to_questions, done_questions, done_tests", "DISTINCT questions.*", "done_tests.users_LOGIN='$login' AND done_questions.questions_ID=questions.id AND done_questions.done_tests_ID=done_tests.id AND tests_to_questions.tests_ID = done_tests.tests_ID AND tests_to_questions.questions_ID = questions.id AND done_tests.tests_ID=$tests_ID","questions.content_ID,questions.id ASC");
    }
 }
    return $questions_order;
}


function eF_getDirections($directions_ID = 0) {
    $directions = eF_getTableData("directions", "*");
    foreach ($directions as $direction) {                                   //Assign directions to temp, having keys match direction ids
        $temp[$direction['id']] = $direction;
    }

    foreach ($temp as $direction_id => &$direction) {

    }
}


function eF_getDirectionsTree($directions_ID){

$dir_name = eF_getTableData("directions","name","id=".$directions_ID);
    while ($directions_ID != 0){
        $res = eF_getTableData("directions","parent_direction_ID,name","id=".$directions_ID);
        $directions_ID = $res[0]['parent_direction_ID'];
        $directions[]= array("name" => $res[0]['name']);

    }

return array_reverse($directions);

}

function eF_getDirections_Subtree($directions_ID) { // afti apothikeyei ta sub_directions enos epipedou pio mesa
$result = eF_getTableData("directions", "*","id=".$directions_ID);
$subtree = $result[0];
$subtree['lessons'] = eF_getTableData("lessons,users_to_lessons", "id, name, price, lessons.active, languages_NAME, info, completed","directions_ID=".$directions_ID." AND lessons.id=users_to_lessons.lessons_ID AND users_LOGIN='".$_SESSION['s_login']."' AND users_to_lessons.active=1 AND users_to_lessons.from_timestamp!=0 AND lessons.active=1 AND course_only=0","name asc");

for($k=0; $k< sizeof($subtree['lessons']); $k++){
    $subtree['lessons'][$k]['user_type'] = eF_getUserBasicType($_SESSION['s_login'], $subtree['lessons'][$k]['id']);
    $subtree['lessons'][$k]['role'] = eF_getUserRole($_SESSION['s_login'], $subtree['lessons'][$k]['id']);
}
$subtree['sub_directions'] = eF_getTableData("directions", "id, name, active", "parent_direction_ID=".$directions_ID,"name asc");

    $courses_list = eF_getTableData("courses,users_to_courses as uc", "courses.*,uc.user_type as role","uc.courses_ID=courses.ID AND uc.active=1 AND uc.from_timestamp!=0 AND directions_ID=$directions_ID AND uc.users_LOGIN='".$_SESSION['s_login']."'","courses.name asc");

    for ($k = 0; $k < sizeof($courses_list); $k++){
        $subtree['courses_list'][$k]['id'] = $courses_list[$k]['id'];
        $subtree['courses_list'][$k]['name'] = $courses_list[$k]['name'];
        $lessons_array = unserialize($courses_list[$k]['lessons']);
        $status        = eF_getCourseRulesPassed($courses_list[$k]['id'], $lessons_array, $_SESSION['s_login']);
        $lessons       = eF_getTableData("lessons", "id,name,info,price", "id IN ( ".implode(",", $lessons_array)." )", "name asc");

        $selectable_lessons     = array();
        $non_selectable_lessons = array();
        for ($i = 0; $i < sizeof($lessons); $i++){
            $content     = getLessonContentUnits($lessons[$i]['id']);
            $new_content = sizeof(eF_getNewContent($_SESSION['s_login'], $lessons[$i]['id'], 'content'));

            $temp = array("id"          => $lessons[$i]['id'],
                          "name"        => $lessons[$i]['name'],
                          "info"        => unserialize($lessons[$i]['info']),
                          "new_content" => $new_content,
                          "content"     => $content,
                          "completed"   => $lessons_list[$lessons[$i]['id']],
                          "selectable"  => $status[$lessons[$i]['id']]['selectable'],
                          "dependence"  => $status[$lessons[$i]['id']]['conditions']);
            $status[$lessons[$i]['id']]['selectable'] ? $selectable_lessons[] = $temp : $non_selectable_lessons[] = $temp;    //in order to discriminate selectable from non-selectable lessons, we create 2 separate arrays, based on the selectable attribute, which we then merge.

        }

        $subtree['courses_list'][$k]['lessons_names'] = array_merge($selectable_lessons, $non_selectable_lessons);

        if ($courses_list[$k]['role'] != "student" && $courses_list[$k]['role'] != "professor"){
            $temp = eF_getTableData("user_types", "basic_user_type", "user_type='".$courses_list[$k]['role']."'");
            $subtree['courses_list'][$k]['user_type'] = $temp[0]['basic_user_type'];
        } else {
           $subtree['courses_list'][$k]['user_type'] = $courses_list[$k]['role'];
        }
        $subtree['courses_list'][$k]['role'] = $courses_list[$k]['role'];
    }
//pr($subtree);
return $subtree;
}

function eF_getSurveyInfo($lesson_id){
    $survey_about = eF_getTableData("surveys","id,survey_code,survey_name,start_date,end_date,status","lessons_ID=".$lesson_id);
    $survey_questions = array();
    for($i = 0 ; $i < sizeof($survey_about) ; $i ++)
        $survey_questions[$i] = eF_getTableData("questions_to_surveys","count(*)","surveys_ID=".$survey_about[$i]['id']);
    $data = array('survey_info' => $survey_about,
                  'survey_questions' => $survey_questions);
    return $data;
}

function eF_getSurveyQuestions($survey_id){
    $data = eF_getTableData("questions_to_surveys","id,surveys_ID,father_ID,type,question,answers,created,info","surveys_ID=".$survey_id,"father_ID ASC");
    if(sizeof($data) == 0)
        return 0;
    else
    return $data;
}

function eF_getSurveyStatistics($survey_id){
    $survey_questions = eF_getTableData("questions_to_surveys","type,question,answers","surveys_ID=".$survey_id,"father_ID ASC");
    $done_users = eF_getTableData("users_to_done_surveys","users_LOGIN","surveys_ID=".$survey_id);

    $votes = array();
    for($i = 0 ; $i < sizeof($done_users) ; $i +=1){
        $user = '"'.$done_users[$i]['users_LOGIN'].'"';
        $user_answers = eF_getTableData("survey_questions_done sqd ,questions_to_surveys qts","sqd.user_answers,qts.type","sqd.question_ID = qts.id AND sqd.surveys_ID=".$survey_id." AND qts.surveys_ID=".$survey_id." AND qts.surveys_ID=sqd.surveys_ID AND sqd.users_LOGIN=".$user,"qts.father_ID ASC");
        $vote = array();

        for($j = 0 ; $j < sizeof($survey_questions) ; $j+=1){
            if($user_answers[$j]['type'] == 'multiple_many'){
                $choices = unserialize($survey_questions[$j]['answers']);
                $type = array_keys($choices);
                $keys = array_keys($choices[$type[0]]);
                $needles = unserialize($user_answers[$j]['user_answers']);
                for($k = 0 ; $k < sizeof($choices[$type[0]]) ; $k +=1){
                    $place = array_search($needles[$k],$choices[$type[0]]);
                    if((string)$place != ''){
                        $vote[$j][$keys[$k]] =$place;
                    }else{
                        $vote[$j][$keys[$k]] == -1;
                    }
                }
            }else{
                if($user_answers[$j]['type'] != 'development'){
                    $choices = unserialize($survey_questions[$j]['answers']);
                    $needle = unserialize($user_answers[$j]['user_answers']);
                    $type = array_keys($choices);
                    $keys = array_keys($choices[$type[0]]);
                    $place = array_search($needle,$choices[$type[0]]);
                    $vote[$j]=$place;
                }else{
                    $vote[$j] =1;
                }
            }
        }
        $votes[$i] = $vote;
    }

    return array('questions' => $survey_questions , 'votes' => $votes);
}

/**
* Check if the specified needle exists in the database
*
* This function is used to check against the database if the specified needle already exists.
* Type may be one of login, mail
* Example:<br>
* <code>
* eF_checkNotExist('john', 'login');                  //returns true if it exists
* eF_checkNotExist('jdoe@somewhere.net', 'mail');     //returns true if it exists
* </code>
*
* @param string $needle The string to check for
* @param string type The data type
* @return bool True if the string exists
* @version 1.0
*/
function eF_checkNotExist($needle, $type) {
    switch ($type) {
        case 'login':
            $result = eF_getTableData("users", "login", "login='$needle'");
            break;
        case 'email':
            $result = eF_getTableData("users", "email", "email='$needle'");
            break;
        case 'user_type':
            $result = eF_getTableData("user_types", "user_type", "user_type='$needle'");
            break;
        case 'course':
            $result = eF_getTableData("courses", "name", "name='$needle'");
            break;
        default:
            $result = array();
            break;
    }

    if (sizeof($result) > 0) {
        return false;
    } else {
        return true;
    }
}

function eF_passwdCheck($password){
    if( mb_strlen($password) < 6){
        return false;
    }
    return true;
}

/**
* Get the current periods
*
* This function returns the current periods, based on the current date
*
* @param int $lesson_id the current lesson id, defaults to corresponding session variable
* @return array The current periods ids
* @version 0.1
* @todo must become compatible with periods per student
*/
function eF_getCurrentPeriods($lesson_id = false) {
    if (!$lesson_id) {
        $lesson_id = $_SESSION['s_lessons_ID'];
    }

    $result = eF_getTableData("periods", "id, name", "lessons_ID = ".$lesson_id." and from_timestamp < ".time()." and to_timestamp > ".time());

    return $result;
}


/**
* Return the custom css
*
* This function checks whether there is a custom css file set and returns its name.
*
* @return string The custom CSS name
* @version 0.1
*/
function eF_getCustomCss() {
    $css = eF_getTableData("configuration", "value", "name='css'");
    if ($css && eF_checkParameter($css[0]['value'], 'filename') && is_file(G_ROOTPATH.'www/css/custom_css/'.$css[0]['value'])) {
        return $css[0]['value'];
    } else {
        return 'normal.css';
    }
}


function pr($ar) {
    echo "<pre>";print_r($ar);echo "</pre>";
}




/**
* Get student Progress
*
* This function can be used to get the progress information for every student lesson
* Example:<br>
* <code>
* $progress = eF_getStudentProgress('jdoe');        //$progress now holds an array of the form 'lessons_ID' => 'progress' for every student lesson
* $progress = eF_getStudentProgress('jdoe', 13);    //$progress now holds a 1 element array of the form 'lessons_ID' => 'progress', for the lesson with id 13
* $progress = eF_getStudentProgress('jdoe', array(13,15,17));    //$progress now holds a 3-elements array of the form 'lessons_ID' => 'progress', for the lessons specified
* </code>
*
* @param string $login The student login
* @param string $lessons The array of lessons id (may be a single value for just 1 lesson)
* @return array The progress array
* @version 1.0
*/
function eF_getStudentProgress($login, $lessons = false) {
    if (!$lessons) {                                                                                    //If $lessons is not specified, all lessons are assumed
        $lessons = eF_getTableDataFlat("users_to_lessons", "lessons_ID", "users_LOGIN='".$login."'");   //Get all user lessons
    } elseif (!is_array($lessons)) {                                                                    //If $lessons is a single value, convert it to 1-element array
        $lessons = array('lessons_ID' => array($lessons));
    }

    $progress = array();                                                                                //Initialize variable, so it never returns undefined value
    foreach ($lessons['lessons_ID'] as $lessons_ID) {                                                   //Get the progress for each lesson id
        $content      = getLessonContentUnits($lessons_ID, true);                                             //Get the content units
        $seen_content = eF_getSeenContent($login, $lessons_ID);                                         //Get the seen content
        array_sum($content) > 0 ? $progress[$lessons_ID] = round(100 - 100 * (array_sum($content) - sizeof($seen_content)) / array_sum($content)) : $progress[$lessons_ID] = '0';   //Calculate the progress percentage
    }

    return $progress;
}

/**
* Get student progress in relation to lesson conditions
*
* This function checks for the given students, which of the lesson conditions are met
* to decide whether or not he has completed the lesson
*
* @param mixes $login The user or an array of users
* @param int $lessons_ID The lesson id
* @param array $tree The content tree
* @return array The conditions and lesson status
*/
function eF_getStudentsLessonStatus($lessons_ID, $users = false, $tree = false) {
    if (!$tree) {
        $tree  = eF_getContentTree($nouse, $lessons_ID, 0, false, true, false, true, false);
    }

    $num_units = 0;                                                                 //We need num_units, because some tree units may be empty. in this case, they must not be calculated in the total tree units
    foreach ($tree as $key => $value) {
        if ($value['data'] || $value['ctg_type'] == 'tests') {                                                       //Get the actual number of units, excluding the empty ones.
            $num_units++;
        }
    }

    if (!$users) {
        $lesson_users = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login, u.name, u.surname, ul.completed, ul.score, ul.comments, ul.issued_certificate", "ul.user_type = 'student' and ul.lessons_ID=".$lessons_ID." and ul.users_LOGIN=u.login");
    } else if (!is_array($users)) {                                                        //In case of a single login, convert it to array
        $lesson_users = eF_getTableDataFlat("users u, users_to_lessons ul", "u.login, u.name, u.surname, ul.completed, ul.score, ul.comments, ul.issued_certificate", "u.login='".$users."' and ul.users_LOGIN=u.login and ul.lessons_ID=".$lessons_ID);
    }

    $conditions = eF_getTableData("lesson_conditions", "*", "lessons_ID=".$lessons_ID);

    foreach ($lesson_users['login'] as $key => $login) {
        unset($lesson_passed);
        $seen_content  = eF_getSeenContent($login, $lessons_ID);
        $number_passed = 0;
        for ($i = 0; $i < sizeof($conditions); $i++) {
            $conditions = eF_getTableData("lesson_conditions", "*", "lessons_ID=".$lessons_ID);
            $conditions[$i]['options'] = unserialize($conditions[$i]['options']);

            switch ($conditions[$i]['type']) {
                case 'all_units':
                    sizeof($seen_content) >= $num_units ? $passed = true : $passed = false;
                    break;
                case 'percentage_units':
                    round($num_units * $conditions[$i]['options'][0] / 100) > sizeof($seen_content) ? $passed = true : $passed = false;
                    break;
                case 'specific_unit':
                    isset($seen_content[$conditions[$i]['options'][0]]) ? $passed = true : $passed = false;
                    break;
                case 'all_tests':
                    $done_tests = eF_getTableDataFlat("done_tests as dt,tests as t, content as c", "dt.score", "dt.users_LOGIN='".$login."' AND t.content_ID = c.id AND c.lessons_ID = ".$lessons_ID." AND t.id=dt.tests_ID");
                    round(100 * array_sum($done_tests['score']) / sizeof($done_tests['score']), 2) >= $conditions[$i]['options'][0] ? $passed = true : $passed = false;
                    break;
                case 'specific_test':
                    $done_tests = eF_getTableDataFlat("done_tests as dt,tests as t, content as c", "dt.score", "dt.users_LOGIN='".$login."' AND t.content_ID = c.id AND c.lessons_ID = ".$lessons_ID." AND t.id=dt.tests_ID and c.id=".$conditions[$i]['options'][0]);
                    $done_tests['score'][0] >= $conditions[$i]['options'][1] ? $passed = true : $passed = false;
                    break;
                default:
                    break;
            }
            $conditions[$i]['passed'] = $passed;
            if ($passed) {
                $number_passed++;
            }
            if (isset($lesson_passed)) {
                $conditions[$i]['relation'] == 'and' ? $lesson_passed = $lesson_passed & $passed : $lesson_passed = $lesson_passed | $passed;
            } else {
                $lesson_passed = $passed;
            }
        }
        $lesson_users['total_conditions'][$key]  = sizeof($conditions);
        $lesson_users['conditions_passed'][$key] = $number_passed;
        $lesson_users['lesson_passed'][$key]     = $lesson_passed;
    }

    return $lesson_users;

}

if (! function_exists("array_fill_keys")) {
    function array_fill_keys(array $keys, $value) {
        return array_combine($keys, array_fill(0, count($keys), $value));
    }
}

/**
* Get type of user
*
* This function returns the user type  for the given lesson
*
* @param login $login, defaults to corresponding session variable
* @param int $lesson_id the current lesson id, defaults to corresponding session variable
* @return user_type of login
* @version 2.6
* @from now on, user_type is lesson specific.if value in users_to_lesson is NULL (because of an import), we take default user type from table users
*/
function eF_getUserRole($login = false, $lessons_ID = false){

    if($login == false){
        $login = $_SESSION['s_login'];
    }
    if($lessons_ID == false){
        if (isset($_SESSION['s_lessons_ID']))
            $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    if ($lessons_ID != ""){
        $res = eF_getTableData("users_to_lessons","user_type","users_LOGIN='".$login."' AND lessons_ID=".$lessons_ID);
    }
    if(isset($res[0]['user_type']) && $res[0]['user_type']!= ""){
        $user_type = $res[0]['user_type'];
    }else{
        $res2 = eF_getTableData("users","user_type","login='".$login."'");
        $user_type = $res2[0]['user_type'];
    }

    return $user_type;
}

/**
* Get basic type of user  ...which is enum(student,professor,administrator)
*
* This function returns the basic user type  for the given lesson
*
* @param login $login, defaults to corresponding session variable
* @param int $lesson_id the current lesson id, defaults to corresponding session variable
* @return basic user_type of login
* @version 2.6
* @from now on, basic user_type is lesson specific.if value in users_to_lesson is NULL (because of an import), we take default basic user type from table users
*/
/*
function eF_getUserBasicType($login = false, $lessons_ID = false){ //backup... below easier

    if($login == false){
        $login = $_SESSION['s_login'];
    }
    if($lessons_ID == false){
        $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    if ($lessons_ID != ""){
        $res = eF_getTableData("users_to_lessons as ul,user_types as us","us.basic_user_type","ul.user_type=us.user_type AND ul.users_LOGIN='".$login."' AND ul.lessons_ID=".$lessons_ID);
    }
    if($res[0]['basic_user_type'] != ""){
        $user_type = $res[0]['basic_user_type'];
    }else{
        $res2 = eF_getTableData("users","user_type","login='".$login."'");
        $user_type = $res2[0]['user_type'];
        if ($user_type != "student" && $user_type != "professor" && $user_type != "administrator" ){
                $res2 = eF_getTableData("users as u, user_types as us","basic_user_type","us.user_type=u.user_type AND login='".$login."'");
                $user_type = $res2[0]['basic_user_type'];
        }

    }

    return $user_type;
}*/
function eF_getUserBasicType($login = false, $lessons_ID = false){

    if($login == false){
        $login = $_SESSION['s_login'];
    }
    if($lessons_ID == false){
        if (isset($_SESSION['s_lessons_ID']))
            $lessons_ID = $_SESSION['s_lessons_ID'];
    }
    $role = eF_getUserRole($login, $lessons_ID);
    if ($role != "student" && $role != "professor" && $role != "administrator" ){
        $res2 = eF_getTableData("user_types","basic_user_type","user_type='".$role."'");
        $user_type = $res2[0]['basic_user_type'];
    }else{
        $user_type = $role;
    }

    return $user_type;
}

/**
* Create new lesson
*
* This function is used to create a new lesson
*
* @param array $fields_insert the lesson database fields
* @return array An array with the first element holding the id, the second the message and the third the message type
*/

function eF_createLesson($fields_insert) {

    $id = eF_insertTableData("lessons", $fields_insert);

    if ($id) {
        $message      = _SUCCESFULLYCREATEDLESSON;
        $message_type = 'success';

        EfrontSearch :: insertText($fields_insert['name'], $id, "lessons", "data");
        EfrontSearch :: insertText($fields_insert['name'], $id, "lessons", "title");

        $fields_forum = array('title'       => $fields_insert['name'],
                              'lessons_ID'  => $id,
                              'parent_id'   => 0,
                              'status'      => 'public',
                              'users_LOGIN' => isset($_SESSION['s_login']) ? $_SESSION['s_login'] : 'admin',
                              'comments'    => '');

        eF_insertTableData("f_forums", $fields_forum);

        $fields_chat = array('name'             => $fields_insert['name'],
                             'create_timestamp' => time(),
                             'users_LOGIN'      => $_SESSION['s_login'],
                             'type'             => 'public',
                             'active'           => 1,
                             'lessons_ID'       => $id);

        eF_insertTableData("chatrooms", $fields_chat);

        $folder  = G_LESSONSPATH.$id;
        if (!(is_dir($folder))) {
            if (!(mkdir($folder, 0700))) {
                $message      = _THELESSONFOLDERCOULDNOTBECREATED;
                $message_type = 'failure';
                eF_deleteLesson($id);
                $id = false;
            }
        } else {
            $message      = _THELESSONNOTCREATEDFOLDERSAMENAME.': '.$id.'. '._PLEASEDELETEFOLDERSORCONTACTADMIN;
            $message_type = 'failure';
            eF_deleteLesson($id);
            $id = false;
        }
    } else {
            $message      = _SOMEPROBLEMOCCURED;
            $message_type = 'failure';
            $id = false;
    }

    return array($id, $message, $message_type);

}

/**
* Delete lesson
*
* This function deletes a lesson and all of its data
* @param int $id The lesson id
* @return bool True if the lesson was deleted succesfully
* @todo change return so as to return false as well
* @version 1.0
*/
function eF_deleteLesson($id) {
    $result = eF_getTableData("content", "id", "lessons_ID=$id");
    $list   = eF_makeListIDs($result);

    eF_deleteTableData("lessons", "id=$id");
    eF_deleteTableData("content", "lessons_ID=$id");
    eF_deleteTableData("news", "lessons_ID=$id");
    eF_deleteTableData("periods", "lessons_ID=$id");
    eF_deleteTableData("users_to_lessons", "lessons_ID=$id");

    if ($list) {
        eF_deleteTableData("current_content", "content_ID IN ($list)");
        eF_deleteTableData("scorm_data", "content_ID IN ($list)");
        eF_deleteTableData("comments", "content_ID IN ($list)");
        eF_deleteTableData("questions", "content_ID IN ($list)");
        eF_deleteTableData("rules", "content_ID IN ($list)");
        eF_deleteTableData("rules", "rule_content_ID IN ($list)");
        eF_deleteTableData("search_keywords", "table_name='content' AND foreign_ID IN ($list)");

        $list_as_text = "'".str_replace(", ", "', '", $list)."'";
        eF_deleteTableData("logs", "comments IN ($list_as_text)");

        $result     = eF_getTableData("tests", "id", "content_ID IN ($list)");
        $tests_list = eF_makeListIDs($result);
        eF_deleteTableData("tests", "content_ID IN ($list)");

        if ($tests_list) {
            eF_deleteTableData("tests_to_questions", "tests_ID IN ($tests_list)");
            $result          = eF_getTableData("done_tests", "id", "tests_ID IN ($tests_list)");
            $done_tests_list = eF_makeListIDs($result);
            $list_as_text    = "'".str_replace(", ", "', '", $list)."'";
            eF_deleteTableData("logs", "action='tests' AND comments IN ($list_as_text)");

            if ($done_tests_list) {
                eF_deleteTableData("done_tests", "id IN ($done_tests_list)");
                eF_deleteTableData("done_questions", "done_tests_ID IN ($done_tests_list)");
            }
        }
    }

/*
    $category = eF_getTableData("f_categories", "id", "lessons_ID=".$id);                                       //Get the forum category, so that it is deleted also
    if (sizeof($category) > 0) {
        $topics = eF_getTableData("f_topics", "id", "f_categories_ID=".$category[0]['id']);                           //Get the category topics
        if (sizeof($topics) > 0) {
            foreach ($topics as $topic) {
                eF_deleteTableData("f_messages", "f_topics_ID=".$topic['id']);                                      //Delete all the topic messages
                eF_deleteTableData("f_topics", "id=".$topic['id']);                                                  //Delete the topic
            }
        }
        eF_deleteTableData("f_categories", "id=".$category[0]['id']);                                           //Delete the category
    }
*/
    $chatroom = eF_getTableData("chatrooms", "id", "lessons_ID=".$id);                                          //Get the lesson chat room
    if (sizeof($chatroom) > 0) {
        eF_deleteTableData("chatmessages", "chatrooms_ID=".$chatroom[0]['id']);                                 //Delete the chat room messages
        eF_deleteTableData("chatrooms", "id=".$chatroom[0]['id']);                                              //Delete the lesson chatroom
    }

    $courses = eF_getTableData("courses");
    for ($i = 0; $i < sizeof($courses); $i++) {
        $course_lessons = unserialize($courses[$i]['lessons']);
        if (($key = array_search($id, $course_lessons)) !== false) {
            $course_rules = unserialize($courses[$i]['rules']);
            unset($course_rules[$id]);
            unset($course_lessons[$key]);
            $course_rules   ? $course_rules   = serialize($course_rules)   : $course_rules   = '';
            $course_lessons ? $course_lessons = serialize($course_lessons) : $course_lessons = '';
            eF_updateTableData("courses", array("lessons" => $course_lessons, "rules" => $course_rules), "id=".$courses[$i]['id']);
        }
    }

    $ok = eF_deleteFolder(G_LESSONSPATH.$id);

    if ($ok) {
        return true;
    } else {
        return false;
    }
}

/**
* Set the lesson parameters
*
* This function is used to read any specific lesson parameters stored in the database and
* assign them to the session.
*
* @return bool True if succes
* @version 1.0
* @deprecated
*/
function eF_setLessonParameters() {

    $result = eF_getTableData("lessons", "options", "id=".$_SESSION['s_lessons_ID']);                                      // Get the lesson options (eg if the chat and forum are activated)
    if ($result[0]['options'] != "") {                                                                                     //If there are no specific options for this lesson, get default options
        $_SESSION['s_lesson_options'] = unserialize($result[0]['options']);
    } else {
        $_SESSION['s_lesson_options'] = $GLOBALS['DEF_OPTIONS'];
    }

    return true;
}

/**
* Set the role parameters
*
* This function sets the session user parameters that define user privileges
* based on its user type (role)
*
* @return bool true if success
* @version 1.0
* @deprecated
*/
function eF_setRoleParameters() {

    $role = eF_getUserRole();
    $_SESSION['s_role'] = $role;
    $basic_type = eF_getUserBasicType();

    $result = eF_getTableData("user_types", "characteristics", "user_type='".$role."'");

    if (sizeof($result) > 0 && $result[0]['characteristics'] != "" && $role != $basic_type) {
        $_SESSION['s_role_options'] = unserialize($result[0]['characteristics']);
    } else {
        $_SESSION['s_role_options'] = $GLOBALS['DEF_ROLE_OPTIONS'];
    }

    return true;
}

function eF_testExport($tests_content_ID){
    $res = eF_getTableData("tests", "id", "content_ID=".$tests_content_ID);
    $tests_ID = $res[0]['id'];
    $test_users = eF_getTableData("users_to_done_tests utdt","utdt.users_LOGIN","utdt.tests_ID=".$tests_ID);
    for($i = 0 ; $i < sizeof($test_users) ; $i ++){
        $data[$i] = array('login' => $test_users[$i]['users_LOGIN'],
                     'info'  => eF_getTableData("done_tests dt,done_questions dq,tests_to_questions tq,questions q,users u","q.text,q.options,dq.answer,dt.score,tq.weight,u.name,u.surname,u.login","dt.tests_ID=".$tests_ID." AND dt.users_LOGIN=\"".$test_users[$i]['users_LOGIN']."\" AND dt.tests_ID = tq.tests_ID AND dq.questions_ID = tq.questions_ID AND q.id = dq.questions_ID AND q.id = tq.questions_ID AND u.login = dt.users_LOGIN"));
    }
    return $data;
}
/**
* Fixes table users_to_lessons while updating lessons to courses
*
* This function  called automatically when we update lessons to courses form in order to update users to lessons table
*
* @version 3.0 4/6/2007
*/
function eF_fixUserstoCoursesLessons($course_ID){

    $result        = eF_getTableData("users_to_courses", "users_LOGIN,user_type", "courses_ID=".$course_ID);
    $result2       = eF_getTableData("courses", "lessons", "id=".$course_ID);
    $lessons_array = unserialize($result2[0]["lessons"]);

    for($i = 0; $i <sizeof($result); $i++){// for each user has access in this course ....
        foreach ($lessons_array as $value){ // for each lesson of this course ...

            $result_lessons = eF_getTableData("users_to_lessons", "count(*) AS count", "users_LOGIN ='".$result[$i]["users_LOGIN"]."' AND lessons_ID=".$value);
            if($result_lessons[0]["count"] == 0){  // ...if there is no row in users_to_lessons...
                eF_insertTableData("users_to_lessons", array(   "users_LOGIN"       => $result[$i]["users_LOGIN"],
                                                                "lessons_ID"        => $value,
                                                                "active"            => 1,
                                                                "from_timestamp"    => time(),
                                                                "user_type"         => $result[$i]["user_type"]));
            }
        }
    }
}

/**
* Get mime types
*
* This function is used to retrieve mime types
*
* @version 0.1
*/
function eF_getMimeTypes() {
    $mime_types = array(
        'ai'        =>   'application/postscript',
        'aif'       =>   'audio/x-aiff',
        'aifc'      =>   'audio/x-aiff',
        'aiff'      =>   'audio/x-aiff',
        'asc'       =>   'text/plain',
        'atom'      =>   'application/atom+xml',
        'au'        =>   'audio/basic',
        'avi'       =>   'video/x-msvideo',
        'bcpio'     =>   'application/x-bcpio',
        'bin'       =>   'application/octet-stream',
        'bmp'       =>   'image/bmp',
        'cdf'       =>   'application/x-netcdf',
        'cgm'       =>   'image/cgm',
        'class'     =>   'application/octet-stream',
        'cpio'      =>   'application/x-cpio',
        'cpt'       =>   'application/mac-compactpro',
        'csh'       =>   'application/x-csh',
        'css'       =>   'text/css',
        'dcr'       =>   'application/x-director',
        'dif'       =>   'video/x-dv',
        'dir'       =>   'application/x-director',
        'djv'       =>   'image/vnd.djvu',
        'djvu'      =>   'image/vnd.djvu',
        'dll'       =>   'application/octet-stream',
        'dmg'       =>   'application/octet-stream',
        'dms'       =>   'application/octet-stream',
        'doc'       =>   'application/msword',
        'dtd'       =>   'application/xml-dtd',
        'dv'        =>   'video/x-dv',
        'dvi'       =>   'application/x-dvi',
        'dxr'       =>   'application/x-director',
        'eps'       =>   'application/postscript',
        'etx'       =>   'text/x-setext',
        'exe'       =>   'application/octet-stream',
        'ez'        =>   'application/andrew-inset',
        'gif'       =>   'image/gif',
        'gram'      =>   'application/srgs',
        'grxml'     =>   'application/srgs+xml',
        'gtar'      =>   'application/x-gtar',
        'hdf'       =>   'application/x-hdf',
        'hqx'       =>   'application/mac-binhex40',
        'htm'       =>   'text/html',
        'html'      =>   'text/html',
        'ice'       =>   'x-conference/x-cooltalk',
        'ico'       =>   'image/x-icon',
        'ics'       =>   'text/calendar',
        'ief'       =>   'image/ief',
        'ifb'       =>   'text/calendar',
        'iges'      =>   'model/iges',
        'igs'       =>   'model/iges',
        'jnlp'      =>   'application/x-java-jnlp-file',
        'jp2'       =>   'image/jp2',
        'jpe'       =>   'image/jpeg',
        'jpeg'      =>   'image/jpeg',
        'jpg'       =>   'image/jpeg',
        'js'        =>   'application/x-javascript',
        'kar'       =>   'audio/midi',
        'latex'     =>   'application/x-latex',
        'lha'       =>   'application/octet-stream',
        'lzh'       =>   'application/octet-stream',
        'm3u'       =>   'audio/x-mpegurl',
        'm4a'       =>   'audio/mp4a-latm',
        'm4b'       =>   'audio/mp4a-latm',
        'm4p'       =>   'audio/mp4a-latm',
        'm4u'       =>   'video/vnd.mpegurl',
        'm4v'       =>   'video/x-m4v',
        'mac'       =>   'image/x-macpaint',
        'man'       =>   'application/x-troff-man',
        'mathml'    =>   'application/mathml+xml',
        'me'        =>   'application/x-troff-me',
        'mesh'      =>   'model/mesh',
        'mid'       =>   'audio/midi',
        'midi'      =>   'audio/midi',
        'mif'       =>   'application/vnd.mif',
        'mov'       =>   'video/quicktime',
        'movie'     =>   'video/x-sgi-movie',
        'mp2'       =>   'audio/mpeg',
        'mp3'       =>   'audio/mpeg',
        'mp4'       =>   'video/mp4',
        'mpe'       =>   'video/mpeg',
        'mpeg'      =>   'video/mpeg',
        'mpg'       =>   'video/mpeg',
        'mpga'      =>   'audio/mpeg',
        'ms'        =>   'application/x-troff-ms',
        'msh'       =>   'model/mesh',
        'mxu'       =>   'video/vnd.mpegurl',
        'nc'        =>   'application/x-netcdf',
        'oda'       =>   'application/oda',
        'ogg'       =>   'application/ogg',
        'pbm'       =>   'image/x-portable-bitmap',
        'pct'       =>   'image/pict',
        'pdb'       =>   'chemical/x-pdb',
        'pdf'       =>   'application/pdf',
        'pgm'       =>   'image/x-portable-graymap',
        'pgn'       =>   'application/x-chess-pgn',
        'pic'       =>   'image/pict',
        'pict'      =>   'image/pict',
        'png'       =>   'image/png',
        'pnm'       =>   'image/x-portable-anymap',
        'pnt'       =>   'image/x-macpaint',
        'pntg'      =>   'image/x-macpaint',
        'ppm'       =>   'image/x-portable-pixmap',
        'ppt'       =>   'application/vnd.ms-powerpoint',
        'ps'        =>   'application/postscript',
        'qt'        =>   'video/quicktime',
        'qti'       =>   'image/x-quicktime',
        'qtif'      =>   'image/x-quicktime',
        'ra'        =>   'audio/x-pn-realaudio',
        'ram'       =>   'audio/x-pn-realaudio',
        'ras'       =>   'image/x-cmu-raster',
        'rdf'       =>   'application/rdf+xml',
        'rgb'       =>   'image/x-rgb',
        'rm'        =>   'application/vnd.rn-realmedia',
        'roff'      =>   'application/x-troff',
        'rtf'       =>   'text/rtf',
        'rtx'       =>   'text/richtext',
        'sgm'       =>   'text/sgml',
        'sgml'      =>   'text/sgml',
        'sh'        =>   'application/x-sh',
        'shar'      =>   'application/x-shar',
        'silo'      =>   'model/mesh',
        'sit'       =>   'application/x-stuffit',
        'skd'       =>   'application/x-koan',
        'skm'       =>   'application/x-koan',
        'skp'       =>   'application/x-koan',
        'skt'       =>   'application/x-koan',
        'smi'       =>   'application/smil',
        'smil'      =>   'application/smil',
        'snd'       =>   'audio/basic',
        'so'        =>   'application/octet-stream',
        'spl'       =>   'application/x-futuresplash',
        'src'       =>   'application/x-wais-source',
        'sv4cpio'   =>   'application/x-sv4cpio',
        'sv4crc'    =>   'application/x-sv4crc',
        'svg'       =>   'image/svg+xml',
        'swf'       =>   'application/x-shockwave-flash',
        't'         =>   'application/x-troff',
        'tar'       =>   'application/x-tar',
        'tcl'       =>   'application/x-tcl',
        'tex'       =>   'application/x-tex',
        'texi'      =>   'application/x-texinfo',
        'texinfo'   =>   'application/x-texinfo',
        'tif'       =>   'image/tiff',
        'tiff'      =>   'image/tiff',
        'tr'        =>   'application/x-troff',
        'tsv'       =>   'text/tab-separated-values',
        'txt'       =>   'text/plain',
        'ustar'     =>   'application/x-ustar',
        'vcd'       =>   'application/x-cdlink',
        'vrml'      =>   'model/vrml',
        'vxml'      =>   'application/voicexml+xml',
        'wav'       =>   'audio/x-wav',
        'wbmp'      =>   'image/vnd.wap.wbmp',
        'wbmxl'     =>   'application/vnd.wap.wbxml',
        'wml'       =>   'text/vnd.wap.wml',
        'wmlc'      =>   'application/vnd.wap.wmlc',
        'wmls'      =>   'text/vnd.wap.wmlscript',
        'wmlsc'     =>   'application/vnd.wap.wmlscriptc',
        'wrl'       =>   'model/vrml',
        'xbm'       =>   'image/x-xbitmap',
        'xht'       =>   'application/xhtml+xml',
        'xhtml'     =>   'application/xhtml+xml',
        'xls'       =>   'application/vnd.ms-excel',
        'xml'       =>   'application/xml',
        'xpm'       =>   'image/x-xpixmap',
        'xsl'       =>   'application/xml',
        'xslt'      =>   'application/xslt+xml',
        'xul'       =>   'application/vnd.mozilla.xul+xml',
        'xwd'       =>   'image/x-xwindowdump',
        'xyz'       =>   'chemical/x-xyz',
        'zip'       =>   'application/zip',
    );

    return $mime_types;
}

function utf8_to_unicode( $str ) {

    $unicode = array();
    $values = array();
    $lookingFor = 1;

    for ($i = 0; $i < strlen( $str ); $i++ ) {
        $thisValue = ord( $str[ $i ] );
        pr($str[$i]);
        if ( $thisValue < 128 ) {
            $unicode[] = $thisValue;
            pr($i);
        }
        else {
            if ( count( $values ) == 0 ) $lookingFor = ( $thisValue < 224 ) ? 2 : 3;

                $values[] = $thisValue;

                if ( count( $values ) == $lookingFor ) {

                    $number = ( $lookingFor == 3 ) ?
                        ( ( $values[0] % 16 ) * 4096 ) + ( ( $values[1] % 64 ) * 64 ) + ( $values[2] % 64 ):
                        ( ( $values[0] % 32 ) * 64 ) + ( $values[1] % 64 );

                    $unicode[] = $number;
                    pr($i);
                    $values = array();
                    $lookingFor = 1;

                }
            }
        }
        return $unicode;
    }


    function createToken($length){

        $salt = "abcdefghijklmnpqrstuvwxyzABCDEFGHIJKLMNPQRSTUVWXYZ123456789";  // salt to select chars from
        srand((double)microtime()*1000000); // start the random generator
        $token=""; // set the inital variable
        for ($i=0;$i<$length;$i++)  // loop and create password
            $token = $token . substr ($salt, rand() % strlen($salt), 1);

        return $token;
    }

    function checkToken($token){
        $tmp = ef_getTableData("tokens","status","token='$token'");
        $token = $tmp[0]['status'];
        if ($token == 'logged'){
            return true;
        }
        else
            return false;
    }

/**
*
*/
function eF_highlightWords ($text, $searchdata, $replacedata) {
    $word = $searchdata;
    $text_pieces = preg_split("'(<a.*>.*</a>)|(<.+?>)'", $text, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);

    $count = 0;
    $found = false;
    $i = 0;
    $info = $replacedata;

    foreach ($text_pieces as $piece) {
        if ( (mb_strpos($piece, '<') === FALSE) && ($found==false) ) {
        //echo $piece."<br>";
            $piece = preg_replace_callback($searchdata, 'eF_encodeWords', $piece);
        //echo $piece."<br>";
        }
        $new_text_pieces[$i] = $piece;
        $i++;
    }

    $text = implode('', $new_text_pieces);
    return $text;
}

/**
*
*/
function eF_encodeWords($matches)
{
    $matching_text = $matches[0];
    $words = explode(" ", $matching_text);
    foreach($words as $key => $word) {
        $words[$key] = 'encode*()!768atyj'.$word;
    }
    $new_text = implode(' ',$words);
    return "<a class = 'glossary' onClick = 'togglePopup(this);' href = 'javascript:void(0)'>".$new_text."<img class = 'tooltip' border = '0' src='images/others/tooltip_arrow.gif'/><span><img align = 'right' class='close' border='0' src='images/16x16/error.png'/>yty656hgh".eF_encodeWordsInner($matching_text)."</span></a>";
}

/**
*
*/
function eF_encodeWordsInner($text)
{
    $words = explode(" ", $text);
    foreach($words as $key => $word) {
        $words[$key] = 'encode*()!768atyj'.$word;
    }
    $new_text = implode(' ',$words);
    return $new_text;
    //return "<a class = 'glossary info' href = 'javascript:void(0)'>".$new_text."<img class = 'tooltip' border = '0' src='/images/others/tooltip_arrow.gif'/><span>yty656hgh".$matching_text."</span></a>";
}

/**
*
*/
function eF_getPassedConditions($lessons_ID, $users_LOGIN, $tree = false) {

    if ($tree === false) {
        $tree = eF_getContentTree($nouse, $lessons_ID, 0, $type, true, false, true, false);
    }
    $num_units = 0;                                                                 //We need num_units, because some tree units may be empty. in this case, they must not be calculated in the total tree units
    foreach ($tree as $key => $value) {
        $tree_names[$value['id']] = $value['name'];                                 //Get the tree unit names
        if ($value['data'] || $value['ctg_type'] == 'tests') {                                                       //Get the actual number of units, excluding the empty ones.
            $num_units++;
        }
    }

    $conditions   = eF_getTableData("lesson_conditions", "*", "lessons_ID=".$lessons_ID);
    $seen_content = eF_getSeenContent($users_LOGIN, $lessons_ID);
    for ($i = 0; $i < sizeof($conditions); $i++) {
        $conditions[$i]['options'] = unserialize($conditions[$i]['options']);
        switch ($conditions[$i]['type']) {
            case 'all_units':
                sizeof($seen_content) >= $num_units ? $passed = true : $passed = false;
                break;
            case 'percentage_units':
                round($num_units * $conditions[$i]['options'][0] / 100) > sizeof($seen_content) ? $passed = true : $passed = false;
                break;
            case 'specific_unit':
                isset($seen_content[$conditions[$i]['options'][0]]) ? $passed = true : $passed = false;
                break;
            case 'all_tests':
                $done_tests = eF_getTableDataFlat("done_tests as dt,tests as t, content as c", "dt.score", "dt.users_LOGIN='".$users_LOGIN."' AND t.content_ID = c.id AND c.lessons_ID = ".$lessons_ID." AND t.id=dt.tests_ID");
                round(100 * array_sum($done_tests['score']) / sizeof($done_tests['score']), 2) >= $conditions[$i]['options'][0] ? $passed = true : $passed = false;
                break;
            case 'specific_test':
                $done_tests = eF_getTableDataFlat("done_tests as dt,tests as t, content as c", "dt.score", "dt.users_LOGIN='".$users_LOGIN."' AND t.content_ID = c.id AND c.lessons_ID = ".$lessons_ID." AND t.id=dt.tests_ID and c.id=".$conditions[$i]['options'][0]);
                $done_tests['score'][0] * 100 >= $conditions[$i]['options'][1] ? $passed = true : $passed = false;
                break;
            default:
                break;
        }
        $conditions[$i]['passed'] = $passed;

    }

    return $conditions;
}

/**
* Filters data array
*
* This function is used to filter the specified array according to the given filter.
* Each array element is checked against the filter, and if the filter is not contained
* in any data "row", the row is removed from the array. The function does not reindex
* array keys
*
* @param array $data The 2-dimensional data array
* @param string $filter The search filter
* @return array The new array
* @version 1.0
*/
function eF_filterData($data, $filter) {
    $filter = mb_strtolower($filter);
    foreach ($data as $key => $value) {
        $imploded_string = implode(",", $value);                 //Instead of checking each row value one-by-one, check it all at once
        if (strpos(mb_strtolower($imploded_string), $filter) === false) {
            unset($data[$key]);
        }
    }

    return $data;
}

function eF_applyGlossary($str) {
    $glossary_words = eF_getTableData("glossary_words", "name,info", "lessons_ID=".$_SESSION['s_lessons_ID']);  //Get all the glossary words of this lesson
    $pos = 0;
    $searchdata     = array();
    $searchdatanext = array();
    $replacedata    = array();

    foreach ($glossary_words as $key => $value) {
        $first_letter = mb_substr($value['name'], 0, 1);
        if ($first_letter != '<') {
            $value['name']        = mb_strtolower($value['name']);
            $searchdata[$pos]     = "/\b(".$value['name'].")/si";                     //This used to be "/\b(".$value['name'].")\b/si" but the word boundary \b at the end does not work with utf8 strings!!!
            $searchdatanext[$pos] = "/(yty656hgh".$value['name'].")/si";
            $replacedata[$pos]    = $value['info'];

            $pos = $pos + 1;
        }
    }

    $str = eF_highlightWords($str, $searchdata, $replacedata);
    $str = preg_replace("/encode\*\(\)\!768atyj/", "", $str);
    $str = preg_replace($searchdatanext, $replacedata, $str);

    return $str;
}

/**
*       greeklish
*
*            greeklish.
* ,   <br>, \  '          underscores (_).
* <br>:
* <code>
* $str = "         ";
* echo eF_makeGreeklish($str);
*
* :
* i_megali_kafe_alepoy_pidai_pano_apo_ton_tempeli_skylo
* </code>
*
* @param string $text    
* @param bool $convert_spaces Whether spaces will be also converted
* @return string    greeklish
* @version 1.0
* @Deprecated
*/
function eF_makeGreeklish($text, $convert_spaces = true)
{
    $querywords = trim($text);
    $querywords = mb_strtolower($querywords);

    $greek_array     = array( "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",  "", "", "", "", "", "",  "", "", "", "", "", "", "", "",  "",  "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "", "",  "", "", "", "", "", "",  "", "", "", "", "", "", "", "", "",  "",  "", "<br>", "/", "'");
    $greeklish_array = array( "a", "e", "i", "i", "o", "o", "i", "y", "a", "b", "g", "d", "e", "z", "i", "th", "i", "k", "l", "m", "n", "ks", "o", "p", "r", "s", "t", "y", "f", "ch", "ps", "o", "a", "e", "i", "i", "y", "o", "o", "i", "y", "i", "y", "ai", "ei", "ou", "oi", "gk", "a", "b", "g", "d", "e", "z", "i", "th", "i", "k", "l", "m", "n", "ks", "o", "p", "r", "s", "s", "t", "y", "f", "ch", "ps", "o", " ",    " ", " ");

    $querywords = str_replace($greek_array, $greeklish_array, $querywords);
    $querywords = mb_eregi_replace("[[:space:]]+", " ", $querywords);
    if ($convert_spaces) {
        $querywords = str_replace(" ", "_", $querywords);
    }

    return $querywords;
}
?>