////////////////////////////////////////////////////////////////////////////////
//
//  ADOBE SYSTEMS INCORPORATED
//  Copyright 2005-2007 Adobe Systems Incorporated
//  All Rights Reserved.
//
//  NOTICE: Adobe permits you to use, modify, and distribute this file
//  in accordance with the terms of the license agreement accompanying it.
//
////////////////////////////////////////////////////////////////////////////////

package mx.core
{

/**
 *  SpriteAsset is a subclass of the flash.display.Sprite class which
 *  represents vector graphic images that you embed in a Flex application.
 *  It implements the IFlexDisplayObject interface, which makes it
 *  possible for an embedded vector graphic image to be displayed in an Image
 *  control, or to be used as a container background or a component skin.
 *
 *  <p>The vector graphic image that you're embedding can be in an SVG file.
 *  You can also embed a sprite symbol that is in a SWF file produced
 *  by Flash.
 *  In both cases, the MXML compiler autogenerates a class that extends
 *  SpriteAsset to represent the embedded vector graphic image.</p>
 *
 *  <p>You don't generally have to use the SpriteAsset class directly
 *  when you write a Flex application.
 *  For example, you can embed a sprite symbol from a SWF file and display
 *  it in an Image control by writing the following:</p>
 *
 *  <pre>
 *  &lt;mx:Image id="logo" source="&#64;Embed(source='Assets.swf', symbol='Logo')"/&gt;</pre>
 *
 *  <p>Or use it as the application's background image in CSS syntax
 *  by writing the following:</p>
 *
 *  <pre>
 *  &lt;mx:Style&gt;
 *      Application
 *      {
 *          backgroundImage: Embed(source="Assets.swf", symbol='Logo')
 *      }
 *  &lt;mx:Style/&gt;</pre>
 *
 *  <p>without having to understand that the MXML compiler has created
 *  a subclass of BitmapAsset for you.</p>
 *
 *  <p>However, it may be useful to understand what is happening
 *  at the ActionScript level.
 *  To embed a vector graphic image in ActionScript, you declare a variable
 *  of type Class, and put <code>[Embed]</code> metadata on it.
 *  For example, you embed a sprite symbol from a SWF file like this:</p>
 *
 *  <pre>
 *  [Bindable]
 *  [Embed(source="Assets.swf", symbol="Logo")]
 *  private var logoClass:Class;</pre>
 *
 *  <p>The MXML compiler notices that the Logo symbol in Assets.swf
 *  is a sprite, autogenerates a subclass of the SpriteAsset class
 *  to represent it, and sets your variable to be a reference to this
 *  autogenerated class.
 *  You can then use this class reference to create instances of the
 *  SpriteAsset using the <code>new</code> operator, and use APIs
 *  of the Sprite class on them:</p>
 *
 *  <pre>
 *  var logo:SpriteAsset = SpriteAsset(new logoClass());
 *  logo.rotation=45;</pre>
 *
 *  <p>However, you rarely need to create SpriteAsset instances yourself
 *  because image-related properties and styles can simply be set to an
 *  image-producing class, and components will create image instances
 *  as necessary.
 *  For example, to display this vector graphic image in an Image control,
 *  you can set the Image's <code>source</code> property to
 *  <code>logoClass</code>.
 *  In MXML you could do this as follows:</p>
 *
 *  <pre>
 *  &lt;mx:Image id="logo" source="{logoClass}"/&gt;</pre>
 */
public class SpriteAsset extends FlexSprite
                         implements IFlexAsset, IFlexDisplayObject, IBorder
{
    include "../core/Version.as";

    //--------------------------------------------------------------------------
    //
    //  Constructor
    //
    //--------------------------------------------------------------------------

    /**
     *  Constructor.
     */
    public function SpriteAsset()
    {
        super();

        // Remember initial size as our measured size.
        _measuredWidth = width;
        _measuredHeight = height;
    }

    //--------------------------------------------------------------------------
    //
    //  Properties
    //
    //--------------------------------------------------------------------------

    //----------------------------------
    //  measuredHeight
    //----------------------------------

    /**
     *  @private
     *  Storage for the measuredHeight property.
     */
    private var _measuredHeight:Number;

    /**
     *  @inheritDoc
     */
    public function get measuredHeight():Number
    {
        return _measuredHeight;
    }

    //----------------------------------
    //  measuredWidth
    //----------------------------------

    /**
     *  @private
     *  Storage for the measuredWidth property.
     */
    private var _measuredWidth:Number;

    /**
     *  @inheritDoc
     */
    public function get measuredWidth():Number
    {
        return _measuredWidth;
    }

    //----------------------------------
    //  borderMetrics
    //----------------------------------
    
    /**
     *  @inheritDoc
     */
    public function get borderMetrics():EdgeMetrics
    {       
        if (scale9Grid == null)
        {
            return EdgeMetrics.EMPTY;   
        }
        else
        {
            return new EdgeMetrics(scale9Grid.left,
                                   scale9Grid.top,
                                   Math.ceil(measuredWidth - scale9Grid.right),
                                   Math.ceil(measuredHeight - scale9Grid.bottom));  
        }
    }
    
    //--------------------------------------------------------------------------
    //
    //  Methods
    //
    //--------------------------------------------------------------------------

    /**
     *  @inheritDoc
     */
    public function move(x:Number, y:Number):void
    {
        this.x = x;
        this.y = y;
    }

    /**
     *  @inheritDoc
     */
    public function setActualSize(newWidth:Number, newHeight:Number):void
    {
        width = newWidth;
        height = newHeight;
    }
}

}
