////////////////////////////////////////////////////////////////////////////////
//
//  ADOBE SYSTEMS INCORPORATED
//  Copyright 2009 Adobe Systems Incorporated
//  All Rights Reserved.
//
//  NOTICE: Adobe permits you to use, modify, and distribute this file
//  in accordance with the terms of the license agreement accompanying it.
//
////////////////////////////////////////////////////////////////////////////////

package spark.accessibility
{

import mx.accessibility.AccConst;
import mx.core.UIComponent;
import mx.core.mx_internal;

import spark.components.CheckBox;
import spark.components.supportClasses.ToggleButtonBase;

use namespace mx_internal;

/**
 *  CheckBoxAccImpl is the accessibility implementation class
 *  for spark.components.CheckBox.
 *
 *  <p>When a Spark CheckBox is created,
 *  its <code>accessibilityImplementation</code> property
 *  is set to an instance of this class.
 *  The Flash Player then uses this class to allow MSAA clients
 *  such as screen readers to see and manipulate the CheckBox.
 *  See the mx.accessibility.AccImpl and
 *  flash.accessibility.AccessibilityImplementation classes
 *  for background information about accessibility implementation
 *  classes and MSAA.</p>
 *
 *  <p><b>Children</b></p>
 *
 *  <p>A CheckBox has no MSAA children.</p>
 *
 *  <p><b>Role</b></p>
 *
 *  <p>The MSAA Role of a CheckBox is ROLE_SYSTEM_CHECKBOX.</p>
 *
 *  <p><b>Name</b></p>
 *
 *  <p>The MSAA Name of a CheckBox is, by default, the label that it displays.
 *  When wrapped in a FormItem element,
 *  this label will be combined with the FormItem's label.
 *  To override this behavior,
 *  set the CheckBox's <code>accessibilityName</code> property.</p>
 *
 *  <p>When the Name changes,
 *  a CheckBox dispatches the MSAA event EVENT_OBJECT_NAMECHANGE.</p>
 *
 *  <p><b>Description</b></p>
 *
 *  <p>The MSAA Description of a CheckBox is, by default, the empty string,
 *  but you can set the CheckBox's <code>accessibilityDescription</code>
 *  property.</p>
 *
 *  <p><b>State</b></p>
 *
 *  <p>The MSAA State of a CheckBox is a combination of:
 *  <ul>
 *    <li>STATE_SYSTEM_UNAVAILABLE (when enabled is false)</li>
 *    <li>STATE_SYSTEM_FOCUSABLE (when enabled is true)</li>
 *    <li>STATE_SYSTEM_FOCUSED (when enabled is true
 *    and the CheckBox has focus)</li>
 *    <li>STATE_SYSTEM_CHECKED (when selected is true)</li>
 *  </ul></p>
 *
 *  <p>When the Name changes,
 *  a CheckBox dispatches the MSAA event EVENT_OBJECT_STATECHANGE.</p>
 *
 *  <p><b>Value</b></p>
 *
 *  <p>A CheckBox does not have an MSAA Value.</p>
 *
 *  <p><b>Location</b></p>
 *
 *  <p>The MSAA Location of a CheckBox is its bounding rectangle.</p>
 *
 *  <p><b>Default Action</b></p>
 *
 *  <p>The MSAA DefaultAction of a CheckBox is "Check" or "UnCheck",
 *  depending on whether it is currently checked or not.</p>
 *
 *  <p>When an MSAA client tells the CheckBox to perform this action,
 *  KEY_DOWN and KEY_UP MouseEvents for the SPACE key are generated,
 *  to simulate pressing the CheckBox via the keyboard,
 *  if the CheckBox is enabled.</p>
 *
 *  <p><b>Focus</b></p>
 *
 *  <p>A CheckBox accepts focus.
 *  When it does so, it dispatches the MSAA event EVENT_OBJECT_FOCUS.</p>
 *
 *  <p><b>Selection</b></p>
 *
 *  <p>A CheckBox does not support selection in the MSAA sense.</p>
 *
 *  @langversion 3.0
 *  @playerversion Flash 10
 *  @playerversion AIR 1.5
 *  @productversion Flex 4
 */
public class CheckBoxAccImpl extends ButtonBaseAccImpl
{
    include "../core/Version.as";

    //--------------------------------------------------------------------------
    //
    //  Class methods
    //
    //--------------------------------------------------------------------------

    /**
     *  Enables accessibility in the CheckBox class.
     *
     *  <p>This method is called by application startup code
     *  that is autogenerated by the MXML compiler.
     *  Afterwards, when instances of CheckBox are initialized,
     *  their <code>accessibilityImplementation</code> property
     *  will be set to an instance of this class.</p>
     *
     *  @langversion 3.0
     *  @playerversion Flash 10
     *  @playerversion AIR 1.5
     *  @productversion Flex 4
     */
    public static function enableAccessibility():void
    {
        CheckBox.createAccessibilityImplementation =
            createAccessibilityImplementation;
    }

    /**
     *  @private
     *  Creates a CheckBox's AccessibilityImplementation object.
     *  This method is called from UIComponent's
     *  initializeAccessibility() method.
     */
    mx_internal static function createAccessibilityImplementation(
                                component:UIComponent):void
    {
        component.accessibilityImplementation =
            new CheckBoxAccImpl(component);
    }

    //--------------------------------------------------------------------------
    //
    //  Constructor
    //
    //--------------------------------------------------------------------------

    /**
     *  Constructor.
     *
     *  @param master The UIComponent instance that this AccImpl instance
     *  is making accessible.
     *
     *  @langversion 3.0
     *  @playerversion Flash 10
     *  @playerversion AIR 1.5
     *  @productversion Flex 4
     */
    public function CheckBoxAccImpl(master:UIComponent)
    {
        super(master);

        role = AccConst.ROLE_SYSTEM_CHECKBUTTON;
    }

    //--------------------------------------------------------------------------
    //
    //  Overridden methods: AccessibilityImplementation
    //
    //--------------------------------------------------------------------------

    /**
     *  @private
     *  IAccessible method for returning the state of the CheckBox.
     *  States are predefined for all the components in MSAA.
     *  Values are assigned to each state.
     *  Depending upon whether the CheckBox is checked or unchecked,
     *  a value is returned.
     *
     *  @param childID uint
     *
     *  @return State Whether the CheckBox is checked or unchecked.
     */
    override public function get_accState(childID:uint):uint
    {
        var accState:uint = getState(childID);

        if (ToggleButtonBase(master).selected)
            accState |= AccConst.STATE_SYSTEM_CHECKED;

        return accState;
    }

    /**
     *  @private
     *  IAccessible method for returning the default action of
     *  the CheckBox, which is Check or UnCheck depending on the state.
     *
     *  @param childID uint
     *
     *  @return DefaultAction Check or UnCheck.
     */
    override public function get_accDefaultAction(childID:uint):String
    {
        return ToggleButtonBase(master).selected ? "UnCheck" : "Check";
    }
}

}
