////////////////////////////////////////////////////////////////////////////////
//
//  ADOBE SYSTEMS INCORPORATED
//  Copyright 2005-2007 Adobe Systems Incorporated
//  All Rights Reserved.
//
//  NOTICE: Adobe permits you to use, modify, and distribute this file
//  in accordance with the terms of the license agreement accompanying it.
//
////////////////////////////////////////////////////////////////////////////////

package mx.core
{

/**
 *  MovieClipAsset is a subclass of the flash.display.MovieClip class
 *  which represents movieclip symbols that you embed in a Flex application
 *  from a SWF file produced by Flash.
 *  It implements the IFlexDisplayObject interface, which makes it
 *  possible for the MovieClip to be displayed in an Image control,
 *  or to be used as a container background or a component skin.
 *
 *  <p>The MovieClip that you're embedding must be a movieclip symbol
 *  that is in a SWF file.
 *  A common reason for using an embedded movieclip is that you have created
 *  a frame-based animation in Flash and want to use it in a Flex application.
 *  The MXML compiler autogenerates a class that extends MovieClipAsset
 *  to represent the embedded animation.</p>
 *
 *  <p>You don't generally have to use the MovieClipAsset class directly
 *  when you write a Flex application.
 *  For example, you can use a movieclip animation as an application's
 *  background image by writing the following:</p>
 *
 *  <pre>
 *  &lt;mx:Application backgroundImage="&#64;Embed(source='Assets.swf', symbol='BackgroundAnimation')"/&gt;</pre>
 *
 *  <p>or</p>
 *
 *  <pre>
 *  &lt;fx:Style&gt;
 *      &#64;namespace mx "library://ns.adobe.com/flex/mx"
 *      mx|Application {
 *          backgroundImage: Embed(source="Assets.swf", symbol="BackgroundAnimation")
 *      }
 *  &lt;fx:Style/&gt;</pre>
 *
 *  <p>without having to understand that the MXML compiler has created
 *  a subclass of MovieClipAsset for you.</p>
 *
 *  <p>However, it may be useful to understand what is happening
 *  at the ActionScript level.
 *  To embed a movieclip in ActionScript, you declare a variable
 *  of type Class, and put <code>[Embed]</code> metadata on it.
 *  For example:</p>
 *
 *  <pre>
 *  [Bindable]
 *  [Embed(source="Assets.swf", symbol="BackgroundAnimation")]
 *  private var backgroundAnimationClass:Class;</pre>
 *
 *  <p>The MXML compiler notices that the BackgroundAnimation symbol
 *  in Assets.swf is a movie clip, autogenerates a subclass of the
 *  MovieClipAsset class to represent it, and sets your variable
 *  to be a reference to this autogenerated class.
 *  You can then use this class reference to create instances of the
 *  MovieClipAsset using the <code>new</code> operator, and you can use
 *  APIs of the MovieClip class on them:</p>
 *
 *  <pre>
 *  var backgroundAnimation:MovieClipAsset =
 *      MovieClipAsset(new backgroundAnimationClass());
 *  var n:int = backgroundAnimation.totalFrames;</pre>
 *
 *  <p>However, you rarely need to create MovieClipAsset instances yourself
 *  because image-related properties and styles can be set to an
 *  image-producing class, and components will create instances as necessary.
 *  For example, to set the application background to this animation,
 *  you can simply write the following:</p>
 *
 *  <pre>
 *  &lt;mx:Application backgroundImage="{backgroundAnimationClass}"/&gt;</pre>
 *  
 *  @langversion 3.0
 *  @playerversion Flash 9
 *  @playerversion AIR 1.1
 *  @productversion Flex 3
 */
public class MovieClipAsset extends FlexMovieClip
                            implements IFlexAsset, IFlexDisplayObject, IBorder
{
    include "../core/Version.as";

    //--------------------------------------------------------------------------
    //
    //  Constructor
    //
    //--------------------------------------------------------------------------

    /**
     *  Constructor.
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public function MovieClipAsset()
    {
        super();

        // Remember initial size as our measured size.
        _measuredWidth = width;
        _measuredHeight = height;
    }

    //--------------------------------------------------------------------------
    //
    //  Properties
    //
    //--------------------------------------------------------------------------

    //----------------------------------
    //  measuredHeight
    //----------------------------------

    /**
     *  @private
     *  Storage for the measuredHeight property.
     */
    private var _measuredHeight:Number;

    /**
     *  @inheritDoc
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public function get measuredHeight():Number
    {
        return _measuredHeight;
    }

    //----------------------------------
    //  measuredWidth
    //----------------------------------

    /**
     *  @private
     *  Storage for the measuredWidth property.
     */
    private var _measuredWidth:Number;

    /**
     *  @inheritDoc
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public function get measuredWidth():Number
    {
        return _measuredWidth;
    }

    //----------------------------------
    //  borderMetrics
    //----------------------------------
    
    /**
     *  @inheritDoc
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public function get borderMetrics():EdgeMetrics
    {
        if (scale9Grid == null)
        {
            return EdgeMetrics.EMPTY;   
        }
        else
        {
            return new EdgeMetrics(scale9Grid.left,
                                   scale9Grid.top,
                                   Math.ceil(measuredWidth - scale9Grid.right),
                                   Math.ceil(measuredHeight - scale9Grid.bottom));  
        }
    }


    //--------------------------------------------------------------------------
    //
    //  Methods
    //
    //--------------------------------------------------------------------------

    /**
     *  @inheritDoc
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public function move(x:Number, y:Number):void
    {
        this.x = x;
        this.y = y;
    }

    /**
     *  @inheritDoc
     *  
     *  @langversion 3.0
     *  @playerversion Flash 9
     *  @playerversion AIR 1.1
     *  @productversion Flex 3
     */
    public function setActualSize(newWidth:Number, newHeight:Number):void
    {
        width = newWidth;
        height = newHeight;
    }
}

}
