/*
 * Decompiled with CFR 0.152.
 */
package net.sf.freecol.server.generator;

import java.awt.Rectangle;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.logging.Logger;
import net.sf.freecol.common.model.Game;
import net.sf.freecol.common.model.Map;
import net.sf.freecol.common.model.Region;
import net.sf.freecol.common.model.Resource;
import net.sf.freecol.common.model.ResourceType;
import net.sf.freecol.common.model.Specification;
import net.sf.freecol.common.model.Tile;
import net.sf.freecol.common.model.TileImprovement;
import net.sf.freecol.common.model.TileImprovementType;
import net.sf.freecol.common.model.TileItemContainer;
import net.sf.freecol.common.model.TileType;
import net.sf.freecol.common.option.OptionGroup;
import net.sf.freecol.common.util.RandomChoice;
import net.sf.freecol.server.generator.River;
import net.sf.freecol.server.generator.RiverSection;
import net.sf.freecol.server.model.ServerRegion;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class TerrainGenerator {
    private static final Logger logger = Logger.getLogger(TerrainGenerator.class.getName());
    public static final int LAND_REGIONS_SCORE_VALUE = 1000;
    public static final int LAND_REGION_MIN_SCORE = 5;
    public static final int PACIFIC_SCORE_VALUE = 100;
    public static final int LAND_REGION_MAX_SIZE = 75;
    private final OptionGroup mapOptions;
    private final Random random;
    private ArrayList<TileType> landTileTypes = null;
    private ArrayList<TileType> oceanTileTypes = null;
    private ServerRegion[] geographicRegions = null;

    public TerrainGenerator(OptionGroup mapGeneratorOptions, Random random) {
        this.mapOptions = mapGeneratorOptions;
        this.random = random;
    }

    private int limitToRange(int value, int lower, int upper) {
        return Math.max(lower, Math.min(value, upper));
    }

    private int getApproximateLandCount() {
        return this.mapOptions.getInteger("model.option.mapWidth") * this.mapOptions.getInteger("model.option.mapHeight") * this.mapOptions.getInteger("model.option.landMass") / 100;
    }

    private void setGeographicRegion(ServerRegion sr) {
        if (this.geographicRegions == null) {
            return;
        }
        for (ServerRegion gr : this.geographicRegions) {
            Map.Position cen = sr.getCenter();
            if (!gr.getBounds().contains(cen.getX(), cen.getY())) continue;
            sr.setParent(gr);
            gr.addChild(sr);
            gr.setSize(gr.getSize() + sr.getSize());
            break;
        }
    }

    private Tile createTile(Game game, int x, int y, boolean[][] landMap, int latitude) {
        return landMap[x][y] ? new Tile(game, this.getRandomLandTileType(game, latitude), x, y) : new Tile(game, this.getRandomOceanTileType(game, latitude), x, y);
    }

    private TileType getRandomLandTileType(Game game, int latitude) {
        if (this.landTileTypes == null) {
            this.landTileTypes = new ArrayList();
            for (TileType type : game.getSpecification().getTileTypeList()) {
                if (type.isElevation() || type.isWater()) continue;
                this.landTileTypes.add(type);
            }
        }
        return this.getRandomTileType(game, this.landTileTypes, latitude);
    }

    private TileType getRandomOceanTileType(Game game, int latitude) {
        if (this.oceanTileTypes == null) {
            this.oceanTileTypes = new ArrayList();
            for (TileType type : game.getSpecification().getTileTypeList()) {
                if (!type.isWater() || !type.isHighSeasConnected() || type.isDirectlyHighSeasConnected()) continue;
                this.oceanTileTypes.add(type);
            }
        }
        return this.getRandomTileType(game, this.oceanTileTypes, latitude);
    }

    private TileType getRandomTileType(Game game, List<TileType> candidates, int latitude) {
        TileType type;
        int forestChance = this.mapOptions.getInteger("model.option.forestNumber");
        int temperaturePreference = this.mapOptions.getInteger("model.option.temperature");
        int poleTemperature = -20;
        int equatorTemperature = 40;
        switch (temperaturePreference) {
            case 0: {
                poleTemperature = -20;
                equatorTemperature = 25;
                break;
            }
            case 1: {
                poleTemperature = -20;
                equatorTemperature = 30;
                break;
            }
            case 2: {
                poleTemperature = -10;
                equatorTemperature = 35;
                break;
            }
            case 3: {
                poleTemperature = -5;
                equatorTemperature = 40;
                break;
            }
            case 4: {
                poleTemperature = 0;
                equatorTemperature = 40;
            }
        }
        Specification spec = game.getSpecification();
        int temperatureRange = equatorTemperature - poleTemperature;
        int localeTemperature = poleTemperature + (90 - Math.abs(latitude)) * temperatureRange / 90;
        int temperatureDeviation = 7;
        localeTemperature += this.random.nextInt(temperatureDeviation * 2) - temperatureDeviation;
        localeTemperature = this.limitToRange(localeTemperature, -20, 40);
        int localeHumidity = spec.getRangeOption("model.option.humidity").getValue();
        int humidityDeviation = 20;
        localeHumidity += this.random.nextInt(humidityDeviation * 2) - humidityDeviation;
        localeHumidity = this.limitToRange(localeHumidity, 0, 100);
        ArrayList<TileType> candidateTileTypes = new ArrayList<TileType>(candidates);
        int i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.TEMPERATURE, localeTemperature)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for temperature==" + localeTemperature);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        i = 0;
        while (i < candidateTileTypes.size()) {
            type = (TileType)candidateTileTypes.get(i);
            if (!type.withinRange(TileType.RangeType.HUMIDITY, localeHumidity)) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        switch (candidateTileTypes.size()) {
            case 0: {
                throw new RuntimeException("No TileType for humidity==" + localeHumidity);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        boolean forested = this.random.nextInt(100) < forestChance;
        i = 0;
        while (i < candidateTileTypes.size()) {
            TileType type2 = (TileType)candidateTileTypes.get(i);
            if (type2.isForested() != forested) {
                candidateTileTypes.remove(i);
                continue;
            }
            ++i;
        }
        i = candidateTileTypes.size();
        switch (i) {
            case 0: {
                throw new RuntimeException("No TileType for forested==" + forested);
            }
            case 1: {
                return (TileType)candidateTileTypes.get(0);
            }
        }
        return (TileType)candidateTileTypes.get(this.random.nextInt(i));
    }

    public Tile getRandomLandTile(Map map, Random random) {
        int x = map.getWidth() < 10 ? random.nextInt(map.getWidth()) : random.nextInt(map.getWidth() - 10) + 5;
        int y = map.getHeight() < 10 ? random.nextInt(map.getHeight()) : random.nextInt(map.getHeight() - 10) + 5;
        for (Tile t : map.getCircleTiles(map.getTile(x, y), true, Integer.MAX_VALUE)) {
            if (!t.isLand()) continue;
            return t;
        }
        return null;
    }

    public void createMap(Game game, boolean[][] landMap) {
        this.createMap(game, null, landMap);
    }

    public void createMap(Game game, Game importGame, boolean[][] landMap) {
        Specification spec = game.getSpecification();
        int width = landMap.length;
        int height = landMap[0].length;
        boolean importTerrain = importGame != null && this.mapOptions.getBoolean("model.option.importTerrain");
        boolean importBonuses = importGame != null && this.mapOptions.getBoolean("model.option.importBonuses");
        boolean mapHasLand = false;
        Tile[][] tiles = new Tile[width][height];
        Map map = new Map(game, tiles);
        int minimumLatitude = this.mapOptions.getInteger("model.option.minimumLatitude");
        int maximumLatitude = this.mapOptions.getInteger("model.option.maximumLatitude");
        minimumLatitude = this.limitToRange(minimumLatitude, -90, 90);
        maximumLatitude = this.limitToRange(maximumLatitude, -90, 90);
        map.setMinimumLatitude(Math.min(minimumLatitude, maximumLatitude));
        map.setMaximumLatitude(Math.max(minimumLatitude, maximumLatitude));
        HashMap<String, ServerRegion> regionMap = new HashMap<String, ServerRegion>();
        if (importTerrain) {
            ServerRegion region;
            String ids = "";
            for (Region r : importGame.getMap().getRegions()) {
                region = new ServerRegion(game, r);
                map.putRegion(region);
                regionMap.put(r.getId(), region);
                ids = ids + " " + region.getNameKey();
            }
            for (Region r : importGame.getMap().getRegions()) {
                region = (ServerRegion)regionMap.get(r.getId());
                Region x = r.getParent();
                if (x != null) {
                    x = (Region)regionMap.get(x.getId());
                }
                region.setParent(x);
                for (Region c : r.getChildren()) {
                    x = (Region)regionMap.get(c.getId());
                    if (x == null) continue;
                    region.addChild(x);
                }
            }
            logger.info("Imported regions: " + ids);
        }
        ArrayList<Tile> fixRegions = new ArrayList<Tile>();
        for (int y = 0; y < height; ++y) {
            int latitude = map.getLatitude(y);
            for (int x = 0; x < width; ++x) {
                Tile t;
                if (landMap[x][y]) {
                    mapHasLand = true;
                }
                Tile importTile = null;
                if (importTerrain && importGame.getMap().isValid(x, y) && (importTile = importGame.getMap().getTile(x, y)) != null && importTile.isLand() == landMap[x][y]) {
                    Region r;
                    String id = importTile.getType().getId();
                    t = new Tile(game, spec.getTileType(id), x, y);
                    if (importTile.getMoveToEurope() != null) {
                        t.setMoveToEurope(importTile.getMoveToEurope());
                    }
                    if (importTile.getTileItemContainer() != null) {
                        TileItemContainer container = new TileItemContainer(game, t);
                        container.copyFrom(importTile.getTileItemContainer(), importBonuses, true);
                        t.setTileItemContainer(container);
                    }
                    if ((r = importTile.getRegion()) == null) {
                        fixRegions.add(t);
                    } else {
                        ServerRegion ours = (ServerRegion)regionMap.get(r.getId());
                        if (ours == null) {
                            logger.warning("Could not set tile region " + r.getId() + " for tile: " + t);
                            fixRegions.add(t);
                        } else {
                            ours.addTile(t);
                        }
                    }
                } else {
                    t = this.createTile(game, x, y, landMap, latitude);
                }
                tiles[x][y] = t;
            }
        }
        game.setMap(map);
        this.geographicRegions = this.getStandardRegions(map);
        if (importTerrain) {
            if (!fixRegions.isEmpty()) {
                this.createOceanRegions(map);
                this.createLakeRegions(map);
                this.createLandRegions(map);
            }
        } else {
            this.createOceanRegions(map);
            this.createHighSeas(map);
            if (mapHasLand) {
                this.createMountains(map);
                this.createRivers(map);
                this.createLakeRegions(map);
                this.createLandRegions(map);
            }
        }
        for (Tile tile : map.getAllTiles()) {
            this.perhapsAddBonus(tile, !importBonuses);
            if (tile.isLand()) continue;
            TerrainGenerator.encodeStyle(tile);
        }
        map.resetContiguity();
        map.resetHighSeasCount();
    }

    private void createOceanRegions(Map map) {
        int y;
        Game game = map.getGame();
        ServerRegion pacific = (ServerRegion)map.getRegion("model.region.pacific");
        ServerRegion northPacific = (ServerRegion)map.getRegion("model.region.northPacific");
        ServerRegion southPacific = (ServerRegion)map.getRegion("model.region.southPacific");
        ServerRegion atlantic = (ServerRegion)map.getRegion("model.region.atlantic");
        ServerRegion northAtlantic = (ServerRegion)map.getRegion("model.region.northAtlantic");
        ServerRegion southAtlantic = (ServerRegion)map.getRegion("model.region.southAtlantic");
        int present = 0;
        if (pacific == null) {
            pacific = new ServerRegion(game, "model.region.pacific", Region.RegionType.OCEAN, null);
            pacific.setDiscoverable(true);
            map.putRegion(pacific);
            pacific.setScoreValue(100);
        }
        if (northPacific == null) {
            northPacific = new ServerRegion(game, "model.region.northPacific", Region.RegionType.OCEAN, pacific);
            northPacific.setDiscoverable(false);
            map.putRegion(northPacific);
        } else {
            ++present;
        }
        if (southPacific == null) {
            southPacific = new ServerRegion(game, "model.region.southPacific", Region.RegionType.OCEAN, pacific);
            southPacific.setDiscoverable(false);
            map.putRegion(southPacific);
        } else {
            ++present;
        }
        if (atlantic == null) {
            atlantic = new ServerRegion(game, "model.region.atlantic", Region.RegionType.OCEAN, null);
            atlantic.setPrediscovered(true);
            atlantic.setDiscoverable(false);
            map.putRegion(atlantic);
        }
        if (northAtlantic == null) {
            northAtlantic = new ServerRegion(game, "model.region.northAtlantic", Region.RegionType.OCEAN, atlantic);
            northAtlantic.setPrediscovered(true);
            northAtlantic.setDiscoverable(false);
            map.putRegion(northAtlantic);
        } else {
            ++present;
        }
        if (southAtlantic == null) {
            southAtlantic = new ServerRegion(game, "model.region.southAtlantic", Region.RegionType.OCEAN, atlantic);
            southAtlantic.setPrediscovered(true);
            southAtlantic.setDiscoverable(false);
            map.putRegion(southAtlantic);
        } else {
            ++present;
        }
        if (present == 4) {
            return;
        }
        int maxx = map.getWidth();
        int midx = maxx / 2;
        int maxy = map.getHeight();
        int midy = maxy / 2;
        Map.Position pNP = null;
        Map.Position pSP = null;
        Map.Position pNA = null;
        Map.Position pSA = null;
        for (y = midy - 1; y >= 0; --y) {
            if (pNP == null && !map.getTile(0, y).isLand()) {
                pNP = new Map.Position(0, y);
            }
            if (pNA == null && !map.getTile(maxx - 1, y).isLand()) {
                pNA = new Map.Position(maxx - 1, y);
            }
            if (pNP != null && pNA != null) break;
        }
        for (y = midy; y < maxy; ++y) {
            if (pSP == null && !map.getTile(0, y).isLand()) {
                pSP = new Map.Position(0, y);
            }
            if (pSA == null && !map.getTile(maxx - 1, y).isLand()) {
                pSA = new Map.Position(maxx - 1, y);
            }
            if (pSP != null && pSA != null) break;
        }
        int nNP = 0;
        int nSP = 0;
        int nNA = 0;
        int nSA = 0;
        Rectangle rNP = new Rectangle(0, 0, midx, midy);
        Rectangle rSP = new Rectangle(0, midy, midx, maxy);
        Rectangle rNA = new Rectangle(midx, 0, maxx, midy);
        Rectangle rSA = new Rectangle(midx, midy, maxx, maxy);
        if (pNP != null) {
            nNP += this.fillOcean(map, pNP, northPacific, rNP);
        }
        if (pSP != null) {
            nSP += this.fillOcean(map, pSP, southPacific, rSP);
        }
        if (pNA != null) {
            nNA += this.fillOcean(map, pNA, northAtlantic, rNA);
        }
        if (pSA != null) {
            nSA += this.fillOcean(map, pSA, southAtlantic, rSA);
        }
        Rectangle rN = new Rectangle(0, 0, maxx, midy);
        Rectangle rS = new Rectangle(0, midy, maxx, maxy);
        if (pNP != null) {
            nNP += this.fillOcean(map, pNP, northPacific, rN);
        }
        if (pSP != null) {
            nSP += this.fillOcean(map, pSP, southPacific, rS);
        }
        if (pNA != null) {
            nNA += this.fillOcean(map, pNA, northAtlantic, rN);
        }
        if (pSA != null) {
            nSA += this.fillOcean(map, pSA, southAtlantic, rS);
        }
        Rectangle rAll = new Rectangle(0, 0, maxx, maxy);
        if (pNP != null) {
            nNP += this.fillOcean(map, pNP, northPacific, rAll);
        }
        if (pSP != null) {
            nSP += this.fillOcean(map, pSP, southPacific, rAll);
        }
        if (pNA != null) {
            nNA += this.fillOcean(map, pNA, northAtlantic, rAll);
        }
        if (pSA != null) {
            nSA += this.fillOcean(map, pSA, southAtlantic, rAll);
        }
        if (nNP <= 0) {
            logger.warning("No North Pacific tiles found");
        }
        if (nSP <= 0) {
            logger.warning("No South Pacific tiles found");
        }
        if (nNA <= 0) {
            logger.warning("No North Atlantic tiles found");
        }
        if (nSA <= 0) {
            logger.warning("No South Atlantic tiles found");
        }
        logger.info("Ocean regions complete: " + nNP + " North Pacific, " + nSP + " South Pacific, " + nNA + " North Atlantic, " + nSP + " South Atlantic");
    }

    private int fillOcean(Map map, Map.Position p, ServerRegion region, Rectangle bounds) {
        LinkedList<Map.Position> q = new LinkedList<Map.Position>();
        int n = 0;
        boolean[][] visited = new boolean[map.getWidth()][map.getHeight()];
        visited[p.getX()][p.getY()] = true;
        q.add(p);
        while ((p = (Map.Position)q.poll()) != null) {
            Tile tile = map.getTile(p);
            region.addTile(tile);
            ++n;
            for (Map.Direction direction : Map.Direction.values()) {
                Map.Position next = p.getAdjacent(direction);
                if (!map.isValid(next) || visited[next.getX()][next.getY()] || !bounds.contains(next.getX(), next.getY())) continue;
                visited[next.getX()][next.getY()] = true;
                Tile t = map.getTile(next);
                if (t.getRegion() != null && t.getRegion() != region || t.isLand()) continue;
                q.add(next);
            }
        }
        return n;
    }

    private ServerRegion[] getStandardRegions(Map map) {
        ServerRegion antarctic;
        Game game = map.getGame();
        int arcticHeight = 2;
        int antarcticHeight = map.getHeight() - 2 - 1;
        ServerRegion arctic = (ServerRegion)map.getRegion("model.region.arctic");
        if (arctic == null) {
            arctic = new ServerRegion(game, "model.region.arctic", Region.RegionType.LAND, null);
            arctic.setPrediscovered(true);
            map.putRegion(arctic);
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y = 0; y < 2; ++y) {
                    Tile tile;
                    if (!map.isValid(x, y) || !(tile = map.getTile(x, y)).isLand()) continue;
                    arctic.addTile(tile);
                }
            }
        }
        if ((antarctic = (ServerRegion)map.getRegion("model.region.antarctic")) == null) {
            antarctic = new ServerRegion(game, "model.region.antarctic", Region.RegionType.LAND, null);
            antarctic.setPrediscovered(true);
            map.putRegion(antarctic);
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y = antarcticHeight; y < map.getHeight(); ++y) {
                    Tile tile;
                    if (!map.isValid(x, y) || !(tile = map.getTile(x, y)).isLand()) continue;
                    antarctic.addTile(tile);
                }
            }
        }
        int thirdWidth = map.getWidth() / 3;
        int twoThirdWidth = 2 * thirdWidth;
        int thirdHeight = map.getHeight() / 3;
        int twoThirdHeight = 2 * thirdHeight;
        ServerRegion northWest = (ServerRegion)map.getRegion("model.region.northWest");
        if (northWest == null) {
            northWest = new ServerRegion(game, "model.region.northWest", Region.RegionType.LAND, null);
            map.putRegion(northWest);
        }
        northWest.setBounds(new Rectangle(0, 0, thirdWidth, thirdHeight));
        northWest.setPrediscovered(true);
        ServerRegion north = (ServerRegion)map.getRegion("model.region.north");
        if (north == null) {
            north = new ServerRegion(game, "model.region.north", Region.RegionType.LAND, null);
            map.putRegion(north);
        }
        north.setBounds(new Rectangle(thirdWidth, 0, twoThirdWidth, thirdHeight));
        north.setPrediscovered(true);
        ServerRegion northEast = (ServerRegion)map.getRegion("model.region.northEast");
        if (northEast == null) {
            northEast = new ServerRegion(game, "model.region.northEast", Region.RegionType.LAND, null);
            map.putRegion(northEast);
        }
        northEast.setBounds(new Rectangle(twoThirdWidth, 0, map.getWidth(), thirdHeight));
        northEast.setPrediscovered(true);
        ServerRegion west = (ServerRegion)map.getRegion("model.region.west");
        if (west == null) {
            west = new ServerRegion(game, "model.region.west", Region.RegionType.LAND, null);
            map.putRegion(west);
        }
        west.setBounds(new Rectangle(0, thirdHeight, thirdWidth, twoThirdHeight));
        west.setPrediscovered(true);
        ServerRegion center = (ServerRegion)map.getRegion("model.region.center");
        if (center == null) {
            center = new ServerRegion(game, "model.region.center", Region.RegionType.LAND, null);
            map.putRegion(center);
        }
        center.setBounds(new Rectangle(thirdWidth, thirdHeight, twoThirdWidth, twoThirdHeight));
        center.setPrediscovered(true);
        ServerRegion east = (ServerRegion)map.getRegion("model.region.east");
        if (east == null) {
            east = new ServerRegion(game, "model.region.east", Region.RegionType.LAND, null);
            map.putRegion(east);
        }
        east.setBounds(new Rectangle(twoThirdWidth, thirdHeight, map.getWidth(), twoThirdHeight));
        east.setPrediscovered(true);
        ServerRegion southWest = (ServerRegion)map.getRegion("model.region.southWest");
        if (southWest == null) {
            southWest = new ServerRegion(game, "model.region.southWest", Region.RegionType.LAND, null);
            map.putRegion(southWest);
        }
        southWest.setBounds(new Rectangle(0, twoThirdHeight, thirdWidth, map.getHeight()));
        southWest.setPrediscovered(true);
        ServerRegion south = (ServerRegion)map.getRegion("model.region.south");
        if (south == null) {
            south = new ServerRegion(game, "model.region.south", Region.RegionType.LAND, null);
            map.putRegion(south);
        }
        south.setBounds(new Rectangle(thirdWidth, twoThirdHeight, twoThirdWidth, map.getHeight()));
        south.setPrediscovered(true);
        ServerRegion southEast = (ServerRegion)map.getRegion("model.region.southEast");
        if (southEast == null) {
            southEast = new ServerRegion(game, "model.region.southEast", Region.RegionType.LAND, null);
            map.putRegion(southEast);
        }
        southEast.setBounds(new Rectangle(twoThirdWidth, twoThirdHeight, map.getWidth(), map.getHeight()));
        southEast.setPrediscovered(true);
        return new ServerRegion[]{northWest, north, northEast, west, center, east, southWest, south, southEast};
    }

    private void createLandRegions(Map map) {
        int c;
        int y;
        Game game = map.getGame();
        int continents = 0;
        boolean[][] landmap = new boolean[map.getWidth()][map.getHeight()];
        int[][] continentmap = new int[map.getWidth()][map.getHeight()];
        int landsize = 0;
        for (int x = 0; x < map.getWidth(); ++x) {
            for (y = 0; y < map.getHeight(); ++y) {
                continentmap[x][y] = 0;
                landmap[x][y] = false;
                if (!map.isValid(x, y)) continue;
                Tile tile = map.getTile(x, y);
                boolean bl = landmap[x][y] = tile.isLand() && tile.getRegion() == null;
                if (!tile.isLand()) continue;
                ++landsize;
            }
        }
        for (int y2 = 0; y2 < map.getHeight(); ++y2) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (!landmap[x][y2]) continue;
                ++continents;
                boolean[][] continent = Map.floodFill(landmap, new Map.Position(x, y2));
                for (int yy = 0; yy < map.getHeight(); ++yy) {
                    for (int xx = 0; xx < map.getWidth(); ++xx) {
                        if (!continent[xx][yy]) continue;
                        continentmap[xx][yy] = continents;
                        landmap[xx][yy] = false;
                    }
                }
            }
        }
        logger.info("Number of individual landmasses is " + continents);
        int[] continentsize = new int[continents + 1];
        for (y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                int n = continentmap[x][y];
                continentsize[n] = continentsize[n] + 1;
            }
        }
        int oldcontinents = continents;
        for (int c2 = 1; c2 <= oldcontinents; ++c2) {
            if (continentsize[c2] <= 75) continue;
            boolean[][] splitcontinent = new boolean[map.getWidth()][map.getHeight()];
            Map.Position splitposition = new Map.Position(0, 0);
            for (int x = 0; x < map.getWidth(); ++x) {
                for (int y3 = 0; y3 < map.getHeight(); ++y3) {
                    if (continentmap[x][y3] == c2) {
                        splitcontinent[x][y3] = true;
                        splitposition = new Map.Position(x, y3);
                        continue;
                    }
                    splitcontinent[x][y3] = false;
                }
            }
            while (continentsize[c2] > 75) {
                int targetsize = 75;
                if (continentsize[c2] < 150) {
                    targetsize = continentsize[c2] / 2;
                }
                ++continents;
                boolean[][] newregion = Map.floodFill(splitcontinent, splitposition, targetsize);
                for (int x = 0; x < map.getWidth(); ++x) {
                    for (int y4 = 0; y4 < map.getHeight(); ++y4) {
                        if (newregion[x][y4]) {
                            continentmap[x][y4] = continents;
                            splitcontinent[x][y4] = false;
                            int n = c2;
                            continentsize[n] = continentsize[n] - 1;
                        }
                        if (!splitcontinent[x][y4]) continue;
                        splitposition = new Map.Position(x, y4);
                    }
                }
            }
        }
        logger.info("Number of land regions being created: " + continents);
        ServerRegion[] landregions = new ServerRegion[continents + 1];
        int landIndex = 1;
        for (c = 1; c <= continents; ++c) {
            String id;
            while (map.getRegion(id = "model.region.land" + Integer.toString(landIndex++)) != null) {
            }
            landregions[c] = new ServerRegion(map.getGame(), id, Region.RegionType.LAND, null);
            landregions[c].setDiscoverable(true);
            map.putRegion(landregions[c]);
        }
        for (int y5 = 0; y5 < map.getHeight(); ++y5) {
            for (int x = 0; x < map.getWidth(); ++x) {
                if (continentmap[x][y5] <= 0) continue;
                Tile tile = map.getTile(x, y5);
                landregions[continentmap[x][y5]].addTile(tile);
            }
        }
        for (c = 1; c <= continents; ++c) {
            ServerRegion sr = landregions[c];
            int score = Math.max((int)((float)sr.getSize() / (float)landsize * 1000.0f), 5);
            sr.setScoreValue(score);
            this.setGeographicRegion(sr);
            logger.fine("Created land region " + sr.getNameKey() + " (size " + sr.getSize() + ", score " + sr.getScoreValue() + ", parent " + (sr.getParent() == null ? "(null)" : sr.getParent().getNameKey()) + ")");
        }
        for (ServerRegion gr : this.geographicRegions) {
            logger.fine("Geographic region " + gr.getNameKey() + " (size " + gr.getSize() + ", children " + gr.getChildren().size() + ")");
        }
    }

    private void createHighSeas(Map map) {
        OptionGroup opt = this.mapOptions;
        TerrainGenerator.createHighSeas(map, opt.getInteger("model.option.distanceToHighSea"), opt.getInteger("model.option.maximumDistanceToEdge"));
    }

    public static void determineHighSeas(Map map, int distToLandFromHighSeas, int maxDistanceToEdge) {
        Specification spec = map.getSpecification();
        TileType ocean = spec.getTileType("model.tile.ocean");
        TileType highSeas = spec.getTileType("model.tile.highSeas");
        if (highSeas == null || ocean == null) {
            throw new RuntimeException("Both Ocean and HighSeas TileTypes must be defined");
        }
        for (Tile t : map.getAllTiles()) {
            if (t.getType() != highSeas) continue;
            t.setType(ocean);
        }
        TerrainGenerator.createHighSeas(map, distToLandFromHighSeas, maxDistanceToEdge);
    }

    private static void createHighSeas(Map map, int distToLandFromHighSeas, int maxDistanceToEdge) {
        if (distToLandFromHighSeas < 0 || maxDistanceToEdge < 0) {
            throw new IllegalArgumentException("The integer arguments cannot be negative.");
        }
        Specification spec = map.getSpecification();
        TileType ocean = spec.getTileType("model.tile.ocean");
        TileType highSeas = spec.getTileType("model.tile.highSeas");
        if (highSeas == null) {
            throw new RuntimeException("TileType highSeas must be defined.");
        }
        Tile seaL = null;
        Tile seaR = null;
        int totalL = 0;
        int totalR = 0;
        int distanceL = -1;
        int distanceR = -1;
        for (int y = 0; y < map.getHeight(); ++y) {
            int distance;
            Tile other;
            Tile t;
            int x;
            for (x = 0; x < maxDistanceToEdge && x < map.getWidth() && map.isValid(x, y) && (t = map.getTile(x, y)).getType() == ocean; ++x) {
                other = map.getLandWithinDistance(x, y, distToLandFromHighSeas);
                if (other == null) {
                    t.setType(highSeas);
                    ++totalL;
                    continue;
                }
                distance = t.getDistanceTo(other);
                if (distanceL >= distance) continue;
                distanceL = distance;
                seaL = t;
            }
            for (x = 0; x < maxDistanceToEdge && x < map.getWidth() && map.isValid(map.getWidth() - 1 - x, y) && (t = map.getTile(map.getWidth() - 1 - x, y)).getType() == ocean; ++x) {
                other = map.getLandWithinDistance(map.getWidth() - 1 - x, y, distToLandFromHighSeas);
                if (other == null) {
                    t.setType(highSeas);
                    ++totalR;
                    continue;
                }
                distance = t.getDistanceTo(other);
                if (distanceR >= distance) continue;
                distanceR = distance;
                seaR = t;
            }
        }
        if (totalL <= 0 && seaL != null) {
            seaL.setType(highSeas);
            ++totalL;
        }
        if (totalR <= 0 && seaR != null) {
            seaR.setType(highSeas);
            ++totalR;
        }
        if (totalL <= 0 || totalR <= 0) {
            logger.warning("No high seas on " + (totalL <= 0 && totalR <= 0 ? "either" : (totalL <= 0 ? "left" : (totalR <= 0 ? "right" : "BOGUS"))) + " side of the map." + "  This can cause failures on small test maps.");
        }
    }

    private void createMountains(Map map) {
        int tries;
        float randomHillsRatio = 0.5f;
        int maximumLength = Math.max(this.mapOptions.getInteger("model.option.mapWidth"), this.mapOptions.getInteger("model.option.mapHeight")) / 10;
        int number = (int)((float)(this.getApproximateLandCount() / this.mapOptions.getInteger("model.option.mountainNumber")) * (1.0f - randomHillsRatio));
        logger.info("Number of mountain tiles is " + number);
        logger.fine("Maximum length of mountain ranges is " + maximumLength);
        Specification spec = map.getSpecification();
        TileType hills = spec.getTileType("model.tile.hills");
        TileType mountains = spec.getTileType("model.tile.mountains");
        if (hills == null || mountains == null) {
            throw new RuntimeException("Both Hills and Mountains TileTypes must be defined");
        }
        int counter = 0;
        block0: for (tries = 0; tries < 100; ++tries) {
            if (counter >= number) continue;
            Tile startTile = this.getRandomLandTile(map, this.random);
            if (startTile == null) {
                return;
            }
            if (startTile.getType() == hills || startTile.getType() == mountains) continue;
            for (Tile t : startTile.getSurroundingTiles(3)) {
                if (t.getType() != mountains) continue;
                continue block0;
            }
            for (Tile t : startTile.getSurroundingTiles(2)) {
                if (t.isLand()) continue;
                continue block0;
            }
            ServerRegion mountainRegion = new ServerRegion(map.getGame(), "model.region.mountain" + tries, Region.RegionType.MOUNTAIN, startTile.getRegion());
            mountainRegion.setDiscoverable(true);
            mountainRegion.setClaimable(true);
            map.putRegion(mountainRegion);
            Map.Direction direction = Map.Direction.getRandomDirection("getLand", this.random);
            int length = maximumLength - this.random.nextInt(maximumLength / 2);
            for (int index = 0; index < length; ++index) {
                Tile nextTile = startTile.getNeighbourOrNull(direction);
                if (nextTile == null || !nextTile.isLand()) continue;
                nextTile.setType(mountains);
                mountainRegion.addTile(nextTile);
                ++counter;
                for (Tile neighbour : nextTile.getSurroundingTiles(1)) {
                    if (!neighbour.isLand() || neighbour.getType() == mountains) continue;
                    int r = this.random.nextInt(8);
                    if (r == 0) {
                        neighbour.setType(mountains);
                        mountainRegion.addTile(neighbour);
                        ++counter;
                        continue;
                    }
                    if (r <= 2) continue;
                    neighbour.setType(hills);
                    mountainRegion.addTile(neighbour);
                }
            }
            int scoreValue = 2 * mountainRegion.getSize();
            mountainRegion.setScoreValue(scoreValue);
            logger.fine("Created mountain region (direction " + (Object)((Object)direction) + ", length " + length + ", size " + mountainRegion.getSize() + ", score value " + scoreValue + ").");
        }
        logger.info("Added " + counter + " mountain range tiles.");
        number = (int)((float)this.getApproximateLandCount() * randomHillsRatio) / this.mapOptions.getInteger("model.option.mountainNumber");
        counter = 0;
        block5: for (tries = 0; tries < 1000; ++tries) {
            if (counter >= number) continue;
            Tile t = this.getRandomLandTile(map, this.random);
            if (t == null) {
                return;
            }
            if (t.getType() == hills || t.getType() == mountains) continue;
            for (Tile tile : t.getSurroundingTiles(3)) {
                if (tile.getType() != mountains) continue;
                continue block5;
            }
            for (Tile tile : t.getSurroundingTiles(1)) {
                if (tile.isLand()) continue;
                continue block5;
            }
            t.setType(this.random.nextInt(4) == 0 ? mountains : hills);
            ++counter;
        }
        logger.info("Added " + counter + " random hills tiles.");
    }

    private void createRivers(Map map) {
        Specification spec = map.getSpecification();
        TileImprovementType riverType = spec.getTileImprovementType("model.improvement.river");
        int number = this.getApproximateLandCount() / this.mapOptions.getInteger("model.option.riverNumber");
        int counter = 0;
        HashMap<Map.Position, River> riverMap = new HashMap<Map.Position, River>();
        ArrayList<River> rivers = new ArrayList<River>();
        block0: for (int i = 0; i < number; ++i) {
            block1: for (int tries = 0; tries < 100; ++tries) {
                Tile tile = this.getRandomLandTile(map, this.random);
                if (tile == null) {
                    return;
                }
                if (!tile.getType().canHaveImprovement(riverType)) continue;
                for (Tile neighborTile : tile.getSurroundingTiles(2)) {
                    if (neighborTile.isLand()) continue;
                    continue block1;
                }
                if (riverMap.get(tile.getPosition()) != null) continue;
                ServerRegion riverRegion = new ServerRegion(map.getGame(), "model.region.river" + i, Region.RegionType.RIVER, tile.getRegion());
                riverRegion.setDiscoverable(true);
                riverRegion.setClaimable(true);
                River river = new River(map, riverMap, riverRegion, this.random);
                if (river.flowFromSource(tile.getPosition())) {
                    logger.fine("Created new river with length " + river.getLength());
                    map.putRegion(riverRegion);
                    rivers.add(river);
                    ++counter;
                    continue block0;
                }
                logger.fine("Failed to generate river.");
                continue block0;
            }
        }
        logger.info("Created " + counter + " rivers of maximum " + number);
        for (River river : rivers) {
            ServerRegion region = river.getRegion();
            int scoreValue = 0;
            for (RiverSection section : river.getSections()) {
                scoreValue += section.getSize();
            }
            region.setScoreValue(scoreValue *= 2);
            logger.fine("Created river region (length " + river.getLength() + ", score value " + scoreValue + ").");
        }
    }

    private void createLakeRegions(Map map) {
        Game game = map.getGame();
        TileType lakeType = map.getSpecification().getTileType("model.tile.lake");
        ArrayList<Tile> lakes = new ArrayList<Tile>();
        for (int y = 0; y < map.getHeight(); ++y) {
            for (int x = 0; x < map.getWidth(); ++x) {
                Tile tile;
                if (!map.isValid(x, y) || (tile = map.getTile(x, y)).isLand() || map.getTile(x, y).getRegion() != null) continue;
                lakes.add(tile);
                logger.info("Adding lake at " + x + "," + y);
            }
        }
        int lakeCount = 0;
        while (!lakes.isEmpty()) {
            String id;
            Tile tile = (Tile)lakes.get(0);
            if (tile.getRegion() != null) continue;
            while (game.getFreeColGameObject(id = "model.region.inlandLake" + lakeCount) != null) {
                ++lakeCount;
            }
            ServerRegion lakeRegion = new ServerRegion(game, id, Region.RegionType.LAKE, null);
            this.setGeographicRegion(lakeRegion);
            map.putRegion(lakeRegion);
            lakeRegion.setPrediscovered(false);
            ArrayList<Tile> todo = new ArrayList<Tile>();
            todo.add(tile);
            while (!todo.isEmpty()) {
                Tile t = (Tile)todo.remove(0);
                if (!lakes.contains(t)) continue;
                t.setRegion(lakeRegion);
                t.setType(lakeType);
                lakes.remove(t);
                todo.addAll(t.getSurroundingTiles(1, 1));
            }
        }
    }

    private void perhapsAddBonus(Tile t, boolean generateBonus) {
        Specification spec = t.getSpecification();
        TileImprovementType fishBonusLandType = spec.getTileImprovementType("model.improvement.fishBonusLand");
        TileImprovementType fishBonusRiverType = spec.getTileImprovementType("model.improvement.fishBonusRiver");
        int bonusNumber = this.mapOptions.getInteger("model.option.bonusNumber");
        if (t.isLand()) {
            if (generateBonus && this.random.nextInt(100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        } else {
            int adjacentLand = 0;
            boolean adjacentRiver = false;
            for (Map.Direction direction : Map.Direction.values()) {
                Tile otherTile = t.getNeighbourOrNull(direction);
                if (otherTile == null || !otherTile.isLand()) continue;
                ++adjacentLand;
                if (!otherTile.hasRiver()) continue;
                adjacentRiver = true;
            }
            if (adjacentLand > 2) {
                t.add(new TileImprovement(t.getGame(), t, fishBonusLandType));
            }
            if (!t.hasRiver() && adjacentRiver) {
                t.add(new TileImprovement(t.getGame(), t, fishBonusRiverType));
            }
            if (t.getType().isHighSeasConnected()) {
                if (generateBonus && adjacentLand > 1 && this.random.nextInt(10 - adjacentLand) == 0) {
                    t.addResource(this.createResource(t));
                }
            } else if (this.random.nextInt(100) < bonusNumber) {
                t.addResource(this.createResource(t));
            }
        }
    }

    private Resource createResource(Tile tile) {
        int maxValue;
        if (tile == null) {
            return null;
        }
        ResourceType resourceType = (ResourceType)RandomChoice.getWeightedRandom(null, null, this.random, tile.getType().getWeightedResources());
        if (resourceType == null) {
            return null;
        }
        int minValue = resourceType.getMinValue();
        int quantity = minValue == (maxValue = resourceType.getMaxValue()) ? maxValue : minValue + this.random.nextInt(maxValue - minValue + 1);
        return new Resource(tile.getGame(), tile, resourceType, quantity);
    }

    public static void encodeStyle(Tile tile) {
        Tile t;
        EnumMap<Map.Direction, Boolean> connections = new EnumMap<Map.Direction, Boolean>(Map.Direction.class);
        Map.Direction[] arr$ = Map.Direction.corners;
        int len$ = arr$.length;
        for (int i$ = 0; i$ < len$; ++i$) {
            Map.Direction d;
            t = tile.getNeighbourOrNull(d = arr$[i$]);
            connections.put(d, t != null && t.isLand());
        }
        for (Map.Direction d : Map.Direction.longSides) {
            t = tile.getNeighbourOrNull(d);
            if (t != null && t.isLand()) {
                connections.put(d, true);
                connections.put(d.getNextDirection(), false);
                connections.put(d.getPreviousDirection(), false);
                continue;
            }
            connections.put(d, false);
        }
        int result = 0;
        int index = 0;
        for (Map.Direction d : Map.Direction.corners) {
            if (((Boolean)connections.get((Object)d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        for (Map.Direction d : Map.Direction.longSides) {
            if (((Boolean)connections.get((Object)d)).booleanValue()) {
                result += (int)Math.pow(2.0, index);
            }
            ++index;
        }
        tile.setStyle(result);
    }
}

