// -*- mode: c++; c-set-style: "stroustrup"; tab-width: 4; -*-
//
// CFile.c
//
// Copyright (C) 2004 Koji Nakamaru
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "common.h"
#include <bzlib.h>
#include <lzma.h>
#include <zlib.h>
#include "CFile.h"

// public functions

CFile::CFile()
: _type(kFileNone),
  _fp(NULL),
  _stream(NULL),
  _pos(0)
{
	memset(_name, 0, sizeof(_name));
}

CFile::~CFile()
{
}

bool CFile::open(
	char *name)
{
	if (name == NULL) {
		_fp = stdin;
		_type = kFileStdIn;
	} else {
		FILE *fp;
		if ((fp = fopen(name, "r")) == NULL) {
			return false;
		}
		int c0 = fgetc(fp);
		int c1 = fgetc(fp);
		int c2 = fgetc(fp);
		int c3 = fgetc(fp);
		int c4 = fgetc(fp);
		int c5 = fgetc(fp);
		fclose(fp);
		if (c0 == 'B' && c1 == 'Z') {
			_type = kFileBZ;
			if ((_fp = (void *)BZ2_bzopen(name, "rb")) == NULL) {
				goto err;
			}
		} else if (c0 == 037 && c1 == 0213) {
			_type = kFileGZ;
			if ((_fp = (void *)gzopen(name, "rb")) == NULL) {
				goto err;
			}
		} else if (c0 == 0xFD && c1 == '7' && c2 == 'z' && c3 == 'X' && c4 == 'Z' && c5 == 0x00) {
			_type = kFileXZ;
			if ((_fp = (void *)fopen(name, "r")) == NULL) {
				goto err;
			}
			_stream = new char[sizeof(lzma_stream)];
			memset(_stream, 0, sizeof(lzma_stream));
			if (lzma_stream_decoder((lzma_stream *)_stream, lzma_easy_decoder_memusage(9), 0) != LZMA_OK) {
				goto err;
			}
		} else {
			_type = kFileUZ;
			if ((_fp = (void *)fopen(name, "r")) == NULL) {
				goto err;
			}
		}
		strcpy(_name, name);
	}
	_pos = 0;
	return true;
err:
	close();
	return false;
}

void CFile::close()
{
	_pos = 0;
	if (_stream) {
		lzma_end((lzma_stream *)_stream);
		delete[] _stream;
		_stream = NULL;
	}
	if (_fp) {
		switch (_type) {
		case kFileBZ:
			BZ2_bzclose((BZFILE *)_fp);
			break;
		case kFileGZ:
			gzclose((gzFile)_fp);
			break;
		case kFileXZ:
		case kFileUZ:
			fclose((FILE *)_fp);
			break;
		default:
			break;
		}
		_fp = NULL;
	}
}

bool CFile::stat(
	struct stat *stbuf)
{
	switch (_type) {
	case kFileBZ:
	case kFileGZ:
	case kFileXZ:
	case kFileUZ:
		return ::stat(_name, stbuf) == 0;
	default:
		return false;
	}
}

char *CFile::gets(
	char *buf,
	size_t len)
{
	memset(buf, 0, len--);
	char *bp = buf;
	int c;
	while ((c = getc()) != EOF && bp - buf < (int)len) {
		*bp++ = c;
		if (c == '\n') {
			return buf;
		}
	}
	return (bp > buf) ? buf : NULL;
}

int CFile::getc()
{
	unsigned char buf[1];
	if (read(buf, 1) == 0) {
		return EOF;
	}
	return buf[0];
}

size_t CFile::read(
	void *buf,
	size_t len)
{
	int count = 0;
	switch (_type) {
	case kFileBZ:
		count = BZ2_bzread((BZFILE *)_fp, buf, len);
		break;
	case kFileGZ:
		count = gzread((gzFile)_fp, buf, len);
		break;
	case kFileXZ:
		{
			lzma_stream *stream = (lzma_stream *)_stream;
			stream->next_out = (uint8_t *)buf;
			stream->avail_out = len;
			while (stream->avail_out > 0) {
				if (stream->avail_in == 0) {
					stream->next_in = (uint8_t *)_buf;
					stream->avail_in = fread(_buf, 1, 1, (FILE *)_fp);
				}
				lzma_ret ret = lzma_code(stream, LZMA_RUN);
				if (ret == LZMA_STREAM_END || ret != LZMA_OK) {
					break;
				}
			}
			count = len - stream->avail_out;
		}
		break;
	case kFileStdIn:
	case kFileUZ:
		count = fread(buf, 1, len, (FILE *)_fp);
		break;
	default:
		return 0;
	};
	_pos += count;
	return count;
}

bool CFile::seek(
	size_t pos)
{
	switch (_type) {
	case kFileBZ:
	case kFileGZ:
	case kFileXZ:
		close();
		if (! open(_name)) {
			return false;
		}
		break;
	case kFileStdIn:
		return false;
	case kFileUZ:
	default:
		break;
	}
	switch (_type) {
	case kFileBZ:
	case kFileGZ:
	case kFileXZ:
		while (_pos < pos) {
			char buf[BUFSIZ];
			size_t off = pos - _pos;
			if (read(buf, (off > BUFSIZ) ? BUFSIZ : off) == 0) {
				return false;
			}
		}
		break;
	case kFileUZ:
		if (fseek((FILE *)_fp, pos, SEEK_SET) != 0) {
			return false;
		}
		break;
	default:
		break;
	}
	return false;
}

size_t CFile::tell()
{
	return _pos;
}

// protected functions
// private functions
// local functions
