/**
 * Wizard class for exporting filter reports.
 * 
 * Created 12.04.2008
 * 
 * @author Joanna Muras 
 */
package net.sourceforge.ganttproject.datafilter;

import java.io.File;
import java.util.List;

import javax.swing.SwingUtilities;

import net.sourceforge.ganttproject.GanttOptions;
import net.sourceforge.ganttproject.IGanttProject;
import net.sourceforge.ganttproject.Mediator;
import net.sourceforge.ganttproject.chart.Chart;
import net.sourceforge.ganttproject.export.ExportFinalizationJob;
import net.sourceforge.ganttproject.export.Exporter;
import net.sourceforge.ganttproject.export.FileChooserPage;
import net.sourceforge.ganttproject.export.State;
import net.sourceforge.ganttproject.export.WebPublisher;
import net.sourceforge.ganttproject.export.Stylesheet.StylesheetType;
import net.sourceforge.ganttproject.gui.UIFacade;
import net.sourceforge.ganttproject.gui.projectwizard.WizardImpl;
import net.sourceforge.ganttproject.language.GanttLanguage;

public class ExportFilterWizard extends WizardImpl {

    private UIFacade myUIFacade;

    private IGanttProject myProject;

    private Chart myGanttChart;

    private Chart myResourceChart;

    private Chart myVisibleChart;

    private GanttOptions myOptions;

    private State myState;

    private static Exporter ourLastSelectedExporter;
    private static Exporter[] ourExporters;

    public ExportFilterWizard(UIFacade uiFacade, IGanttProject project,
            GanttOptions options, List tasks) {
        this(uiFacade, project, uiFacade.getGanttChart(),
                uiFacade.getResourceChart(), uiFacade.getActiveChart(), options, tasks);
    }

    public ExportFilterWizard(UIFacade uiFacade,
            IGanttProject project, Chart ganttChart, Chart resourceChart,
            Chart visibleChart, GanttOptions options, List tasks) {
        super(uiFacade, GanttLanguage.getInstance().getText("datafilter.exportWizard.dialog.title"));
        myUIFacade = uiFacade;
        myProject = project;
        myGanttChart = ganttChart;
        myResourceChart = resourceChart;
        myVisibleChart = visibleChart;
        myOptions = options;
        myState = new StateImpl();
        if (ourExporters==null) {
            ourExporters = Mediator.getPluginManager().getExporters();
        }

        /* Choose pdf exporter and style sheets for filter export */
        for (Exporter exporter: ourExporters) {
            if (exporter.getFileNamePattern().equals("pdf")) {
                myState.setExporter(exporter);
                exporter.setContext(project, uiFacade, StylesheetType.filter);
                if (exporter instanceof LegacyOptionsClient) {
                    ((LegacyOptionsClient)exporter).setOptions(myOptions);
                }
                myState.setTaskContext(tasks);	
                addPage(new FileChooserPage(myState, myProject));
            }
        }
    }

    @Override
    protected boolean canFinish() {
        return myState.getExporter() != null && myState.getFile() != null;
    }

    @Override
    protected void onOkPressed() {
        super.onOkPressed();
        SwingUtilities.invokeLater(new Runnable(){
            public void run() {
                try {
                    ExportFinalizationJob finalizationJob = new ExportFinalizationJobImpl();
                    //System.out.println("ExportFilterWizard myState.getTaskContext().size " + myState.getTaskContext().size());
                    myState.getExporter().run(myState.getFile(), myState.getTaskContext(), finalizationJob);
                } catch (Exception e) {
                    getUIFacade().showErrorDialog(e);
                }
            }
        });
    }

    private class ExportFinalizationJobImpl implements ExportFinalizationJob {
        public void run(File[] exportedFiles) {
            if (myState.getPublishInWebOption().isChecked()) {
                WebPublisher publisher = new WebPublisher();
                publisher.run(exportedFiles, myProject.getDocumentManager().getFTPOptions(), getUIFacade());
            }
        }
    }

    private GanttLanguage getLanguage() {
        return GanttLanguage.getInstance();
    }

    private void setStatusText(String text) {
        getUIFacade().setStatusText(text);
    }

    class StateImpl extends State {

        @Override
        public void setFile(File file) {
            myFile = file;
            ExportFilterWizard.this.adjustButtonState();
        }

        @Override
        public void setExporter(Exporter exporter) {
            myExporter = exporter;
            ExportFilterWizard.ourLastSelectedExporter = exporter;
        }

    }

    interface LegacyOptionsClient {
        void setOptions(GanttOptions options);
    }
}
