/**
 \brief
  This module implements the gnocl::??commands module.
 \authors
  Peter G. Baum, William J Giddings
 \date 2001-03:
*/

/* user documentation
 * split each description into its own file..
 */

/** \page page12 gnocl::??command??
  \section sec DESCRIPTION
  Implementation of...
  \subsection subsection1 Implemented XXX Options
  \subsection subsection2 Implemented XXX Commands
  \verbatim
    gnocl::winfo parent id
        Returns the id of the widget parent.

    gnocl::winfo toplevel id
        Returns the toplevel container for widget. This will be non-window
        if the container has not yet been embedded into a GtkWindow.

    gnocl::winfo geometry id
        Returns the position and dimensions of the widget in its containing
  	    window as a list in the form {xwin ywin width height xroot yroot}.
        Where:
            xwin,ywin      positionof the widget in its toplevel window
            width, height  displayed width and height of th widget
            xroot, yroot   position of the widget on the displayed screen

    gnocl::winfo style
        Returns a list of style parameters and their settings as applied to
        the widget.
 \endverbatim

\subsection subsection3 Sample Tcl Script
 \code
   set box [gnocl::box -buttonType 1]
   $box add [gnocl::button -text "%_Do_n't save" -icon "%#No"]
   $box add [gnocl::button -text "%#Cancel"]
   $box add [gnocl::button -text "%#Save"]
   gnocl::window -title "Buttons" -child $box
 \endcode

*/

/****h* gnocl/command
 * NAME
 * commands.c
 * DESCRIPTION
 * This file implements a Tcl interface to GTK+ and Gnome
 * AUTHOR
 *  Peter G. Baum
 *	William J Giddings
 * CREATION DATE
 * PURPOSE
 * USAGE
 * PARAMETERS
 * COMMANDS
 * 	winfo parent id
 *    Returns the id of the widget parent.
 * 	winfo toplevel id
 *    Returns the toplevel container for widget. This will be non-window
 *    if the container has not yet been embedded into a GtkWindow.
 *  winfo geometry id
 *    Returns the position and dimensions of the widget in its containing
 * 	  window as a list in the form {xwin ywin width height xroot yroot}.
 *    Where:
 *      xwin,ywin      positionof the widget in its toplevel window
 *      width, height  displayed width and height of th widget
 *      xroot, yroot   position of the widget on the displayed screen
 *  winfo style
 *    Returns a list of style parameters and their setting as applied to
 *    the widget.
 * OPTIONS
 * EXAMPLE
 * FUNCTION
 * NOTES
 * BUGS
 * SEE ALSO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 *  2009-03: added gnoclWinfoCmd
 *  2009-02: added gnoclResourceCmd
 *  2003-03: split from gnocl.c
 * TODO
 * SOURCE
 *****

/*
 * $Id: commands.c,v 1.9 2005/02/22 23:16:10 baum Exp $
 *
 * This file implements a Tcl interface to GTK+ and Gnome
 *
 * Copyright (c) 2001 - 2005 Peter G. Baum  http://www.dr-baum.net
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

/*
   History:
   2009-02: added gnoclResourceCmd
   2003-03: split from gnocl.c
*/

#include "gnocl.h"
#include <string.h>
#include <ctype.h>
#include <glade/glade.h>



/****f* command/gnoclGladeXMLCmd
 * NAME
 *	gnoclWinfoCmd
 * PURPOSE
 *  Load a Glade file and create user interface.
 *  Register widget, and create variables holding names based upon id given in the glade file.
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 *	02-Mar-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * 	There may be some overlap with features already implemented, e.g. class.
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclGladeXMLCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	static const char *cmd[] = { "glade", "B", NULL };
	enum optIdx { GladeIdx, ToplevelIdx, GeometryIdx, StyleIdx };
	int idx;



	if ( objc != 3 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "option widgetid " );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmd, "glade", TCL_EXACT, &idx ) != TCL_OK )
	{
		return TCL_ERROR;
	}

	g_print ( "objc = %d opt = %s  id = %s\n", objc, Tcl_GetString ( objv[1] ),  Tcl_GetString ( objv[2] ) );

	GtkWidget *window;
	GladeXML *xml;

	xml = glade_xml_new ( Tcl_GetString ( objv[2] ), NULL, NULL );
	window = glade_xml_get_widget ( xml, "window" );

	/* perhaps there is some way of attaching signals handlers
	 * the following function is clearly parsing the same file
	 * that created the widgets..
	 * */
	//glade_xml_signal_autoconnect (xml);

	gtk_widget_show_all ( window );
	/* gtk_main (); */

	if ( 1 )
	{
		return gnoclGladeWindow ( interp, window ) ;
	}

	else
	{
		return TCL_OK;
	}
}



/****f* command/gnoclWinfoCmd
 * NAME
 *	gnoclWinfoCmd
 * PURPOSE
 *  winfo - Return window/widget-related information
 * 	The inclusion of this function to bring the functions into one module
 *  rather than replicating code as commands to specific widgets.
 *  Certain items, if more convieniently handled, have been placed as options,
 *  e.g. class, to return the GtkWidget type for specific objects.
 * AUTHOR
 *	William J Giddings
 * CREATION DATE
 *	01-Mar-09
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * 	There may be some overlap with features already implemented, e.g. class.
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclWinfoCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	static const char *cmd[] = { "parent", "toplevel", "geometry", "style", NULL };
	enum optIdx { ParentIdx, ToplevelIdx, GeometryIdx, StyleIdx };
	int idx;

	if ( objc != 3 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "option widgetid " );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmd, "option", TCL_EXACT, &idx ) != TCL_OK )
	{
		return TCL_ERROR;
	}

	g_print ( "opt = %s  id = %s\n",  Tcl_GetString ( objv[1] ),  Tcl_GetString ( objv[2] ) );

	switch ( idx )
	{
		case ParentIdx:
			{
				char buffer[128];
				GtkWidget *widget = gnoclGetWidgetFromName (  Tcl_GetString ( objv[2] ), interp );
				GtkWidget *parent = gtk_widget_get_parent ( widget );

				sprintf ( buffer, "%s", gnoclGetNameFromWidget ( parent ) );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( buffer, -1 ) );
			}

			break;
		case ToplevelIdx:
			{
				char buffer[128];
				GtkWidget *widget = gnoclGetWidgetFromName (  Tcl_GetString ( objv[2] ), interp );
				GtkWidget *toplevel = gtk_widget_get_toplevel ( widget );
				sprintf ( buffer, "%s", gnoclGetNameFromWidget ( toplevel ) );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( buffer, -1 ) );
			}

			break;
			/* probably best leave this to the widget commands themselves */
		case GeometryIdx:
			{
				char buffer[128];
				sprintf ( buffer, "%s", "Geometry" );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( buffer, -1 ) );
			}

			break;
		case StyleIdx:
			{
				char buffer[128];
				sprintf ( buffer, "%s", "Style" );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( buffer, -1 ) );
			}

			break;
	}

	return TCL_OK;
}

/*****/

/****f* command/gnoclInfoCmd
 * NAME
 *	fuctionName
 * PURPOSE
 *  info -- Retrieve different information.
 * AUTHOR
 *	Peter G. Baum
 *	William J Giddings
 * CREATION DATE
 *	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclInfoCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	static const char *cmd[] = { "version", "gtkVersion",
								 "hasGnomeSupport", "allStockItems", "breakpoint", NULL
							   };
	enum optIdx { VersionIdx, GtkVersionIdx,
				  HasGnomeIdx, AllStockItems, BreakpointIdx
				};
	int idx;

	if ( objc != 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "option" );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmd, "option", TCL_EXACT,
							   &idx ) != TCL_OK )
		return TCL_ERROR;

	switch ( idx )
	{
		case VersionIdx:
			Tcl_SetObjResult ( interp, Tcl_NewStringObj ( VERSION, -1 ) );
			break;
		case GtkVersionIdx:
			{
				char buffer[128];
				sprintf ( buffer, "%d.%d.%d", gtk_major_version,
						  gtk_minor_version, gtk_micro_version );
				Tcl_SetObjResult ( interp, Tcl_NewStringObj ( buffer, -1 ) );
			}

			break;
		case HasGnomeIdx:
			Tcl_SetObjResult ( interp, Tcl_NewBooleanObj (
#ifdef GNOCL_USE_GNOME
								   1
#else
								   0
#endif
							   ) );
			break;
		case AllStockItems:
			{
				Tcl_Obj *res = Tcl_NewListObj ( 0, NULL );
				GSList *ids = gtk_stock_list_ids();
				GSList *p;

				for ( p = ids; p != NULL; p = p->next )
				{
					char *txt = p->data;
					int skip = 0;
					/* FIXME: gtk-missing-image, gtk-dnd-multiple and gtk-dnd
					          fail lookup, why?
					{
					   GtkStockItem sp;
					   printf( "%s lookup: %d\n", txt,
					         gtk_stock_lookup( txt, &sp ) );
					}
					*/

					/* see createStockName and gnoclGetStockName */

					if ( strncmp ( txt, "gtk", 3 ) == 0 )
						skip = 3;

#ifdef GNOCL_USE_GNOME
					else if ( strncmp ( txt, "gnome-stock", 11 ) == 0 )
						skip = 11;

#endif
					if ( skip > 0 )
					{
						GString *name = g_string_new ( NULL );
						char *tp = txt + skip;

						for ( ; *tp; ++tp )
						{
							if ( *tp == '-' )
							{
								++tp;
								g_string_append_c ( name, toupper ( *tp ) );
							}

							else
								g_string_append_c ( name, *tp );
						}

						Tcl_ListObjAppendElement ( interp, res,

												   Tcl_NewStringObj ( name->str, -1 ) );
						/* printf( "%s -> %s\n", (char *)p->data, name->str ); */
						g_string_free ( name, 1 );
					}

					else
						Tcl_ListObjAppendElement ( interp, res,
												   Tcl_NewStringObj ( txt, -1 ) );

					g_free ( p->data );
				}

				g_slist_free ( ids );

				Tcl_SetObjResult ( interp, res );
			}

			break;
		case BreakpointIdx:
			/* this is only for debugging */
			G_BREAKPOINT();
			break;
	}

	return TCL_OK;
}

/*****/


/****f* command/gnoclUpdateCmd
 * NAME
 *	fuctionName
 * PURPOSE
 * AUTHOR
 *	Peter G. Baum
 *	William J Giddings
 * CREATION DATE
 *	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclUpdateCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	int nMax = 500;
	int n;

	if ( objc != 1 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, NULL );
		return TCL_ERROR;
	}

	for ( n = 0; n < nMax && gtk_events_pending(); ++n )
		gtk_main_iteration_do ( 0 );

	Tcl_SetObjResult ( interp, Tcl_NewIntObj ( n ) );

	return TCL_OK;
}

/*****/


/****f* command/gnoclResourceFileCmd
 * NAME
 *	gnoclResourceFileCmd
 * PURPOSE
 * AUTHOR
 *	Peter G. Baum
 * CREATION DATE
 *	25-FEB-2009
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclResourceFileCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	g_print ( "gnoclResourceFileCmd\n" ) ;

	int nMax = 500;
	int n;

	if ( objc != 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, NULL );
		return TCL_ERROR;
	}

	char * str = Tcl_GetString ( objv[1] );

	gtk_rc_parse ( str );

	Tcl_SetObjResult ( interp, Tcl_NewIntObj ( n ) );

	return TCL_OK;
}

/*****/

/****f* command/gnoclConfigureCmd
 * NAME
 *	fuctionName
 * PURPOSE
 * AUTHOR
 *	Peter G. Baum
 *	William J Giddings
 * CREATION DATE
 *	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclConfigureCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	GnoclOption options[] =
	{
		{ "-tooltip", GNOCL_BOOL, NULL },
		{ "-defaultIcon", GNOCL_OBJ, NULL },
		{ NULL }
	};
	const int tooltipIdx     = 0;
	const int defaultIconIdx = 1;

	int ret = TCL_ERROR;

	if ( gnoclParseOptions ( interp, objc, objv, options ) != TCL_OK )
		goto cleanExit;

	if ( options[defaultIconIdx].status == GNOCL_STATUS_CHANGED )
	{
		GnoclStringType type = gnoclGetStringType (
								   options[defaultIconIdx].val.obj );

		switch ( type )
		{
			case GNOCL_STR_EMPTY:
				gtk_window_set_default_icon_list ( NULL );
				break;
			case GNOCL_STR_FILE:
				{
					GdkPixbuf *pix = gnoclPixbufFromObj ( interp,
														  options + defaultIconIdx );
					GList *list = NULL;

					if ( pix == NULL )
						goto cleanExit;

					list = g_list_append ( list, pix );

					gtk_window_set_default_icon_list ( list );

					g_list_free ( list );
				}

				break;
			default:
				Tcl_AppendResult ( interp, "Unknown type for \"",
								   Tcl_GetString ( options[defaultIconIdx].val.obj ),
								   "\" must be of type FILE (%/) or empty", NULL );
				goto cleanExit;
		}
	}

	if ( options[tooltipIdx].status == GNOCL_STATUS_CHANGED )
	{
		if ( options[tooltipIdx].val.b )
			gtk_tooltips_enable ( gnoclGetTooltips() );
		else
			gtk_tooltips_disable ( gnoclGetTooltips() );
	}

	ret = TCL_OK;

cleanExit:
	gnoclClearOptions ( options );
	return ret;
}

/*****/

/****f* command/gnoclClipboardCmd
 * NAME
 *	fuctionName
 * PURPOSE
 * AUTHOR
 *	Peter G. Baum
 *	William J Giddings
 * CREATION DATE
 *	When created
 * USAGE
 *	how this function is used
 * ARGUMENTS
 * RETURN VALUE
 * NOTES
 * TODO
 * USES
 * USED BY
 * MODIFICATION HISTORY
 * SOURCE
 */
int gnoclClipboardCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj * const objv[] )
{
	GnoclOption options[] =
	{
		{ "-primary", GNOCL_BOOL, NULL },
		{ NULL }
	};
	const int usePrimaryIdx = 0;

	static const char *cmd[] = { "hasText", "setText", "getText", "clear",
								 NULL
							   };
	enum optIdx { HasTextIdx, SetTextIdx, GetTextIdx, ClearIdx };
	int idx;
	int optNum;
	GtkClipboard *clip;
	int usePrimary = 0;

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "option" );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmd, "option", TCL_EXACT,
							   &idx ) != TCL_OK )
		return TCL_ERROR;

	if ( idx == SetTextIdx )
	{
		optNum = 2;

		if ( objc < 3 )
		{
			Tcl_WrongNumArgs ( interp, 1, objv, "text ?option value?" );
			return TCL_ERROR;
		}
	}

	else
	{
		optNum = 1;

		if ( objc < 2 )
		{
			Tcl_WrongNumArgs ( interp, 1, objv, NULL );
			return TCL_ERROR;
		}
	}

	if ( gnoclParseOptions ( interp, objc - optNum, objv + optNum, options )
			!= TCL_OK )
		return TCL_ERROR;

	if ( options[usePrimaryIdx].status == GNOCL_STATUS_CHANGED )
		usePrimary = options[usePrimaryIdx].val.b;

	clip = gtk_clipboard_get ( usePrimary ? gdk_atom_intern ( "PRIMARY", 1 )
							   : GDK_NONE );

	switch ( idx )
	{
		case HasTextIdx:
			{
				int ret = gtk_clipboard_wait_is_text_available ( clip );
				Tcl_SetObjResult ( interp, Tcl_NewBooleanObj ( ret ) );
			}

			break;
		case SetTextIdx:
			gtk_clipboard_set_text ( clip, Tcl_GetString ( objv[2] ), -1 );
			break;
		case GetTextIdx:
			{
				char *txt = gtk_clipboard_wait_for_text ( clip );

				if ( txt )
				{
					Tcl_SetObjResult ( interp, Tcl_NewStringObj ( txt, -1 ) );
					g_free ( txt );
				}

				/* FIXME? else error? */
			}

			break;
		case ClearIdx:
			gtk_clipboard_clear ( clip );
			break;
	}

	return TCL_OK;
}

/*****/
