/**
 \brief
  This module implements the gnocl::image widget.
 \authors
  Peter G. Baum, William J Giddings
 \date 2001-03:
*/

/* user documentation */

/**
\page page23 gnocl::image
\section sec gnocl::image
  Implementation of gnocl::image
\subsection subsection1 Implemented gnocl::image Options
    \verbinclude image_options.txt
\subsection subsection2 Implemented gnocl::image Commands
    \verbinclude image_commands.txt
\subsection subsection3 Sample Tcl Script
    \include image_example.tcl
\subsection subsection4 Produces
    \image html "image.png"
    \example image_example.tcl
    \image html "image.png"
*/

/****h* widget/image
* NAME
* DESCRIPTION
* 	This file contains the code to produce the gnocl::image widget and
* 	all other functions associated with image manipulation.
* AUTHOR
*	Peter G. Baum
*	William J Giddings
* CREATION DATE
* 	2003-02
* PURPOSE
* USAGE
* PARAMETERS
*
* COMMANDS
*  id delete
*    Deletes the widget and the associated tcl command.
*  id configure [-option value...]
*    Configures the widget. Option may have any of the values accepted on creation of the widget.
*
* OPTIONS

*
* EXAMPLE

*
* |html <B>PRODUCES</B><P>
* |html <image src="../pics/image.png">
*
* FUNCTION
* NOTES
* BUGS
* SEE ALSO
* USES
* USED BY
* MODIFICATION HISTORY
*  2008-10: added command, class
*       04: renamed -size to -stockSize, added -size
*  2003-02: Begin of developement
* TODO
* SOURCE
 *****
 */

/*
* $Id: image.c,v 1.9 2005/08/16 20:57:45 baum Exp $
 *
* This file implements the image widget
 *
* Copyright (c) 2001 - 2005 Peter G. Baum  http://www.dr-baum.net
 *
* See the file "license.terms" for information on usage and redistribution
* of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

#include "gnocl.h"
#include <string.h>
#include <assert.h>

/****v* image/imageOptions
* NAME
*  fuctionName
* PURPOSE
* AUTHOR
*  Peter G. Baum
*  William J Giddings
* CREATION DATE
*  When created
* USAGE
*  how this function is used
* ARGUMENTS
* RETURN VALUE
* NOTES
* TODO
* USES
* USED BY
* MODIFICATION HISTORY
* SOURCE
 */
static GnoclOption imageOptions[] =
{
	{ "-image", GNOCL_OBJ, NULL },
	{ "-stockSize", GNOCL_OBJ, NULL },
	{ "-size", GNOCL_OBJ, NULL },
	{ "-align", GNOCL_OBJ, "?align", gnoclOptBothAlign },
	{ "-xPad", GNOCL_OBJ, "xpad", gnoclOptPadding },
	{ "-yPad", GNOCL_OBJ, "ypad", gnoclOptPadding },
	{ "-name", GNOCL_STRING, "name" },
	{ "-visible", GNOCL_BOOL, "visible" },
	{ "-sensitive", GNOCL_BOOL, "sensitive" },
	{ NULL }
};
/*****/

/****v* image/statics
* NAME
*  fuctionName
* PURPOSE
* AUTHOR
*  Peter G. Baum
* CREATION DATE
*  When created
* USAGE
*  how this function is used
* ARGUMENTS
* RETURN VALUE
* NOTES
* TODO
* USES
* USED BY
* MODIFICATION HISTORY
* SOURCE
 */
static const int imageIdx = 0;
static const int stockSizeIdx  = 1;
static const int sizeIdx = 2;
/*****/

/****f* image/staticFuncs/getIconSize
* NAME
*  fuctionName
* PURPOSE
* AUTHOR
*  Peter G. Baum
* CREATION DATE
*  When created
* USAGE
*  how this function is used
* ARGUMENTS
* RETURN VALUE
* NOTES
* TODO
* USES
* USED BY
* MODIFICATION HISTORY
* SOURCE
 */
static int getIconSize (
	Tcl_Interp *interp,
	Tcl_Obj *obj,
	GtkIconSize *size )
{
	const char *txt[] = { "menu", "smallToolBar",
						  "largeToolBar", "button", "dnd", "dialog", NULL
						};
	GtkIconSize modes[] = { GTK_ICON_SIZE_MENU, GTK_ICON_SIZE_SMALL_TOOLBAR,
							GTK_ICON_SIZE_LARGE_TOOLBAR, GTK_ICON_SIZE_BUTTON,
							GTK_ICON_SIZE_DND, GTK_ICON_SIZE_DIALOG
						  };

	int idx;

	if ( Tcl_GetIndexFromObj ( interp, obj, txt, "icon size",
							   TCL_EXACT, &idx ) != TCL_OK )
		return TCL_ERROR;

	*size = modes[idx];

	return TCL_OK;
}

/*****/

/****f* image/staticFuncs/configure
* NAME
*  fuctionName
* PURPOSE
* AUTHOR
*  Peter G. Baum
* 	William J Giddings
* CREATION DATE
*  When created
* USAGE
*  how this function is used
* ARGUMENTS
* RETURN VALUE
* NOTES
* TODO
* USES
* USED BY
* MODIFICATION HISTORY
* SOURCE
 */
static int configure (
	Tcl_Interp *interp,
	GtkImage *image,
	GnoclOption options[] )
{
	/*
	   TODO:
	   GdkPixbuf*  gdk_pixbuf_new_from_data        (const guchar *data,
	                                             GDK_COLORSPACE_RGB,
	                                             gboolean has_alpha,
	                                             int bits_per_sample,
	                                             int width,
	                                             int height,
	                                             int rowstride,
	                                             GdkPixbufDestroyNotify destroy_fn,
	                                             gpointer destroy_fn_data);
	         %m25x10x8
	         %i-RGB-25x10x10-
	         %i-RGBA-25x10x16

	GdkPixbuf*  gdk_pixbuf_new_from_xpm_data    (const char **data);


	*/
	if ( options[imageIdx].status == GNOCL_STATUS_CHANGED )
	{
		GnoclStringType type = gnoclGetStringType ( options[imageIdx].val.obj );

		switch ( type & ( GNOCL_STR_FILE | GNOCL_STR_STOCK ) )
		{
			case GNOCL_STR_FILE:
				{
					GError *error = NULL;
					char *txt = gnoclGetString ( options[imageIdx].val.obj );
					GdkPixbufAnimation *ani = gdk_pixbuf_animation_new_from_file (
												  txt, &error );

					if ( ani == NULL )
					{
						Tcl_SetResult ( interp, error->message, TCL_VOLATILE );
						g_error_free ( error );
						return TCL_ERROR;
					}

					if ( gdk_pixbuf_animation_is_static_image ( ani ) )
					{
						GdkPixbuf *pix = gdk_pixbuf_animation_get_static_image ( ani );
						gtk_image_set_from_pixbuf ( image, pix );
					}

					else
						gtk_image_set_from_animation ( image, ani );

					g_object_unref ( ani );
				}

				break;
			case GNOCL_STR_STOCK:
				{
					GtkIconSize size = GTK_ICON_SIZE_BUTTON;
					GtkStockItem item;

					if ( gnoclGetStockItem ( options[imageIdx].val.obj, interp,
											 &item ) != TCL_OK )
						return TCL_ERROR;

					if ( options[stockSizeIdx].status == GNOCL_STATUS_CHANGED )
					{
						if ( getIconSize ( interp, options[stockSizeIdx].val.obj,
										   &size ) != TCL_OK )
						{
							return TCL_ERROR;
						}
					}

					else if ( gtk_image_get_storage_type ( image ) == GTK_IMAGE_STOCK )
						gtk_image_get_stock ( image, NULL, &size );

					gtk_image_set_from_stock ( image, item.stock_id, size );
				}

				break;
			default:
				Tcl_AppendResult ( interp, "Unknown type for \"",
								   Tcl_GetString ( options[imageIdx].val.obj ),
								   "\" must be of type FILE (%/) or STOCK (%#)", NULL );
				return TCL_ERROR;
		}
	}

	else if ( options[stockSizeIdx].status == GNOCL_STATUS_CHANGED )
	{
		char        *id;
		GtkIconSize size;

		if ( gtk_image_get_storage_type ( image ) != GTK_IMAGE_STOCK )
		{
			Tcl_SetResult ( interp, "Size can only be changed for stock images.",
							TCL_STATIC );
			return TCL_ERROR;
		}

		gtk_image_get_stock ( image, &id, &size );

		if ( getIconSize ( interp, options[stockSizeIdx].val.obj, &size )
				!= TCL_OK )
			return TCL_ERROR;

		gtk_image_set_from_stock ( image, id, size );
	}

	if ( options[sizeIdx].status == GNOCL_STATUS_CHANGED )
	{
		GdkPixbuf *src, *dest;
		int       width, height;

		if ( gtk_image_get_storage_type ( image ) != GTK_IMAGE_PIXBUF )
		{
			Tcl_SetResult ( interp, "Only pixbuf images can be sized.",
							TCL_STATIC );
			return TCL_ERROR;
		}

		if ( gnoclGet2Int ( interp, options[sizeIdx].val.obj,
							&width, &height ) != TCL_OK )
			return TCL_ERROR;

		if ( width <= 0  || height <= 0 )
		{
			Tcl_SetResult ( interp, "Size must be greater zero.", TCL_STATIC );
			return TCL_ERROR;
		}

		src = gtk_image_get_pixbuf ( image );

		dest = gdk_pixbuf_scale_simple ( src, width, height,
										 GDK_INTERP_BILINEAR );

		if ( dest == NULL )
		{
			Tcl_SetResult ( interp, "Error in scaling. Not enough memory?",
							TCL_STATIC );
			return TCL_ERROR;
		}

		gtk_image_set_from_pixbuf ( image, dest );

		g_object_unref ( dest );
	}

	return TCL_OK;
}

/*****/

/****f* image/staticFuncs/imageFunc
* NAME
*  imageFunc
* PURPOSE
* AUTHOR
*  Peter G. Baum
* CREATION DATE
*  When created
* USAGE
*  how this function is used
* ARGUMENTS
* RETURN VALUE
* NOTES
* TODO
* USES
* USED BY
* MODIFICATION HISTORY
* SOURCE
 */
static int imageFunc (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj* const objv[] )
{

	static const char *cmds[] = { "delete", "configure", "class", NULL };
	enum cmdIdx { DeleteIdx, ConfigureIdx, ClassIdx };
	int idx;
	GtkImage *image = ( GtkImage* ) data;

	if ( objc < 2 )
	{
		Tcl_WrongNumArgs ( interp, 1, objv, "command" );
		return TCL_ERROR;
	}

	if ( Tcl_GetIndexFromObj ( interp, objv[1], cmds, "command", TCL_EXACT, &idx ) != TCL_OK )
	{
		return TCL_ERROR;
	}

	switch ( idx )
	{
		case ClassIdx:
			Tcl_SetObjResult ( interp, Tcl_NewStringObj ( "image", -1 ) );
			break;
		case DeleteIdx:
			return gnoclDelete ( interp, GTK_WIDGET ( image ), objc, objv );

		case ConfigureIdx:
			{
				int ret = TCL_ERROR;

				if ( gnoclParseAndSetOptions ( interp, objc - 1, objv + 1,
											   imageOptions, G_OBJECT ( image ) ) == TCL_OK )
				{
					ret = configure ( interp, image, imageOptions );
				}

				gnoclClearOptions ( imageOptions );

				return ret;
			}

			break;
	}

	return TCL_OK;
}

/*****/

/****f* image/gnoclImageCmd
* NAME
*  gnoclImageCmd
* PURPOSE
* AUTHOR
*  Peter G. Baum
* CREATION DATE
*  When created
* USAGE
*  how this function is used
* ARGUMENTS
* RETURN VALUE
* NOTES
* TODO
* USES
* USED BY
* MODIFICATION HISTORY
* SOURCE
 */
int gnoclImageCmd (
	ClientData data,
	Tcl_Interp *interp,
	int objc,
	Tcl_Obj* const objv[] )
{
	GtkImage *image;
	int      ret;

	if ( gnoclParseOptions ( interp, objc, objv, imageOptions )
			!= TCL_OK )
	{
		gnoclClearOptions ( imageOptions );
		return TCL_ERROR;
	}

	image = GTK_IMAGE ( gtk_image_new( ) );

	ret = gnoclSetOptions ( interp, imageOptions, G_OBJECT ( image ), -1 );

	if ( ret == TCL_OK )
	{
		ret = configure ( interp, image, imageOptions );
	}

	gnoclClearOptions ( imageOptions );

	if ( ret != TCL_OK )
	{
		gtk_widget_destroy ( GTK_WIDGET ( image ) );
		return TCL_ERROR;
	}

	gtk_widget_show ( GTK_WIDGET ( image ) );

	return gnoclRegisterWidget ( interp, GTK_WIDGET ( image ), imageFunc );
}

/*****/
